/*
 * UPnP Browser for Maemo
 *
 * helper.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license with patent exclusion,
 * see file COPYING.
 */

#ifdef MAEMO
#include <hildon-fm/hildon-widgets/hildon-file-chooser-dialog.h>
#include <hildon-widgets/gtk-infoprint.h>
#endif

#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <curl/curl.h>
#include <unistd.h>
#include <string.h>

#include <cybergarage/http/chttp.h>
#include <cybergarage/net/curl.h>
#include <cybergarage/net/curi.h>
#include <cybergarage/util/cstring.h>

#include "helper.h"
#include "interface.h"
#include "browser.h"
#include "resource_selection.h"

extern UPnPBrowserWidgets* widgets;

/**
 * Struct that is used to pass the uri and FILE* for
 * the file downloader
 */
typedef struct _DownloadFile
{
	gchar* uri;
	FILE* file;
} DownloadFile;

/**
 * Extract host from the given absolute URI.
 *
 * @param uri The absolute URI to extract from
 */
gchar* extract_host_from_uri(gchar* uri)
{
	gchar* host = NULL;
	
	int head = 0;
	int tail = 0;
	
	g_return_val_if_fail(uri != NULL, FALSE);
	
	/* Start of address "://..." */
	head = cg_strstr(uri, CG_NET_URI_PROTOCOL_DELIM);
	head = head + strlen(CG_NET_URI_PROTOCOL_DELIM);
	
	/* End of address "...:" */
	tail = cg_strstr(uri + head, CG_NET_URI_COLON_DELIM);
	tail = tail + head;
	
	if (head != -1 && tail != -1)
	{
		host = g_new0(gchar, tail - head + 1);
		strncpy(host, uri + head, tail - head);
	}
	else
	{
		host = NULL;
	}

	return host;
}

/**
 * Extract port from the given absolute URI.
 *
 * @param uri The absolute URI to extract from
 */
gchar* extract_port_from_uri(gchar* uri)
{
	gchar* port = NULL;
	
	int head = 0;
	int tail = 0;
	
	g_return_val_if_fail(uri != NULL, FALSE);

	/* Skip "://..." */
	head = cg_strstr(uri, CG_NET_URI_PROTOCOL_DELIM);
	head = head + strlen(CG_NET_URI_PROTOCOL_DELIM);
	
	/* Start of port ":" */
	tail = cg_strstr(uri + head, CG_NET_URI_COLON_DELIM);
	head = tail + head + strlen(CG_NET_URI_COLON_DELIM);

	/* End of port "/" */
	tail = cg_strstr(uri + head, CG_NET_URI_SLASH_DELIM);
	tail = tail + head;
	
	if (head != -1 && tail != -1)
	{
		port = g_new0(gchar, tail - head + 1);
		strncpy(port, uri + head, tail - head);
	}
	else
	{
		port = NULL;
	}
	
	return port;
}

/**
 * Extract the relative URI from the given absolute URI.
 *
 * @param uri The absolute URI to extract from
 */
gchar* extract_rel_uri_from_uri(gchar* uri)
{
	gchar* rel_uri = NULL;
	
	int head = 0;
	int tail = 0;
	
	g_return_val_if_fail(uri != NULL, FALSE);

	/* Skip "://..." */
	head = cg_strstr(uri, CG_NET_URI_PROTOCOL_DELIM);
	head = head + strlen(CG_NET_URI_PROTOCOL_DELIM);
	
	/* Start of URI "/" */
	tail = cg_strstr(uri + head, CG_NET_URI_SLASH_DELIM);
	head = tail + head;

	/* All the way to the end */
	tail = strlen(uri);

	if (head != -1 && tail != -1)
	{
		rel_uri = g_new0(gchar, tail - head + 1);
		strncpy(rel_uri, uri + head, tail - head);
	}
	else
	{
		rel_uri = NULL;
	}
	
	return rel_uri;
}

/**
 * Extract the file name from the given absolute URI.
 *
 * @param uri The absolute URI to extract from
 */
gchar* extract_filename_from_uri(gchar* uri)
{
	gchar* name = NULL;
	
	int head = 0;
	int tail = 0;
	
	g_return_val_if_fail(uri != NULL, FALSE);

	for (head = strlen(uri) - 1; head >= 0; head--)
	{
		if (uri[head] == '/')
		{
			head++; /* Don't take the slash */
			break;
		}
	}
	
	tail = strlen(uri);
	
	if (head != -1 && tail != -1)
	{
		name = g_new0(gchar, tail - head + 1);
		strncpy(name, uri + head, tail - head);
	}
	else
	{
		name = NULL;
	}
	
	return name;
}

/**
 * File downloader function
 */
gboolean download_idle(gpointer user_data)
{
	CURL* curl = NULL;
	DownloadFile* df = (DownloadFile*) user_data;
	
	curl = curl_easy_init();
	if (!curl)
	{
		fclose(df->file);
		free(df->uri);
		free(df);
		
		return FALSE;
	}
		
	curl_easy_setopt(curl, CURLOPT_URL, df->uri);
	curl_easy_setopt(curl, CURLOPT_WRITEDATA, df->file);
	curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);
	
	if (curl_easy_perform(curl) != CURLE_OK)
	{
#ifdef MAEMO
		gtk_infoprint(NULL, "Unable to download!");
#else
		fprintf(stderr, "\nUnable to download!\n");
#endif
	}
	else
	{
#ifdef MAEMO
		gtk_infoprint(NULL, "Download complete");
#else
		fprintf(stderr, "\nDownload complete\n");
#endif
	}

	fclose(df->file);
	free(df->uri);
	free(df);
		
	curl_easy_cleanup(curl);
		
	return FALSE;
}

/**
 * Save data to a file
 *
 */
gboolean save_data_to_file()
{
	GtkWidget* dialog = NULL;
	gboolean result = FALSE;
	ResourceSelectionDialog* rsd = NULL;
	
	char* name = NULL;
	char* uri = NULL;
	char* node_class = NULL;
	char* id = NULL;
	char* srv_udn = NULL;
	gchar* selected_uri = NULL;
	
	DownloadFile* df = NULL;
	
	if (get_current_content_item(&name, &uri, &node_class, &id, &srv_udn) == FALSE)
	{
#ifdef MAEMO
		gtk_infoprint(NULL, "Nothing selected");
#else
		fprintf(stderr, "Nothing selected\n");
#endif
		g_free(name);
		g_free(uri);
		g_free(node_class);
		g_free(id);
		g_free(srv_udn);

		return FALSE;
	}

	if (resource_selection_dialog_run(uri, &selected_uri) == FALSE)
	{
#ifdef MAEMO
		gtk_infoprint(NULL, "Download aborted");
#else
		fprintf(stderr, "Download aborted\n");
#endif

		g_free(name);
		g_free(uri);
		g_free(node_class);
		g_free(id);
		g_free(srv_udn);
		
		return FALSE;
	}

#ifdef MAEMO
	dialog = hildon_file_chooser_dialog_new(NULL,
						GTK_FILE_CHOOSER_ACTION_SAVE);
#else
	dialog = gtk_file_chooser_dialog_new("Choose save location",
					     GTK_WINDOW(widgets->appview),
					     GTK_FILE_CHOOSER_ACTION_SAVE,
					     GTK_STOCK_CANCEL,
					     GTK_RESPONSE_CANCEL,
					     GTK_STOCK_SAVE,
					     GTK_RESPONSE_OK,
					     NULL);
#endif
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), name);

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
	{
		name = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
		
		df = (DownloadFile*) malloc (sizeof(DownloadFile));
		
		df->file = fopen(name, "wb");
		if (df->file)
		{
			df->uri = g_strdup(selected_uri);
						
			/* Download file in the background */
			g_idle_add(download_idle, df);

#ifdef MAEMO
			gtk_infoprint(NULL, "Downloading...");
#endif
			
			result = TRUE;
		}
		else
		{
#ifdef MAEMO
			gtk_infoprint(NULL, "Unable to create file!");
#else
			fprintf(stderr, "Unable to create file!\n");
#endif
			result = FALSE;
		}
	}
	else
	{
#ifdef MAEMO
		gtk_infoprint(NULL, "Save cancelled");
#else
		fprintf(stderr, "Save cancelled\n");
#endif
		result = FALSE;
	}
	
	g_free(name);
	g_free(uri);
	g_free(node_class);
	g_free(id);
	g_free(srv_udn);
	g_free(selected_uri);
	
	resource_selection_dialog_delete(rsd);
	gtk_widget_destroy(dialog);
	
	return result;
}

/**
 * Find a device from the tree model by the device's UDN.
 *
 * @param model The Tree model to search from
 * @param udn The UDN to search for
 * @param udn_col The column where to get the UDN from
 * @param iter The GtkTreeIter that points to the device node
 * @return TRUE if the device was found; otherwise FALSE
 */
gboolean find_device_from_model(GtkTreeModel* model, 
				char* udn, int udn_col,
				GtkTreeIter* iter)
{
	gchar* tmp_udn = NULL;
	gboolean found = FALSE;
	
	if (udn == NULL || model == NULL)
	{
		return FALSE;
	}

	if (!gtk_tree_model_get_iter_first(model, iter))
	{
		return FALSE;
	}
	
	do
	{
		gtk_tree_model_get(model, iter, udn_col, &tmp_udn, -1);

		if (cg_strcmp(tmp_udn, udn) == 0)
		{
			found = TRUE;
			break;
		}
			
		g_free(tmp_udn);
	}
	while (gtk_tree_model_iter_next(model, iter) == TRUE);

	return found;
}

/**
 * Convert a CurrentTrackDuration data type to long int (seconds).
 * Doesn't care about fractions (.F+ or .F0/F1).
 *
 * @param duration A CurrentTrackDuration string (H+:MM:SS)
 * @return duration as a long int (0 if an error occurs)
 */
long track_duration_to_long(gchar* duration)
{
	int len = 0;
	int i = 0;
	int head = 0;
	int tail = 0;
	long result = 0;
	gchar* tmp = NULL;
       
	g_return_val_if_fail(duration != NULL, 0);
       
	len = strlen(duration);
	tmp = g_new0(gchar, sizeof(gchar) * len);

	/* Find the first colon (it can be anywhere) */
	for (i = 0; i < len; i++)
	{
		if (duration[i] == ':')
		{
			tail = i;
			break;
		}
	}
	
	/* Bail out if tail goes too far or head is bigger than tail */
	if (tail > len || head > tail)
	{
		g_free(tmp);
		return 0;
	}

	/* Extract hours */
	memcpy(tmp, duration + head, tail - head);
	tmp[tail - head + 1] = '\0';
               
	result += 3600 * atoi(tmp);

	/* The next colon should be exactly 2 chars right */
	head = tail + 1;
	tail = head + 2;

	/* Bail out if tail goes too far or head is bigger than tail */
	if (tail > len || head > tail)
	{
		g_free(tmp);
		return 0;
	}
       
	/* Extract minutes */
	memcpy(tmp, duration + head, tail - head);
	tmp[tail - head + 1] = '\0';
       
	result += 60 * atoi(tmp);

	/* The next colon should again be exactly 2 chars right */
	head = tail + 1;
	tail = head + 2;

	/* Bail out if tail goes too far or head is bigger than tail */
	if (tail > len || head > tail)
	{
		g_free(tmp);
		return 0;
	}

	/* Extract seconds */
	memcpy(tmp, duration + head, tail - head);
	tmp[tail - head + 1] = '\0';
	result += atoi(tmp);

	g_free(tmp);
       
	return result;
}

/**
 * Convert a long int to CurrentTrackDuration data type (seconds)
 * Doesn't care about fractions (.F+ or .F0/F1).
 *
 * @param duration A CurrentTrackDuration string (H+:MM:SS)
 * @return A newly-created string containing the duration
 */
gchar* long_to_track_duration(long duration)
{
	int hr = 0;
	int min = 0;
	int sec = 0;
	int size = 4; /* -/+ in front, two colons and terminating NULL */
	gchar* result = NULL;
       
	/* Number of hours, cast loses all decimals automatically */
	hr = (int) (duration / 3600);
	size += (hr / 10) + 1;
       
	/* Reduce the amount of hours (in seconds) from the source */
	duration = duration - (hr * 3600);
       
	/* Number of minutes, cast loses all decimals automatically */
	min = (int) duration / 60;
	size += 2;
       
	/* Reduce the amount of minutes (in seconds) from the source */
	duration = duration - (min * 60);

	/* All we should have left is seconds */
	sec = duration;
	size += 2;
       
	result = g_new0(gchar, sizeof(gchar) * size);
	sprintf(result, "%d:%.2d:%.2d", hr, min, sec);

	return result;
}
