/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Leonid Zolotarev <leonid.zolotarev@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#include <glib-object.h>
#include "gwebwidget.h"
#include "gwebhistorylist.h"

#ifndef __GWEBENGINE_H__
#define __GWEBENGINE_H__

G_BEGIN_DECLS

/** GWebEngineSecurityMode.
  * Enumerates security modes.
  */
typedef enum
{
	G_WEBENGINE_NO_SECURITY = 0,
	G_WEBENGINE_LOW_SECURITY,
	G_WEBENGINE_MEDIUM_SECURITY,
	G_WEBENGINE_HIGH_SECURITY,
	G_WEBENGINE_UNKNOWN_SECURITY
} GWebEngineSecurityMode;

/** GWebEnginePopupMenuType.
  * Enumerates popup menu types.
  */
typedef enum
{
	/**< No type of menu */
	G_WEBENGINE_POPUPMENU_NOTYPE         = 0x0000,
	/**< Context menu for document */
	G_WEBENGINE_POPUPMENU_DOCUMENT       = 0x0001,
	/**< Context menu for link */
	G_WEBENGINE_POPUPMENU_LINK           = 0x0002,
	/**< Context menu for image */
	G_WEBENGINE_POPUPMENU_IMAGE          = 0x0004,
	/**< Context menu for text selection */
	G_WEBENGINE_POPUPMENU_TEXTSELECTION  = 0x0008,
	/**< Context menu for mailto */
	G_WEBENGINE_POPUPMENU_MAILTO         = 0x0010,
	/**< Context menu for iframes */
	G_WEBENGINE_POPUPMENU_IFRAME	     = 0x0020,
	/**< Context menu for unknown protocol */
	G_WEBENGINE_POPUPMENU_UNKNOWN_PROTOCOL = 0x0040
} GWebEnginePopupMenuType;

/** GWebEngineLoadingFinishStatus.
  * Enumerates loading finish status types.
  */
typedef enum
{
	/**< Loading of page is finished, with success */
	G_WEBENGINE_LOADING_FINISH_SUCCESS,
	/**< Loading of page is finished, error while loading page */
	G_WEBENGINE_LOADING_FINISH_COULDNT_CONNECT,
	/**< Loading of page is finished, status unknown */
	G_WEBENGINE_LOADING_FINISH_UNKNOWN
}  GWebEngineLoadingFinishStatus;


/** GWebEngineAuthenticationType.
  * Enumerates authentication types.
  */
typedef enum
{
	/**< basic http authentication */
	G_WEBENGINE_AUTHENTICATE = 0,
	/**< http authentication failed (and the question was asked again) */
	G_WEBENGINE_AUTHENTICATION_WRONG,
	/**< proxy authentication */
	G_WEBENGINE_PROXY_AUTHENTICATE,
	/**< proxy authentication failed (and the question was asked again) */
	G_WEBENGINE_PROXY_AUTHENTICATION_WRONG
} GWebEngineAuthenticationType;

#define G_TYPE_WEB_ENGINE             (g_web_engine_get_type ())
#define G_WEB_ENGINE(obj)             (G_TYPE_CHECK_INSTANCE_CAST ((obj), G_TYPE_WEB_ENGINE, GWebEngine))
#define G_WEB_ENGINE_CLASS(vtable)    (G_TYPE_CHECK_CLASS_CAST ((vtable), G_TYPE_WEB_ENGINE, GWebEngineIface))
#define G_IS_WEB_ENGINE(obj)          (G_TYPE_CHECK_INSTANCE_TYPE ((obj), G_TYPE_WEB_ENGINE))
#define G_IS_WEB_ENGINE_CLASS(vtable) (G_TYPE_CHECK_CLASS_TYPE ((vtable), G_TYPE_WEB_ENGINE))
#define G_WEB_ENGINE_GET_CLASS(inst)  (G_TYPE_INSTANCE_GET_INTERFACE ((inst), G_TYPE_WEB_ENGINE, GWebEngineIface))

typedef struct _GWebEngine GWebEngine; /* dummy object */
typedef struct _GWebEngineIface GWebEngineIface;

/** @struct GWebEngineIface.
  * Defines web engine interface.
  *
  * Implementation of this type maps to the Browser window.
  * It constructs window and handles window related functions.
  * It requests for actions from UI to engine.
  */
struct _GWebEngineIface {
	GTypeInterface parent;

	void (*destroy) (GWebEngine *self);
	void (*reset) (GWebEngine *self);
	void (*load_url) (GWebEngine *self, const gchar *url); 
	GWebStatus (*load_follow_link) (GWebEngine *self, const gchar *url); 
	GWebStatus (*load_url_with_css) (GWebEngine *self, const gchar *url, const gchar *cssfile);
	void (*load_url_with_js) (GWebEngine *self, gboolean  enable);
	void (*load_home_page) (GWebEngine *self); 
	void (*stop_load) (GWebEngine *self);
	void (*reload) (GWebEngine *self); 
	void (*go_back) (GWebEngine *self, guint nsteps);
	void (*go_forward) (GWebEngine *self, guint nsteps); 
	gboolean (*can_go_back) (GWebEngine *self);
	gboolean (*can_go_forward) (GWebEngine *self);
	void (*set_zoom_level) (GWebEngine *self, gint level);
	void (*zoom_by_steps) (GWebEngine *self, guint relative_zoom, guint* legal_zoom_value);
	void (*new_zoom_by_steps) (GWebEngine *self, gboolean up, guint* legal_zoom_value);
	GWebStatus (*set_zoom_level_around_point) (GWebEngine *self, guint x, guint y, gint level);
	GWebStatus (*compare_frames_zoom_level) (GWebEngine *self, gboolean *equal);
	GWebStatus (*get_image_properties) (GWebEngine *self, const gchar *url, gchar **file_type, guint *file_size, guint *width, guint *height);
	GWebStatus (*get_frame_properties) (GWebEngine *self, gchar **file_type, guint *file_size, GWebEngineEncoding *file_encoding, guint *security, gboolean *automatic_encoding);
	GWebStatus (*load_image) (GWebEngine *self, const gchar* url);
	gint (*get_back_window_history_list) (GWebEngine *self, GWebHistoryList **history);
	gint (*get_forward_window_history_list) (GWebEngine *self, GWebHistoryList **history);
	guint (*get_zoom_level) (GWebEngine *self);
	void (*set_frame_zoom_level) (GWebEngine *self, guint level);
	guint (*get_frame_zoom_level) (GWebEngine *self);
	void (*set_layout_mode) (GWebEngine *self, GWebEngineLayoutMode mode);
	GWebEngineLayoutMode (*get_layout_mode) (GWebEngine *self);
	void (*set_load_images) (GWebEngine *self, GWebEngineImagePolicy level);
	GWebEngineImagePolicy (*get_load_images) (GWebEngine *self); 
	void (*set_frame_stacking) (GWebEngine *self, gboolean enabled);
	gboolean (*get_frame_stacking) (GWebEngine *self);
	GWebStatus (*save_web_page) (GWebEngine *self, GWebEngineSaveAction setting, const gchar *file_name_with_path);
	GWebStatus (*save_current_frame) (GWebEngine *self, GWebEngineSaveAction setting, const gchar *file_name_with_path);
	const gchar* (*get_frame_title) (GWebEngine *self, gint frame_number);
	gint (*get_frame_number) (GWebEngine *self);
	GWebStatus (*save_web_page_image) (GWebEngine *self, const gchar *url, const gchar* directory); 
	void (*select_all_text) (GWebEngine *self);
	void (*set_language_encoding) (GWebEngine *self, GWebEngineEncoding encoding);
	void (*set_frame_encoding) (GWebEngine *self, GWebEngineEncoding encoding);
	gboolean (*find_text) (GWebEngine *self, const gchar *string, gboolean reverse, gboolean whole_word, gboolean case_sensitive, gboolean restart);
	void (*authenticate) (GWebEngine *self, gboolean ok, const gchar *username, const gchar *password);
	GWebStatus (*cut) (GWebEngine *self);
	GWebStatus (*copy) (GWebEngine *self);
	GWebStatus (*paste) (GWebEngine *self);
	gboolean (*can_cut) (GWebEngine *self);
	gboolean (*can_paste) (GWebEngine *self);
	gboolean (*can_copy) (GWebEngine *self);
	gboolean (*can_selecttext) (GWebEngine *self);
	gchar* (*get_favicon_name) (GWebEngine *self, const gchar *favicon_folder);
	GObject* (*get_engine_widget) (GWebEngine *self);
	GObject* (*get_engine_notifier) (GWebEngine *self);
	void (*set_open_link_mode) (GWebEngine *self, gboolean in_new_window);
	GWebStatus (*set_true_zoom_mode) (GWebEngine *self, gboolean status);
	void (*notify_plugins) (GWebEngine *self, gint event);
	void (*preload_current_skin) (GWebEngine *self);
	void (*invoke_selectmatch) (GWebEngine *self);
	GObject* (*get_server_certificate) (GWebEngine *self);
};

GType g_web_engine_get_type (void);

/** Destroy object.
  * @param self Object pointer
  */
void g_web_engine_destroy (GWebEngine *self);

/** Loads and displays the page specified by the URL.
  * @param self Object pointer
  * @param url URL of the page to be loaded
  */
void g_web_engine_load_url (GWebEngine *self, const gchar *url); 

/** Loads and displays the page specified by the url,
  * mimicking the behaviour of the user clicking or navigating to the url.
  * @param self Object pointer
  * @param url URL of the page to be loaded
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_load_follow_link (GWebEngine *self, const gchar *url); 

/** Loads and displays the page specified by the url,
  * using the specified style sheet.
  * @param self Object pointer
  * @param url URL of the page to be loaded
  * @param cssfile The style sheet
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_load_url_with_css (GWebEngine *self, const gchar *url, const gchar *cssfile);

/** Enables JS support if ECMAScript is enabled in this window.
  * This setting is the local setting for the window, not any global setting.
  * @param self Object pointer
  * @param enable TRUE if the window local setting for ECMAScript should be set
  * to allow ECMAScript, FALSE otherwise.
  */
void g_web_engine_load_url_with_js (GWebEngine *self, gboolean  enable);

/** Loads and displays home page
  * @param self Object pointer
  */
void g_web_engine_load_home_page (GWebEngine *self); 

/** Stop the loading of the current page
  * @param self Object pointer
  */
void g_web_engine_stop_load (GWebEngine *self);

/** Reloads the current page
  * @param self Object pointer
  */
void g_web_engine_reload (GWebEngine *self); 

/** Display a page that is before the current page in the history list. 
  * An attempt to move back beyond the first page in the history list causes
  * the first page to be displayed.
  * Nothing will change if the current page is the first page in
  * the history list.
  * @param self Object pointer
  * @param nsteps Number of pages to go back from current page.
  * 1 means previous page.
  */
void g_web_engine_go_back (GWebEngine *self, guint nsteps);

/** Display the page that is after the current page in the history list.
  * An attempt to move forward beyond the last page in the history list causes
  * the last page to be displayed.
  * Nothing will change if the current page is the last page in the history list.
  * @param self Object pointer
  * @param nsteps Number of pages to go forward from current page.
  * 1 means move forward one page.
  */
void g_web_engine_go_forward (GWebEngine *self, guint nsteps); 

/** Check if the current page is not the first page in the history list.
  * @param self Object pointer
  * @return TRUE, if it can go back to previous page, FALSE otherwise.
  */
gboolean g_web_engine_can_go_back (GWebEngine *self);

/** Check if the current page is not the last page the history list.
  * @param self Object pointer
  * @return TRUE, if it can go forward to next page, FALSE otherwise.
  */
gboolean g_web_engine_can_go_forward (GWebEngine *self);

/** Set the zoom level by steps
  * @param self Object pointer
  * @param relative zoom level in percent
  */
void g_web_engine_zoom_by_steps (GWebEngine *self, guint relative_zoom, guint* legal_zoom_value);

/** Set the new zoom level by steps
  * @param self Object pointer
  * @param up parameter, TRUE to increase and vice versa
  */
void g_web_engine_new_zoom_by_steps (GWebEngine *self, gboolean up, guint* legal_zoom_value);


/** Set the zoom level in percent
  * @param self Object pointer
  * @param level The zoom level in percent
  */
void g_web_engine_set_zoom_level (GWebEngine *self, gint level);

/** Zoom page around a point.
  * @param self Object pointer
  * @param x X coordinate.
  * @param y Y coordinate.
  * @param level The zoom level in percent
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_set_zoom_level_around_point (GWebEngine *self, guint x, guint y, gint level);

/** Compare the zoom levels of frames/iframes.
  * @param self Object pointer
  * @param equal Return argument, TRUE if all zoom levels are equal
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_compare_frames_zoom_level (GWebEngine *self, gboolean *equal);

/** Get the image properties.
  * @param self Object pointer
  * @param url URL of the image.
  * @param file_type Type of the image.
  * @param file_size Size of the image.
  * @param width Width of the Image.
  * @param height Height of the Image.
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_get_image_properties (GWebEngine *self, const gchar *url, gchar **file_type, guint *file_size, guint *width, guint *height);

/** Get the frame properties.
  * @param self Object pointer
  * @param file_type Return argument, the frame type
  * @param file_size Return argument, the frame size in bytes
  * @param file_encoding Return argument, the frame encoding
  * @param security Return argument, the frame security mode
  * @param automatic_encoding 	Return argument, true if the default encoding of the page was used.
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_get_frame_properties (GWebEngine *self, gchar **file_type, guint *file_size, GWebEngineEncoding *file_encoding, guint *security, gboolean *automatic_encoding);

/** Load the image.
  * @param self Object pointer
  * @param url URL of the image.
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_load_image (GWebEngine *self, const gchar* url);

/** Gets history information for all pages before the current page in the
  * history list.
  * @param self Object pointer
  * @param history  Return argument for the history information.
  *                 It is set to @c NULL if the list is empty, or
  *                 if an error occurs.
  *
  * @return The number of elements in the history list or a negative value on error.
  *         The error values are @c GWebStatus values.
  */
gint g_web_engine_get_back_window_history_list (GWebEngine *self, GWebHistoryList **history);

/** Gets history information for all pages after the current page in the
  * history list.
  * @param self Object pointer
  * @param history  Return argument for the history information.
  *                 It is set to @c NULL if the list is empty, or
  *                 if an error occurs.
  * @return The number of elements in the history list or a negative value on error.
  *         The error values are @c GWebStatus values.
  */
gint g_web_engine_get_forward_window_history_list (GWebEngine *self, GWebHistoryList **history);

/** Get the zoom level in percent.
  * @param self Object pointer
  * @return The zoom level in percent.
  */
guint g_web_engine_get_zoom_level (GWebEngine *self);

/** Set the zoom level for frame in percent.
  * @param self Object pointer
  * @param level The zoom level in percent for frame
  */
void g_web_engine_set_frame_zoom_level (GWebEngine *self, guint level);

/** Get the frame zoom level value.
  * @param self Object pointer
  * @return Frame zoom level in percent
  */
guint g_web_engine_get_frame_zoom_level (GWebEngine *self);

/** Sets the layout mode to use when displaying pages.
  * @param self Object pointer
  * @param mode The layout mode to use(Normal or MSR)
  */
void g_web_engine_set_layout_mode (GWebEngine *self, GWebEngineLayoutMode mode);

/** Get the layout mode when displaying pages.
  * @param self Object pointer
  * @return The layout modes for specifying how pages should be displayed.
  */
GWebEngineLayoutMode g_web_engine_get_layout_mode (GWebEngine *self);

/** Set the Image policy value(Always or Only loaded or Never).
  * @param self Object pointer
  * @param level Image policy value
  */
void g_web_engine_set_load_images (GWebEngine *self, GWebEngineImagePolicy level);

/** Get the Images policy value(Always or Only loaded or Never).
  * @param self Object pointer
  * @return Image policy
  */
GWebEngineImagePolicy g_web_engine_get_load_images (GWebEngine *self); 

/** Turn on/off frame stacking.
  * @param self Object pointer
  * @param enabled TRUE to enable the frame stacking, FALSE otherwise
  */
void g_web_engine_set_frame_stacking (GWebEngine *self, gboolean enabled);

/** Get the status of frame stacking mode(ON/OFF).
  * @param self Object pointer
  * @return TRUE if the frame stacking is on, FALSE otherwise
  */
gboolean g_web_engine_get_frame_stacking (GWebEngine *self);

/** Save the current displayed web page.
  * @param self Object pointer
  * @param setting G_WEBENGINE_SAVE_FRAMES or G_WEBENGINE_SAVE_NO_INLINES or G_WEBENGINE_SAVE_ALL_INLINES
  * @param file_name_with_path The filename with path to save.
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_save_web_page (GWebEngine *self, GWebEngineSaveAction setting, const gchar *file_name_with_path);

/** Save the current focused frame.
  * @param self Object pointer
  * @param setting G_WEBENGINE_SAVE_FRAMES or G_WEBENGINE_SAVE_NO_INLINES or G_WEBENGINE_SAVE_ALL_INLINES
  * @param file_name_with_path The filename with path to save.
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_save_current_frame (GWebEngine *self, GWebEngineSaveAction setting, const gchar *file_name_with_path);

/** Get the focused frame title.
  * @param self Object pointer
  * @param frame_number Focused frame number
  * @return Focused frame title string.
  */
const gchar* g_web_engine_get_frame_title (GWebEngine *self, gint frame_number);

/** Get the focused frame number.
  * @param self Object pointer
  * @return Focused frame number.
  */
gint g_web_engine_get_frame_number (GWebEngine *self);

/** Save the image.
  * @param self Object pointer
  * @param url The URL to save
  * @param directory The directory to save to. The directory is created if does not exist.
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_save_web_page_image (GWebEngine *self, const gchar *url, const gchar* directory);

/** Select all text in current page.
  * @param self Object pointer
  */
void g_web_engine_select_all_text (GWebEngine *self);

/** Set language encoding value.
  * @param self Object pointer
  * @param encoding GWebEngineEncoding value
  */
void g_web_engine_set_language_encoding (GWebEngine *self, GWebEngineEncoding encoding);

/** Set frame encoding value.
  * @param self Object pointer
  * @param encoding GWebEngineEncoding value
  */
void g_web_engine_set_frame_encoding (GWebEngine *self, GWebEngineEncoding encoding);

/** Search for a text string in current page.
  * @param self Object pointer
  * @param string The text to search for
  * @param reverse If TRUE, search up/left instead of down/right
  * @param whole_word Match only whole words
  * @param case_sensitive Accept uppercase/lowercase differences as a match.
  * @param restart If TRUE, search from the beginning of the document. If FALSE,
  * continue search from previous match
  * @return TRUE if can find specified text. FALSE, if can't
  */
gboolean g_web_engine_find_text (GWebEngine *self, const gchar *string, gboolean reverse, gboolean whole_word, gboolean case_sensitive, gboolean restart);

/** Submit a pending http authentication request.
  * @param self Object pointer
  * @param ok TRUE if ok was clicked, FALSE if cancel
  * @param username The username
  * @param password The password
  */
void g_web_engine_authenticate (GWebEngine *self, gboolean ok, const gchar *username, const gchar *password);

/** Cut selected text.
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_cut (GWebEngine *self);

/** Copy selected text.
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_copy (GWebEngine *self);

/** Paste selected text.
  * @param self Object pointer
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_paste (GWebEngine *self);

/** Checks if it's meaningful to call the g_web_engine_cut.
  * @param self Object pointer
  * @return TRUE if it seems to be meaningful to cut text. 
  */
gboolean g_web_engine_can_cut (GWebEngine *self);

/** Checks if it's meaningful to call the g_web_engine_paste.
  * @param self Object pointer
  * @return TRUE if it seems to be meaningful to paste text. 
  */
gboolean g_web_engine_can_paste (GWebEngine *self);

/** Checks if it's meaningful to call the g_web_engine_copy.
  * @param self Object pointer
  * @return TRUE if it seems to be meaningful to copy text. 
  */
gboolean g_web_engine_can_copy (GWebEngine *self);

/** Checks if it's meaningful to call the g_web_engine_cut.
  * @param self Object pointer
  * @return TRUE if it seems to be meaningful to cut text.
  */
gboolean g_web_engine_can_selecttext (GWebEngine *self);

/** Get the icon of the currently loaded page.
  * @param self Object pointer
  * @param favicon_folder Favicon folder name
  * @return Name of the icon.
  */
gchar* g_web_engine_get_favicon_name (GWebEngine *self, const gchar *favicon_folder);

/** Gets the engine widget.
  * @param self Object pointer
  * @return Returns the engine widget
  */
GObject* g_web_engine_get_engine_widget (GWebEngine *self);

/** Gets the engine notifier.
  * @param self Object pointer
  * @return Returns the engine notifier object that will emit signals
  */
GObject* g_web_engine_get_engine_notifier (GWebEngine *self);

/** Change the way links are opened when clicked or selected when navigated to.
  * Can either be opened in a new window or the same.
  * @param self Object pointer
  * @param in_new_window If TRUE, open links in new window.
  * Otherwise clicked/selected links are opened in the same window
  */
void g_web_engine_set_open_link_mode (GWebEngine *self, gboolean in_new_window);

/** (De)activate true zoom for a webpage
  * @param self Object pointer
  * @param status TRUE if activate true zoom
  * @return @c G_WEB_STATUS_OK on success.
  */
GWebStatus g_web_engine_set_true_zoom_mode (GWebEngine *self, gboolean status);

/** Sends a event message to all plugins in the current window
  * Using the NPP_HandleEvent signal
  *
  * @param self Object pointer
  * @param event The event to be sent to the plugin
  * (of type GWebPluginEvent)
  */
void g_web_engine_notify_plugins(GWebEngine *self, gint event);	

/** Normally the skin elements are loaded when needed, but loading skin
  * might be time consuming if file I/O is very slow. This function
  * makes the skins load immediately, to avoid delays later.
  *
  * @param self Object pointer
  */
void g_web_engine_preload_current_skin(GWebEngine *self);

/** Invoke the select username dialog to select the 
  * stored username and password is filled in the forms element.
  * 
  * @param self Object pointer
  */
void g_web_engine_invoke_selectmatch(GWebEngine *self);

/** Gets the server certificate
  * @param self Object pointer
  *  
  * @return Returns the certificate.
  */
GObject* g_web_engine_get_server_certificate (GWebEngine *self);

G_END_DECLS

#endif /* __GWEBENGINE_H__ */
