
/*
 * This file is part of maemopad+
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <ui/callbacks.h>
#include <ui/interface.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libintl.h>

/*
 * strlen needed from string.h 
 */
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <hildon-widgets/hildon-banner.h>
#include <libgnomevfs/gnome-vfs.h>
#include <osso-browser-interface.h>

#include <config.h>

/*
 * Privates: 
 */
gboolean read_file_to_buffer(MainView * mainview);
void write_buffer_to_file(MainView * mainview);
void new_node_dialog(nodeType typ, MainView * mainview);

/*
st==0: turn off/zero refcount
st==1: turn on/increase ref
st==2: decrease refcount and turn off if necessary
*/
void setBusy(MainView *mainview, gchar st)
{
if (st==0) mainview->busyrefcount=0;
else if (st==1) mainview->busyrefcount++;
else if (st==2 && mainview->busyrefcount>0) mainview->busyrefcount--;
gdk_window_set_cursor(GTK_WIDGET(mainview->data->main_view)->window, (mainview->busyrefcount>0)?mainview->cursorBusy:NULL);
}

gboolean isBusy(MainView *mainview)
{
if (mainview->busyrefcount>0) return(TRUE);
return(FALSE);	
}

void prepareUIforNodeChange(MainView * mainview, nodeType typ)
{
	if (typ == NODE_TEXT)
		{
			gtk_widget_show(GTK_WIDGET(mainview->font_tb));
			gtk_widget_show(mainview->tools_font);
		}
	else
		{
			gtk_widget_hide(GTK_WIDGET(mainview->font_tb));
			gtk_widget_hide(mainview->tools_font);
		}

	if (typ == NODE_SKETCH)
		{
			gtk_widget_show(GTK_WIDGET(mainview->colorbutton_tb));
			gtk_widget_show(GTK_WIDGET(mainview->eraser_tb));
			gtk_widget_show(GTK_WIDGET(mainview->brushsize_tb));
			gtk_widget_show(GTK_WIDGET(mainview->sketchlines_tb));
			gtk_widget_show(GTK_WIDGET(mainview->shape_tb));
			gtk_widget_show(GTK_WIDGET(mainview->undo_tb));
			gtk_widget_show(GTK_WIDGET(mainview->redo_tb));
			gtk_widget_show(mainview->tools_color);
			gtk_widget_show(mainview->tools_brushsize);
			gtk_widget_show(mainview->tools_pagestyle);
			gtk_widget_show(mainview->tools_shape);
		}
	else
		{
			gtk_widget_hide(GTK_WIDGET(mainview->colorbutton_tb));
			gtk_widget_hide(GTK_WIDGET(mainview->eraser_tb));
			gtk_widget_hide(GTK_WIDGET(mainview->brushsize_tb));
			gtk_widget_hide(GTK_WIDGET(mainview->sketchlines_tb));
			gtk_widget_hide(GTK_WIDGET(mainview->shape_tb));
			gtk_widget_hide(GTK_WIDGET(mainview->undo_tb));
			gtk_widget_hide(GTK_WIDGET(mainview->redo_tb));
			gtk_widget_hide(mainview->tools_color);
			gtk_widget_hide(mainview->tools_brushsize);
			gtk_widget_hide(mainview->tools_pagestyle);
			gtk_widget_hide(mainview->tools_shape);
		}

	if (typ == NODE_TEXT)
		{
			gtk_widget_hide(sketchwidget_get_mainwidget(mainview->sk));
			gtk_widget_show(GTK_WIDGET(mainview->scrolledwindow));
			gtk_widget_show(mainview->tools_item);
		}
	else if (typ == NODE_SKETCH)
		{
			gtk_widget_hide(GTK_WIDGET(mainview->scrolledwindow));
			gtk_widget_show(sketchwidget_get_mainwidget(mainview->sk));
			gtk_widget_show(mainview->tools_item);
		}
	else
		{
			gtk_widget_hide(GTK_WIDGET(mainview->scrolledwindow));
			gtk_widget_hide(sketchwidget_get_mainwidget(mainview->sk));
			gtk_widget_hide(mainview->tools_item);
		}
}

nodeData *getSelectedNode(MainView * mainview)
{
	GtkTreeIter iter;
	GtkTreeModel *model;

	nodeData *nd;

	GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(mainview->treeview));

	if (!gtk_tree_selection_get_selected(selection, &model, &iter))
		return (NULL);

	gtk_tree_model_get(model, &iter, NODE_DATA, &nd, -1);

	return (nd);
}

void saveCurrentData(MainView * mainview)
{
	nodeData *selnode = getSelectedNode(mainview);
	saveDataToNode(mainview, selnode);
}

void saveDataToNode(MainView * mainview, nodeData *selnode)
{
	if (selnode == NULL)
		return;

	if ((selnode->typ == NODE_SKETCH && sketchwidget_get_edited(mainview->sk) == FALSE) || (selnode->typ == NODE_TEXT && gtk_text_buffer_get_modified(GTK_TEXT_BUFFER(mainview->buffer)) == FALSE))
		{
			fprintf(stderr, "node not edited, not saving\n");
			return;
		}

	mainview->file_edited = TRUE;

	setBusy(mainview, 1);
	fprintf(stderr, "savecurrentdata!\n");

	gboolean goterr = TRUE;
	gchar *textdata = NULL;
	GdkPixbuf *pixbuf = NULL;
	gchar *sketchdata = NULL;
	gsize datalen = 0;
	char tq[512];

	if (selnode->typ == NODE_TEXT)
		{
			GtkTextIter start, end;

			gtk_text_buffer_get_bounds(GTK_TEXT_BUFFER(mainview->buffer), &start, &end);
			textdata = gtk_text_buffer_get_slice(GTK_TEXT_BUFFER(mainview->buffer), &start, &end, TRUE);

			snprintf(tq, sizeof(tq), "UPDATE %s SET bodytype=%d, bodyblob=NULL, body=?, flags=%d WHERE nodeid=%d", datatable_tmpname, selnode->typ, selnode->flags, selnode->sql3id);
		}
	else if (selnode->typ == NODE_SKETCH)
		{
			GError *err = NULL;
			GdkPixmap *skpix = sketchwidget_get_Pixmap(mainview->sk);
			GtkWidget *skdr = sketchwidget_get_drawingarea(mainview->sk);

			pixbuf = gdk_pixbuf_get_from_drawable(NULL, GDK_DRAWABLE(skpix), NULL, 0, 0, 0, 0, skdr->allocation.width, skdr->allocation.height);
			if (pixbuf == NULL)
				{
					fprintf(stderr, "Error saving: pixbuf is null\n");
				}
			else
				{
				double w, h;
				GdkPixbuf *pixbuf2 = sketchwidget_trim_image(pixbuf, skdr->allocation.width, skdr->allocation.height, &w, &h, FALSE);

				if (pixbuf2!=NULL)
					{
					if (gdk_pixbuf_save_to_buffer(pixbuf2, &sketchdata, &datalen, "png", &err, NULL) == FALSE)
						{
							sketchdata = NULL;
							datalen = 0;
							fprintf(stderr, "Error saving sketch! %s\n", err->message);
							g_error_free(err);
						}
					gdk_pixbuf_unref(pixbuf2);
					}
				}
			snprintf(tq, sizeof(tq), "UPDATE %s SET bodytype=%d, body=NULL, bodyblob=?, flags=%d WHERE nodeid=%d", datatable_tmpname, selnode->typ, selnode->flags, selnode->sql3id);
			fprintf(stderr, "datalen:%d\n", datalen);
			if (skpix && G_IS_OBJECT(skpix)) g_object_unref(skpix);
		}

	/*
	 * fprintf(stderr, "query is *%s*\n", tq);
	 */

	do
		{
			sqlite3_stmt *stmt = NULL;
			const char *dum;
			int rc = sqlite3_prepare(mainview->db, tq, strlen(tq), &stmt, &dum);

			if (rc)
				{
					fprintf(stderr, "Error updating: %s\n", sqlite3_errmsg(mainview->db));
					break;
				}
			if (selnode->typ == NODE_TEXT)
				sqlite3_bind_text(stmt, 1, textdata, strlen(textdata), SQLITE_TRANSIENT);
			else if (selnode->typ == NODE_SKETCH)
				sqlite3_bind_blob(stmt, 1, sketchdata, datalen, SQLITE_TRANSIENT);

			rc = SQLITE_BUSY;
			while(rc == SQLITE_BUSY)
				{
					rc = sqlite3_step(stmt);
					if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE || rc == SQLITE_DONE)
						break;
				}
			sqlite3_finalize(stmt);
			if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE)
				{
					fprintf(stderr, "Error saving node: %s\n", sqlite3_errmsg(mainview->db));
				}
			else
				{
					if (selnode->typ == NODE_TEXT)
						gtk_text_buffer_set_modified(GTK_TEXT_BUFFER(mainview->buffer), FALSE);
					else if (selnode->typ == NODE_SKETCH)
						sketchwidget_set_edited(mainview->sk, FALSE);
					goterr = FALSE;
				}
		}
	while(FALSE);

	if (textdata)
		g_free(textdata);
	if (sketchdata)
		g_free(sketchdata);
	if (goterr == TRUE)
		{
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Error saving memo"));
		}
	setBusy(mainview, 2);
}

void callback_treeview_celldatafunc(GtkTreeViewColumn * tree_column, GtkCellRenderer * cell, GtkTreeModel * tree_model, GtkTreeIter * iter, gpointer data)
{
	MainView *mainview = NULL;
	mainview = ( MainView * ) data;
	g_assert(mainview != NULL && mainview->data != NULL );
	
	nodeData *nd;

	gtk_tree_model_get(tree_model, iter, NODE_DATA, &nd, -1);
	if (nd == NULL)
		return;

	if (nd->namepix == NULL)
		{
			g_object_set(cell, "visible", FALSE, NULL);
		}
	else
		{
			g_object_set(cell, "visible", TRUE, NULL);
			g_object_set(cell, "width", SKETCHNODE_RX, NULL);
			g_object_set(cell, "height", SKETCHNODE_RY, NULL);
		}
}

gboolean callback_treeview_testcollapse(GtkTreeView * treeview, GtkTreeIter * arg1, GtkTreePath * arg2, gpointer user_data)
{
	return (TRUE);
}

void callback_treeview_change(GtkTreeSelection * selection, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeData *nd = getSelectedNode(mainview);

/*
	if (nd==NULL)
		fprintf(stderr, "changed called but no selectednode\n");
	else
		fprintf(stderr, "changed called, selectednode is %d\n", nd->sql3id);
*/
#ifdef NEW_SEL_LOGIC
	guint tm=time(NULL);
	if (mainview->cansel_time>0 && mainview->cansel_time+1.0<tm) mainview->cansel_node=NULL;

	if (nd==NULL)
		{
		if (mainview->cansel_node!=NULL)
			{
			fprintf(stderr, "[SELLOGIC] saving %d to unselect all nodes\n", (mainview->cansel_node)->sql3id);
			saveDataToNode(mainview, (mainview->cansel_node));
			mainview->cansel_node=NULL;
			}
		return;
		}

	if (mainview->cansel_node!=NULL)
		{
		if (nd->sql3id == (mainview->cansel_node)->sql3id)
			{
			mainview->cansel_node=NULL;
			fprintf(stderr, "[SELLOGIC] doubly selected %d, doing nothing\n", nd->sql3id);
			return;
			}
		else
			{
			fprintf(stderr, "[SELLOGIC] saving %d to load new node\n", (mainview->cansel_node)->sql3id);
			saveDataToNode(mainview, (mainview->cansel_node));
			mainview->cansel_node=NULL;
			}
		}
#endif

	if (nd == NULL) return;

	setBusy(mainview, 1);

	gboolean goterr = TRUE;
	char *textdata = NULL;
	char *blob = NULL;
	int blobsize = 0;

	char tq[512];

	snprintf(tq, sizeof(tq), "SELECT bodytype, body, bodyblob, flags FROM %s WHERE nodeid=%d", datatable_tmpname, nd->sql3id);
	sqlite3_stmt *stmt = NULL;
	const char *dum;
	int rc = sqlite3_prepare(mainview->db, tq, strlen(tq), &stmt, &dum);

	if (rc)
		{
			fprintf(stderr, "Error reading (1) %s\n", sqlite3_errmsg(mainview->db));
		}
	else
		{
			rc = SQLITE_BUSY;
			while(rc == SQLITE_BUSY || rc == SQLITE_ROW)
				{
					rc = sqlite3_step(stmt);
					if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE || rc == SQLITE_DONE)
						break;
					else if (rc == SQLITE_ROW)
						{
							nd->typ = sqlite3_column_int(stmt, 0);
							nd->flags = sqlite3_column_int(stmt, 3);

							prepareUIforNodeChange(mainview, nd->typ);
							if (nd->typ == NODE_TEXT)
								{
									textdata = (char *)sqlite3_column_text(stmt, 1);
									blobsize = sqlite3_column_bytes(stmt, 1);

									if (textdata == NULL || g_utf8_validate(textdata, blobsize, NULL) == FALSE)
										gtk_text_buffer_set_text(GTK_TEXT_BUFFER(mainview->buffer), "", -1);
									else
										{
											gtk_text_buffer_set_text(GTK_TEXT_BUFFER(mainview->buffer), textdata, blobsize);
										}

									gtk_text_buffer_set_modified(GTK_TEXT_BUFFER(mainview->buffer), FALSE);
									goterr = FALSE;
								}
							else if (nd->typ == NODE_SKETCH)
								{
									sketchwidget_wipe_undo(mainview->sk);
									sketchwidget_set_fillmode(mainview->sk, FALSE);
									sketchwidget_set_shift(mainview->sk, FALSE);
									gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainview->shapemenuitems[1]), TRUE);
									gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainview->shapemenuitems[0]), TRUE);

									blobsize = sqlite3_column_bytes(stmt, 2);
									blob = (char *)sqlite3_column_blob(stmt, 2);
									gboolean clear = TRUE;

									if (blob == NULL)
										goterr = FALSE;
									if (blob != NULL)
										{
											fprintf(stderr, "blobsize:%d\n", blobsize);
											GdkPixbufLoader *pl = gdk_pixbuf_loader_new_with_type("png", NULL);
											GError *err = NULL;

											gdk_pixbuf_loader_write(pl, (guchar *) blob, blobsize, &err);
											if (err != NULL)
												{
													fprintf(stderr, "Error loading sketch! %s\n", err->message);
													g_error_free(err);
													err = NULL;
												}
											gdk_pixbuf_loader_close(pl, NULL);
											GdkPixbuf *pixbuf = gdk_pixbuf_loader_get_pixbuf(pl);

											if (GDK_IS_PIXBUF(pixbuf))
												{
													GtkWidget *skdr = sketchwidget_get_drawingarea(mainview->sk);
													GtkPixmap *skpix = (GtkPixmap *) sketchwidget_get_Pixmap(mainview->sk);

													int w=gdk_pixbuf_get_width(pixbuf);
													int h=gdk_pixbuf_get_height(pixbuf);
													if (w!=skdr->allocation.width || h!=skdr->allocation.height)
														{
														if (w>skdr->allocation.width) w=skdr->allocation.width;
														if (h>skdr->allocation.height) h=skdr->allocation.height;
														sketchwidget_clear_real(mainview->sk);
														}
													gdk_draw_pixbuf(GDK_DRAWABLE(skpix), NULL, pixbuf, 0, 0, 0, 0, w, h, GDK_RGB_DITHER_NONE, 0, 0);

													clear = FALSE;
													goterr = FALSE;

													if (skpix && G_IS_OBJECT(skpix)) g_object_unref(skpix);
												}
											else
												{
													fprintf(stderr, "error loading pixbuf\n");
												}
											g_object_unref(pl);
										}
									if (clear == TRUE)
										{
											fprintf(stderr, "clearing pix area\n");
											sketchwidget_clear_real(mainview->sk);
										}
									gtk_widget_queue_draw(sketchwidget_get_drawingarea(mainview->sk));
								}

							if ((nd->flags & NODEFLAG_SKETCHLINES) > 0)
								gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainview->sketchlinesmenuitems[1]), TRUE);
							else if ((nd->flags & NODEFLAG_SKETCHGRAPH) > 0)
								gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainview->sketchlinesmenuitems[2]), TRUE);
							else
								{
									gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainview->sketchlinesmenuitems[0]), TRUE);
									callback_sketchlines(NULL, mainview->sketchlinesmenuitems[0]);	/*FIXME:ugly */
								}

							break;
						}
				}
			if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE)
				fprintf(stderr, "Error reading (2) %s\n", sqlite3_errmsg(mainview->db));

			sqlite3_finalize(stmt);
		}

	setBusy(mainview, 2);

	if (goterr == TRUE)
		{
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Error loading memo"));
		}
}

gboolean treeview_canselect(GtkTreeSelection * selection, GtkTreeModel * model, GtkTreePath * path, gboolean path_currently_selected, gpointer userdata)
{
	MainView *mainview = NULL;

	mainview = (MainView *) userdata;
	g_assert(mainview != NULL && mainview->data != NULL);

	if (mainview->loading==FALSE)
		{
#ifndef EXPANDING_ROWS
			if (path_currently_selected)
				return (FALSE);
#endif
		}
#ifndef NEW_SEL_LOGIC
	saveCurrentData(mainview);
#else
	if (path_currently_selected)
		{
		GtkTreeIter iter;
	  gtk_tree_model_get_iter(model, &iter, path);
	
	  gtk_tree_model_get(model, &iter, NODE_DATA, &(mainview->cansel_node), -1);
		mainview->cansel_time=time(NULL);
/*  	fprintf(stderr, "[SELLOGIC] canselect called for node %d\n", nd->sql3id);*/
		}
#endif

	return (TRUE);
}

gboolean newnodedlg_key_press_cb(GtkWidget * widget, GdkEventKey * event, GtkWidget * dlg)
{
	SketchWidget *s = gtk_object_get_data(GTK_OBJECT(dlg), "sk");

	switch (event->keyval)
		{
		case GDK_F7:
			sketchwidget_redo(s);
			return TRUE;
		case GDK_F8:
			sketchwidget_undo(s);
			return TRUE;
			return TRUE;
		}
	return FALSE;
}

void new_node_dialog(nodeType typ, MainView * mainview)
{
	GtkWidget *dialog, *label, *entry, *but_ok, *but_cancel, *vbox, *hbox, *al, *cb, *rb1, *rb2;

	dialog = gtk_dialog_new();

	g_signal_connect(G_OBJECT(dialog), "key_press_event", G_CALLBACK(newnodedlg_key_press_cb), dialog);

	GtkWidget *notebook = gtk_notebook_new();

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(vbox), 15);

	hbox = gtk_hbox_new(FALSE, 0);

	rb1=gtk_radio_button_new_with_label(NULL, _("Text memo"));
	gtk_box_pack_start(GTK_BOX(hbox), rb1, FALSE, FALSE, 0);
	gtk_object_set_data(GTK_OBJECT(dialog), "rbt", rb1);

	rb2=gtk_radio_button_new_with_label(gtk_radio_button_get_group(GTK_RADIO_BUTTON(rb1)), _("Sketch memo"));
	gtk_box_pack_start(GTK_BOX(hbox), rb2, FALSE, FALSE, 0);
	gtk_object_set_data(GTK_OBJECT(dialog), "rbs", rb2);
	
	if (typ==NODE_TEXT) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(rb1), TRUE);
								 else gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(rb2), TRUE);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), hbox);

	label = gtk_label_new(_("Text name"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
	label = gtk_label_new(_("Please name the memo:"));
	entry = gtk_entry_new_with_max_length(64);
	gtk_entry_set_text(GTK_ENTRY(entry), _("New memo"));
	gtk_editable_select_region(GTK_EDITABLE(entry), 0, -1);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(vbox), 15);
	label = gtk_label_new(_("Sketch name"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);

	label = gtk_label_new(_("Please name the memo:"));

	al = gtk_alignment_new(0.5, 0.5, 0, 0);

	SketchWidget *s = sketchwidget_new(SKETCHNODE_X, SKETCHNODE_Y, TRUE);

	sketchwidget_set_brushsize(s, 2);
	sketchwidget_set_backstyle(s, SKETCHBACK_GRAPH);

	gtk_widget_set_size_request(sketchwidget_get_mainwidget(s), SKETCHNODE_X, SKETCHNODE_Y);
	gtk_container_add(GTK_CONTAINER(al), sketchwidget_get_mainwidget(s));

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), al, FALSE, FALSE, 0);

	but_ok = gtk_button_new_with_label(_("Create"));
	but_cancel = gtk_button_new_with_label(_("Cancel"));
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), but_ok);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), but_cancel);

	gtk_object_set_user_data(GTK_OBJECT(dialog), entry);
	gtk_object_set_data(GTK_OBJECT(dialog), "mainview", mainview);
	gtk_object_set_data(GTK_OBJECT(dialog), "sk", s);
	gtk_object_set_data(GTK_OBJECT(dialog), "n", notebook);

	nodeType *tmp = g_malloc(sizeof(nodeType));	/*FIXME memleak:not freed on widget_destroy */
	*tmp = typ;
	gtk_object_set_data(GTK_OBJECT(dialog), "t", tmp);

	gtk_signal_connect_object(GTK_OBJECT(but_cancel), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), dialog);
	g_signal_connect(G_OBJECT(but_ok), "clicked", G_CALLBACK(callback_new_node_real), dialog);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), notebook);

	cb = gtk_check_button_new_with_label("Create as childnode");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cb), mainview->newnodedialog_createchild);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), cb);
	gtk_object_set_data(GTK_OBJECT(dialog), "cb", cb);

	gtk_widget_grab_focus(entry);

	gtk_widget_show_all(dialog);
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
}

void add_new_node(nodeData * node, MainView * mainview, gboolean ischild)
{
	GtkTreeIter parentiter, newiter;
	GtkTreeModel *model;
	void *ptr = NULL;

	GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(mainview->treeview));

	if (gtk_tree_selection_get_selected(selection, &model, &parentiter))
		ptr = &parentiter;

	GtkTreePath *path = NULL;

	unsigned int parentnodeid = 0;

	if (ptr != NULL)
		{
			path = gtk_tree_model_get_path(model, &parentiter);

			if (ischild == FALSE)
				{
					gboolean tmp = gtk_tree_path_up(path);

					if (gtk_tree_path_get_depth(path) > 0)
						{
							if (gtk_tree_model_get_iter(model, &parentiter, path))
								ptr = &parentiter;
						}
					else if (tmp == TRUE)
						{
							gtk_tree_path_down(path);	/*restore path so expand() works */
						}
				}
		}

	if (ptr != NULL)
		{
			nodeData *nd;

			gtk_tree_model_get(model, ptr, NODE_DATA, &nd, -1);
			if (nd)
				parentnodeid = nd->sql3id;
		}

	node->sql3id = 0;
	do
		{
			sqlite3_stmt *stmt = NULL;
			const char *dum;
			char tq[512];

			/*
			 * FIXME: ord 
			 */
			snprintf(tq, sizeof(tq), "INSERT INTO %s (parent, bodytype, name, nameblob, ord) VALUES (%d, %d, ?, ?, 0);", datatable_tmpname, parentnodeid, node->typ);
			int rc = sqlite3_prepare(mainview->db, tq, strlen(tq), &stmt, &dum);

			if (rc)
				{
					fprintf(stderr, "Error inserting(1): %s\n", sqlite3_errmsg(mainview->db));
					break;
				}
			if (node->name != NULL)
				sqlite3_bind_text(stmt, 1, node->name, strlen(node->name), SQLITE_TRANSIENT);
			else
				sqlite3_bind_text(stmt, 1, NULL, 0, SQLITE_TRANSIENT);

			if (node->namepix != NULL)
				{
					gchar *namepixdata = NULL;
					gsize datalen = 0;

					GError *err = NULL;

					if (gdk_pixbuf_save_to_buffer(node->namepix, &namepixdata, &datalen, "png", &err, NULL) == FALSE)
						{
							namepixdata = NULL;
							datalen = 0;
							fprintf(stderr, "Error saving name! %s\n", err->message);
							g_error_free(err);
						}
					sqlite3_bind_blob(stmt, 2, namepixdata, datalen, SQLITE_TRANSIENT);
				}
			else
				sqlite3_bind_blob(stmt, 2, NULL, 0, SQLITE_TRANSIENT);

			rc = SQLITE_BUSY;
			while(rc == SQLITE_BUSY)
				{
					rc = sqlite3_step(stmt);
					if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE || rc == SQLITE_DONE)
						break;
				}
			sqlite3_finalize(stmt);
			if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE)
				{
					fprintf(stderr, "Error inserting(2): %s\n", sqlite3_errmsg(mainview->db));
					break;
				}
			node->sql3id = sqlite3_last_insert_rowid(mainview->db);
		}
	while(FALSE);

	if (node->sql3id == 0)
		{
			if (node->name)
				g_free(node->name);
			if (node->namepix)
				g_object_unref(node->namepix);
			g_free(node);
			if (path)
				gtk_tree_path_free(path);
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, "Error creating node");
			return;
		}

	fprintf(stderr, "new node id is %d\n", node->sql3id);


	gtk_tree_store_append(GTK_TREE_STORE(model), &newiter, ptr);

	gtk_tree_store_set(GTK_TREE_STORE(model), &newiter, NODE_NAME, node->name, NODE_PIXBUF, node->namepix, NODE_DATA, node, -1);

	if (path)
		{
			mainview->loading=TRUE; /*only when we have a valid parent*/
			gtk_tree_view_expand_row(GTK_TREE_VIEW(mainview->treeview), path, FALSE);
			gtk_tree_path_free(path);
		}

	gtk_tree_selection_select_iter(selection, &newiter);

	mainview->loading=FALSE;
}

void callback_new_node_real(GtkAction * action, gpointer data)
{
	MainView *mainview;

	GtkWidget *dialog = data;
	GtkWidget *entry = gtk_object_get_user_data(GTK_OBJECT(dialog));

	mainview = gtk_object_get_data(GTK_OBJECT(dialog), "mainview");
	SketchWidget *s = gtk_object_get_data(GTK_OBJECT(dialog), "sk");
	GtkWidget *notebook = gtk_object_get_data(GTK_OBJECT(dialog), "n");
	GtkWidget *cb = gtk_object_get_data(GTK_OBJECT(dialog), "cb");
/*	nodeType *typ = gtk_object_get_data(GTK_OBJECT(dialog), "t");*/

	nodeType typ = NODE_TEXT;

/*	GtkWidget *rbt = gtk_object_get_data(GTK_OBJECT(dialog), "rbt");*/
	GtkWidget *rbs = gtk_object_get_data(GTK_OBJECT(dialog), "rbs");
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(rbs))) typ=NODE_SKETCH;


	gint t = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));

	gchar *txt = NULL;
	GdkPixbuf *pixbuf = NULL;

	if (t == 0)
		{
			txt = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
			if (strcmp(txt, "") == 0)
				{
					g_free(txt);
					return;
				}
		}
	else
		{
			GtkWidget *sdr = sketchwidget_get_drawingarea(s);

			GdkPixmap *spix = sketchwidget_get_Pixmap(s);

			pixbuf = gdk_pixbuf_get_from_drawable(NULL, GDK_DRAWABLE(spix), NULL, 0, 0, 0, 0, sdr->allocation.width, sdr->allocation.height);
			g_object_unref(spix);
			double w, h;
			GdkPixbuf *pixbuf2 = sketchwidget_trim_image(pixbuf, SKETCHNODE_X, SKETCHNODE_Y, &w,
																									 &h, TRUE);
			if (pixbuf2==NULL) return;

			GdkPixbuf *pixbuf3 = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, SKETCHNODE_RX,
																					SKETCHNODE_RY);

			gdk_pixbuf_fill(pixbuf3, 0xffffffff);

			if (w <= SKETCHNODE_RX && h <= SKETCHNODE_RY)
				{
					gdk_pixbuf_copy_area(pixbuf2, 0, 0, w, h, pixbuf3, 0, (SKETCHNODE_RY - h) / 2);
				}
			else
				{
					double neww, newh;

					if (w > h)
						{
							neww = SKETCHNODE_RX;
							newh = (h / w) * SKETCHNODE_RX;
						}
					else
						{
							newh = SKETCHNODE_RY;
							neww = (w / h) * SKETCHNODE_RY;
						}
					if (newh > SKETCHNODE_RY)
						newh = SKETCHNODE_RY;	/*FIXME: newh exceeds SKETCHNODE_RY sometimes */

					GdkPixbuf *tmpbuf = gdk_pixbuf_scale_simple(pixbuf2, neww, newh,
																											GDK_INTERP_BILINEAR);

					gdk_pixbuf_copy_area(tmpbuf, 0, 0, neww, newh, pixbuf3, 0, (SKETCHNODE_RY - newh) / 2);

					gdk_pixbuf_unref(tmpbuf);
				}

			pixbuf = pixbuf3;
			gdk_pixbuf_unref(pixbuf2);
		}

	nodeData *node;

	node = g_malloc(sizeof(nodeData));
	node->typ = typ;
/*	node->typ = *typ;
	g_free(typ);*/
	node->name = NULL;
	node->namepix = NULL;

	if (t == 0)
		{
			node->name = txt;
		}
	else if (t == 1)
		{
			node->namepix = pixbuf;
		}

	node->lastMod = 0;
	node->flags = 0;
	node->sql3id = 0;

	mainview->newnodedialog_createchild = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cb));
	add_new_node(node, mainview, mainview->newnodedialog_createchild);

	sketchwidget_destroy(s);
	gtk_widget_destroy(dialog);
	mainview->file_edited = TRUE;
}

/*
 * delete node 
 */
void callback_file_delete_node(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	if (getSelectedNode(mainview) == NULL)
		{
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Select a node first"));
			return;
		}

	GtkWidget *dialog, *label, *but_ok, *but_cancel;

	dialog = gtk_dialog_new();

	label = gtk_label_new(_("Are you sure?"));

	but_ok = gtk_button_new_with_label(_("Yes, Delete"));
	but_cancel = gtk_button_new_with_label(_("Cancel"));

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), but_ok);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), but_cancel);

	gtk_object_set_user_data(GTK_OBJECT(dialog), mainview);

	gtk_signal_connect_object(GTK_OBJECT(but_cancel), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), dialog);
	g_signal_connect(G_OBJECT(but_ok), "clicked", G_CALLBACK(callback_delete_node_real), dialog);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), label);

	gtk_widget_show_all(dialog);
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
}

void callback_file_expand_collapse_node(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	GtkTreeIter iter;
	GtkTreeModel *model;

	GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(mainview->treeview));

	if (!gtk_tree_selection_get_selected(selection, &model, &iter))
		{
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Select a node first"));
			return;
		}

	GtkTreePath *path = gtk_tree_model_get_path(model, &iter);
	if (gtk_tree_view_row_expanded(GTK_TREE_VIEW(mainview->treeview), path)==FALSE)
		{
		gtk_tree_view_expand_row(GTK_TREE_VIEW(mainview->treeview), path, FALSE);
		}
	else
		{
		gtk_tree_view_collapse_row(GTK_TREE_VIEW(mainview->treeview), path);
		}

	gtk_tree_path_free(path);
}

void callback_file_export_node(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeData *nd=getSelectedNode(mainview);
	if (nd == NULL)
		{
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Select a node first"));
			return;
		}

	gchar *nodename=nd->name;
	if (nodename==NULL) nodename="saved node";

	if (nd->typ == NODE_TEXT)
		{
		GtkTextIter begin, end;
		gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER(mainview->buffer), &begin, &end);
		gchar *text = gtk_text_buffer_get_slice(GTK_TEXT_BUFFER(mainview->buffer), &begin, &end, TRUE);
		if (text==NULL || !strcmp(text, ""))
			{
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Node is empty"));
			}
		else
			{
			gchar *fn = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE, nodename, "txt");
			if (fn!=NULL)
				{
				GnomeVFSResult vfs_result;
				GnomeVFSHandle *handle = NULL;
				GnomeVFSFileSize out_bytes;
  			vfs_result = gnome_vfs_create(&handle, fn, GNOME_VFS_OPEN_WRITE, 0, 0600);
		    if ( vfs_result != GNOME_VFS_OK ) {
					hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Export failed"));
		    	}
    		else
    			{
    			gnome_vfs_write(handle, text, strlen(text), &out_bytes);
    			gnome_vfs_close(handle);
    			if (out_bytes==strlen(text)) hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Exported"));
    															else hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Export incomplete"));
    			}
				g_free(fn);
				}
			}
		g_free(text);
		}
	else if (nd->typ == NODE_SKETCH)
		{
		GdkPixmap *skpix = sketchwidget_get_Pixmap(mainview->sk);
		GtkWidget *skdr = sketchwidget_get_drawingarea(mainview->sk);
		GdkPixbuf *pixbuf = gdk_pixbuf_get_from_drawable(NULL, GDK_DRAWABLE(skpix), NULL, 0, 0, 0, 0, skdr->allocation.width, skdr->allocation.height);
		if (pixbuf==NULL)
			{
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Node is empty"));
			}		
		else
			{
			gchar *fn = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE, nodename, "png");
			if (fn!=NULL)
				{
				if (gdk_pixbuf_save(pixbuf, fn, "png", NULL, NULL)==FALSE)
					{
					hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Export failed"));
					}
				else
					{
					hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Exported"));
					}
				g_free(fn);
				}
			}
		g_object_unref(skpix);
		}
}

gboolean ref2iter(GtkTreeModel * model, GtkTreeRowReference * ref, GtkTreeIter * iter)
{
	gboolean res = FALSE;
	GtkTreePath *path = gtk_tree_row_reference_get_path(ref);

	if (gtk_tree_model_get_iter(model, iter, path))
		{
			res = TRUE;
		}
	gtk_tree_path_free(path);
	return (res);
}

GtkTreeRowReference *iter2ref(GtkTreeModel * model, GtkTreeIter * iter)
{
	GtkTreeRowReference *ref;

	GtkTreePath *path = gtk_tree_model_get_path(model, iter);

	ref = gtk_tree_row_reference_new(model, path);
	gtk_tree_path_free(path);
	return (ref);
}

void move_nodes_up(GtkTreeModel * model, GtkTreeRowReference * topnode, GtkTreeRowReference * newtop)
{
	GtkTreeIter topiter;

	fprintf(stderr, "here2\n");

	if (ref2iter(model, topnode, &topiter) == FALSE)
		return;

	fprintf(stderr, "here3\n");

	GtkTreeIter child;

	if (gtk_tree_model_iter_children(model, &child, &topiter))
		{
			fprintf(stderr, "here4\n");
			GtkTreeRowReference *ref;
			GList *rr_list = NULL, *node;

			do
				{
					ref = iter2ref(model, &child);
					rr_list = g_list_append(rr_list, ref);
				}
			while(gtk_tree_model_iter_next(model, &child));

			/*
			 * got a reflist for all children
			 */

			fprintf(stderr, "here5\n");
			for(node = rr_list; node; node = node->next)
				{
					ref = (GtkTreeRowReference *) (node->data);
					if (ref2iter(model, ref, &child))
						{
							GtkTreeIter newtopiter, newiter;
							GtkTreeIter *newtopiterptr;

							if (ref2iter(model, newtop, &newtopiter))
								newtopiterptr = &newtopiter;
							else
								newtopiterptr = NULL;

							nodeData *node;

							gtk_tree_model_get(model, &child, NODE_DATA, &node, -1);

							gtk_tree_store_append(GTK_TREE_STORE(model), &newiter, newtopiterptr);
							gtk_tree_store_set(GTK_TREE_STORE(model), &newiter, NODE_NAME, node->name, NODE_PIXBUF, node->namepix, NODE_DATA, node, -1);

							GtkTreeRowReference *newref = iter2ref(model, &newiter);

							move_nodes_up(model, ref, newref);
							gtk_tree_row_reference_free(newref);

							gtk_tree_store_remove(GTK_TREE_STORE(model), &child);
						}
					gtk_tree_row_reference_free(ref);
				}
			fprintf(stderr, "here6\n");

			g_list_free(rr_list);
		}

	fprintf(stderr, "here7\n");

}

void callback_delete_node_real(GtkAction * action, gpointer data)
{
	MainView *mainview;

	GtkWidget *dialog = data;

	mainview = gtk_object_get_user_data(GTK_OBJECT(dialog));

	GtkTreeIter iter;
	GtkTreeModel *model;

	GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(mainview->treeview));

	if (!gtk_tree_selection_get_selected(selection, &model, &iter))
		return;

	nodeData *nd;

	gtk_tree_model_get(model, &iter, NODE_DATA, &nd, -1);
	if (!nd)
		return;

	mainview->file_edited = TRUE;

	unsigned int sql3id = nd->sql3id;

	if (nd->name)
		g_free(nd->name);

	/*
	 * g_free(nd->data);
	 * if (nd->pix) g_object_unref(nd->pix);
	 */
	g_free(nd);

	fprintf(stderr, "here1\n");
	GtkTreeRowReference *upref = NULL, *ref = NULL;

	GtkTreePath *path = gtk_tree_model_get_path(model, &iter);

	ref = gtk_tree_row_reference_new(model, path);
	if (gtk_tree_path_up(path))
		upref = gtk_tree_row_reference_new(model, path);
	gtk_tree_path_free(path);

	g_object_ref(model);
	gtk_tree_view_set_model(GTK_TREE_VIEW(mainview->treeview), NULL);

	fprintf(stderr, "here! 1\n");
	move_nodes_up(model, ref, upref);

	fprintf(stderr, "here! 2\n");
	if (ref2iter(model, ref, &iter))
		{
			char tq[512];

			snprintf(tq, sizeof(tq), "SELECT parent FROM %s WHERE nodeid=%d", datatable_tmpname, sql3id);
			sqlite3_stmt *stmt = NULL;
			const char *dum;
			int rc = sqlite3_prepare(mainview->db, tq, strlen(tq), &stmt, &dum);
			unsigned int sql3parentid = 0;

			if (rc)
				{
					fprintf(stderr, "Error %s\n", sqlite3_errmsg(mainview->db));
				}
			else
				{
					rc = SQLITE_BUSY;
					while(rc == SQLITE_BUSY || rc == SQLITE_ROW)
						{
							rc = sqlite3_step(stmt);
							if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE || rc == SQLITE_DONE)
								break;
							else if (rc == SQLITE_ROW)
								{
									sql3parentid = sqlite3_column_int(stmt, 0);
									break;
								}
						}
					sqlite3_finalize(stmt);

					snprintf(tq, sizeof(tq), "UPDATE %s SET parent=%d WHERE parent=%d;", datatable_tmpname, sql3parentid, sql3id);
					if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
						fprintf(stderr, "ERROR moving nodes up!\n");
					else
						{
							snprintf(tq, sizeof(tq), "DELETE FROM %s WHERE nodeid=%d;", datatable_tmpname, sql3id);
							if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
								fprintf(stderr, "ERROR deleting node!\n");
						}
				}
			gtk_tree_store_remove(GTK_TREE_STORE(model), &iter);
		}

	gtk_tree_view_set_model(GTK_TREE_VIEW(mainview->treeview), model);
	g_object_unref(model);

	fprintf(stderr, "here! 3\n");
	gtk_tree_row_reference_free(ref);
	gtk_tree_row_reference_free(upref);

	gtk_tree_view_expand_all(GTK_TREE_VIEW(mainview->treeview));

	fprintf(stderr, "here10\n");
	gtk_widget_destroy(dialog);
}

void callback_edit_clear(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeData *nd = getSelectedNode(mainview);

	if (nd->typ == NODE_TEXT) gtk_text_buffer_set_text(GTK_TEXT_BUFFER(mainview->buffer), "", 0);
	else if (nd->typ == NODE_SKETCH) sketchwidget_clear(mainview->sk);
}

/*
 * cut 
 */
void callback_edit_cut(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeData *nd = getSelectedNode(mainview);

	if (nd->typ == NODE_TEXT)
		gtk_text_buffer_cut_clipboard(GTK_TEXT_BUFFER(mainview->buffer), mainview->clipboard, TRUE);
	else if (nd->typ == NODE_SKETCH)
		if (sketchwidget_cut(mainview->sk, mainview->clipboard)==FALSE)
								hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Error cutting"));
}

/*
 * copy 
 */
void callback_edit_copy(GtkAction * action, gpointer data)
{
	/*
	 * connect pointer to our MainView struct 
	 */
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeData *nd = getSelectedNode(mainview);

	if (nd->typ == NODE_TEXT)
		gtk_text_buffer_copy_clipboard(GTK_TEXT_BUFFER(mainview->buffer), mainview->clipboard);
	else if (nd->typ == NODE_SKETCH)
		if (sketchwidget_copy(mainview->sk, mainview->clipboard)==FALSE)
					hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Error copying"));
}

/*
 * paste 
 */
void callback_edit_paste(GtkAction * action, gpointer data)
{
	/*
	 * connect pointer to our MainView struct 
	 */
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeData *nd = getSelectedNode(mainview);

	if (nd->typ == NODE_TEXT)
		gtk_text_buffer_paste_clipboard(GTK_TEXT_BUFFER(mainview->buffer), mainview->clipboard, NULL, TRUE);
	else if (nd->typ == NODE_SKETCH)
		if (sketchwidget_paste(mainview->sk, mainview->clipboard)==FALSE)
							hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Error pasting"));

	mainview->file_edited = TRUE;
}

gint cb_popup(GtkWidget * widget, GdkEvent * event)
{
	GtkMenu *menu;
	GdkEventButton *event_button;

	/*
	 * The "widget" is the menu that was supplied when 
	 * * g_signal_connect_swapped() was called.
	 */
	menu = GTK_MENU(widget);
	event_button = (GdkEventButton *) event;
	if (event->type == GDK_BUTTON_PRESS && event_button->button == 3)
		{
			gtk_menu_popup(menu, NULL, NULL, NULL, NULL, event_button->button, event_button->time);
			return TRUE;
		}
	return FALSE;
}

/*
 * close 
 */
gboolean closefile(MainView * mainview)
{
	gint answer;

	saveCurrentData(mainview);

	/*
	 * save changes note if file is edited 
	 */
	if (mainview->file_edited)
		{
			answer = interface_save_changes_note(mainview);
			if (answer == CONFRESP_CANCEL)
				return (FALSE);
			else if (answer == CONFRESP_YES)
				{
					if (mainview->file_name == NULL)
						{
							mainview->file_name = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE, "maemopaddata", "db");
						}
					write_buffer_to_file(mainview);
				}
		}

	if (mainview->db)
		sqlite3_close(mainview->db);
	mainview->db = NULL;
	return (TRUE);
}

gboolean callback_file_close(GtkAction * action, gpointer data)
{

	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);
	if (closefile(mainview) == FALSE)
		return(FALSE);

	gtk_main_quit();
	return(TRUE);
}

void callback_file_new_node(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeType typ = NODE_SKETCH;

	nodeData *nd = getSelectedNode(mainview);

	if (nd != NULL)
		typ = nd->typ;

	new_node_dialog(typ, mainview);
}

/*
 * new 
 */
void callback_file_new(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	gchar *filename = NULL;

	if (closefile(mainview) == FALSE)
		return;

	filename = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE, "newfile", "db");
	if (filename == NULL)
		return;

	new_file(mainview);

	do
		{
		setBusy(mainview, 1);

		int rc;
	
		rc = sqlite3_open(filename, &mainview->db);
		if (rc)
			{
				hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Error 1"));
				fprintf(stderr, "Can't create database %s: %s\n", filename, sqlite3_errmsg(mainview->db));
				break;
			}
		char tq[512];
	
		snprintf(tq, sizeof(tq), "CREATE TABLE %s%s", misctable_name, misctable);
		sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);
	
		snprintf(tq, sizeof(tq), "CREATE TABLE %s%s", datatable_name, datatable);
		if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
			{
				fprintf(stderr, "ERROR creating data table\n");
				hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Error 2"));
				break;
			}
	
		if (mainview->db)
			sqlite3_close(mainview->db);
		mainview->db = NULL;
	
		mainview->file_name = filename;
		mainview->file_edited = FALSE;
		read_file_to_buffer(mainview);
	}while(FALSE);
	setBusy(mainview, 0);
}

gboolean reset_ctree(GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
	nodeData *node;

	gtk_tree_model_get(model, iter, NODE_DATA, &node, -1);
	if (node)
		{
			if (node->name)
				g_free(node->name);
			if (node->namepix)
				g_object_unref(node->namepix);
			g_free(node);

			gtk_tree_store_set(GTK_TREE_STORE(model), iter, NODE_DATA, NULL, -1);
		}
	return (FALSE);
}

void new_file(MainView * mainview)
{
	setBusy(mainview, 1);
	/*
	 * clear buffer, filename and free buffer text 
	 */
	gtk_text_buffer_set_text(GTK_TEXT_BUFFER(mainview->buffer), "", -1);
	mainview->file_name = NULL;
	mainview->file_edited = FALSE;
	mainview->newnodedialog_createchild = TRUE;

	GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(mainview->treeview));

	g_object_ref(model);
	gtk_tree_view_set_model(GTK_TREE_VIEW(mainview->treeview), NULL);

	gtk_tree_model_foreach(model, (GtkTreeModelForeachFunc) reset_ctree, (gpointer) mainview);

	/*
	 * crashing bastard
	 * gtk_tree_store_clear(GTK_TREE_STORE(model));
	 */
	GtkTreePath *path = gtk_tree_path_new_from_indices(0, -1);
	GtkTreeIter iter;

	if (gtk_tree_model_get_iter(model, &iter, path))
		{
			do
				{
					gtk_tree_store_remove(GTK_TREE_STORE(model), &iter);
				}
			while(gtk_tree_store_iter_is_valid(GTK_TREE_STORE(model), &iter));
		}
	gtk_tree_path_free(path);

	gtk_tree_view_set_model(GTK_TREE_VIEW(mainview->treeview), model);
	g_object_unref(model);

	prepareUIforNodeChange(mainview, NODE_UNKNOWN);
	setBusy(mainview, 2);
}

/*
 * open 
 */
void callback_file_open(GtkAction * action, gpointer data)
{
	gchar *filename = NULL;
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	if (closefile(mainview) == FALSE)
		return;

	/*
	 * open new file 
	 */
	filename = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_OPEN, NULL, NULL);

	/*
	 * if we got a file name from chooser -> open file 
	 */
	open_file(filename, mainview);
	g_free(filename);
}

gboolean open_file(gchar * filename, MainView * mainview)
{
	gboolean ret=FALSE;

	setBusy(mainview, 1);

	while(filename != NULL)
		{
			struct stat s;

			if (stat(filename, &s) == -1) break;

			mainview->file_name = g_strdup(filename);
			gboolean res = read_file_to_buffer(mainview);

			if (res == FALSE)
				{
					g_free(mainview->file_name);
					mainview->file_name = NULL;
					break;
				}
			mainview->file_edited = FALSE;
			ret=TRUE;
			break;
		}

	setBusy(mainview, 2);
	return(ret);
}

void callback_about_link(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);
	
	osso_rpc_run_with_defaults(mainview->data->osso, "osso_browser", OSSO_BROWSER_OPEN_NEW_WINDOW_REQ, NULL,
																	DBUS_TYPE_STRING, "http://garage.maemo.org/projects/maemopadplus/", DBUS_TYPE_INVALID);
}
                                             
void callback_about(GtkAction * action, gpointer data)
{
#if 1
	MainView *mainview = NULL;
	mainview = (MainView *) data;

	GtkWidget *w, *d;

	d = gtk_dialog_new();
	gtk_container_border_width(GTK_CONTAINER(d), 15);

	char tmp[32];

	snprintf(tmp, sizeof(tmp), "Maemopad+ %s", VERSION);
  w = gtk_button_new_with_label(tmp);
  g_object_set(w, "relief", GTK_RELIEF_NONE, NULL);
  g_object_set(w, "focus-on-click", FALSE, NULL);
	g_signal_connect(G_OBJECT(w), "clicked", G_CALLBACK(callback_about_link), mainview);
	gtk_container_add (GTK_CONTAINER(GTK_DIALOG(d)->vbox), w);

	w = gtk_label_new("\nLicense: LGPL\nAuthor: Kemal 'disq' Hadimli <disqkk@gmail.com>\n");
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(d)->vbox), w);

	w = gtk_button_new_with_label(_("Close"));
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(d)->action_area), w);
	gtk_signal_connect_object(GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), d);

	gtk_widget_show_all(d);
	gtk_window_set_modal(GTK_WINDOW(d), TRUE);

#else

	MainView *mainview=(MainView *)data;
	const char *authors[]={"Kemal 'disq' Hadimli <disqkk@gmail.com>", NULL};
	gtk_show_about_dialog(GTK_WINDOW(mainview->data->main_view), "name", "Maemopad+", "authors", authors, "copyright", "\xc2\xa9 2006 Kemal 'disq' Hadimli <disqkk@gmail.com>", "license", "GPL", "version", VERSION, "website", "http://garage.maemo.org/projects/maemopadplus/", "comments", "Based on Nokia's MaemoPad", NULL);

#endif
}

/*
 * save 
 */
void callback_file_save(GtkAction * action, gpointer data)
{
	gchar *filename = NULL;
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	/*
	 * check is we had a new file 
	 */
	if (mainview->file_name != NULL)
		{
			write_buffer_to_file(mainview);
		}
	else
		{
			filename = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE, "maemopaddata", "db");
			/*
			 * if we got a file name from chooser -> save file 
			 */
			if (filename != NULL)
				{
					mainview->file_name = filename;
					write_buffer_to_file(mainview);
					mainview->file_edited = FALSE;
				}
		}
}

void callback_shapemenu(GtkAction * action, GtkWidget * wid)
{
	int style = (int)gtk_object_get_user_data(GTK_OBJECT(wid));
	MainView *mainview = gtk_object_get_data(GTK_OBJECT(wid), "mainview");

	g_assert(mainview != NULL);

	if (style==0) sketchwidget_set_shape(mainview->sk, SKETCHSHAPE_FREEHAND);
	else if (style==1) sketchwidget_set_shape(mainview->sk, SKETCHSHAPE_LINE);
	else if (style==2) sketchwidget_set_shape(mainview->sk, SKETCHSHAPE_RECT);
	else if (style==3) sketchwidget_set_shape(mainview->sk, SKETCHSHAPE_ELLIPSE);
}



void callback_eraser(GtkAction * action, MainView * mainview)
{
	g_assert(mainview != NULL && mainview->data != NULL);

	if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainview->eraser_tb)) == TRUE)
		{
			GdkColor c2;

			c2.red = 65535;
			c2.green = 65535;
			c2.blue = 65535;

			sketchwidget_set_brushcolor(mainview->sk, c2);
			mainview->brushsize_backup = sketchwidget_get_brushsize(mainview->sk);
			guint ers=(mainview->brushsize_backup*2)+4;
			sk_set_brushsize(mainview, ers);
			sketchwidget_set_brushsize(mainview->sk, ers); /*fixme:to override max brush size, not pretty*/
		}
	else
		{
			GdkColor *col = hildon_color_button_get_color(HILDON_COLOR_BUTTON(mainview->colorbutton));

			GdkColor c2;

			c2.red = col->red;
			c2.green = col->green;
			c2.blue = col->blue;

			sketchwidget_set_brushcolor(mainview->sk, c2);
			sk_set_brushsize(mainview, mainview->brushsize_backup);
		}
}

void callback_menu(GtkAction * action, GtkWidget * menu)
{
	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, GDK_CURRENT_TIME);
}

void callback_brushsizetb(GtkAction * action, MainView *mainview)
{
	g_assert(mainview != NULL && mainview->data != NULL);

	if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainview->eraser_tb)) == TRUE)
		{
			gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainview->eraser_tb), FALSE);
		}
	else
		{
			callback_menu(NULL, mainview->brushsizemenu);
		}
}

void callback_brushsize(GtkAction * action, GtkWidget * wid)
{
	int bsize = (int)gtk_object_get_user_data(GTK_OBJECT(wid));
	MainView *mainview = gtk_object_get_data(GTK_OBJECT(wid), "mainview");

	g_assert(mainview != NULL && mainview->data != NULL);

	sketchwidget_set_brushsize(mainview->sk, bsize);

	GtkWidget *pix = gtk_object_get_data(GTK_OBJECT(wid), "icon");

	gtk_widget_show(pix);
	gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(mainview->brushsize_tb), pix);
}

void callback_sketchlines(GtkAction * action, GtkWidget * wid)
{
	int style = (int)gtk_object_get_user_data(GTK_OBJECT(wid));
	MainView *mainview = gtk_object_get_data(GTK_OBJECT(wid), "mainview");

	g_assert(mainview != NULL);

	nodeData *nd = getSelectedNode(mainview);
	gboolean doit = FALSE;

	if (nd != NULL && nd->typ == NODE_SKETCH)
		{
			nd->flags &= ~NODEFLAG_SKETCHLINES;
			nd->flags &= ~NODEFLAG_SKETCHGRAPH;
			sketchwidget_set_edited(mainview->sk, TRUE);
			doit = TRUE;
		}

	if (style == 0)
		{
			sketchwidget_set_backstyle(mainview->sk, SKETCHBACK_NONE);
		}
	else if (style == 1)
		{
			sketchwidget_set_backstyle(mainview->sk, SKETCHBACK_LINES);
			if (doit == TRUE)
				nd->flags |= NODEFLAG_SKETCHLINES;
		}
	else if (style == 2)
		{
			sketchwidget_set_backstyle(mainview->sk, SKETCHBACK_GRAPH);
			if (doit == TRUE)
				nd->flags |= NODEFLAG_SKETCHGRAPH;
		}

	GtkWidget *pix = gtk_object_get_data(GTK_OBJECT(wid), "icon");

	gtk_widget_show(pix);
	gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(mainview->sketchlines_tb), pix);
}

void callback_color(HildonColorButton * colorButton, MainView * mainview)
{
	g_assert(mainview != NULL && mainview->data != NULL);

	gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainview->eraser_tb), FALSE);

	GdkColor *col = hildon_color_button_get_color(colorButton);

	GdkColor c2;

	c2.red = col->red;
	c2.green = col->green;
	c2.blue = col->blue;

	sketchwidget_set_brushcolor(mainview->sk, c2);
}

void callback_color_invoke(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	HildonColorButton *hc = HILDON_COLOR_BUTTON(mainview->colorbutton);

	gtk_button_clicked(GTK_BUTTON(&hc->button));
}

/*
 * font
 */
void callback_font(GtkAction * action, gpointer data)
{
	PangoFontDescription *new_font = NULL;
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeData *nd = getSelectedNode(mainview);

	if (nd == NULL)
		return;
	if (nd->typ == NODE_SKETCH)
		return;

	new_font = interface_font_chooser(mainview);

	/*
	 * if we got a new font from chooser -> change font 
	 */
	if (new_font != NULL)
		{
			mainview->font_desc = new_font;
			gtk_widget_modify_font(mainview->textview, mainview->font_desc);
		}
}

void callback_undo(GtkAction * action, MainView * mainview)
{
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeData *nd = getSelectedNode(mainview);

	if (nd == NULL || nd->typ != NODE_SKETCH)
		return;

	sketchwidget_undo(mainview->sk);
}

void callback_redo(GtkAction * action, MainView * mainview)
{
	g_assert(mainview != NULL && mainview->data != NULL);

	nodeData *nd = getSelectedNode(mainview);

	if (nd == NULL || nd->typ != NODE_SKETCH)
		return;

	sketchwidget_redo(mainview->sk);
}

void callback_undotoggle(SketchWidget * sk, gboolean st, MainView * mainview)
{
	g_assert(mainview != NULL && mainview->data != NULL);

	gtk_widget_set_sensitive(GTK_WIDGET(mainview->undo_tb), st);
}

void callback_redotoggle(SketchWidget * sk, gboolean st, MainView * mainview)
{
	g_assert(mainview != NULL && mainview->data != NULL);

	gtk_widget_set_sensitive(GTK_WIDGET(mainview->redo_tb), st);
}

gboolean close_cb(GtkWidget * widget, GdkEventAny * event, MainView * mainview)
{
	callback_file_close(NULL, mainview);
	return (TRUE);
}

gboolean key_press_cb(GtkWidget * widget, GdkEventKey * event, MainView * mainview)
{
	switch (event->keyval)
		{

			/*
			 * case GDK_Up:
			 * gtk_infoprint(GTK_WINDOW(app), "Navigation Key Up");
			 * return TRUE;
			 * 
			 * case GDK_Down:
			 * gtk_infoprint(GTK_WINDOW(app), "Navigation Key Down");
			 * return TRUE;
			 * 
			 * case GDK_Left:
			 * gtk_infoprint(GTK_WINDOW(app), "Navigation Key Left");
			 * return TRUE;
			 * 
			 * case GDK_Right:
			 * gtk_infoprint(GTK_WINDOW(app), "Navigation Key Right");
			 * return TRUE;
			 * 
			 * case GDK_Return:
			 * gtk_infoprint(GTK_WINDOW(app), "Navigation Key select");
			 * return TRUE;
			 */
/*code below messes up when you have a textview*/
/*
		case GDK_Left:
		case GDK_Right:
			{
			gtk_widget_child_focus(widget, event->keyval==GDK_Left?GTK_DIR_TAB_BACKWARD:GTK_DIR_TAB_FORWARD);
			return TRUE;
			}
*/
		case GDK_Left:
			{
			nodeData *selnode = getSelectedNode(mainview);
			if (selnode!=NULL && selnode->typ==NODE_SKETCH)
				{
				hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Squared shapes ON"));
				sketchwidget_set_shift(mainview->sk, TRUE);
				return TRUE;
				}
			return FALSE;
			}
		case GDK_Right:
			{
			nodeData *selnode = getSelectedNode(mainview);
			if (selnode!=NULL && selnode->typ==NODE_SKETCH)
				{
				hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Squared shapes OFF"));
				sketchwidget_set_shift(mainview->sk, FALSE);
				return TRUE;
				}
			return FALSE;
			}
		case GDK_Down:
			{
			nodeData *selnode = getSelectedNode(mainview);
			if (selnode!=NULL && selnode->typ==NODE_SKETCH)
				{
				hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Filled shapes OFF"));
				sketchwidget_set_fillmode(mainview->sk, FALSE);
				return TRUE;
				}
			return FALSE;
			}
		case GDK_Up:
			{
			nodeData *selnode = getSelectedNode(mainview);
			if (selnode!=NULL && selnode->typ==NODE_SKETCH)
				{
				hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, _("Filled shapes ON"));
				sketchwidget_set_fillmode(mainview->sk, TRUE);
				return TRUE;
				}
			return FALSE;
			}
		case GDK_F6:
			callback_fullscreen(NULL, mainview);
			return TRUE;
		case GDK_F7:
			{
			nodeData *selnode = getSelectedNode(mainview);
			if (selnode!=NULL && selnode->typ==NODE_SKETCH)
				{
				callback_redo(NULL, mainview);
				return TRUE;
				}
			return FALSE;
			}
		case GDK_F8:
			{
			nodeData *selnode = getSelectedNode(mainview);
			if (selnode!=NULL && selnode->typ==NODE_SKETCH)
				{
				callback_undo(NULL, mainview);
				return TRUE;
				}
			return FALSE;
			}
		case GDK_Escape:
			if (mainview->fullscreen > 0)
				{
					mainview->fullscreen = 4;
					callback_fullscreen(NULL, mainview);
				}
			else
				{
					callback_file_close(NULL, mainview);
				}
			return TRUE;
		}

	return FALSE;
}

/*
 * fullscreen 
 */
void callback_fullscreenmenu(GtkAction * action, GtkWidget * wid)
{
	int style = (int)gtk_object_get_user_data(GTK_OBJECT(wid));
	MainView *mainview = gtk_object_get_data(GTK_OBJECT(wid), "mainview");

	g_assert(mainview != NULL);

	setBusy(mainview, 1);

	mainview->fullscreen=style;
	if (mainview->fullscreen <= 1)
		{
		gtk_window_unfullscreen(GTK_WINDOW(mainview->data->main_view));			
		}
	else
		{
		gtk_window_fullscreen(GTK_WINDOW(mainview->data->main_view));			
		}
	gtk_widget_show(mainview->scrolledtree);
	gtk_widget_show(mainview->toolbar);
	if (mainview->fullscreen == 1 || mainview->fullscreen >= 3)
		gtk_widget_hide(mainview->scrolledtree);
	if (mainview->fullscreen == 4)
		gtk_widget_hide(mainview->toolbar);
	
	setBusy(mainview, 2);
}

void callback_fullscreen(GtkAction * action, gpointer data)
{
	MainView *mainview = NULL;
	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	mainview->fullscreen++;
	if (mainview->fullscreen > 4)
		mainview->fullscreen = 0;
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mainview->fullscreenmenuitems[mainview->fullscreen]), TRUE);
}

/*
 * buffer edited 
 */
void callback_buffer_modified(GtkAction * action, gpointer data)
{
	/*
	 * connect pointer to our MainView struct 
	 */
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	/*
	 * change state 
	 */
	mainview->file_edited = TRUE;
}

/*
 * Callback for exit D-BUS event 
 */
void exit_event_handler(gboolean die_now, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	if (!die_now)
		{
		if (callback_file_close(NULL, mainview)==FALSE) gtk_main_quit(); /*make sure we call gtk_main_quit*/
		}
	else
		{
		gtk_main_quit();
		}	
	/*
	 * application_exit(mainview->data);
	 */
}

/*
 * Callback for hardware D-BUS events 
 */
void hw_event_handler(osso_hw_state_t * state, gpointer data)
{
	MainView *mainview = NULL;

	mainview = (MainView *) data;
	g_assert(mainview != NULL && mainview->data != NULL);

	/*
	 * if (state->shutdown_ind)
	 * {
	 * gtk_infoprint(GTK_WINDOW(mainview->data->app),
	 * "Shutdown event!");
	 * }
	 * if (state->memory_low_ind)
	 * {
	 * gtk_infoprint(GTK_WINDOW(mainview->data->app),
	 * "Memory low event!");
	 * }
	 */
	if (state->save_unsaved_data_ind)
		{
			fprintf(stderr, "Saving unsaved data!\n");
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, "Saving unsaved data!");
			callback_file_save(NULL, mainview);
		}

	/*
	 * if (state->system_inactivity_ind)
	 * {
	 * gtk_infoprint(GTK_WINDOW(mainview->data->app),
	 * "Minimize application inactivity event!");
	 * }
	 */
}

GtkTreeRowReference *read_sqlite3_data(MainView * mainview, unsigned int parentid, GtkTreeRowReference * parenttree, unsigned int selected, GtkTreeStore * model)
{
	GtkTreeRowReference *resref = NULL;

	char q[256];

	snprintf(q, sizeof(q), "SELECT nodeid, bodytype, name, nameblob, lastmodified, flags FROM %s WHERE parent=%d ORDER BY ord", datatable_tmpname, parentid);

	sqlite3_stmt *stmt = NULL;
	const char *dum;
	int rc = sqlite3_prepare(mainview->db, q, strlen(q), &stmt, &dum);

	if (rc)
		{
			fprintf(stderr, "Error %s\n", sqlite3_errmsg(mainview->db));
			return (NULL);
		}

	rc = SQLITE_BUSY;
	while(rc == SQLITE_BUSY || rc == SQLITE_ROW)
		{
			rc = sqlite3_step(stmt);
			if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE || rc == SQLITE_DONE)
				break;
			else if (rc == SQLITE_ROW)
				{
					int nodeid = sqlite3_column_int(stmt, 0);
					int typ = sqlite3_column_int(stmt, 1);
					const unsigned char *name = sqlite3_column_text(stmt, 2);
					const unsigned char *nameblob = sqlite3_column_text(stmt, 3);
					int lastmod = sqlite3_column_int(stmt, 4);
					int flags = sqlite3_column_int(stmt, 5);

					/*
					 * fprintf(stderr, "CARD=%s TYPE=%d\n", name, typ);
					 */
					if ((typ != NODE_TEXT && typ != NODE_SKETCH) || (name == NULL && nameblob == NULL))
						{

							/*
							 * fprintf(stderr, "invalid card, skipping\n");
							 */
							continue;
						}

					nodeData *node = g_malloc(sizeof(nodeData));

					node->sql3id = nodeid;
					node->typ = typ;
					node->flags = flags;
					node->name = NULL;
					node->namepix = NULL;
					if (name != NULL)
						node->name = g_strdup((char *)name);
					if (nameblob != NULL)
						{
							int blobsize = sqlite3_column_bytes(stmt, 3);

							GdkPixbufLoader *pl = gdk_pixbuf_loader_new_with_type("png", NULL);
							GError *err = NULL;

							gdk_pixbuf_loader_write(pl, (guchar *) nameblob, blobsize, &err);
							if (err != NULL)
								{
									fprintf(stderr, "Error loading nodename! %s\n", err->message);
									g_error_free(err);
									err = NULL;
								}
							gdk_pixbuf_loader_close(pl, NULL);
							GdkPixbuf *pixbuf = gdk_pixbuf_loader_get_pixbuf(pl);

							if (GDK_IS_PIXBUF(pixbuf))
								node->namepix = pixbuf;
						}
					node->lastMod = lastmod;

					GtkTreeIter parentiter, newiter;
					void *par = NULL;

					if (parenttree != NULL)
						{
							GtkTreePath *pa = gtk_tree_row_reference_get_path(parenttree);

							gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &parentiter, pa);
							gtk_tree_path_free(pa);
							par = &parentiter;
						}

					gtk_tree_store_append(model, &newiter, par);
					gtk_tree_store_set(model, &newiter, NODE_NAME, node->name, NODE_PIXBUF, node->namepix, NODE_DATA, node, -1);

					GtkTreePath *pa = gtk_tree_model_get_path(GTK_TREE_MODEL(model), &newiter);

					GtkTreeRowReference *newref = gtk_tree_row_reference_new(GTK_TREE_MODEL(model), pa);

					if (selected == nodeid)
						resref = newref;

					gtk_tree_path_free(pa);
					GtkTreeRowReference *r = read_sqlite3_data(mainview, nodeid, newref, selected,
																										 model);

					if (resref != newref)
						gtk_tree_row_reference_free(newref);

					if (r != NULL)
						{
							if (resref == NULL)
								resref = r;
							else
								gtk_tree_row_reference_free(r);	/*safeguard */
						}
				}
		}

	if (stmt)
		sqlite3_finalize(stmt);

	return (resref);							/*ref to supposed-to-be-selected treeitem */
}

/*
 * read file 
 */
gboolean read_file_to_buffer(MainView * mainview)
{
	char tq[512];

	g_assert(mainview != NULL);
	gboolean res = FALSE;

	gchar *filename = mainview->file_name;

	new_file(mainview);
	mainview->file_name = filename;
	mainview->loading=TRUE;

	fprintf(stderr, "read:*%s*\n", filename);

	int rc;
	sqlite3_stmt *stmt = NULL;

	rc = sqlite3_open(filename, &mainview->db);
	do
		{
			if (rc)
				{
					fprintf(stderr, "Can't open database %s: %s\n", filename, sqlite3_errmsg(mainview->db));
					break;
				}

			char *q = "SELECT skey, sval FROM settings";
			const char *dum;

			rc = sqlite3_prepare(mainview->db, q, strlen(q), &stmt, &dum);
			if (rc)
				{
					fprintf(stderr, "Error %s\n", sqlite3_errmsg(mainview->db));
					break;
				}

			unsigned int selectedCard = 0;

			/*
			 * fprintf(stderr, "start config\n");
			 */
			unsigned int curDataVersion = 0;

			rc = SQLITE_BUSY;
			while(rc == SQLITE_BUSY || rc == SQLITE_ROW)
				{
					rc = sqlite3_step(stmt);
					if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE || rc == SQLITE_DONE)
						break;
					else if (rc == SQLITE_ROW)
						{
							const unsigned char *col_key = sqlite3_column_text(stmt, 0);
							const unsigned char *col_val = sqlite3_column_text(stmt, 1);

							/*
							 * fprintf(stderr, "%s=%s\n", col_key, col_val);
							 */
							if (!strcmp(col_key, "leftPanedPos"))
								{
									gint panedPos = atoi((char *)col_val);

									if (panedPos > 0)
										gtk_paned_set_position(GTK_PANED(mainview->hpaned), panedPos);
								}
							if (!strcmp(col_key, "selectedNode"))
								{
									gint tmp = atoi((char *)col_val);

									if (tmp > 0)
										selectedCard = tmp;
								}
							if (!strcmp(col_key, "dataVersion"))
								{
									gint tmp = atoi((char *)col_val);

									if (tmp > 0)
										curDataVersion = tmp;
								}
							if (!strcmp(col_key, "newNodeDlgCreateChild"))
								{
									gint tmp = atoi((char *)col_val);

									mainview->newnodedialog_createchild = TRUE;
									if (tmp == 0)
										mainview->newnodedialog_createchild = FALSE;
								}
							if (!strcmp(col_key, "fullScreen"))
								{
									gint tmp = atoi((char *)col_val);
									if (tmp<0 || tmp>4) tmp=0;
									if (tmp==0) tmp=4;
												 else tmp--;
									mainview->fullscreen=tmp;
									callback_fullscreen(NULL, mainview);
								}
							if (!strcmp(col_key, "brushSize"))
								{
									gint tmp = atoi((char *)col_val);
									if (tmp>0) sk_set_brushsize(mainview, tmp);
								}
							if (!strcmp(col_key, "brushColor"))
								{
									unsigned long tmp = atol((char *)col_val);
									GdkColor c2;
						
									c2.red = ((tmp & 0xFF0000) >> 16) << 8;
									c2.green = ((tmp & 0xFF00) >> 8) << 8;
									c2.blue = (tmp & 0xFF) << 8;
/*	fprintf(stderr, "READ BRUSHCOLOR is %ul (%d,%d,%d)\n", tmp, c2.red, c2.green, c2.blue);*/
				
									sketchwidget_set_brushcolor(mainview->sk, c2);
									hildon_color_button_set_color(HILDON_COLOR_BUTTON(mainview->colorbutton), &c2);
								}

						}
				}
			if (rc == SQLITE_ERROR || rc == SQLITE_MISUSE)
				{
					fprintf(stderr, "Error2 %s\n", sqlite3_errmsg(mainview->db));
					break;
				}

			/*
			 * fprintf(stderr, "end config\n");
			 */
			if (stmt)
				sqlite3_finalize(stmt);

			gboolean resback = FALSE;

			while(curDataVersion < datatableversion)
				{
					if (curDataVersion == 0)
						{
							snprintf(tq, sizeof(tq), "DROP TABLE %s", datatable_backupname);
							sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

							snprintf(tq, sizeof(tq), "ALTER TABLE %s RENAME TO %s", datatable_name, datatable_backupname);
							if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
								{
									fprintf(stderr, "ERROR backing up table!\n");
									break;
								}
							resback = TRUE;

							snprintf(tq, sizeof(tq), "CREATE TABLE %s%s", datatable_name, datatable);
							if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
								{
									fprintf(stderr, "ERROR creating table!\n");
									break;
								}
							snprintf(tq, sizeof(tq), "INSERT INTO %s SELECT nodeid, parent, bodytype, name, body, nameblob, bodyblob, lastmodified, ord, 0 FROM %s", datatable_name, datatable_backupname);
							if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
								{
									fprintf(stderr, "ERROR copying data!\n");
									break;
								}

							snprintf(tq, sizeof(tq), "DROP TABLE %s", datatable_backupname);
							sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

							curDataVersion = datatableversion;
						}
				}
			if (curDataVersion != datatableversion)
				{
					fprintf(stderr, "Data version mismatch\n");

					if (resback == TRUE)
						{
							snprintf(tq, sizeof(tq), "DROP TABLE %s", datatable_name);
							sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);
							snprintf(tq, sizeof(tq), "ALTER TABLE %s RENAME TO %s", datatable_backupname, datatable_name);
							sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);
						}

					break;
				}

			GtkTreeStore *model = GTK_TREE_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(mainview->treeview)));

			g_object_ref(model);
			gtk_tree_view_set_model(GTK_TREE_VIEW(mainview->treeview), NULL);

			do
				{
					char tq[512];

					snprintf(tq, sizeof(tq), "CREATE%s TABLE %s%s", TEMPTABLE_KEYWORD, datatable_tmpname, datatable);
					if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
						{
							fprintf(stderr, "ERROR creating temp table!\n");
							break;
						}
					snprintf(tq, sizeof(tq), "CREATE INDEX %s_index ON %s %s", datatable_tmpname, datatable_tmpname, dataindex);
					if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
						{
							fprintf(stderr, "ERROR creating temp index!\n");
							break;
						}
					snprintf(tq, sizeof(tq), "INSERT INTO %s SELECT * FROM %s", datatable_tmpname, datatable_name);
					if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
						{
							fprintf(stderr, "ERROR copying data to temp table!\n");
							break;
						}
				}
			while(FALSE);

			GtkTreeRowReference *selectedRef = read_sqlite3_data(mainview, 0, NULL, selectedCard, model);

			gtk_tree_view_set_model(GTK_TREE_VIEW(mainview->treeview), GTK_TREE_MODEL(model));
			g_object_unref(model);
			gtk_tree_view_expand_all(GTK_TREE_VIEW(mainview->treeview));
			
			if (selectedRef != NULL)
				{
					GtkTreeIter seliter;

					if (ref2iter(GTK_TREE_MODEL(model), selectedRef, &seliter) == TRUE)
						{
							GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(mainview->treeview));
							gtk_tree_selection_select_iter(selection, &seliter);
						}

					gtk_tree_row_reference_free(selectedRef);
				}
			res = TRUE;
		}
	while(FALSE);

	if (stmt)
		sqlite3_finalize(stmt);


	mainview->loading=FALSE;
	
	return (res);
}

/*
 * write to file 
 */
void write_buffer_to_file(MainView * mainview)
{
	fprintf(stderr, "write:*%s*\n", mainview->file_name);
	saveCurrentData(mainview);

	setBusy(mainview, 1);

	char tq[512];

	snprintf(tq, sizeof(tq), "DROP TABLE %s", misctable_name);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	snprintf(tq, sizeof(tq), "CREATE TABLE %s%s", misctable_name, misctable);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	gint panedPos = gtk_paned_get_position(GTK_PANED(mainview->hpaned));

	snprintf(tq, sizeof(tq), "INSERT INTO %s VALUES('leftPanedPos', '%d');", misctable_name, panedPos);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	gint nndcc = 1;

	if (mainview->newnodedialog_createchild == FALSE)
		nndcc = 0;
	snprintf(tq, sizeof(tq), "INSERT INTO %s VALUES('newNodeDlgCreateChild', '%d');", misctable_name, nndcc);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	snprintf(tq, sizeof(tq), "INSERT INTO %s VALUES('fullScreen', '%d');", misctable_name, mainview->fullscreen);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	nodeData *node = getSelectedNode(mainview);

	if (node)
		{
			snprintf(tq, sizeof(tq), "INSERT INTO %s VALUES('selectedNode', '%d');", misctable_name, node->sql3id);
			sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);
		}

	guint bsize;
	if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(mainview->eraser_tb)) == TRUE)
		{
			bsize=mainview->brushsize_backup;
		}
	else
		{
			bsize=sketchwidget_get_brushsize(mainview->sk);
		}
	snprintf(tq, sizeof(tq), "INSERT INTO %s VALUES('brushSize', '%d');", misctable_name, bsize);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	GdkColor *col = hildon_color_button_get_color(HILDON_COLOR_BUTTON(mainview->colorbutton));
	unsigned long bcol=((col->red >> 8) << 16) | ((col->green >> 8) << 8) | (col->blue >> 8);
/*	fprintf(stderr, "BRUSHCOLOR is %d (%d,%d,%d)\n", bcol, col->red, col->green, col->blue);*/
	snprintf(tq, sizeof(tq), "INSERT INTO %s VALUES('brushColor', '%lu');", misctable_name, bcol);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	snprintf(tq, sizeof(tq), "INSERT INTO %s VALUES('dataVersion', '%d');", misctable_name, datatableversion);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	snprintf(tq, sizeof(tq), "DROP TABLE %s", datatable_backupname);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);
	snprintf(tq, sizeof(tq), "CREATE TABLE %s%s", datatable_backupname, datatable);
	if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
		{
			fprintf(stderr, "ERROR creating backup table!\n");
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, "Error 1");
			return;
		}
	snprintf(tq, sizeof(tq), "INSERT INTO %s SELECT * FROM %s", datatable_backupname, datatable_name);
	if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
		{
			fprintf(stderr, "ERROR backing up table!\n");
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, "Error 2");
			return;
		}
	snprintf(tq, sizeof(tq), "DELETE FROM %s", datatable_name);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	snprintf(tq, sizeof(tq), "INSERT INTO %s SELECT * FROM %s", datatable_name, datatable_tmpname);
	if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
		{
			fprintf(stderr, "ERROR saving table!\n");
			hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), NULL, "Error 3");

			snprintf(tq, sizeof(tq), "DELETE FROM %s", datatable_name);
			sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

			snprintf(tq, sizeof(tq), "INSERT INTO %s SELECT * FROM %s", datatable_name, datatable_backupname);
			if (sqlite3_exec(mainview->db, tq, NULL, NULL, NULL) != 0)
				{
					fprintf(stderr, "ERROR restoring backup! data lost!\n");
				}
			setBusy(mainview, 2);
			return;
		}

	snprintf(tq, sizeof(tq), "DROP TABLE %s", datatable_backupname);
	sqlite3_exec(mainview->db, tq, NULL, NULL, NULL);

	mainview->file_edited = FALSE;
	setBusy(mainview, 2);
	hildon_banner_show_information(GTK_WIDGET(mainview->data->main_view), GTK_STOCK_SAVE, _("Saved"));
}
