/*
  This file is part of "WhoAmI" - A "Guess whats on the photo" riddle game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <cstdlib>

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Panel.h>
#include <Lum/Toggle.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>

#include <Lum/OS/Main.h>

#include <Lum/OS/Manager/Repository.h>

#include "Control.h"
#include "DownloadImage.h"
#include "DownloadLists.h"
#include "Image.h"
#include "ImageInfo.h"

#include "config.h"

#include <iostream>

static Lum::Def::AppInfo info;

class MainDialog : public Lum::Dialog
{
private:
  Lum::Model::ActionRef newImageAction;
  Lum::Model::ActionRef blockGroupAction;
  Lum::Model::ActionRef imageInfoAction;
  Lum::Model::ActionRef aboutAction;
  Lum::Model::ActionRef networkStateChangeAction;
  Lum::Model::ActionRef waitForNetworkTimer;
  Control               *control;

  std::list<Image*>     images;
  Image*                currentImage;

  DownloadListsTask     downloadListsTask;
  DownloadImageTask     downloadImageTask;

  bool                  downloadRequested;

  Lum::OS::Manager::NetworkManager* nm;

public:
  MainDialog()
  : newImageAction(new Lum::Model::Action),
    blockGroupAction(new Lum::Model::Action),
    imageInfoAction(new Lum::Model::Action),
    aboutAction(new Lum::Model::Action),
    networkStateChangeAction(new Lum::Model::Action),
    waitForNetworkTimer(new Lum::Model::Action),
    control(NULL),
    currentImage(NULL),
    downloadRequested(false),
    nm(Lum::OS::Manager::repository->GetNetworkManager())
  {
    Observe(newImageAction);
    Observe(blockGroupAction);
    Observe(imageInfoAction);
    Observe(aboutAction);
    Observe(GetOpenedAction());
    Observe(GetClosedAction());

    if (nm!=NULL) {
      Observe(nm->GetNetworkStateChangeAction());
      Observe(waitForNetworkTimer);
    }

    blockGroupAction->Disable();
    imageInfoAction->Disable();

    downloadListsTask.SetParent(this);
    downloadImageTask.SetParent(this);

    Observe(downloadListsTask.GetFinishedAction());
    Observe(downloadListsTask.GetAbortedAction());

    Observe(downloadImageTask.GetFinishedAction());
  }

  void PreInit()
  {
    Lum::Panel       *horiz,*vert;

    horiz=Lum::HPanel::Create(true,true);

    control=new Control();
    control->SetFlex(true,true);
    horiz->Add(control);
    // TODO

    horiz->AddSpace();

    vert=Lum::VPanel::Create(false,true);
    vert->Add(Lum::Button::Create(L"_New image",newImageAction,true,false));
    vert->AddSpace();
    vert->Add(Lum::Toggle::Create(L"Peek",control->GetShowAll(),true,false));
    vert->AddSpace();
    vert->Add(Lum::Button::Create(L"_Block group",blockGroupAction,true,false));
    vert->AddSpace();
    vert->Add(Lum::Button::Create(L"_Image info",imageInfoAction,true,false));

    horiz->Add(vert);

    SetMain(horiz);

  Lum::Def::Menu *menu=Lum::Def::Menu::Create();

  menu
    ->GroupProject()
      ->ActionQuit(GetClosedAction())
    ->End()
    ->GroupHelp()
      //->ActionHelp()
      ->ActionAbout(aboutAction)
    ->End();
    SetMenu(menu);

    Dialog::PreInit();
  }

  void BlockGroup()
  {
    assert(currentImage!=NULL);

    for (std::list<Image*>::const_iterator iter=images.begin(); iter!=images.end(); ++iter) {
      if (!(*iter)->HasBeenShown()) {
        if (currentImage->GetAuthor()==(*iter)->GetAuthor()) {
          currentImage->SetShown();
          currentImage->SetImage(NULL);
        }
      }
    }
  }

  size_t GetAvailableImages()
  {
    size_t availableImages=0;

    for (std::list<Image*>::const_iterator iter=images.begin(); iter!=images.end(); ++iter) {
      if (!(*iter)->HasBeenShown()) {
        availableImages++;
      }
    }

    return availableImages;
  }

  void SetNewImage()
  {
    control->SetImage(currentImage->GetImage());

    currentImage->SetShown();
    currentImage->SetImage(NULL);

    blockGroupAction->Enable();
    imageInfoAction->Enable();
  }

  void ShowNewImage()
  {
    size_t                      pos;
    std::list<Image*>::iterator iter;

    pos=(size_t)(double(GetAvailableImages())*rand()/RAND_MAX);

    iter=images.begin();
    while (true) {
      if (pos==0 && !(*iter)->HasBeenShown()) {
        break;
      }

      if (!(*iter)->HasBeenShown()) {
        pos--;
      }

      iter++;
    }

    currentImage=(*iter);

    assert(!currentImage->HasBeenShown());

    if (!currentImage->HasImage()) {
      downloadImageTask.SetImage(currentImage);
      downloadImageTask.Start();

      return;
    }

    SetNewImage();
  }

  void TriggerDownloadLists()
  {
    downloadListsTask.SetImageList(images);
    downloadListsTask.Start();
  }

  void TriggerNewImage()
  {
    if (GetAvailableImages()<10) {
      TriggerDownloadLists();
      return;
    }

    ShowNewImage();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() &&
        GetOpenedAction()->IsFinished()) {
      // no code
    }
    else if (model==GetClosedAction() &&
             GetClosedAction()->IsFinished()) {
      // no code
    }
    else if (model==newImageAction &&
             newImageAction->IsFinished() &&
             newImageAction->IsEnabled()) {
      std::cout << "New image requested!" << std::endl;

      if (nm!=NULL &&
          nm->GetNetworkState()!=Lum::OS::Manager::NetworkManager::NetworkStateUnknown &&
          nm->GetNetworkState()!=Lum::OS::Manager::NetworkManager::NetworkStateConnected) {
        std::cout << "Not connected => requesting connection and waiting for connect" << std::endl;

        nm->Connect();

        downloadRequested=true;

        newImageAction->Disable();

        Lum::OS::display->AddTimer(30,0,waitForNetworkTimer);

        Dialog::Resync(model,msg);

          return;
      }

      TriggerNewImage();
    }
    else if (model==blockGroupAction &&
             blockGroupAction->IsFinished()) {
      BlockGroup();
    }
    else if (model==imageInfoAction &&
             imageInfoAction->IsFinished()) {
      ImageInfo *imageInfo;

      imageInfo=new ImageInfo(currentImage);
      imageInfo->SetParent(this);

      if (imageInfo->Open()) {
        imageInfo->EventLoop();
        imageInfo->Close();

        delete imageInfo;
      }
    }
    else if (model==aboutAction &&
             aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==downloadListsTask.GetFinishedAction() &&
             downloadListsTask.GetFinishedAction()->IsFinished()) {
      downloadListsTask.GetImageList(images);

      if (downloadListsTask.HasError()) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"Error while downloading RSS feeds!",
                              L"There was an error while downloading the RSS feeds:\n"+
                              downloadListsTask.GetError());
      }
      else {
        if (GetAvailableImages()>0) {
          ShowNewImage();
        }
        else {
          Lum::Dlg::Msg::ShowOk(this,
                                L"No new images available!",
                                L"Sorry, there are currently no new images available!\n"
                                L"Please, try again later!");
        }
      }
    }
    else if (model==downloadListsTask.GetAbortedAction() &&
             downloadListsTask.GetAbortedAction()->IsFinished()) {
      // no code
    }
    else if (model==downloadImageTask.GetAbortedAction() &&
             downloadImageTask.GetAbortedAction()->IsFinished()) {
      // no code
    }
    else if (model==downloadImageTask.GetFinishedAction() &&
             downloadImageTask.GetFinishedAction()->IsFinished()) {
      if (downloadImageTask.HasError()) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"Error while downloading image!",
                              L"There was an error during download of image data:\n"+
                              downloadImageTask.GetError());
      }
      else {
        SetNewImage();
      }
    }
    else if (nm!=NULL &&
             model==nm->GetNetworkStateChangeAction() &&
             nm->GetNetworkStateChangeAction()->IsFinished()) {
      std::cout << "Network status change!" << std::endl;
      if (downloadRequested) {
        if (nm->GetNetworkState()==Lum::OS::Manager::NetworkManager::NetworkStateConnected) {
          std::cout << "We are connected and a new image was requested => download" << std::endl;

          Lum::OS::display->RemoveTimer(waitForNetworkTimer);
          downloadRequested=false;

          newImageAction->Enable();

          TriggerNewImage();
        }
        else if (nm->GetNetworkState()==Lum::OS::Manager::NetworkManager::NetworkStateDisconnecting ||
                 nm->GetNetworkState()==Lum::OS::Manager::NetworkManager::NetworkStateDisconnected ||
                 nm->GetNetworkState()==Lum::OS::Manager::NetworkManager::NetworkStateUnknown) {
          std::cout << "We are disconnected and a new image was requested => abort" << std::endl;

          Lum::OS::display->RemoveTimer(waitForNetworkTimer);
          downloadRequested=false;

          newImageAction->Enable();
        }
      }
    }
    else if (model==waitForNetworkTimer &&
             waitForNetworkTimer->IsFinished()) {
      std::cout << "Waiting for network connection timed out => abort" << std::endl;

      downloadRequested=false;

      newImageAction->Enable();
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  Main()
  {
    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"What do you see?");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    srand(time(NULL));
  }

  bool Prepare()
  {
    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  void Cleanup()
  {
    Lum::OS::MainDialog<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

