/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Global includes
#include <libosso-abook/osso-abook.h>
#include <rtcom-eventlogger/eventlogger.h>
#include <uuid/uuid.h>

// Local includes
#include "export.hpp"
#include "qt_c_api.h"

// Defines
#define PROGRAMNAME    "web2sms"
#define PROGRAMVERSION "0.1"

/*

   This file only exists until the abook API has been fixed on MADDE.
   With MADDE 0.6.14 it's not possible to include osso-abook.h in a cpp file.
   Thus we have to keep the qt_c_api workaround here :(

 */

#ifndef Q_OS_WIN
#undef PROVIDERS_API
#define PROVIDERS_API
#endif

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

void PROVIDERS_API init_contacts()
{
  osso_context_t* osso_ctx;

  // Init osso (wheat means "osso"?)
  osso_ctx = osso_initialize(PROGRAMNAME, PROGRAMVERSION, FALSE, NULL);

  // Init the phonebook
  osso_abook_init_with_name(PROGRAMNAME, osso_ctx);

  g_type_init();
}

//////////////////////////////////////////////////////////////////////////

void PROVIDERS_API deinit_contacts()
{
  // Deinitialize osso
  osso_deinitialize( osso_abook_get_osso_context() );
}

//////////////////////////////////////////////////////////////////////////

int PROVIDERS_API get_contact_for_sms(const struct QString* dialogTitle, struct ContactInfoList* contacts)
{
  int success = FALSE;
  GtkWidget* chooser;
  GList* selection;
  struct QByteArray* utf8Title;

  utf8Title = QString_toUtf8(dialogTitle);

  // Create a new contact chooser
  chooser = osso_abook_contact_chooser_new(NULL, QByteArray_getValue(utf8Title));

  // Run the contact chooser
  gtk_dialog_run(GTK_DIALOG(chooser));
  gtk_widget_hide(chooser);

  // Get the selection
  selection = osso_abook_contact_chooser_get_selection(OSSO_ABOOK_CONTACT_CHOOSER(chooser));

  if ( selection )
  {
    EVCardAttribute* selectedAttribute = NULL;
    GtkWidget* numberChooser = NULL;

    OssoABookContactDetailStore* detailStore = osso_abook_contact_detail_store_new(selection->data, OSSO_ABOOK_CONTACT_DETAIL_SMS);
    GSequence* sequence = osso_abook_contact_detail_store_get_fields(detailStore);

    // Get the field count (sequence may be NULL if the contact has no mobile numbers)
    gint fieldCount = sequence ? g_sequence_get_length(sequence) : 0;
    if ( fieldCount > 1 )
    {
      // Show a new dialog where the user chooses a phone number
      numberChooser = osso_abook_contact_detail_selector_new_for_contact(NULL, selection->data, OSSO_ABOOK_CONTACT_DETAIL_SMS);

      gtk_dialog_run(GTK_DIALOG(numberChooser));
      gtk_widget_hide(numberChooser);

      // Get the selected attribute
      selectedAttribute = osso_abook_contact_detail_selector_get_detail(OSSO_ABOOK_CONTACT_DETAIL_SELECTOR(numberChooser));
    }
    else if ( fieldCount == 1 )
    {
      GSequenceIter* iter = g_sequence_get_begin_iter(sequence);
      selectedAttribute = osso_abook_contact_field_get_attribute( g_sequence_get(iter) );
    }

    if ( selectedAttribute )
    {
      ContactInfoList_addContact(contacts,
                                 osso_abook_contact_get_display_name(selection->data), // Get the display name of the contact
                                 e_vcard_attribute_get_value(selectedAttribute), // Get the number of the contact
                                 e_contact_get_const(E_CONTACT(selection->data), E_CONTACT_UID)); // Get the uid of the contact

      g_printf("Contact chosen: %s (%s)\n",
               osso_abook_contact_get_display_name(selection->data),
               e_vcard_attribute_get_value(selectedAttribute));

      success = TRUE;
    }

    // Destroy the number chooser
    if ( numberChooser )
      gtk_widget_destroy(numberChooser);
    g_object_unref(detailStore);
  }

  // Free some memory
  g_list_free(selection);
  gtk_widget_destroy(chooser);
  QByteArray_delete(utf8Title);

  return success;
}

//////////////////////////////////////////////////////////////////////////

int PROVIDERS_API add_to_sent_sms(const struct QString* name, const struct QString* number, const struct QString* uid, const struct QString* message)
{
  RTComEl* el;
  RTComElEvent* event;

  // Init RTCom
  el = rtcom_el_new();
  if ( !el )
  {
    g_printf("cannot init rtcom\n");
    return FALSE;
  }

  struct QByteArray* nameBytes = QString_toUtf8(name);
  struct QByteArray* numberBytes = QString_toUtf8(number);
  struct QByteArray* uidBytes = QString_toUtf8(uid);
  struct QByteArray* messageBytes = QString_toUtf8(message);

  char* remote_uid = QByteArray_getValue(numberBytes);
  char* group_uid;
  if ((int) (strlen (remote_uid) - 7) >= 0)
  {
    group_uid = remote_uid + strlen (remote_uid) - 7;
  }
  else
  {
    group_uid = remote_uid;
  }

  g_printf("Group uid: %s\n", group_uid);

  // Create a new event
  event = rtcom_el_event_new();
  if ( !event )
    g_printf("Event creation failed\n");

  // Fill some fields
  RTCOM_EL_EVENT_SET_FIELD(event, service, "RTCOM_EL_SERVICE_SMS");
  RTCOM_EL_EVENT_SET_FIELD(event, event_type, "RTCOM_EL_EVENTTYPE_SMS_OUTBOUND");
  RTCOM_EL_EVENT_SET_FIELD(event, start_time, time(NULL));
  RTCOM_EL_EVENT_SET_FIELD(event, end_time, time(NULL));
  RTCOM_EL_EVENT_SET_FIELD(event, is_read, TRUE);
  RTCOM_EL_EVENT_SET_FIELD(event, local_uid, "ring/tel/ring");
  RTCOM_EL_EVENT_SET_FIELD(event, local_name, "<SelfHandle>");
  RTCOM_EL_EVENT_SET_FIELD(event, remote_uid, QByteArray_getValue(numberBytes));
  RTCOM_EL_EVENT_SET_FIELD(event, remote_name, QByteArray_getValue(nameBytes));
  RTCOM_EL_EVENT_SET_FIELD(event, remote_ebook_uid, QByteArray_getValue(uidBytes));
  RTCOM_EL_EVENT_SET_FIELD(event, group_uid, group_uid);
  RTCOM_EL_EVENT_SET_FIELD(event, free_text, QByteArray_getValue(messageBytes));

  // Add the event to the event logger
  gint id = rtcom_el_add_event(el, event, NULL);
  if ( id < 0 )
  {
    g_printf("event adding failed: %d\n", id);
  }
  else
  {
    /*
     * Added because if a message token isn't added it means you can't open the message in conversations
     * to forward it.
     */

    uuid_t uuid;
    char key[36 + 1];
    uuid_generate_random(uuid);
    uuid_unparse(uuid, key);

    gchar* token = g_strdup_printf("web2sms-%s", key);

    gint header_id = rtcom_el_add_header(el, id, "message-token", token, NULL);
    g_printf("Header id: %d\n", header_id);

    g_printf("token: %s\n", token);

    g_free(token);
  }

  // Free the event
  rtcom_el_event_free(event);
  g_object_unref(el);

  // Free the QByteArrays
  QByteArray_delete(nameBytes);
  QByteArray_delete(numberBytes);
  QByteArray_delete(uidBytes);
  QByteArray_delete(messageBytes);

  return FALSE;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
