/*
 * vkontakte-plugin: An account plugin for Gabble
 * Copyright (C) 2010 Ivan Daniluk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <gtk/gtk.h>

#include "vkontakte-plugin.h"

#include <librtcom-accounts-widgets/rtcom-param-string.h>
#include <librtcom-accounts-widgets/rtcom-param-int.h>
#include <librtcom-accounts-widgets/rtcom-param-bool.h>
#include <librtcom-accounts-widgets/rtcom-login.h>
#include <librtcom-accounts-widgets/rtcom-edit.h>

#include <hildon/hildon-entry.h>
#include <hildon/hildon-defines.h>

#include <telepathy-glib/connection-manager.h>
#include <telepathy-glib/util.h>

#define H_(x) dgettext("hildon-libs", x)

ACCOUNT_DEFINE_PLUGIN (VkPlugin, vk_plugin, RTCOM_TYPE_ACCOUNT_PLUGIN);

#define VK_PRIV(ap) (G_TYPE_INSTANCE_GET_PRIVATE ((ap), \
            VK_TYPE, VkPrivate))

typedef struct {
  gchar *name;
  guint caps;
} VkPrivate;

enum /* properties */
{
  PROP_NAME = 1,
  PROP_CAPS,
  LAST_PROP
};

static void
vk_plugin_init (VkPlugin *self)
{
  g_object_set (self,
      "name", "vkontakte",
      "caps", RTCOM_PLUGIN_CAPABILITY_SCREEN_NAME |
              RTCOM_PLUGIN_CAPABILITY_REGISTER,
      NULL);
}

static GtkWidget *
get_login_page (VkPlugin *plugin,
    RtcomAccountItem *account)
{
    const static gchar *invalid_chars_re = "[:'\"<>&;#\\s]";
    return g_object_new (RTCOM_TYPE_LOGIN,
        "username-field", "account",
        "username-invalid-chars-re", invalid_chars_re,
        "username-prefill", RTCOM_ACCOUNT_PLUGIN (plugin)->username_prefill,
        "items-mask", RTCOM_ACCOUNT_PLUGIN (plugin)->capabilities,
        "account", account,
        NULL);
}

static GtkWidget *
get_edit_page (VkPlugin *plugin,
    RtcomAccountItem *account)
{
    const static gchar *invalid_chars_re = "[:'\"<>&;#\\s]";

    return g_object_new (RTCOM_TYPE_EDIT,
        "username-field", "account",
        "username-invalid-chars-re", invalid_chars_re,
        "items-mask", RTCOM_ACCOUNT_PLUGIN (plugin)->capabilities,
        "account", account,
        NULL);
}

static void
context_init (RtcomAccountPlugin *plugin,
    RtcomDialogContext *context)
{
  gboolean editing;
  GtkWidget *page = NULL;
  RtcomAccountItem *account;

  editing = account_edit_context_get_editing (ACCOUNT_EDIT_CONTEXT (context));
  account = RTCOM_ACCOUNT_ITEM (
      account_edit_context_get_account (ACCOUNT_EDIT_CONTEXT (context)));

  if (editing)
  {
      page = get_edit_page (
          VK_PLUGIN (plugin), account);

      g_assert (RTCOM_IS_EDIT (page));
  }
  else
  {
      page = get_login_page (
          VK_PLUGIN (plugin), account);

      g_assert (RTCOM_IS_LOGIN (page));
  }

  rtcom_dialog_context_set_start_page (context, page);
}

static void
vk_set_property (GObject *object,
    guint prop_id,
    const GValue *value,
    GParamSpec *pspec)
{
  VkPlugin *self = VK_PLUGIN (object);
  VkPrivate *priv = VK_PRIV (self);

  switch (prop_id)
    {
      case PROP_NAME:
        g_free (priv->name);
        priv->name = g_value_dup_string (value);
        break;
      case PROP_CAPS:
        priv->caps = g_value_get_uint (value);
        break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
vk_get_property (GObject *object,
    guint prop_id,
    GValue *value,
    GParamSpec *pspec)
{
  VkPlugin *self = VK_PLUGIN (object);
  VkPrivate *priv = VK_PRIV (self);

  switch (prop_id)
    {
      case PROP_NAME:
        g_value_set_string (value, priv->name);
        break;
      case PROP_CAPS:
        g_value_set_uint (value, priv->caps);
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
vk_finalize (GObject *object)
{
  VkPrivate *priv = VK_PRIV (object);

  if (priv->name != NULL)
    {
      g_free (priv->name);
      priv->name = NULL;
    }
}

static void
vk_constructed (GObject *object)
{
  RtcomAccountPlugin *rtcom_plugin = RTCOM_ACCOUNT_PLUGIN (object);
  VkPrivate *priv = VK_PRIV (object);
  GList *profiles, *list;
  gchar *cfg_name;

  rtcom_plugin->name = priv->name;
  rtcom_plugin->capabilities = priv->caps;

  cfg_name = g_strdup_printf ("%s-plugin", priv->name);

  profiles = mc_profiles_list ();
  for (list = profiles; list; list = list->next)
    {
      McProfile *profile = MC_PROFILE (list->data);
      const gchar *name, *cfg_ui;

      cfg_ui = mc_profile_get_configuration_ui (profile);
      if (cfg_ui != NULL && strcmp (cfg_ui, cfg_name) == 0)
        {
            name = mc_profile_get_unique_name (profile);
            rtcom_account_plugin_add_service (rtcom_plugin, name);
        }
    }

  mc_profiles_free_list (profiles);
  g_free (cfg_name);
}

static void
vk_plugin_class_init (VkPluginClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  RtcomAccountPluginClass *rtcom_class = RTCOM_ACCOUNT_PLUGIN_CLASS (klass);

  object_class->get_property = vk_get_property;
  object_class->set_property = vk_set_property;
  object_class->constructed = vk_constructed;
  object_class->finalize = vk_finalize;
  
  rtcom_class->context_init = context_init;

  g_object_class_install_property (object_class, PROP_NAME,
      g_param_spec_string ("name", "name", "name", "",
          G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));

  g_object_class_install_property (object_class, PROP_CAPS,
      g_param_spec_uint ("caps", "caps", "caps", 0, G_MAXUINT, 0,
          G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));

  g_type_class_add_private (object_class, sizeof (VkPrivate));
}
