#include "mainwindow.h"
#include <QMessageBox>
#include <QDesktopWidget>

bool ishexnstring(const QString& string)
{
  for (int i = 0; i < string.length(); i++) {
    if (isxdigit(string[i] == 0))
      return false;
  }
  return true;
}

QString unescapeJson( const QByteArray& ba, bool* ok )
{
  *ok = false;
  QString res;
  QByteArray seg;
  bool bs = false;
  for ( int i = 0, size = ba.size(); i < size; ++i ) {
    const char ch = ba[i];
    if ( !bs ) {
      if ( ch == '\\' )
        bs = true;
      else
        seg += ch;
    } else {
      bs = false;
      switch ( ch ) {
        case 'b':
          seg += '\b';
          break;
        case 'f':
          seg += '\f';
          break;
        case 'n':
          seg += '\n';
          break;
        case 'r':
          seg += '\r';
          break;
        case 't':
          seg += '\t';
          break;
        case 'u':
        {
          res += QString::fromUtf8( seg );
          seg.clear();

          if ( i > size - 5 ) {
            //error
            return QString();
          }

          const QString hex_digit1 = QString::fromUtf8( ba.mid( i + 1, 2 ) );
          const QString hex_digit2 = QString::fromUtf8( ba.mid( i + 3, 2 ) );
          i += 4;

          if ( !ishexnstring( hex_digit1 ) || !ishexnstring( hex_digit2 ) ) {
            qCritical() << "Not an hex string:" << hex_digit1 << hex_digit2;
            return QString();
          }
          bool hexOk;
          const ushort hex_code1 = hex_digit1.toShort( &hexOk, 16 );
          if (!hexOk) {
            qCritical() << "error converting hex value to short:" << hex_digit1;
            return QString();
          }
          const ushort hex_code2 = hex_digit2.toShort( &hexOk, 16 );
          if (!hexOk) {
            qCritical() << "error converting hex value to short:" << hex_digit2;
            return QString();
          }

          res += QChar(hex_code2, hex_code1);
          break;
        }
        case '\\':
          seg  += '\\';
          break;
        default:
          seg += ch;
          break;
      }
    }
  }
  res += QString::fromUtf8( seg );
  *ok = true;
  return res;
}

QString unescapeXML(const QString &str)
{
    QString data = str;
    data.replace( "&#39;", "'" );
    data.replace( "&quot;", "\"" );
    return data;
}

/************************ Main Window ********************************/

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent)
{
    QNetworkProxyFactory::setUseSystemConfiguration(true);
    _accessManager = new QNetworkAccessManager(this);
    _toWindow = 0;

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif

    createMenu();
    createComponents();
    createPortraitLayout();
    createLandscapeLayout();
    createAutoLayout();

    selfAssignLayout();

    createConnects();
    createLangCodes();

    retranslateUi();
    loadSettings();
}

void MainWindow::createMenu()
{
    actionLeft_from_clipboard = new QAction(this);
    actionLeft_from_clipboard->setObjectName(QString::fromUtf8("actionLeft_from_clipboard"));
    actionRight_from_clipboard = new QAction(this);
    actionRight_from_clipboard->setObjectName(QString::fromUtf8("actionRight_from_clipboard"));
    actionLeft_to_clipboard = new QAction(this);
    actionLeft_to_clipboard->setObjectName(QString::fromUtf8("actionLeft_to_clipboard"));
    actionRight_to_clipboard = new QAction(this);
    actionRight_to_clipboard->setObjectName(QString::fromUtf8("actionRight_to_clipboard"));
    actionLeft_as_sms = new QAction(this);
    actionLeft_as_sms->setObjectName(QString::fromUtf8("actionLeft_as_sms"));
    actionRight_as_sms = new QAction(this);
    actionRight_as_sms->setObjectName(QString::fromUtf8("actionRight_as_sms"));
    actionLeft_as_e_mail = new QAction(this);
    actionLeft_as_e_mail->setObjectName(QString::fromUtf8("actionLeft_as_e_mail"));
    actionRight_as_e_mail = new QAction(this);
    actionRight_as_e_mail->setObjectName(QString::fromUtf8("actionRight_as_e_mail"));
    actionAbout_qtranslate = new QAction(this);
    actionAbout_qtranslate->setObjectName(QString::fromUtf8("actionAbout_qtranslate"));
    //actionDonate = new QAction(this);
    //actionDonate->setObjectName(QString::fromUtf8("actionDonate"));

    QMenuBar * menuBar = new QMenuBar(this);

    menuBar->addAction(actionLeft_from_clipboard);
    menuBar->addAction(actionRight_from_clipboard);
    menuBar->addAction(actionLeft_to_clipboard);
    menuBar->addAction(actionRight_to_clipboard);
    menuBar->addAction(actionLeft_as_sms);
    menuBar->addAction(actionRight_as_sms);
    menuBar->addAction(actionLeft_as_e_mail);
    menuBar->addAction(actionRight_as_e_mail);
    menuBar->addAction(actionAbout_qtranslate);
    setMenuBar(menuBar);
    //Fremantle->addAction(actionDonate);
}

void MainWindow::createComponents()
{
    texteditLeft = new QTextEdit();
    texteditLeft->setObjectName(QString::fromUtf8("texteditLeft"));
    texteditRight = new QTextEdit();
    texteditRight->setObjectName(QString::fromUtf8("texteditRight"));

    comboboxLang_left = new QComboBox();
    comboboxLang_left->setObjectName(QString::fromUtf8("comboboxLang_left"));

    comboboxLang_right = new QComboBox();
    comboboxLang_right->setObjectName(QString::fromUtf8("comboboxLang_right"));

    buttonLeft_clear = new QPushButton();
    buttonLeft_clear->setObjectName(QString::fromUtf8("buttonLeft_clear"));
    buttonLeft_clear->setMaximumSize(QSize(100, 16777215));

    buttonRight_clear = new QPushButton();
    buttonRight_clear->setObjectName(QString::fromUtf8("buttonRight_clear"));
    buttonRight_clear->setMaximumSize(QSize(100, 16777215));

    buttonLeft_to_right = new QToolButton();
    buttonLeft_to_right->setObjectName(QString::fromUtf8("buttonLeft_to_right"));
    QIcon icon;
    icon.addFile(QString::fromUtf8(":/images/images/arrowright.png"), QSize(), QIcon::Normal, QIcon::Off);
    buttonLeft_to_right->setIcon(icon);

    buttonRight_to_left = new QToolButton();
    buttonRight_to_left->setObjectName(QString::fromUtf8("buttonRight_to_left"));
    QIcon icon1;
    icon1.addFile(QString::fromUtf8(":/images/images/arrowleft.png"), QSize(), QIcon::Normal, QIcon::Off);
    buttonRight_to_left->setIcon(icon1);
}

void MainWindow::createPortraitLayout()
{
    layoutPortrait = new QVBoxLayout;

    QVBoxLayout * verticalLayout_left = new QVBoxLayout();
    verticalLayout_left->addWidget(texteditLeft);
    QHBoxLayout * horizontalLayout_left = new QHBoxLayout();
    horizontalLayout_left->addWidget(comboboxLang_left);
    horizontalLayout_left->addWidget(buttonLeft_clear);
    verticalLayout_left->addLayout(horizontalLayout_left);

    layoutPortrait->addLayout(verticalLayout_left);

    QHBoxLayout * horizontalLayout_center = new QHBoxLayout();
    horizontalLayout_center->addWidget(buttonLeft_to_right);
    horizontalLayout_center->addWidget(buttonRight_to_left);

    layoutPortrait->addLayout(horizontalLayout_center);

    QVBoxLayout * verticalLayout_right = new QVBoxLayout();
    verticalLayout_right->addWidget(texteditRight);
    QHBoxLayout * horizontalLayout_right = new QHBoxLayout();
    horizontalLayout_right->addWidget(comboboxLang_right);
    horizontalLayout_right->addWidget(buttonRight_clear);
    verticalLayout_right->addLayout(horizontalLayout_right);

    layoutPortrait->addLayout(verticalLayout_right);
}

void MainWindow::createLandscapeLayout()
{
    layoutLandscape = new QHBoxLayout;

    QVBoxLayout * verticalLayout_left = new QVBoxLayout();
    verticalLayout_left->addWidget(texteditLeft);
    QHBoxLayout * horizontalLayout_left = new QHBoxLayout();
    horizontalLayout_left->addWidget(comboboxLang_left);
    horizontalLayout_left->addWidget(buttonLeft_clear);
    verticalLayout_left->addLayout(horizontalLayout_left);

    layoutLandscape->addLayout(verticalLayout_left);

    QVBoxLayout * verticalLayout_center = new QVBoxLayout();
    verticalLayout_center->addWidget(buttonLeft_to_right);
    verticalLayout_center->addWidget(buttonRight_to_left);

    layoutLandscape->addLayout(verticalLayout_center);

    QVBoxLayout * verticalLayout_right = new QVBoxLayout();
    verticalLayout_right->addWidget(texteditRight);
    QHBoxLayout * horizontalLayout_right = new QHBoxLayout();
    horizontalLayout_right->addWidget(buttonRight_clear);
    horizontalLayout_right->addWidget(comboboxLang_right);
    verticalLayout_right->addLayout(horizontalLayout_right);

    layoutLandscape->addLayout(verticalLayout_right);
}

void MainWindow::createAutoLayout()
{
    autoLayout = new PLLayout();
    autoLayout->setPLayout(layoutPortrait);
    autoLayout->setLLayout(layoutLandscape);
}

void MainWindow::selfAssignLayout()
{
    centralWidget = new QWidget();
    centralWidget->setLayout(autoLayout);
    this->setCentralWidget(centralWidget);
}

void MainWindow::createConnects()
{
    connect(actionLeft_from_clipboard, SIGNAL(triggered()), this, SLOT(leftFromClipboard()));
    connect(actionRight_from_clipboard, SIGNAL(triggered()), this, SLOT(rightFromClipboard()));
    connect(actionLeft_to_clipboard, SIGNAL(triggered()), this, SLOT(leftToClipboard()));
    connect(actionRight_to_clipboard, SIGNAL(triggered()), this, SLOT(rightToClipboard()));
    connect(buttonLeft_to_right, SIGNAL(clicked()), this, SLOT(translateLeftToRight()));
    connect(buttonRight_to_left, SIGNAL(clicked()), this, SLOT(translateRightToLeft()));
    connect(actionLeft_as_e_mail, SIGNAL(triggered()), this, SLOT(leftAsEmail()));
    connect(actionRight_as_e_mail, SIGNAL(triggered()), this, SLOT(rightAsEmail()));
    connect(actionLeft_as_sms, SIGNAL(triggered()), this, SLOT(leftAsSms()));
    connect(actionRight_as_sms, SIGNAL(triggered()), this, SLOT(rightAsSms()));
    connect(buttonLeft_clear, SIGNAL(clicked()), this, SLOT(leftClear()));
    connect(buttonRight_clear, SIGNAL(clicked()), this, SLOT(rightClear()));
    connect(actionAbout_qtranslate, SIGNAL(triggered()), this, SLOT(aboutQTranslate()));
    //connect(actionDonate, SIGNAL(triggered()), this, SLOT(donate()));

    connect(_accessManager, SIGNAL(finished(QNetworkReply*)), this, SLOT(translateRequestFinished(QNetworkReply*)));
}

void MainWindow::retranslateUi()
{
    setWindowTitle(QApplication::translate("MainWindow", "qTranslate", 0, QApplication::UnicodeUTF8));
    actionLeft_from_clipboard->setText(QApplication::translate("MainWindow", "Left from clipboard", 0, QApplication::UnicodeUTF8));
    actionRight_from_clipboard->setText(QApplication::translate("MainWindow", "Right from clipboard", 0, QApplication::UnicodeUTF8));
    actionLeft_to_clipboard->setText(QApplication::translate("MainWindow", "Left to clipboard", 0, QApplication::UnicodeUTF8));
    actionRight_to_clipboard->setText(QApplication::translate("MainWindow", "Right to clipboard", 0, QApplication::UnicodeUTF8));
    actionLeft_as_sms->setText(QApplication::translate("MainWindow", "Left as SMS", 0, QApplication::UnicodeUTF8));
    actionRight_as_sms->setText(QApplication::translate("MainWindow", "Right as SMS", 0, QApplication::UnicodeUTF8));
    actionLeft_as_e_mail->setText(QApplication::translate("MainWindow", "Left as e-mail", 0, QApplication::UnicodeUTF8));
    actionRight_as_e_mail->setText(QApplication::translate("MainWindow", "Right as e-mail", 0, QApplication::UnicodeUTF8));
    actionAbout_qtranslate->setText(QApplication::translate("MainWindow", "About qTranslate", 0, QApplication::UnicodeUTF8));
    //actionDonate->setText(QApplication::translate("MainWindow", "Donate", 0, QApplication::UnicodeUTF8));
    buttonLeft_clear->setText(QApplication::translate("MainWindow", "Clear", 0, QApplication::UnicodeUTF8));
    buttonRight_clear->setText(QApplication::translate("MainWindow", "Clear", 0, QApplication::UnicodeUTF8));
}

void MainWindow::saveSettings() {
    QSettings settings("Peter Bouda", "qTranslate");
    QString currentLeftLang = comboboxLang_left->itemText(comboboxLang_left->currentIndex());
    QString currentRightLang = comboboxLang_right->itemText(comboboxLang_right->currentIndex());
    settings.setValue("leftLanguage", currentLeftLang);
    settings.setValue("rightLanguage", currentRightLang);
}

void MainWindow::loadSettings() {
    QSettings settings("Peter Bouda", "qTranslate");
    QString currentLeftLang = settings.value("leftLanguage", "de").toString();
    QString currentRightLang = settings.value("rightLanguage", "en").toString();
    comboboxLang_left->setCurrentIndex(comboboxLang_left->findText(currentLeftLang));
    comboboxLang_right->setCurrentIndex(comboboxLang_right->findText(currentRightLang));
}

void MainWindow::closeEvent(QCloseEvent *event)
 {
    saveSettings();
    event->accept();
 }

/*void MainWindow::donate()
{
    QMessageBox donate(this);
    donate.setTextFormat(Qt::RichText);
    donate.setWindowTitle(tr("Donate for qTranslate"));
    donate.setText(tr("Please donate as much as you like if you find this software useful.<br/><b>Do you want to donate via Paypal now?</b>"));
    donate.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
    donate.setDefaultButton(QMessageBox::Yes);
    int ret = donate.exec();
    if (ret == QMessageBox::Yes)
        QDesktopServices::openUrl(QUrl("https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=VXN9T26P6HVLG"));
}*/

void MainWindow::aboutQTranslate()
{
    QMessageBox::information(this, tr("About qTranslate"),tr("qTranslate - A translation software for Maemo, Symbian and MeeGo<br>") + "v1.0.3<br>" + tr("based on Google Translate<br>Find more information and support on <a href=\"http://mobileqt.de/apps/qtranslate\">the qTranslate website</a>."));
}

void MainWindow::leftClear()
{
    texteditLeft->clear();
}

void MainWindow::rightClear()
{
    texteditRight->clear();
}

void MainWindow::leftAsSms()
{
    QString text = texteditLeft->toPlainText();
    QDesktopServices::openUrl("sms:?body="+text);
}

void MainWindow::rightAsSms()
{
    QString text = texteditRight->toPlainText();
    QDesktopServices::openUrl("sms:?body="+text);
}

void MainWindow::leftAsEmail()
{
    QString text = texteditLeft->toPlainText();
    QDesktopServices::openUrl("mailto:?body="+text);
}

void MainWindow::rightAsEmail()
{
    QString text = texteditRight->toPlainText();
    QDesktopServices::openUrl("mailto:?body="+text);
}

void MainWindow::translateLeftToRight()
{
    QString text = texteditLeft->toPlainText();
    int iLangFrom = comboboxLang_left->currentIndex();
    QString langFrom = comboboxLang_left->itemData(iLangFrom).toString();
    int iLangTo = comboboxLang_right->currentIndex();
    QString langTo = comboboxLang_right->itemData(iLangTo).toString();
    if (langTo == "auto") {
            QMessageBox::critical(this, tr("Choose target language"),tr("Detecting the language is not possible for target language."));
    } else {
        _toWindow = 1;
        googleTranslate(text, langFrom, langTo);
    }

}

void MainWindow::translateRightToLeft()
{
    QString text = texteditRight->toPlainText();
    int iLangFrom = comboboxLang_right->currentIndex();
    QString langFrom = comboboxLang_right->itemData(iLangFrom).toString();
    int iLangTo = comboboxLang_left->currentIndex();
    QString langTo = comboboxLang_left->itemData(iLangTo).toString();
    if (langTo == "auto") {
            QMessageBox::critical(this, tr("Choose target language"),tr("Detecting the language is not possible for target language."));
    } else {
        _toWindow = 0;
        googleTranslate(text, langFrom, langTo);
    }

}

void MainWindow::googleTranslate(QString text, QString langFrom, QString langTo)
{

    QNetworkRequest request;
    request.setUrl( QUrl("http://ajax.googleapis.com/ajax/services/language/translate?v=1.0&q="+text+"&langpair="+langFrom+"|"+langTo));
    request.setRawHeader("Host", "ajax.googleapis.com");
    request.setRawHeader("User-Agent", "qTranslateForMaemo");
    request.setRawHeader("Accept-Charset", "utf-8;q=0.7,*;q=0.7");
    request.setRawHeader("Referer", "qTranslateForMaemo");
    _accessManager->get(request);

    /*QByteArray params("v=1.0");
    params.append("&q="+QUrl::toPercentEncoding(text));
    params.append("&langpair="+langFrom+"%7C"+langTo);
    qDebug(params);*/
    //_translation = new QBuffer(this);
    //_http->request(header, QByteArray(), _translation);
    //QMessageBox::information(this, tr("Tiddlers on Rails"),tr("Get a free wiki at <a href=\"http://aufundapp.de/tor\">Tiddler on Rails</a>. Access it everywhere."));
}

void MainWindow::leftFromClipboard()
{
    fromClipboard(0);
}

void MainWindow::rightFromClipboard()
{
    fromClipboard(1);
}

void MainWindow::leftToClipboard()
{
    toClipboard(0);
}

void MainWindow::rightToClipboard()
{
    toClipboard(1);
}

void MainWindow::fromClipboard(int side)
{
    QClipboard *clipboard = QApplication::clipboard();
    QString clipboardText = clipboard->text();
    if (side == 0)
        texteditLeft->setText(clipboardText);
    else
        texteditRight->setText(clipboardText);
}

void MainWindow::toClipboard(int side)
{
    QClipboard *clipboard = QApplication::clipboard();
    QString text;
    if (side == 0)
        text = texteditLeft->toPlainText();
    else
        text = texteditRight->toPlainText();
    clipboard->setText(text);
}

MainWindow::~MainWindow()
{
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        retranslateUi();
        break;
    default:
        break;
    }
}

void MainWindow::resizeEvent(QResizeEvent* event) {
    // Call the autoLayout convenience function for the event handling
    autoLayout->resizeEvent(event);
    // Call base class resize event handler
    QMainWindow::resizeEvent(event);

#ifdef Q_OS_SYMBIAN
    showMaximized();
#endif

    QRect qs=QApplication::desktop()->availableGeometry();
    if (event->size().height() > event->size().width()) {
        QIcon icon;
        icon.addFile(QString::fromUtf8(":/images/images/arrowdown.png"), QSize(), QIcon::Normal, QIcon::Off);
        buttonLeft_to_right->setIcon(icon);
        QIcon icon2;
        icon2.addFile(QString::fromUtf8(":/images/images/arrowup.png"), QSize(), QIcon::Normal, QIcon::Off);
        buttonRight_to_left->setIcon(icon2);
    } else {
        QIcon icon;
        icon.addFile(QString::fromUtf8(":/images/images/arrowright.png"), QSize(), QIcon::Normal, QIcon::Off);
        buttonLeft_to_right->setIcon(icon);
        QIcon icon2;
        icon2.addFile(QString::fromUtf8(":/images/images/arrowleft.png"), QSize(), QIcon::Normal, QIcon::Off);
        buttonRight_to_left->setIcon(icon2);
    }
}

void MainWindow::createLangCodes()
{
    QMap<QString, QString> langCodes;

    langCodes[tr("Afrikaans")] = "af";
    langCodes[tr("Albanian")] = "sq";
    langCodes[tr("Arabic")] = "ar";
    langCodes[tr("Bulgarian")] = "bg";
    langCodes[tr("Catalan")] = "ca";
    langCodes[tr("Chinese")] = "zh-CN";
    langCodes[tr("Croatian")] = "hr";
    langCodes[tr("Czech")] = "cs";
    langCodes[tr("Dutch")] = "nl";
    langCodes[tr("English")] = "en";
    langCodes[tr("Estonian")] = "et";
    langCodes[tr("Farsi")] = "fa";
    langCodes[tr("Finnish")] = "fi";
    langCodes[tr("French")] = "fr";
    langCodes[tr("Galician")] = "gl";
    langCodes[tr("German")] = "de";
    langCodes[tr("Greek")] = "el";
    langCodes[tr("Hebrew")] = "iw";
    langCodes[tr("Hindi")] = "hi";
    langCodes[tr("Hungarian")] = "hu";
    langCodes[tr("Icelandic")] = "is";
    langCodes[tr("Indonesian")] = "id";
    langCodes[tr("Irish")] = "ga";
    langCodes[tr("Italian")] = "it";
    langCodes[tr("Japanese")] = "ja";
    langCodes[tr("Korean")] = "ko";
    langCodes[tr("Lettic")] = "lv";
    langCodes[tr("Lithuanian")] = "lt";
    langCodes[tr("Macedonian")] = "mk";
    langCodes[tr("Malay")] = "ms";
    langCodes[tr("Maltese")] = "mt";
    langCodes[tr("Norwegian")] = "no";
    langCodes[tr("Polish")] = "pl";
    langCodes[tr("Portuguese")] = "pt";
    langCodes[tr("Romanian")] = "ro";
    langCodes[tr("Russian")] = "ru";
    langCodes[tr("Serbian")] = "sr";
    langCodes[tr("Slovakian")] = "sk";
    langCodes[tr("Slovenian")] = "sl";
    langCodes[tr("Spanish")] = "es";
    langCodes[tr("Swahili")] = "sw";
    langCodes[tr("Swedish")] = "sv";
    langCodes[tr("Tagalog")] = "tl";
    langCodes[tr("Thai")] = "th";
    langCodes[tr("Turkish")] = "tr";
    langCodes[tr("Ukrainian")] = "uk";
    langCodes[tr("Vietnamese")] = "vi";
    langCodes[tr("Welsh")] = "cy";
    langCodes[tr("Yiddish")] = "yi";

    QMapIterator<QString, QString> i(langCodes);
    while (i.hasNext()) {
         i.next();
         comboboxLang_left->addItem(i.key(), i.value());
         comboboxLang_right->addItem(i.key(), i.value());
    }
}

void MainWindow::translateRequestFinished(QNetworkReply * reply)
 {
    if (reply->error() != QNetworkReply::NoError) {
        QMessageBox::information(this, tr("HTTP"),
                                  tr("Could not connect to the translation server. Please check that offline mode is disabled and device is able to connect to a network. (%1)")
                                  .arg(reply->errorString()));
    } else {
        bool ok = false;
        QString jsonTranslation = unescapeJson(reply->readAll(), &ok).trimmed();
        //qDebug() << jsonTranslation;
        QString translation = QString("");
        QRegExp regexp = QRegExp("\"translatedText\":\"(.*)\"");
        regexp.setPatternSyntax(QRegExp::RegExp2);
        regexp.setMinimal(true);
        if (jsonTranslation.contains(regexp))
        {
            translation = regexp.cap(1);
            translation = unescapeXML(translation);
            //qDebug() << translation;
            //translation = translation.fromUtf8(translation);
        }
        else
        {
            //qDebug("not found");
        }

        if (_toWindow == 0)
            texteditLeft->setText(translation);
        else
            texteditRight->setText(translation);
    }
    reply->deleteLater();
}

