/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtgui_python.h"

#include "qgraphicslayoutitem_wrapper.h"

// Extra includes
#include <qgraphicsitem.h>
#include <qgraphicslayoutitem.h>
#include <qrect.h>
#include <qsize.h>
#include <qsizepolicy.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QGraphicsLayoutItemWrapper::QGraphicsLayoutItemWrapper(QGraphicsLayoutItem * parent, bool isLayout) : QGraphicsLayoutItem(parent, isLayout) {
    // ... middle
}

void QGraphicsLayoutItemWrapper::setGeometry(const QRectF & rect)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "setGeometry"));
    if (py_override.isNull()) {
        return this->QGraphicsLayoutItem::setGeometry(rect);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QRectF & >::toPython(rect)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

QSizeF QGraphicsLayoutItemWrapper::sizeHint(Qt::SizeHint which, const QSizeF & constraint) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "sizeHint"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QGraphicsLayoutItem.sizeHint()' not implemented.");
        return QSizeF();
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<Qt::SizeHint >::toPython(which),
        Shiboken::Converter<QSizeF & >::toPython(constraint)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QSizeF();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QSizeF >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsLayoutItem.sizeHint", SbkType<QSizeF >()->tp_name, py_result->ob_type->tp_name);
        return QSizeF();
    }
    QSizeF cpp_result(Shiboken::Converter<QSizeF >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsLayoutItemWrapper::updateGeometry()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "updateGeometry"));
    if (py_override.isNull()) {
        return this->QGraphicsLayoutItem::updateGeometry();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

QGraphicsLayoutItemWrapper::~QGraphicsLayoutItemWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQGraphicsLayoutItem_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QGraphicsLayoutItemWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);
    SbkBaseWrapperType* type = reinterpret_cast<SbkBaseWrapperType*>(self->ob_type);
    SbkBaseWrapperType* myType = reinterpret_cast<SbkBaseWrapperType*>(SbkPySide_QtGuiTypes[SBK_QGRAPHICSLAYOUTITEM_IDX]);
    if (type == myType) {
        PyErr_SetString(PyExc_NotImplementedError,
            "'QGraphicsLayoutItem' represents a C++ abstract class and cannot be instantiated");
        return -1;
    }


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "QGraphicsLayoutItem", 0, 2, &(pyargs[0]), &(pyargs[1])))
        return -1;

    if (numArgs == 0) {
        // QGraphicsLayoutItem(QGraphicsLayoutItem*,bool)
        cptr = new QGraphicsLayoutItemWrapper();
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(pyargs[0], self);
    } else if (Shiboken::Converter<QGraphicsLayoutItem * >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QGraphicsLayoutItem* cpp_arg0 = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // QGraphicsLayoutItem(QGraphicsLayoutItem*,bool)
            cptr = new QGraphicsLayoutItemWrapper(cpp_arg0);
            //CppGenerator::writeParentChildManagement
            Shiboken::setParent(pyargs[0], self);
        } else if (PyNumber_Check(pyargs[1])) {
            bool cpp_arg1 = Shiboken::Converter<bool >::toCpp(pyargs[1]);
            // QGraphicsLayoutItem(QGraphicsLayoutItem*,bool)
            cptr = new QGraphicsLayoutItemWrapper(cpp_arg0, cpp_arg1);
            //CppGenerator::writeParentChildManagement
            Shiboken::setParent(pyargs[0], self);
        } else goto SbkQGraphicsLayoutItem_Init_TypeError;
    } else goto SbkQGraphicsLayoutItem_Init_TypeError;

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QGraphicsLayoutItem >(), cptr)) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQGraphicsLayoutItem_Init_TypeError;

    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;

    SbkQGraphicsLayoutItem_Init_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsLayoutItem = None, bool = false", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsLayoutItem", overloads);
        return -1;
}

static PyObject* SbkQGraphicsLayoutItemFunc_contentsRect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // contentsRect()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    QRectF cpp_result = cppSelf->QGraphicsLayoutItem::contentsRect();
    py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_effectiveSizeHint(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "effectiveSizeHint", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<Qt::SizeHint >::isConvertible(pyargs[0])) {
        Qt::SizeHint cpp_arg0 = Shiboken::Converter<Qt::SizeHint >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // effectiveSizeHint(Qt::SizeHint,QSizeF)const
            QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
            QSizeF cpp_result = cppSelf->QGraphicsLayoutItem::effectiveSizeHint(cpp_arg0);
            py_result = Shiboken::Converter<QSizeF >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QSizeF & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QSizeF > cpp_arg1_auto_ptr;
            QSizeF* cpp_arg1 = Shiboken::Converter<QSizeF* >::toCpp(pyargs[1]);
            if (!SbkQSizeF_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QSizeF >(cpp_arg1);
            // effectiveSizeHint(Qt::SizeHint,QSizeF)const
            QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
            QSizeF cpp_result = cppSelf->QGraphicsLayoutItem::effectiveSizeHint(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QSizeF >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQGraphicsLayoutItemFunc_effectiveSizeHint_TypeError;
    } else goto SbkQGraphicsLayoutItemFunc_effectiveSizeHint_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQGraphicsLayoutItemFunc_effectiveSizeHint_TypeError:
        const char* overloads[] = {"PySide.QtCore.Qt.SizeHint, PySide.QtCore.QSizeF = QSizeF()", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsLayoutItem.effectiveSizeHint", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_geometry(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // geometry()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    QRectF cpp_result = cppSelf->QGraphicsLayoutItem::geometry();
    py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_graphicsItem(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // graphicsItem()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    QGraphicsItem * cpp_result = cppSelf->QGraphicsLayoutItem::graphicsItem();
    py_result = Shiboken::Converter<QGraphicsItem * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_isLayout(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isLayout()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsLayoutItem::isLayout();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_maximumHeight(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // maximumHeight()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsLayoutItem::maximumHeight();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_maximumSize(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // maximumSize()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    QSizeF cpp_result = cppSelf->QGraphicsLayoutItem::maximumSize();
    py_result = Shiboken::Converter<QSizeF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_maximumWidth(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // maximumWidth()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsLayoutItem::maximumWidth();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_minimumHeight(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // minimumHeight()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsLayoutItem::minimumHeight();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_minimumSize(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // minimumSize()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    QSizeF cpp_result = cppSelf->QGraphicsLayoutItem::minimumSize();
    py_result = Shiboken::Converter<QSizeF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_minimumWidth(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // minimumWidth()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsLayoutItem::minimumWidth();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_ownedByLayout(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // ownedByLayout()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsLayoutItem::ownedByLayout();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_parentLayoutItem(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // parentLayoutItem()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    QGraphicsLayoutItem * cpp_result = cppSelf->QGraphicsLayoutItem::parentLayoutItem();
    py_result = Shiboken::Converter<QGraphicsLayoutItem * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_preferredHeight(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // preferredHeight()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsLayoutItem::preferredHeight();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_preferredSize(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // preferredSize()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    QSizeF cpp_result = cppSelf->QGraphicsLayoutItem::preferredSize();
    py_result = Shiboken::Converter<QSizeF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_preferredWidth(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // preferredWidth()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsLayoutItem::preferredWidth();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setGeometry(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QRectF & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(arg);
        if (!SbkQRectF_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // setGeometry(QRectF)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setGeometry(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setGeometry_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setGeometry_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setGeometry", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setGraphicsItem(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsItem * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // setGraphicsItem(QGraphicsItem*)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setGraphicsItem(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setGraphicsItem_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setGraphicsItem_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setGraphicsItem", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setMaximumHeight(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setMaximumHeight(qreal)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setMaximumHeight(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setMaximumHeight_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setMaximumHeight_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setMaximumHeight", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setMaximumSize(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setMaximumSize", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // setMaximumSize(qreal,qreal)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setMaximumSize(cpp_arg0, cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QSizeF & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QSizeF > cpp_arg0_auto_ptr;
        QSizeF* cpp_arg0 = Shiboken::Converter<QSizeF* >::toCpp(pyargs[0]);
        if (!SbkQSizeF_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QSizeF >(cpp_arg0);
        // setMaximumSize(QSizeF)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setMaximumSize(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setMaximumSize_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setMaximumSize_TypeError:
        const char* overloads[] = {"PySide.QtCore.QSizeF", "float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsLayoutItem.setMaximumSize", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setMaximumWidth(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setMaximumWidth(qreal)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setMaximumWidth(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setMaximumWidth_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setMaximumWidth_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setMaximumWidth", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setMinimumHeight(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setMinimumHeight(qreal)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setMinimumHeight(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setMinimumHeight_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setMinimumHeight_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setMinimumHeight", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setMinimumSize(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setMinimumSize", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // setMinimumSize(qreal,qreal)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setMinimumSize(cpp_arg0, cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QSizeF & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QSizeF > cpp_arg0_auto_ptr;
        QSizeF* cpp_arg0 = Shiboken::Converter<QSizeF* >::toCpp(pyargs[0]);
        if (!SbkQSizeF_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QSizeF >(cpp_arg0);
        // setMinimumSize(QSizeF)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setMinimumSize(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setMinimumSize_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setMinimumSize_TypeError:
        const char* overloads[] = {"PySide.QtCore.QSizeF", "float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsLayoutItem.setMinimumSize", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setMinimumWidth(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setMinimumWidth(qreal)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setMinimumWidth(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setMinimumWidth_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setMinimumWidth_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setMinimumWidth", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setOwnedByLayout(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setOwnedByLayout(bool)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setOwnedByLayout(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setOwnedByLayout_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setOwnedByLayout_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setOwnedByLayout", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setParentLayoutItem(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsLayoutItem * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsLayoutItem* cpp_arg0 = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp(arg);
        // setParentLayoutItem(QGraphicsLayoutItem*)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setParentLayoutItem(cpp_arg0);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, arg);
    } else goto SbkQGraphicsLayoutItemFunc_setParentLayoutItem_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setParentLayoutItem_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsLayoutItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setParentLayoutItem", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setPreferredHeight(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setPreferredHeight(qreal)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setPreferredHeight(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setPreferredHeight_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setPreferredHeight_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setPreferredHeight", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setPreferredSize(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setPreferredSize", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // setPreferredSize(qreal,qreal)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setPreferredSize(cpp_arg0, cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QSizeF & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QSizeF > cpp_arg0_auto_ptr;
        QSizeF* cpp_arg0 = Shiboken::Converter<QSizeF* >::toCpp(pyargs[0]);
        if (!SbkQSizeF_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QSizeF >(cpp_arg0);
        // setPreferredSize(QSizeF)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setPreferredSize(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setPreferredSize_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setPreferredSize_TypeError:
        const char* overloads[] = {"PySide.QtCore.QSizeF", "float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsLayoutItem.setPreferredSize", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setPreferredWidth(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setPreferredWidth(qreal)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setPreferredWidth(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsLayoutItemFunc_setPreferredWidth_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setPreferredWidth_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsLayoutItem.setPreferredWidth", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_setSizePolicy(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "setSizePolicy", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 1 && Shiboken::Converter<QSizePolicy & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QSizePolicy* cpp_arg0 = Shiboken::Converter<QSizePolicy* >::toCpp(pyargs[0]);
        // setSizePolicy(QSizePolicy)
        QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsLayoutItem::setSizePolicy(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QSizePolicy::Policy >::isConvertible(pyargs[0]) && Shiboken::Converter<QSizePolicy::Policy >::isConvertible(pyargs[1])) {
        QSizePolicy::Policy cpp_arg0 = Shiboken::Converter<QSizePolicy::Policy >::toCpp(pyargs[0]);
        QSizePolicy::Policy cpp_arg1 = Shiboken::Converter<QSizePolicy::Policy >::toCpp(pyargs[1]);
        if (numArgs == 2) {
            // setSizePolicy(QSizePolicy::Policy,QSizePolicy::Policy,QSizePolicy::ControlType)
            QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsLayoutItem::setSizePolicy(cpp_arg0, cpp_arg1);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QSizePolicy::ControlType >::isConvertible(pyargs[2])) {
            QSizePolicy::ControlType cpp_arg2 = Shiboken::Converter<QSizePolicy::ControlType >::toCpp(pyargs[2]);
            // setSizePolicy(QSizePolicy::Policy,QSizePolicy::Policy,QSizePolicy::ControlType)
            QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsLayoutItem::setSizePolicy(cpp_arg0, cpp_arg1, cpp_arg2);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQGraphicsLayoutItemFunc_setSizePolicy_TypeError;
    } else goto SbkQGraphicsLayoutItemFunc_setSizePolicy_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsLayoutItemFunc_setSizePolicy_TypeError:
        const char* overloads[] = {"PySide.QtGui.QSizePolicy.Policy, PySide.QtGui.QSizePolicy.Policy, PySide.QtGui.QSizePolicy.ControlType = QSizePolicy.DefaultType", "PySide.QtGui.QSizePolicy", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsLayoutItem.setSizePolicy", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_sizeHint(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "sizeHint", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<Qt::SizeHint >::isConvertible(pyargs[0])) {
        Qt::SizeHint cpp_arg0 = Shiboken::Converter<Qt::SizeHint >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // sizeHint(Qt::SizeHint,QSizeF)const
            if (SbkBaseWrapper_containsCppWrapper(self)) {
                PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QGraphicsLayoutItem.sizeHint()' not implemented.");
                return 0;
            }
            QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
            QSizeF cpp_result = cppSelf->sizeHint(cpp_arg0);
            py_result = Shiboken::Converter<QSizeF >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QSizeF & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QSizeF > cpp_arg1_auto_ptr;
            QSizeF* cpp_arg1 = Shiboken::Converter<QSizeF* >::toCpp(pyargs[1]);
            if (!SbkQSizeF_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QSizeF >(cpp_arg1);
            // sizeHint(Qt::SizeHint,QSizeF)const
            if (SbkBaseWrapper_containsCppWrapper(self)) {
                PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QGraphicsLayoutItem.sizeHint()' not implemented.");
                return 0;
            }
            QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
            QSizeF cpp_result = cppSelf->sizeHint(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QSizeF >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQGraphicsLayoutItemFunc_sizeHint_TypeError;
    } else goto SbkQGraphicsLayoutItemFunc_sizeHint_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQGraphicsLayoutItemFunc_sizeHint_TypeError:
        const char* overloads[] = {"PySide.QtCore.Qt.SizeHint, PySide.QtCore.QSizeF = QSizeF()", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsLayoutItem.sizeHint", overloads);
        return 0;
}

static PyObject* SbkQGraphicsLayoutItemFunc_sizePolicy(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // sizePolicy()const
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    QSizePolicy cpp_result = cppSelf->QGraphicsLayoutItem::sizePolicy();
    py_result = Shiboken::Converter<QSizePolicy >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsLayoutItemFunc_updateGeometry(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // updateGeometry()
    QGraphicsLayoutItem* cppSelf = Shiboken::Converter<QGraphicsLayoutItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsLayoutItem::updateGeometry();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyMethodDef SbkQGraphicsLayoutItem_methods[] = {
    {"contentsRect", (PyCFunction)SbkQGraphicsLayoutItemFunc_contentsRect, METH_NOARGS},
    {"effectiveSizeHint", (PyCFunction)SbkQGraphicsLayoutItemFunc_effectiveSizeHint, METH_VARARGS},
    {"geometry", (PyCFunction)SbkQGraphicsLayoutItemFunc_geometry, METH_NOARGS},
    {"graphicsItem", (PyCFunction)SbkQGraphicsLayoutItemFunc_graphicsItem, METH_NOARGS},
    {"isLayout", (PyCFunction)SbkQGraphicsLayoutItemFunc_isLayout, METH_NOARGS},
    {"maximumHeight", (PyCFunction)SbkQGraphicsLayoutItemFunc_maximumHeight, METH_NOARGS},
    {"maximumSize", (PyCFunction)SbkQGraphicsLayoutItemFunc_maximumSize, METH_NOARGS},
    {"maximumWidth", (PyCFunction)SbkQGraphicsLayoutItemFunc_maximumWidth, METH_NOARGS},
    {"minimumHeight", (PyCFunction)SbkQGraphicsLayoutItemFunc_minimumHeight, METH_NOARGS},
    {"minimumSize", (PyCFunction)SbkQGraphicsLayoutItemFunc_minimumSize, METH_NOARGS},
    {"minimumWidth", (PyCFunction)SbkQGraphicsLayoutItemFunc_minimumWidth, METH_NOARGS},
    {"ownedByLayout", (PyCFunction)SbkQGraphicsLayoutItemFunc_ownedByLayout, METH_NOARGS},
    {"parentLayoutItem", (PyCFunction)SbkQGraphicsLayoutItemFunc_parentLayoutItem, METH_NOARGS},
    {"preferredHeight", (PyCFunction)SbkQGraphicsLayoutItemFunc_preferredHeight, METH_NOARGS},
    {"preferredSize", (PyCFunction)SbkQGraphicsLayoutItemFunc_preferredSize, METH_NOARGS},
    {"preferredWidth", (PyCFunction)SbkQGraphicsLayoutItemFunc_preferredWidth, METH_NOARGS},
    {"setGeometry", (PyCFunction)SbkQGraphicsLayoutItemFunc_setGeometry, METH_O},
    {"setGraphicsItem", (PyCFunction)SbkQGraphicsLayoutItemFunc_setGraphicsItem, METH_O},
    {"setMaximumHeight", (PyCFunction)SbkQGraphicsLayoutItemFunc_setMaximumHeight, METH_O},
    {"setMaximumSize", (PyCFunction)SbkQGraphicsLayoutItemFunc_setMaximumSize, METH_VARARGS},
    {"setMaximumWidth", (PyCFunction)SbkQGraphicsLayoutItemFunc_setMaximumWidth, METH_O},
    {"setMinimumHeight", (PyCFunction)SbkQGraphicsLayoutItemFunc_setMinimumHeight, METH_O},
    {"setMinimumSize", (PyCFunction)SbkQGraphicsLayoutItemFunc_setMinimumSize, METH_VARARGS},
    {"setMinimumWidth", (PyCFunction)SbkQGraphicsLayoutItemFunc_setMinimumWidth, METH_O},
    {"setOwnedByLayout", (PyCFunction)SbkQGraphicsLayoutItemFunc_setOwnedByLayout, METH_O},
    {"setParentLayoutItem", (PyCFunction)SbkQGraphicsLayoutItemFunc_setParentLayoutItem, METH_O},
    {"setPreferredHeight", (PyCFunction)SbkQGraphicsLayoutItemFunc_setPreferredHeight, METH_O},
    {"setPreferredSize", (PyCFunction)SbkQGraphicsLayoutItemFunc_setPreferredSize, METH_VARARGS},
    {"setPreferredWidth", (PyCFunction)SbkQGraphicsLayoutItemFunc_setPreferredWidth, METH_O},
    {"setSizePolicy", (PyCFunction)SbkQGraphicsLayoutItemFunc_setSizePolicy, METH_VARARGS},
    {"sizeHint", (PyCFunction)SbkQGraphicsLayoutItemFunc_sizeHint, METH_VARARGS},
    {"sizePolicy", (PyCFunction)SbkQGraphicsLayoutItemFunc_sizePolicy, METH_NOARGS},
    {"updateGeometry", (PyCFunction)SbkQGraphicsLayoutItemFunc_updateGeometry, METH_NOARGS},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQGraphicsLayoutItem_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtGui.QGraphicsLayoutItem",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQGraphicsLayoutItem_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQGraphicsLayoutItem_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QGraphicsLayoutItem >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static SbkBaseWrapperType* SbkQGraphicsLayoutItem_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkQGraphicsLayoutItem_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<QGraphicsLayoutItem*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}


PyAPI_FUNC(void) init_QGraphicsLayoutItem(PyObject* module)
{
    SbkPySide_QtGuiTypes[SBK_QGRAPHICSLAYOUTITEM_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQGraphicsLayoutItem_Type);

    // Fill type discovery information
    SbkQGraphicsLayoutItem_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkQGraphicsLayoutItem_Type.type_discovery->addTypeDiscoveryFunction(&SbkQGraphicsLayoutItem_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkQGraphicsLayoutItem_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQGraphicsLayoutItem_Type));
    PyModule_AddObject(module, "QGraphicsLayoutItem",
        ((PyObject*)&SbkQGraphicsLayoutItem_Type));

    Shiboken::TypeResolver::createObjectTypeResolver<QGraphicsLayoutItem >("QGraphicsLayoutItem*");
    Shiboken::TypeResolver::createObjectTypeResolver<QGraphicsLayoutItem >(typeid(QGraphicsLayoutItem).name());
}


} // extern "C"

