#! /usr/bin/env python
# -*- coding: UTF-8 -*-

import os, sys, time, dbus, telepathy
import telepathy_support as service #TODO: Update

from PySide.QtCore import *
from PySide.QtGui  import *

from config import *

class AccountButton(QCheckBox):
  def __init__(self, account, name, icon):
    QCheckBox.__init__(self, name)

    self.setProperty('account', account)
    self.setIcon(QIcon(os.path.join(HILDON_ICON_PATH, icon + '.png')))

class Form(QScrollArea):
  def __init__(self, bus, config, media_profile = u'Music'):
    QScrollArea.__init__(self)

    self.setWidgetResizable(True)

    self.__bus      = bus
    self.__config   = config
    self.__buttons  = list()
    self.__updating = False

    self.vbox       = QVBoxLayout()
    self.contents   = QWidget()

    self.enabledBtn = QCheckBox(u'Enabled')
    self.enabledBtn.setChecked(self.__config.is_enabled(media_profile))
    self.vbox.addWidget(self.enabledBtn)

    self.messageEdt = QLineEdit()
    self.messageEdt.setText(self.__config.template(media_profile))
    self.messageEdt.setEnabled(self.enabledBtn.isChecked())
    self.vbox.addWidget(self.messageEdt)

    QObject.connect(self.messageEdt, SIGNAL('textEdited(const QString &)'), self.update_template)

    QObject.connect(self.enabledBtn, SIGNAL('toggled(bool)'),
                    lambda v: self.__config.set_enabled(self.current_profile, v, False))

    QObject.connect(self.enabledBtn, SIGNAL('toggled(bool)'),
                    self.messageEdt, SLOT('setEnabled(bool)'))

    connections = ConfigParser.RawConfigParser()
    connections.read(RTCOM_ACCOUNTS_CONFIG)

    am          = self.__bus.get_object(telepathy.ACCOUNT_MANAGER,
                                        service.TELEPATHY_ACCOUNT_MANAGER_PATH)
    am_iface    = dbus.Interface(am, dbus.PROPERTIES_IFACE)

    for account in am_iface.Get(telepathy.ACCOUNT_MANAGER, 'ValidAccounts'):
      ac_name   = account.replace(service.TELEPATHY_ACCOUNT_PREFIX, '')
      pf_name   = connections.get(ac_name, 'Profile')

      ac        = bus.get_object(telepathy.ACCOUNT_MANAGER, account)
      ac_iface  = dbus.Interface(ac, dbus.PROPERTIES_IFACE)

      if IGNORE_ACCOUNTS.count(ac_name) > 0:
        continue

      profile   = ConfigParser.RawConfigParser()
      profile.read(os.path.join(RTCOM_PROFILES_PATH, pf_name + '.profile'))

      button    = AccountButton(ac_name,
                                ac_iface.Get(telepathy.ACCOUNT, 'NormalizedName'),
                                profile.get('Profile', 'IconName'))

      button.setChecked(config.accounts(media_profile).count(ac_name) > 0)
      button.setEnabled(self.enabledBtn.isChecked())

      QObject.connect(self.enabledBtn, SIGNAL('toggled(bool)'),
                      button,          SLOT('setEnabled(bool)'))

      QObject.connect(button, SIGNAL('toggled(bool)'),
                      enclosure(self.__update_account, ac_name))

      self.__buttons.append(button)
      self.vbox.addWidget(button)

    self.current_profile = media_profile

    self.contents.setLayout(self.vbox)
    self.setWidget(self.contents)

  def set_media_profile(self, media_profile):
    self.__updating       = True
    self.current_profile  = media_profile

    self.enabledBtn.setChecked(self.__config.is_enabled(media_profile))
    self.messageEdt.setText(self.__config.template(media_profile))
    self.messageEdt.setEnabled(self.enabledBtn.isChecked())

    for button in self.__buttons:
      button.setChecked(self.__config.accounts(media_profile).count(button.property('account')) > 0)
      button.setEnabled(self.enabledBtn.isChecked())

    self.__updating = False

  def __update_account(self, state, account):
    #   Test to see if we're switching profiles to avoid erroneous reconfiguring
    # during this process.
    if not self.__updating:
      accounts = self.__config.accounts(self.current_profile)

      if state:
        accounts.append(account)
      else:
        accounts.remove(account)

  def update_template(self, text):
    if not self.__updating:
      self.__config.set_template(self.current_profile, str(text), False)

class MediaSelector(QWidget):
  def __init__(self, controller):
    QWidget.__init__(self)

    self.hbox     = QHBoxLayout()
    self.buttons  = list()

    for profile,icon,enabled,template in MEDIA_PROFILES:
      button = QPushButton(profile)
      button.setCheckable(True)
      button.setIcon(QIcon(os.path.join(HILDON_ICON_PATH, icon + '.png')))
      button.setAutoExclusive(True)

      self.buttons.append(button)
      self.hbox.addWidget(button)

      QObject.connect(button, SIGNAL('clicked()'),
                      enclosure(controller.show_media_profile, profile))

    self.buttons[0].setChecked(True)
    self.setLayout(self.hbox)

class DialogContent(QWidget):
  def __init__(self, bus, config):
    QWidget.__init__(self)

    self.vbox           = QVBoxLayout()
    self.form           = Form(bus, config)
    self.media_selector = MediaSelector(self)

    self.vbox.setContentsMargins(0, 0, 0, 0)

    self.vbox.addWidget(self.media_selector)
    self.vbox.addWidget(self.form)

    self.setLayout(self.vbox)

  def show_media_profile(self, profile):
    self.form.set_media_profile(profile)

class DialogButtons(QDialogButtonBox):
  def __init__(self):
    QDialogButtonBox.__init__(self, Qt.Vertical)

    self.donate_btn = self.addButton(u'Donate', QDialogButtonBox.AcceptRole)
    self.save_btn   = self.addButton(u'Save',   QDialogButtonBox.AcceptRole)

class Dialog(QDialog):
  def __init__(self, bus, config):
    QDialog.__init__(self)

    self.setWindowTitle(u'Media IM Status Updater')

    self.bus      = bus
    self.config   = config

    self.hbox     = QHBoxLayout()
    self.content  = DialogContent(bus, config)
    self.buttons  = DialogButtons()

    QObject.connect(self.buttons.save_btn,    SIGNAL('clicked()'), self.do_save)
    QObject.connect(self.buttons.donate_btn,  SIGNAL('clicked()'), self.do_donate)

    self.hbox.addWidget(self.content)
    self.hbox.addWidget(self.buttons)

    self.setLayout(self.hbox)

  def do_save(self):
    self.config.save()
    self.done(0)

  def do_donate(self):
    self.done(0)
    proxy = self.bus.get_object('com.nokia.osso_browser', '/com/nokia/osso_browser/request')
    iface = dbus.Interface(proxy, 'com.nokia.osso_browser')
    iface.top_application()
    iface.load_url('https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=RLC92AUZKFPU2')

if __name__ == '__main__':
  import dbus.glib

  DEBUG = True

  config  = Config(CONFIG_PATH)
  config.load()

  bus     = dbus.SessionBus()
  app     = QApplication(sys.argv)
  dialog  = Dialog(bus, config)

  dialog.show()

  sys.exit(dialog.exec_())

