#! /usr/bin/env python
# -*- coding: UTF-8 -*-

import os, sys, dbus, telepathy, ConfigParser

import media_im_status_updater as service

from PySide.QtCore  import *
from PySide.QtGui   import *

RTCOM_PROFILES_PATH   = '/usr/share/osso-rtcom'
RTCOM_ACCOUNTS_CONFIG = '/home/user/.rtcom-accounts/accounts.cfg'

HILDON_ICON_PATH      = '/opt/usr/share/icons/hicolor/48x48/hildon'

class AccountButton(QCheckBox):
  def __init__(self, account, name, icon):
    QCheckBox.__init__(self, name)

    self.setProperty('account', account)
    self.setIcon(QIcon(os.path.join(HILDON_ICON_PATH, icon + '.png')))


class Form(QScrollArea):
  __config  = None
  __buttons = None

  def __init__(self, bus, config):
    QScrollArea.__init__(self)

    self.setWidgetResizable(True)

    self.__config   = config

    self.vbox       = QVBoxLayout()
    self.contents   = QWidget()

    self.__buttons  = list()

    self.enabledBtn = QCheckBox(u'Enabled')
    self.enabledBtn.setChecked(config.is_enabled())
    self.vbox.addWidget(self.enabledBtn)

    self.messageEdt = QLineEdit()
    self.messageEdt.setText(config.template())
    self.messageEdt.setEnabled(self.enabledBtn.isChecked())
    self.vbox.addWidget(self.messageEdt)

    self.connect(self.enabledBtn, SIGNAL('toggled(bool)'),
                 self.messageEdt, SLOT('setEnabled(bool)'))

    connections = ConfigParser.RawConfigParser()
    connections.read(RTCOM_ACCOUNTS_CONFIG)

    am        = bus.get_object(telepathy.ACCOUNT_MANAGER,
                               service.TELEPATHY_ACCOUNT_MANAGER_PATH)

    am_iface  = dbus.Interface(am, dbus.PROPERTIES_IFACE)

    for account in am_iface.Get(telepathy.ACCOUNT_MANAGER, 'ValidAccounts'):
      ac_name   = account.replace(service.TELEPATHY_ACCOUNT_PREFIX, '')
      pf_name   = connections.get(ac_name, 'Profile')

      ac        = bus.get_object(telepathy.ACCOUNT_MANAGER, account)
      ac_iface  = dbus.Interface(ac, dbus.PROPERTIES_IFACE)

      # Skip GSM phone "account".
      if ac_name == 'ring/tel/ring':
        continue

      profile = ConfigParser.RawConfigParser()
      profile.read(os.path.join(RTCOM_PROFILES_PATH, pf_name + '.profile'))

      button  = AccountButton(ac_name,
                              ac_iface.Get(telepathy.ACCOUNT, 'NormalizedName'),
                              profile.get('Profile', 'IconName'))
      button.setChecked(config.accounts().count(ac_name) > 0)
      button.setEnabled(self.enabledBtn.isChecked())

      self.connect(self.enabledBtn, SIGNAL('toggled(bool)'),
                   button, SLOT('setEnabled(bool)'))

      self.__buttons.append(button)
      self.vbox.addWidget(button)

    self.contents.setLayout(self.vbox)
    self.setWidget(self.contents)

  def save_config(self):
    accounts = list()

    for button in self.__buttons:
      if button.isChecked():
        accounts.append(str(button.property('account').toString()))

    self.__config.set_enabled(self.enabledBtn.isChecked(),  save = False)
    self.__config.set_template(self.messageEdt.text(),      save = False)
    self.__config.set_accounts(accounts,                    save = False)
    self.__config.save()


class Dialog(QDialog):
  __config  = None

  def __init__(self, bus, config):
    QDialog.__init__(self)

    self.__config = config

    self.hbox     = QHBoxLayout()

    self.form     = Form(bus, config)
    self.hbox.addWidget(self.form)

    self.btnBox   = QDialogButtonBox(Qt.Vertical)
    self.saveBtn  = QPushButton(u'Save')

    self.saveBtn.setDefault(True)

    self.btnBox.addButton(self.saveBtn, QDialogButtonBox.ActionRole)
    self.hbox.addWidget(self.btnBox)

    self.setWindowTitle(u'Media IM Status Updater')
    self.setLayout(self.hbox)

    self.connect(self.saveBtn, SIGNAL('clicked()'), self, SLOT('do_save()'))

  def do_save(self):
    self.form.save_config()
    self.done(0)

if __name__ == '__main__':
  config      = service.Config(service.CONFIG_PATH)
  config.load()

  session_bus = dbus.SessionBus()
  application = QApplication(sys.argv)

  dialog      = Dialog(session_bus, config)
  dialog.show()

  sys.exit(dialog.exec_())

