/*
 * gpstest.c
 *
 * simple tool to fake a gpsd
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <math.h>

#include <sys/socket.h> /* for socket(), bind(), and connect() */
#include <arpa/inet.h>  /* for sockaddr_in and inet_ntoa() */
#include <string.h>     /* for memset() */

typedef struct {
  double latitude, longitude;
} pos_t;

#define NAN (0.0/0.0)

GtkWidget *window = NULL;
GThread* thread_p = NULL;
GMutex *mutex = NULL;

/* initial position */
pos_t cur_pos = { 48.8812, 8.50773 };
float altitude = 100.1;
gboolean fix = TRUE, fix3d = TRUE;

#define PORT  2947 

pos_t geomath_translate(pos_t p, double dist, double angle) {
  pos_t result;

  double gcrad = 6371000.0;  // great circle in meters

  // from: http://www.movable-type.co.uk/scripts/latlong.html
  result.latitude = asin(sin(p.latitude/180*M_PI) * cos(dist/gcrad) + 
		       cos(p.latitude/180*M_PI) * sin(dist/gcrad) * 
		       cos(angle/180*M_PI) )/M_PI*180;
  result.longitude = p.longitude + atan2(sin(angle/180*M_PI)*sin(dist/gcrad)*
				    cos(p.latitude/180*M_PI), 
				    cos(dist/gcrad)-sin(p.latitude/180*M_PI)*
				    sin(result.latitude/180*M_PI))/M_PI*180;
  // normalise to -180...+180
  result.longitude = fmodf(result.longitude+180,360) - 180;

  return result;
}

void pos_lon_str(char *str, int len, double longitude) {
  char c = 'E';
  double integral, fractional;

  if(longitude < 0) { longitude = fabs(longitude); c = 'W'; }
  fractional = modf(longitude, &integral);

  snprintf(str, len, "%c %03d° %06.3f'", c, (int)integral, fractional*60.0);
}

void pos_lat_str(char *str, int len, double latitude) {
  char c = 'N';
  double integral, fractional;

  if(latitude < 0) { latitude = fabs(latitude); c = 'S'; }
  fractional = modf(latitude, &integral);

  snprintf(str, len, "%c %02d° %06.3f'", c, (int)integral, fractional*60.0);
}

void pos_alt_str(char *str, int len, double altitude) {
  snprintf(str, len, "%.02fm", altitude);
}


GtkWidget *pos_lat(double latitude) {
  char str[32];
  pos_lat_str(str, sizeof(str), latitude);
  return gtk_label_new(str);
}

GtkWidget *pos_lon(double longitude) {
  char str[32];
  pos_lon_str(str, sizeof(str), longitude);
  return gtk_label_new(str);
}

GtkWidget *pos_alt(double altitude) {
  char str[32];
  pos_alt_str(str, sizeof(str), altitude);
  return gtk_label_new(str);
}

#define DEFAULT_DIST 10.0
#define DEFAULT_DIR   0.0
GtkWidget *navtool_dist, *navtool_dir, *navtool_go;
GtkWidget *navtool_lat, *navtool_lon, *navtool_fix, *navtool_fix3d;
GtkWidget *navtool_alt;

void go_clicked(GtkButton *button, gpointer data) {
  char str[32];
  float dist, dir;
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(navtool_dist));
  if(sscanf(p, "%f", &dist) != 1)
    dist = NAN;

  p = (char*)gtk_entry_get_text(GTK_ENTRY(navtool_dir));
  if(sscanf(p, "%f", &dir) != 1)
    dir = NAN;

  printf("%s: going %.2fm heading %.1f°\n", __func__, dist, dir);

  g_mutex_lock(mutex);
  cur_pos = geomath_translate(cur_pos, dist, dir);
  g_mutex_unlock(mutex);

  pos_lat_str(str, sizeof(str), cur_pos.latitude);
  gtk_label_set_text(GTK_LABEL(navtool_lat), str);
  pos_lon_str(str, sizeof(str), cur_pos.longitude);
  gtk_label_set_text(GTK_LABEL(navtool_lon), str);
}

void fix_toggled(GtkToggleButton *togglebutton, gpointer user_data) {
  fix =  gtk_toggle_button_get_active(togglebutton);
}

void fix3d_toggled(GtkToggleButton *togglebutton, gpointer user_data) {
  fix3d =  gtk_toggle_button_get_active(togglebutton);
}

GtkWidget *new_navtool(void) {
  char str[32];
  GtkWidget *hbox, *table = gtk_table_new(3,3,FALSE);

  hbox = gtk_hbox_new(FALSE,0);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), navtool_dist = gtk_entry_new());
  gtk_box_pack_start_defaults(GTK_BOX(hbox), gtk_label_new("m"));
  gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 1, 0, 1);
  snprintf(str, sizeof(str), "%.2f", DEFAULT_DIST);
  gtk_entry_set_text(GTK_ENTRY(navtool_dist), str);
  gtk_entry_set_width_chars(GTK_ENTRY(navtool_dist), 6);

  hbox = gtk_hbox_new(FALSE,0);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), navtool_dir = gtk_entry_new());
  gtk_box_pack_start_defaults(GTK_BOX(hbox), gtk_label_new("°"));
  gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 1, 1, 2);
  snprintf(str, sizeof(str), "%.1f", DEFAULT_DIR);
  gtk_entry_set_text(GTK_ENTRY(navtool_dir), str);
  gtk_entry_set_width_chars(GTK_ENTRY(navtool_dir), 6);

  gtk_table_attach_defaults(GTK_TABLE(table),
	    navtool_go = gtk_button_new_with_label("Go"), 1, 2, 0, 2);
  gtk_signal_connect(GTK_OBJECT(navtool_go), "clicked",
		     (GtkSignalFunc)go_clicked, NULL);

  gtk_table_attach_defaults(GTK_TABLE(table),
		    navtool_lat = pos_lat(cur_pos.latitude), 2, 3, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table),
		    navtool_lon = pos_lon(cur_pos.longitude), 2, 3, 1, 2);

  navtool_fix = gtk_check_button_new_with_label("Fix");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(navtool_fix), fix);
  g_signal_connect(navtool_fix, "toggled",
		   G_CALLBACK(fix_toggled), NULL);
  gtk_table_attach_defaults(GTK_TABLE(table),
			    navtool_fix, 0, 1, 2, 3);

  navtool_fix3d = gtk_check_button_new_with_label("Fix3d");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(navtool_fix3d), fix3d);
  g_signal_connect(navtool_fix3d, "toggled",
		   G_CALLBACK(fix3d_toggled), NULL);
  gtk_table_attach_defaults(GTK_TABLE(table),
			    navtool_fix3d, 1, 2, 2, 3);

  gtk_table_attach_defaults(GTK_TABLE(table),
		    navtool_alt = pos_alt(altitude), 2, 3, 2, 3);


  return table;
}

pos_t gui_navtool_get_pos(void) {
  g_mutex_lock(mutex);
  pos_t cpos = cur_pos;
  g_mutex_unlock(mutex);

  return cpos;
}


/*
 * Terminate the main loop.
 */
static void on_destroy (GtkWidget * widget, gpointer data) {
  gtk_main_quit ();
}

gpointer comm_thread(gpointer data) {
  char buf[1024];
  int sock;                 /* socket to create */
  struct sockaddr_in saddr; /* Local address */

  /* open a listen socket */
  /* Create socket for incoming connections */
  if ((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
    perror("socket()");
    exit(1);
  }
      
  /* Construct local address structure */
  memset(&saddr, 0, sizeof(saddr));   /* Zero out structure */
  saddr.sin_family = AF_INET;         /* Internet address family */
  saddr.sin_addr.s_addr = htonl(INADDR_ANY); /* Any incoming interface */
  saddr.sin_port = htons(PORT);              /* Local port */

  /* Bind to the local address */
  if (bind(sock, (struct sockaddr *) &saddr, sizeof(saddr)) < 0) {
    perror("bind()");
    exit(1);
  }

  /* Mark the socket so it will listen for incoming connections */
  if (listen(sock, 1) < 0) {
    perror("listen()");
    exit(1);
  }

  printf("Server listening on port %d\n", PORT);
    
  while(1) {
    printf("Waiting for client ...\n");

    int csock;                /* Socket descriptor for client */
    struct sockaddr_in caddr; /* Client address */
    unsigned int clen;        /* Length of client address data structure */

    /* Set the size of the in-out parameter */
    clen = sizeof(caddr);
    
    /* Wait for a client to connect */
    if((csock = accept(sock, (struct sockaddr *) &caddr, &clen)) < 0) {
      perror("accept()");
      exit(1);
    }
    
    printf("Connected by client %s\n", inet_ntoa(caddr.sin_addr)); 

    int reqlen = read(csock, buf, sizeof(buf));
    while(reqlen > 0) {
      buf[reqlen] = 0;
      printf("Got %d bytes request (%s)\n", reqlen, buf);
      
      int i;
      for(i=0;i<reqlen;i++) {
	switch(buf[i]) {
	  char sbuf[512];

	case 0:
	case '\r':
	case '\n':
	case '+':
	  break;
	  
	case 'o':
	case 'O':
	  { pos_t pos = gui_navtool_get_pos();

	  if(fix) {
	    char lat_str[G_ASCII_DTOSTR_BUF_SIZE];
	    char lon_str[G_ASCII_DTOSTR_BUF_SIZE];
	    g_ascii_dtostr(lat_str, sizeof (lat_str), pos.latitude);
	    g_ascii_dtostr(lon_str, sizeof (lon_str), pos.longitude);

	    char alt_str[G_ASCII_DTOSTR_BUF_SIZE];
	    g_ascii_dtostr(alt_str, sizeof (alt_str), altitude);
	    
	    // http://gpsd.berlios.de/gpsd.html
	    snprintf(sbuf, sizeof(sbuf), 
		     "GPSD O=XXX "  // TAG
		     "%d "          // timestr
		     "1 "           // ept, error time 
		     "%s %s "   // pos
		     "%s "          // alt
		     "10.0 100 " /* eph, epv */
		     "0.0 0.0 0.0 " /* track, speed, climb */
		     "1.0 1.0 1.0 " /* epd, eps, epc */
		     "3\n",  /* mode */
		     (int)time(NULL), /* timestr */
		     lat_str, lon_str,
		     fix3d?alt_str:"?");
	  } else
	    strcpy(sbuf, "GPSD O=?\n");

	  //	  printf("sending %s\n", sbuf);

	  write(csock, sbuf, strlen(sbuf));
	  } break;
	  
	  
	default:
	  printf("insupported request %c\n", buf[i]);
	  snprintf(sbuf, sizeof(sbuf), "GPSD %c=?\n", buf[i]);
	  write(csock, sbuf, strlen(sbuf));
	  break;
	}
      }
      reqlen = read(csock, buf, sizeof(buf));
    }

    perror("read()");

    printf("Closing connection\n");
    close(csock);
  }
}

int
main (int argc, char *argv[]) {
  
  gtk_init (&argc, &argv);

  g_thread_init(NULL);

  /* create the main, top level, window */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  
  /* give the window a 20px wide border */
  gtk_container_set_border_width (GTK_CONTAINER (window), 20);
  
  /* give it the title */
  gtk_window_set_title (GTK_WINDOW (window), APP);
  
  /* open it a bit wider so that both the label and title show up */
  //  gtk_window_set_default_size (GTK_WINDOW (window), 200, 300);

  /* Connect the destroy event of the window with our on_destroy function
   * When the window is about to be destroyed we get a notificaiton and
   * stop the main GTK loop
   */
  g_signal_connect(G_OBJECT(window), "destroy", G_CALLBACK(on_destroy), NULL);
  
  gtk_container_add(GTK_CONTAINER(window), new_navtool());

  /* make sure that everything, window and label, are visible */
  gtk_widget_show_all (window);

  /* start communication thread */
  mutex = g_mutex_new();
  thread_p = g_thread_create(comm_thread, NULL, FALSE, NULL);

  /* start the main loop */
  gtk_main();
  
  return 0;
}

