/*
 *  lenscover reminder statusbar plugin.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <fcntl.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#include <gconf/gconf-client.h>

#include "lens-cover-reminder-sp.h"

#define LENS_COVER_STATUS_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
									       TYPE_LENS_COVER_STATUS_PLUGIN, LensCoverStatusPluginPrivate))

#define GC_ROOT "/apps/maemo/lens-cover-reminder/"

struct _LensCoverStatusPluginPrivate
{
  DBusGConnection *dbus_conn;
  DBusGProxy *dbus_proxy_cover;
  DBusGProxy *dbus_proxy_lock;
  gchar* vibra_pattern;
  gchar* led_pattern;
};

HD_DEFINE_PLUGIN_MODULE(LensCoverStatusPlugin, lens_cover_status_plugin, HD_TYPE_STATUS_MENU_ITEM);

static gboolean
lens_cover_status_plugin_cover_is_open()
{
  int fd;
  char state[7];
  fd = open("/sys/devices/platform/gpio-switch/cam_shutter/state", O_RDONLY);
  if(fd)
  {
    read(fd, state, 7 );
    if(!g_str_has_prefix(state, "closed"))
    {
      return TRUE;
    }
    close( fd );
  }
  return FALSE;
}

static void
lens_cover_status_plugin_show_icon(LensCoverStatusPlugin* plugin)
{
  GdkPixbuf* pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), "statusarea_lens_cover_reminder", 18, 0, NULL);
  if(pixbuf)
  {
    hd_status_plugin_item_set_status_area_icon(HD_STATUS_PLUGIN_ITEM(plugin), pixbuf);
    g_object_unref(pixbuf);
  }
}

static void
lens_cover_status_plugin_hide_icon(LensCoverStatusPlugin* plugin)
{
  hd_status_plugin_item_set_status_area_icon(HD_STATUS_PLUGIN_ITEM(plugin), NULL);
}

static void
lens_cover_status_plugin_notification(LensCoverStatusPlugin* plugin,
				      gboolean activate)
{
  GConfClient* client = gconf_client_get_default();
  gchar* vibra_pattern = gconf_client_get_string(client, GC_ROOT "vibra", NULL);
  gchar* led_pattern = gconf_client_get_string(client, GC_ROOT "led", NULL);
  
  DBusGProxy* proxy = dbus_g_proxy_new_for_name(plugin->priv->dbus_conn,
						MCE_SERVICE,
						MCE_REQUEST_PATH,
						MCE_REQUEST_IF);
  if(proxy)
  {
    if(led_pattern != NULL && g_strcmp0(led_pattern, "disabled")!=0)
    {
      if(activate)
      {
	g_free(plugin->priv->led_pattern);
	plugin->priv->led_pattern = g_strdup(led_pattern);
	dbus_g_proxy_call_no_reply(proxy,
				   MCE_ACTIVATE_LED_PATTERN,
				   G_TYPE_STRING, plugin->priv->led_pattern,
				   G_TYPE_INVALID);
      }
      else
      {
	dbus_g_proxy_call_no_reply(proxy,
				   MCE_DEACTIVATE_LED_PATTERN,
				   G_TYPE_STRING, plugin->priv->led_pattern,
				   G_TYPE_INVALID);
	g_free(plugin->priv->led_pattern);
	plugin->priv->led_pattern = NULL;
      }
    }
    if(vibra_pattern == NULL)
    {
      vibra_pattern = g_strdup("PatternIncomingMessage");
    }
    if(g_strcmp0(vibra_pattern, "disabled")!=0)
    {
      if(activate)
      {
	g_free(plugin->priv->vibra_pattern);
	plugin->priv->vibra_pattern = g_strdup(vibra_pattern);
	dbus_g_proxy_call_no_reply(proxy,
				   MCE_ACTIVATE_VIBRATOR_PATTERN,
				   G_TYPE_STRING, plugin->priv->vibra_pattern,
				   G_TYPE_INVALID);
      }
      else
      {
	dbus_g_proxy_call_no_reply(proxy,
				   MCE_DEACTIVATE_VIBRATOR_PATTERN,
				   G_TYPE_STRING, plugin->priv->vibra_pattern,
				   G_TYPE_INVALID);
	g_free(plugin->priv->vibra_pattern);
	plugin->priv->vibra_pattern = NULL;
      }
    }
    g_object_unref(proxy);
  }
  g_free(led_pattern);
  g_free(vibra_pattern);
  g_object_unref(client);
}

static void
handle_lock_mode_changed(DBusGProxy* object,
                         const gchar* tklock_mode,
                         LensCoverStatusPlugin* plugin)
{
  if(lens_cover_status_plugin_cover_is_open())
  {
    if(g_str_has_prefix(tklock_mode, MCE_TK_LOCKED))
      lens_cover_status_plugin_notification(plugin, TRUE);
    else
      lens_cover_status_plugin_notification(plugin, FALSE);
  }
}

static void
optional_lock_screen(LensCoverStatusPlugin* plugin)
{
  GConfClient* client = gconf_client_get_default();
  gboolean lock_device = gconf_client_get_bool(client, GC_ROOT "lock_device", NULL);
  g_object_unref(client);
  if(!lock_device)
    return;
  if(plugin->priv->dbus_conn)
  {
    DBusGProxy* proxy = dbus_g_proxy_new_for_name(plugin->priv->dbus_conn,
                                                  MCE_SERVICE,
                                                  MCE_REQUEST_PATH,
                                                  MCE_REQUEST_IF);
    if(proxy)
    {
      dbus_g_proxy_call_no_reply(proxy,
                                 MCE_TKLOCK_MODE_CHANGE_REQ,
                                 G_TYPE_STRING, MCE_TK_LOCKED_DIM,
                                 G_TYPE_INVALID);
      g_object_unref(proxy);
    }
  }   
}

static void
handle_cover_switch_changed(DBusGProxy* object,
			    const gchar* event,
			    const gchar* source,
			    LensCoverStatusPlugin* plugin)
{
  if(lens_cover_status_plugin_cover_is_open())
  {
    lens_cover_status_plugin_show_icon(plugin);
  }
  else
  {
    lens_cover_status_plugin_hide_icon(plugin);
    lens_cover_status_plugin_notification(plugin, FALSE);
    optional_lock_screen(plugin);
  }
}

static void
lens_cover_status_plugin_finalize(GObject* object)
{
  LensCoverStatusPlugin* plugin = LENS_COVER_STATUS_PLUGIN(object);
  if(plugin->priv->dbus_proxy_cover)
  {
    dbus_g_proxy_disconnect_signal(plugin->priv->dbus_proxy_cover,
				   "Condition",
				   G_CALLBACK(handle_cover_switch_changed),
				   plugin);
  }
  if(plugin->priv->dbus_proxy_lock)
  {
    dbus_g_proxy_disconnect_signal(plugin->priv->dbus_proxy_lock,
                                   MCE_TKLOCK_MODE_SIG,
                                   G_CALLBACK(handle_lock_mode_changed),
                                   plugin);
  }
}

static void
register_dbus_signal_on_cover_switch(LensCoverStatusPlugin* plugin)
{
  plugin->priv->dbus_conn = NULL;
  plugin->priv->dbus_proxy_cover = NULL;
  plugin->priv->dbus_proxy_lock = NULL;
  plugin->priv->dbus_conn = hd_status_plugin_item_get_dbus_g_connection(HD_STATUS_PLUGIN_ITEM(&plugin->parent),
									DBUS_BUS_SYSTEM,
									NULL);

  if(plugin->priv->dbus_conn)
  {
    plugin->priv->dbus_proxy_cover = dbus_g_proxy_new_for_name(plugin->priv->dbus_conn,
							 "org.freedesktop.Hal",
							 "/org/freedesktop/Hal/devices/platform_cam_shutter",
							 "org.freedesktop.Hal.Device");
    dbus_g_proxy_add_signal(plugin->priv->dbus_proxy_cover,
			    "Condition",
			    G_TYPE_STRING,
			    G_TYPE_STRING,
			    G_TYPE_INVALID);
    dbus_g_proxy_connect_signal(plugin->priv->dbus_proxy_cover,
				"Condition",
				G_CALLBACK(handle_cover_switch_changed), plugin, NULL);
							 
    plugin->priv->dbus_proxy_lock = dbus_g_proxy_new_for_name(plugin->priv->dbus_conn,
                                                              MCE_SERVICE,
                                                              MCE_SIGNAL_PATH,
                                                              MCE_SIGNAL_IF);
    dbus_g_proxy_add_signal(plugin->priv->dbus_proxy_lock,
                            MCE_TKLOCK_MODE_SIG,
                            G_TYPE_STRING,
                            G_TYPE_INVALID);
    dbus_g_proxy_connect_signal(plugin->priv->dbus_proxy_lock,
                                MCE_TKLOCK_MODE_SIG,
                                G_CALLBACK(handle_lock_mode_changed), plugin, NULL);

  }
}


static void
lens_cover_status_plugin_class_finalize(LensCoverStatusPluginClass *klass)
{
}

static void
lens_cover_status_plugin_class_init(LensCoverStatusPluginClass *klass)
{
  g_type_class_add_private(klass, sizeof(LensCoverStatusPluginPrivate));
  G_OBJECT_CLASS(klass)->finalize = (GObjectFinalizeFunc)lens_cover_status_plugin_finalize;
}

static void
lens_cover_status_plugin_init(LensCoverStatusPlugin* plugin)
{
  plugin->priv = LENS_COVER_STATUS_PLUGIN_GET_PRIVATE(plugin);
  plugin->priv->led_pattern = NULL;
  plugin->priv->vibra_pattern = NULL;
  register_dbus_signal_on_cover_switch(plugin);
  if(lens_cover_status_plugin_cover_is_open())
  {
    lens_cover_status_plugin_show_icon(plugin);
  }
  else
  {
    lens_cover_status_plugin_hide_icon(plugin);
  }
}

