/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2005 Koos Vriezen <koos.vriezen@xs4all.nl>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _KMPLAYERPROCESS_H_
#define _KMPLAYERPROCESS_H_

#include <libosso.h>
#include <list>

#include "kmplayerplaylist.h"

typedef struct DBusPendingCall DBusPendingCall;
typedef struct _Application Application;

namespace KMPlayer {

class Viewer;
class ProcessNotify;
class UnixProcess;

/*
 * Base class for all backend processes
 */
class KMPLAYER_NO_EXPORT Process {
public:
    enum State {
        NotRunning = 0, Ready, Buffering, Playing, Paused
    };
    Process (ProcessNotify * ctrl, const char * n, bool xv);
    virtual ~Process ();
    //virtual String menuName () const;
    //virtual void setAudioLang (int, const QString &);
    //virtual void setSubtitle (int, const QString &);
    void setPosition (int p);
    bool playing () const;
    unsigned int viewer () const;
    //virtual bool grabPicture (const KURL & url, int pos);
    //bool supports (const char * source) const;
    State state () const { return m_state; }
    Mrl *mrl () { return m_mrl ? m_mrl->mrl () : NULL; }
    void infoMessage (const String &msg);
//signals:
    //void grabReady (const QString & path);
public:// slots:
    virtual bool ready (unsigned int);
    virtual bool play (NodePtr mrl);
    virtual bool stop ();
    virtual bool quit ();
    virtual bool pause ();
    /* seek (pos, abs) seek positon in deci-seconds */
    virtual bool seek (int pos, bool absolute);
    /* volume from 0 to 100 */
    virtual bool volume (int pos, bool absolute);
    /* saturation/hue/contrast/brightness from -100 to 100 */
    virtual bool saturation (int pos, bool absolute);
    virtual bool hue (int pos, bool absolute);
    virtual bool contrast (int pos, bool absolute);
    virtual bool brightness (int pos, bool absolute);
    virtual void setVideoWindow (int x, int y, int w, int h);
    KMPLAYER_NO_EXPORT const char * name () { return m_name; }
    void scheduledStateChanged ();
    ProcessNotify * processNotify () { return m_notify; }
    int userStopPosition () const { return m_user_stop_position; }
    void setUserStop ();
    void setState (State newstate);
    void setWidth (int w) { m_width = w; }
    void setHeight (int h) { m_height = h; }
    void updateAspects ();
    bool hasVideo () const { return playing () && !no_video; }
    void setHasVideo (bool b) { no_video = !b; }
    bool hasXv () const { return has_xv; }
    void setUseXv (bool b) { has_xv = b; }
    bool isLocalFile () const { return is_local_file; }
protected:
    //Source * m_source;
    //Settings * m_settings;
    NodePtrW m_mrl;
    State m_state;
    State m_old_state;
    String m_url;
    ProcessNotify * m_notify;
    unsigned int m_viewer;
    const char * m_name;
    int start_timer;
    int m_length;
    int m_position;
    int m_width;
    int m_height;
    int m_user_stop_position;
    float m_aspect;
    bool no_video;
    bool has_xv;
    bool is_local_file;
    //int m_request_seek;
    //const char ** m_supported_sources;
};

/*
 * Listener for Process events
 */
class ProcessNotify {
public:
    virtual void stateChanged (Process * proc, Process::State os, Process::State ns)=0;
    virtual void errorMsg (const String & msg) = 0;
    virtual void setLoading (int perc) = 0;
    virtual void setPosition (int pos) = 0;
    virtual void setLength (int len) = 0;
    virtual void setAspect (float aspect) = 0;
    virtual Node *downloads () = 0;
    virtual void updateDownloads () = 0;
    virtual void updateDownload (Node *download, int percentage) = 0;
    virtual void videoDetected () = 0;
};

class UnixProcessListener {
public:
    virtual void readEvent (UnixProcess *) = 0;
    virtual void writeEvent (UnixProcess *) = 0;
    virtual void processExited (UnixProcess *) = 0;
};

class UnixProcess {
public:
    enum { NoIO = 0, StdIn = 0x01, StdOut = 0x02 };

    UnixProcess (UnixProcessListener *l, int flags);

    bool start (const String &pwd, const String &cmdline);
    void stop ();
    bool running () const;
    void pollStdIn (bool b);
    void closeStdIn ();
    void closeStdOut ();
    int writeStdIn (const char *data, int len);
    int readStdOut (char *buf, int len);

    void readEvent (GIOChannel *channel);
    void writeEvent ();
    void processExited (GPid pid);

    UnixProcessListener *listener;
    int io_flags;
    GIOChannel *pin, *pout, *perr;
    int win, wout, werr;
    GPid process_id;
    bool eof;
};

/*
 * MM backend using MPlayer
 */
class MPlayer : public Process, public UnixProcessListener {
public:
    MPlayer (ProcessNotify *ctrl, const char *name);
    ~MPlayer ();
    virtual bool play (NodePtr mrl);
    virtual bool pause ();
    virtual bool ready (unsigned int);
    virtual bool stop ();
    virtual bool quit ();
    virtual bool seek (int pos, bool absolute);
    virtual bool volume (int pos, bool absolute);
    virtual bool saturation (int pos, bool absolute);
    virtual bool hue (int pos, bool absolute);
    virtual bool contrast (int pos, bool absolute);
    virtual bool brightness (int pos, bool absolute);

    void setPlayer (const char *p) { path = p; }
    void setAudioDriver (const char *ao) { audio_driver = ao; }
    void setVideoDriver (const char *vo) { video_driver = vo; }
    void setAutoSubTitles (bool b) { auto_sub_titles = b; }

    // for UnixProcessListener functions
    void processExited (UnixProcess *);
    void writeEvent (UnixProcess *);
    void readEvent (UnixProcess *);

protected:
    bool sendCommand (const String &);
    bool removeQueued (const char *cmd);
    typedef std::list <String> StringList;
    StringList commands;
    String command_chunk;
    UnixProcess unix_process;
    int request_seek;
    const char *path;
    const char *audio_driver;
    const char *video_driver;
    State m_transition_state;
    bool request_quit;
    bool auto_sub_titles;
    bool padding;
    char outbuf[512];
};

#ifdef __ARMEL__

class KMPLAYER_NO_EXPORT NpPlayer : public MPlayer {
public:
    NpPlayer (ProcessNotify * ctrl);

    virtual bool play (NodePtr mrl);

    static bool isFlashMimeType (const String &mime);
};

#endif //__ARMEL__

/*
 * Download backend
 */

class Download : public Element, public IOJobListener
{
public:
    Download (NodePtr &doc, ProcessNotify *n, int fd = -1);
    ~Download ();

    virtual void closed();
    virtual void begin();
    virtual const char *nodeName () const { return "download"; }

    virtual void jobData (IOJob *job, ByteArray & data);
    virtual void jobResult (IOJob *job);
    virtual void redirected (IOJob * job, const String & uri);

    String title;
    String file;
    String uri;
    IOJob *job;
    ProcessNotify *notify;
    off_t size;
    off_t bytes;
    int fd;
};

class Downloader : public Process
{
public:
    Downloader (ProcessNotify *ctrl, Application *app);
    ~Downloader ();

    virtual bool play (NodePtr mrl);
    virtual bool stop ();

    void createDownload (const String &url, const String &title);

private:
    Application *application;
};

} // namespace

#endif //_KMPLAYERPROCESS_H_
