import org.maemo.hermes.engine.service

from org.maemo.hermes.engine.names import canonical
from org.maemo.hermes.engine.friend import Friend

class Service(org.maemo.hermes.engine.service.Service):
    """Facebook backend for Hermes.
                
       Copyright (c) Andrew Flegg <andrew@bleb.org> 2010.
       Copyright (c) Fredrik Wendt <fredrik@wendt.se> 2010.
       Released under the Artistic Licence."""
       
    attrs = ['uid', 'name', 'pic_big', 'birthday_date', 'profile_url', 'first_name', 'last_name', 'website']


    # -----------------------------------------------------------------------
    def __init__(self, service_id, facebook, create_birthday_only = False):
        self.fb = facebook
        self._service_id = service_id
        self._create_birthday_only = create_birthday_only
        
        self._friends_by_name = {}
        self._friends_by_url = {}
        self._friends_by_contact = {}
        self._contacts_by_friend = {}
        self._friends_without_contact = set()
        self._known_urls = set()


    # -----------------------------------------------------------------------
    def get_friends_to_create_contacts_for(self):
        friends = []
        if self._create_birthday_only:
            for friend in self._friends_without_contact:
                if friend.has_birthday_date():
                    friends.append(friend)
                    
        return friends
    
    
    # -----------------------------------------------------------------------
    def get_friends(self):
        """Returns all friends on Facebook"""
        
        return self._friends_by_url.values()
    
    
    # -----------------------------------------------------------------------
    def get_contacts_with_match(self):
        """Returns a dict, where each key value pair is a contact (key) that 
           matched a friend (value)"""

        return self._friends_by_contact
    
    
    # -----------------------------------------------------------------------
    def get_unmatched_friends(self):
        """Returns a list of all friends that didn't match a contact."""
         
        return self._friends_without_contact


    # -----------------------------------------------------------------------
    def pre_process_contact(self, contact):
        """Registers URLs of all previous mappings, and makes sure that any 
           friend with such a URL don't get match by name."""
           
        for url in contact.get_urls():
            self._known_urls.add(url)
    
    
    # -----------------------------------------------------------------------
    def process_friends(self):
        """Retreives data from Facebook and parse that into Friend 
           objects."""
        
        def if_defined(data, key, callback):
            if key in data and data[key]:
                callback(data[key])
        
        friends_data = self._get_friends_data()
        for data in friends_data:
            friend = Friend(data['name'])
        
            if 'profile_url' not in data:
                data['profile_url'] = "http://www.facebook.com/profile.php?id=" + str(data['uid'])
        
            if_defined(data, 'website', friend.add_url)
            if_defined(data, 'profile_url', friend.add_url)
            if_defined(data, 'birthday_date', friend.set_birthday_date)

            if_defined(data, 'pic_big', friend.set_photo_url)
            
            url = data['profile_url']
            friend.add_url(url)
            self._register_friend(friend)


    # -----------------------------------------------------------------------
    def process_contact(self, contact):
        """If the contact is matched with a friend, that friend is returned,
           otherwise None."""
           
        matched_friend = None
        if self._friends_by_contact.has_key(contact):
            matched_friend = self._friends_by_contact[contact]
        
        # we might get a hit if the friend has setup a URL with another service,
        # such as putting the id link to Facebook on the Twitter account's profile
        if not matched_friend:
            for url in contact.get_urls():
                if url in self._friends_by_url:
                    matched_friend = self._friends_by_url[url]
                    self._register_match(contact, matched_friend)
                    break

        if not matched_friend:
            for id in contact.get_identifiers():
                if id in self._friends_by_name:
                    matched_friend = self._friends_by_name.pop(id)
                    self._register_match(contact, matched_friend)
                    break
                
        return matched_friend
    

    # -----------------------------------------------------------------------
    def _register_friend(self, friend):
        self._friends_without_contact.add(friend)
        
        for url in friend.get_urls():
            if self.is_profile_url(url):
                self._friends_by_url[url] = friend

        if self._allow_friend_to_match_by_name(friend):
            key = canonical(friend.get_name())
            self._friends_by_name[key] = friend


    # -----------------------------------------------------------------------
    def is_profile_url(self, url):
        """Return True if this is a URL for this service."""

        return url and "facebook.com" in url


    # -----------------------------------------------------------------------
    def _allow_friend_to_match_by_name(self, friend):
        for url in friend.get_urls():
            if url in self._known_urls:
                return False
        return True


    # -----------------------------------------------------------------------
    def _register_match(self, contact, friend):
        friend.set_contact(contact)
        self._friends_without_contact.discard(friend)
        self._friends_by_contact[contact] = friend
        self._contacts_by_friend[friend] = contact


    # -----------------------------------------------------------------------
    def _get_friends_data(self):
        """Returns a list of dicts, where each dict represents the raw data
           of a friend"""
        
        return self.fb.users.getInfo(self.fb.friends.get(), Service.attrs)
