/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <stdlib.h>
#include <string.h>

#include "csvfunc.h"

void addtolist(csvlist *plist, void *str)
{
  csvlistnode *ph;
  csvlistnode *pt;
  csvlistnode *newlist;

  ph = plist->head;
  pt = plist->tail;

  newlist=(csvlistnode *)malloc(sizeof(csvlistnode));
  newlist->data = str;
  if (ph == NULL) {
    ph=newlist;
    pt=ph;
    pt->prev=NULL;
    pt->next=NULL;
    plist->nodes=1;
  }
  else {
    pt->next=newlist;
    newlist->prev=pt;
    pt=newlist;
    pt->next=NULL;
    plist->nodes++;
  }

  plist->tail=pt;
  plist->head=ph;
}

void csvfreerow(csvlist *plist)
{
  csvlistnode *ph;
  csvlistnode *p;

  ph = plist->head;
  p = ph;

  while (ph != NULL) {
    free(ph->data);
    p=ph;
    ph = ph->next;
    free(p);
    plist->nodes--;
  }
  plist->tail = NULL;
  plist->head = NULL;
}

void csvremoverow(csvlist *plist, csvlistnode *prow)
{
  csvlistnode *ph;
  csvlistnode *pt;
  csvlistnode *next;
  csvlistnode *prev;

  ph = plist->head;
  pt = plist->tail;

  if (ph != NULL && pt != NULL) {
    next = prow->next;
    prev = prow->prev;
    csvfreerow((csvlist *)(prow->data));
    if (prev != NULL) {/* removing other than head */
      prev->next = next;
    }
    else /* removing head */
      plist->head = next;
    if (next != NULL) {/* removing other than tail */
      next->prev = prev;
    }
    else /* removing tail */
      plist->tail = prev;
   
    plist->nodes--;
  }
}

/**
 * \fn void csvinit(csvlist *plist)
 * \brief Initializes the csvlist structure
 * \param *plist pointer the the csvlist
 *
 * This is the first function that must be called when parsing a csv
 * file. The csvlist structure must have been allocated before calling
 * this function.
 *
 */
void csvinit(csvlist *plist)
{
  plist->head=NULL;
  plist->tail=NULL;
  plist->nodes=0;
}

/**
 * \fn void csvfree(csvlist *rowlist)
 * \brief Frees the memory allocated for rows inside a csvlist structure
 * \param *rowlist pointer the the csvlist
 *
 * This is the last function that must be called when parsing a csv
 * file. The memory allocated to store the rows in the csvlist
 * structure are released.
 *
 */
void csvfree(csvlist *rowlist)
{
  csvlistnode *prow;
  int i;

  prow=rowlist->head;
  i=1;
  while (i<rowlist->nodes && prow != NULL) {
/*     csvfreerow((csvlist *)(prow->data)->head); */
    csvfreerow((csvlist *)(prow->data));
    prow=prow->next;
  }
  csvfreerow(rowlist);
}


enum csvstate_t {NOTIN,INNORMAL,INQUOTED,QUOTE,STOP};

/**
 * \fn int csvparse(char *line, csvlist *rowlist, int *rows, int *cols)
 * \brief Parses one line of a csv file
 * \param *line pointer to the line
 * \param *rowlist pointer to the structure where the parsed data is appended
 * \param *rows the total number of rows
 * \param *cols the total number of columns
 * \return succesful operation is indicated with 0, otherwise 1
 *
 */
int csvparse(char *line, csvlist *rowlist, int *rows, int *cols)
{
  enum csvstate_t csvstate;
  int i;
  int cstart, cstop;
  int cnum;
  char *newstr;
  int slen;
  csvlist *plist;
  size_t linelen;

  cstart=0;
  cstop=0;
  plist=(csvlist *)malloc(sizeof(csvlist));
  csvinit(plist);
  addtolist(rowlist,(void *)plist);

  (*rows)++;
  cnum=0;
  csvstate = NOTIN;
  linelen=strlen(line);
  if (linelen>0)
    if (line[linelen-1]=='\n') /* dos format */
      linelen--;
  for (i=0;i<linelen;i++) {
    switch (csvstate) {
    case NOTIN: 
      if (line[i]=='"') {
	csvstate = INQUOTED;
	cstart = i+1;
      }
      else if (line[i]==','|| line[i] == '\n' || line[i] == '\r') {
	csvstate = STOP;
	cstart = i;
	cstop = i-1;
      }
      else {
	csvstate = INNORMAL;
	cstart = i;
      }
      break;
    case INNORMAL:
      if (line[i]==',' || line[i] == '\n' || line[i] == '\r') {
	csvstate = STOP;
	cstop = i-1;
      }
      break;
    case INQUOTED:
      if (line[i]=='"') {
	csvstate = QUOTE;
      }
      break;
    case QUOTE:
      if (line[i]=='"') {
	csvstate = INQUOTED;
      }
      else
	if (line[i]==',' || line[i] == '\n' || line[i] == '\r') {
	  cstop = i-2;
	  csvstate = STOP;
	}
	else {
/* 	  printf("error: not quote or comma after quote\n"); */
	  return 1;
	}
    case STOP:
      break;
    }
    if (csvstate==STOP) {
      csvstate = NOTIN;

      slen=cstop-cstart+1;
      newstr=(char *)malloc(sizeof(char)*slen+1);
      memcpy(newstr,&line[cstart],slen);
      newstr[slen]='\0';
      addtolist(plist,(void *)newstr);
      cnum++;
    }
  }
  if ((*cols)<cnum) (*cols)=cnum;

  return 0;
}

/**
 * \fn csvlist *csvgetrow(csvlist *rowlist, int row)
 * \brief Returns a pointer to a row
 * \param *rowlist pointer to the csv structure where the parsed data is stored
 * \param row an index to the requested row
 * \return a pointer to the row
 *
 */
csvlist *csvgetrow(csvlist *rowlist, int row)
{
  csvlistnode *prow;
  int i;

  prow=rowlist->head;
  i=1;
  while (i<row && prow != NULL) {
    prow=prow->next;
    i++;
  }
  if (i==row)
/*     return ((csvlist *)(prow->data))->head; */
    return ((csvlist *)(prow->data));
  else
    return NULL;
}

/**
 * \fn csvlistnode *csvgetrowcol(csvlist *rowlist, int row, int col)
 * \brief Returns a pointer to a row and column
 * \param *rowlist pointer to the csv structure where the parsed data is stored
 * \param row an index to the requested row
 * \param col an index to the requested col
 * \return a pointer to the data in row,col
 *
 */
csvlistnode *csvgetrowcol(csvlist *rowlist, int row, int col)
{
  csvlistnode *pcol;
  int j;

  pcol = csvgetrow(rowlist, row)->head;
  if (pcol != NULL) {
    j=1;
    while (j<col && pcol != NULL) {
      pcol=pcol->next;
      j++;
    }
    if (j==col)
      return pcol;
    else
      return NULL;
  }
  else
    return NULL;
}
