/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <glib-object.h>
#include <libosso.h>
#include <osso-log.h> /* OSSO Debug macros */
#include <libosso-abook/osso-abook-contact.h>
#include <rtcom-eventlogger/eventlogger.h>
#include <rtcom-eventlogger-ui/rtcom-log-columns.h>

#include "tp-status-feed.h"
#include "tp-status-feed-private.h"
#include "eventlogger-util.h"

/**************************************/
/* Private functions                  */
/**************************************/

static void
log_common_status (RTComEl *el,
                   const gchar *serv,
                   const gchar *eventtype,
                   const gchar *localuid,
                   const gchar *remote_euid,
                   const gchar *remoteuid,
                   const gchar *remotename,
                   const gchar* status_message)
{
  RTComElEvent *event = NULL;
  time_t t = 0;

  event = rtcom_el_event_new ();

  if (!event) {
    g_error ("Could not create an event for logging");
    return ;
  }

  t = time(NULL);
  RTCOM_EL_EVENT_SET_FIELD (event, service, (gchar*)serv);
  RTCOM_EL_EVENT_SET_FIELD (event, event_type, (gchar*)eventtype);
  RTCOM_EL_EVENT_SET_FIELD (event, start_time, t);
  RTCOM_EL_EVENT_SET_FIELD (event, local_uid, (gchar*)localuid);
  RTCOM_EL_EVENT_SET_FIELD (event, remote_uid, (gchar*)remoteuid);
  RTCOM_EL_EVENT_SET_FIELD (event, remote_name, (gchar*)remotename);
  RTCOM_EL_EVENT_SET_FIELD (event, remote_ebook_uid, (gchar*)remote_euid);
  RTCOM_EL_EVENT_SET_FIELD (event, free_text, (gchar*)status_message);

  rtcom_el_add_event (el, event, NULL);
  rtcom_el_event_free (event);
}

static void
log_self_status_message (TpStatusFeed *statusfeed)
{
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);
  const gchar *local_uid; /* local and remote uid are both osso-abook-self */
  const gchar *status_message;
  ContactStatus *last_status;

  status_message = osso_abook_presence_get_presence_status_message (
      OSSO_ABOOK_PRESENCE (priv->self_contact));

  if (!status_message)
    return ;

  /* remote_uid = local_uid, both must not be NULL */
  local_uid = osso_abook_contact_get_uid (
      OSSO_ABOOK_CONTACT (priv->self_contact));

  last_status = eventlogger_util_get_last_contact_status (priv->eventlogger,
                                                          local_uid, local_uid);

  /* Log this status if myself does not have no last status or
   * last status is different from this status */
  if (NULL == last_status ||
      g_strcmp0 (last_status->text, status_message)) {
    const gchar *remote_euid; /* osso-abook-self */
    const gchar *remote_name;

    remote_euid = osso_abook_contact_get_persistent_uid (
        OSSO_ABOOK_CONTACT (priv->self_contact));
    remote_name = osso_abook_contact_get_display_name (
        OSSO_ABOOK_CONTACT (priv->self_contact));

    log_common_status (priv->eventlogger, "RTCOM_EL_SERVICE_STATUS",
        "RTCOM_EL_EVENTTYPE_STATUS_MY", local_uid, remote_euid,
        local_uid, remote_name, status_message);
  }

  eventlogger_util_contact_status_free (last_status);
}

/* TODO may receive multiple times for a single update (Sure????????)
 * A hashtable to cache last status of every contact?
 */
static void
log_friend_status_message (TpStatusFeed *statusfeed,
                           OssoABookContact *contact)
{
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);
  const gchar *status_message, *vcard_field;
  gchar *im_field;
  ContactStatus *last_status;
  McAccount *account;

  ULOG_DEBUG_L ("%s", G_STRFUNC);

  /* Get current status message */
  status_message = osso_abook_presence_get_presence_status_message (
        OSSO_ABOOK_PRESENCE (contact));

  if (!status_message)
    return ;

  /* Get last status message from database */
  /* X-Protocol, e.g. X-JABBER */
  vcard_field = osso_abook_contact_get_vcard_field (contact);
  /* account object */
  account = osso_abook_contact_get_account (contact);
  /* e.g. someone@gmail.com */
  im_field = osso_abook_contact_get_value (E_CONTACT (contact), vcard_field);
  last_status = eventlogger_util_get_last_contact_status (priv->eventlogger,
                                                          account->name, im_field);

  /* Log this status if there is no last status for this contact or
   * last status is different from this status */
  if (NULL == last_status ||
      g_strcmp0 (last_status->text, status_message)) {
    const gchar *local_uid, *remote_name, *remote_euid;

    /* local_uid: account name, e.g. gabble/jabber/account0 */
    local_uid = account->name;

    /* remote_euid: persistent UID for this contact */
    remote_euid = osso_abook_contact_get_persistent_uid (contact);

    /* remote_uid: im_field of contact, e,g. someone@gmail.com */

    /* remote_name: display name of contact, e.g. Foo Bar */
    remote_name = osso_abook_contact_get_display_name (contact);

    log_common_status (priv->eventlogger, "RTCOM_EL_SERVICE_STATUS",
        "RTCOM_EL_EVENTTYPE_STATUS_FRIEND", local_uid, remote_euid,
        im_field, remote_name, status_message);
  }

  eventlogger_util_contact_status_free (last_status);
  g_free (im_field);
}

/***** Presence update callback *****/

static void
self_status_message_updated_cb (TpStatusFeed *statusfeed)
{
  ULOG_DEBUG_L ("%s", G_STRFUNC);

  log_self_status_message (statusfeed);
}

static void
friend_status_message_updated_cb (OssoABookContact    *contact,
                                  GParamSpec          *param,
                                  gpointer             userdata)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (userdata);

  ULOG_DEBUG_L ("%s", G_STRFUNC);

  log_friend_status_message (statusfeed, contact);
}

/**************************************/
/* Public functions                   */
/**************************************/

void tp_status_feed_backend_log_self_status_update (TpStatusFeed *statusfeed)
{
  TpStatusFeedPrivate *priv = TP_STATUS_FEED_GET_PRIVATE (statusfeed);

  ULOG_DEBUG_L ("%s", G_STRFUNC);

  if (!priv->self_contact) {
    priv->self_contact = osso_abook_self_contact_get_default ();
    g_signal_connect_swapped (priv->self_contact, "notify::presence-status-message",
        G_CALLBACK (self_status_message_updated_cb), statusfeed);

    log_self_status_message (statusfeed);
  }
}

void
tp_status_feed_backend_contacts_added_cb (OssoABookRoster    *aggregator,
                                          OssoABookContact  **contacts,
                                          gpointer            userdata)
{
  TpStatusFeed *statusfeed = TP_STATUS_FEED (userdata);
  OssoABookContact **p;

  ULOG_DEBUG_L ("%s", G_STRFUNC);

  for (p = contacts; *p; ++p) {
    GList *roster_contacts, *rc;

    /**
     * 1. Get roster contacts attached to this master contact (*p)
     * 2. Connect to notify signal of roster contacts?
     * 3. Log the current presence status message if it's different from
     *    the last one in the database
     * TODO
     * How can I know if any roster contact is removed
     */
    roster_contacts = osso_abook_contact_get_roster_contacts (*p);
    for (rc = roster_contacts; rc; rc = rc->next) {
      OssoABookContact *contact; /* roster contact */
      const char *uid;

      contact = rc->data;
      g_signal_connect (contact, "notify::presence-status-message",
          G_CALLBACK (friend_status_message_updated_cb), statusfeed);

      log_friend_status_message (statusfeed, contact);

      /* UID: gabble/jabber/rabbitrun84_40ovi_2ecom0-rabbitrun84@gmail.com */
      uid = e_contact_get_const (E_CONTACT (contact), E_CONTACT_UID);
      g_debug ("roster contact added: %s", uid);
    }
    g_list_free (roster_contacts);
  }
}
