/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>
#include <hildon-uri.h>

#include "one-status-view.h"
#include "urllabel.h"
#include "common.h"

#define ONE_STATUS_WIDTH_SIZE 540

#define GET_PRIVATE(o) ((OneStatusViewPrivate *)       \
                        ((OneStatusView *)(o))->priv)

typedef struct _OneStatusViewPrivate OneStatusViewPrivate;
struct _OneStatusViewPrivate
{
  GtkWidget *presence_icon; /* online/busy/offline icon */
  GtkWidget *service_icon;
  GtkWidget *image; /* contact's avatar */
  GtkWidget *status_label; /* selected status message */

  OssoABookContact *contact; /* contact-status-view destorys this obj? */
  gchar *status_text;
  gint event_id; /* event_id of the current status */
};

G_DEFINE_TYPE (OneStatusView, one_status_view, GTK_TYPE_ALIGNMENT);

/**************************************/
/* Private functions                  */
/**************************************/

static void
action_started_cb (OssoABookTouchContactStarter *widget,
                   gpointer *data)
{
  GtkWidget *dialog;

  g_debug ("%s: destroying contact starter dialog..", G_STRFUNC);

  dialog = gtk_widget_get_toplevel (GTK_WIDGET(widget));
  gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
avatar_tapped_cb (GtkWidget      *button,
                  gpointer        data)
{
  GtkWidget *dialog;
  GtkWidget *contact_starter;

  OneStatusView *view = (OneStatusView *)data;
  OneStatusViewPrivate *priv = GET_PRIVATE (view);

  if (!priv->contact)
    return ;

  dialog = gtk_dialog_new ();

  contact_starter = osso_abook_touch_contact_starter_new_with_contact (
      GTK_WINDOW (dialog), priv->contact);
  gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dialog)->vbox),
      contact_starter);

  g_signal_connect (contact_starter, "action-started",
                    G_CALLBACK (action_started_cb), view);

  gtk_widget_show (contact_starter);
  gtk_widget_show (dialog);
}

/**
 * Callback for urllabel
 * Click a link in the status will launch the browser
 */
static void
status_url_clicked_cb (GtkWidget *url_label,
                       gchar *url,
                       gpointer userdata)
{
  hildon_uri_open (url, NULL, NULL);
  g_debug ("open URL: %s\n", url);
}

/**************************************/
/* GObject functions                  */
/**************************************/

static void
one_status_view_init (OneStatusView *view)
{
  OneStatusViewPrivate *priv;
  GtkWidget *hbox;
  GtkWidget *avatar_button;

  priv = view->priv = G_TYPE_INSTANCE_GET_PRIVATE(view, ONE_TYPE_STATUS_VIEW,
                                                  OneStatusViewPrivate);
  priv->status_text = NULL;

  hbox = gtk_hbox_new (FALSE, 0);

  /* Avatar of the contact */
  priv->image = osso_abook_avatar_image_new_with_avatar (NULL,
      OSSO_ABOOK_PIXEL_SIZE_AVATAR_MEDIUM);
  avatar_button = hildon_gtk_button_new (HILDON_SIZE_AUTO);
  gtk_button_set_relief (GTK_BUTTON (avatar_button), GTK_RELIEF_NONE);
  gtk_container_add (GTK_CONTAINER (avatar_button), priv->image);
  gtk_box_pack_start (GTK_BOX (hbox), avatar_button, FALSE, FALSE, 0);
  gtk_widget_show (priv->image);

  /* Service icon */
  priv->service_icon = gtk_image_new ();
  gtk_box_pack_start (GTK_BOX (hbox), priv->service_icon, FALSE, FALSE, 0);
  gtk_widget_show (priv->service_icon);

  /* Presence icon */
  priv->presence_icon = osso_abook_presence_icon_new (NULL);
  gtk_box_pack_start (GTK_BOX (hbox), priv->presence_icon, FALSE, FALSE, HILDON_MARGIN_DEFAULT);
  gtk_widget_show (priv->presence_icon);

  /* Selected status message */
  priv->status_label = url_label_new ();
  gtk_widget_set_size_request(priv->status_label, ONE_STATUS_WIDTH_SIZE, -1);
  gtk_label_set_line_wrap (GTK_LABEL(priv->status_label), TRUE);
  gtk_box_pack_start (GTK_BOX (hbox), priv->status_label, FALSE, FALSE, HILDON_MARGIN_DEFAULT);
  gtk_widget_show (priv->status_label);

  /* Add all UI elements to OneStatusView object */
  gtk_container_add (GTK_CONTAINER (view), hbox);

  /* Signals */
  g_signal_connect (avatar_button, "clicked",
      G_CALLBACK (avatar_tapped_cb), view);

  g_signal_connect (GTK_WIDGET (priv->status_label), "url_clicked",
                    G_CALLBACK (status_url_clicked_cb), NULL);
}

/* TODO */
static void
one_status_view_dispose (GObject *obj)
{

}

static void
one_status_view_finalize (GObject *obj)
{
  /* widget destory, g_object_unref */

  G_OBJECT_CLASS (one_status_view_parent_class)->finalize (obj);
}

static void
one_status_view_class_init (OneStatusViewClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = one_status_view_dispose;
  object_class->finalize = one_status_view_finalize;

  g_type_class_add_private (klass, sizeof (OneStatusViewPrivate));
}

/**************************************/
/* Public functions                  */
/**************************************/

GtkWidget *
one_status_view_new (void)
{
  return g_object_new (ONE_TYPE_STATUS_VIEW, NULL);
}

const gchar *
one_status_view_get_status_text (OneStatusView *view)
{
  OneStatusViewPrivate *priv = GET_PRIVATE (view);

  return priv->status_text;
}

void
one_status_view_update_contact (OneStatusView *view,
                                OssoABookContact *contact,
                                gchar *text,
                                gchar *start_time,
                                const GdkPixbuf *service_icon)
{
  OneStatusViewPrivate *priv = GET_PRIVATE (view);
  gchar *result /*, *parsed_text = NULL*/;

  osso_abook_avatar_image_set_avatar (
      OSSO_ABOOK_AVATAR_IMAGE (priv->image),
      contact ? OSSO_ABOOK_AVATAR (contact) : NULL);

  osso_abook_presence_icon_set_presence (
      OSSO_ABOOK_PRESENCE_ICON (priv->presence_icon),
      OSSO_ABOOK_PRESENCE (contact));

  gtk_image_set_from_pixbuf (GTK_IMAGE (priv->service_icon),
      (GdkPixbuf *)service_icon);

  priv->contact = contact;

  /* Update status message */
  g_free (priv->status_text);
  priv->status_text = text;
/*
  parse_message_links (
      text, &parsed_text, NULL);
*/
  result = g_strdup_printf (
      "<span>%s</span>\n"
      "<span foreground=\"%s\" size=\"x-small\"><sup>%s</sup></span>",
      /*parsed_text*/text,
      get_secondary_text_color (), start_time);

  url_label_set_text (URL_LABEL (priv->status_label), result);

//  g_free (parsed_text);
  g_free (start_time);
  g_free (result);
}

void
one_status_view_set_event_id (OneStatusView *view,
                              gint event_id)
{
  OneStatusViewPrivate *priv = GET_PRIVATE (view);

  priv->event_id = event_id;
}

gint
one_status_view_get_event_id (OneStatusView *view)
{
  OneStatusViewPrivate *priv = GET_PRIVATE (view);

  return priv->event_id;
}
