/*
babyphone - A baby monitor application on the Nokia N900.
    Copyright (C) 2011  Roman Morawek <maemo@morawek.at>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "contactchooser.h"
#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "settingsdialog.h"

#include <QDebug>
#include <QMessageBox>
#include <QInputDialog>
#include <QtDBus>


/*!
  The constructor sets up the main user interface and instantiates the
  AudioMonitor, ProfileSwitcher, as well as the application Settings.
  It connects the signals from the AudioMonitor to internal methods.
  As a last step it starts the audio monitoring.
*/
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent), ui(new Ui::MainWindow)
{
    // load settings
    itsSettings = new Settings(this);

    // setup profile switcher
    itsProfileSwitcher = new ProfileSwitcher(itsSettings, this);

    // setup UI, orientation and audio graphs
    setupGui();

    itsActivationDelayTimer.setSingleShot(true);
    connect(&itsActivationDelayTimer, SIGNAL(timeout()),
            this, SLOT(activationTimerExpired()));

    // setup display status monitor
    bool result = QDBusConnection::systemBus().connect("",
                          "", "com.nokia.mce.signal", "display_status_ind",
                          this, SLOT(displayDimmed(const QDBusMessage&)));
    if (result == false)
        qWarning() << "Cannot connect to display status: " << QDBusConnection::systemBus().lastError();

    // show information message at first startup
    if (itsSettings->itsFirstRun)
        showFirstRunInfo();

    // start babyphone engine
    itsBabyphone = new Babyphone(itsSettings, this);
    // register for audio data to update display
    connect(itsBabyphone, SIGNAL(newAudioData(int,int)),
            this, SLOT(newAudioData(int,int)));
    // register to phone call info
    connect(itsBabyphone, SIGNAL(newCallStatus(bool,bool)),
            this, SLOT(newCallStatus(bool,bool)));
    // register for end of phone application to put babyphone to forground again
    connect(itsBabyphone, SIGNAL(phoneApplicationFinished()),
            this, SLOT(bringWindowToFront()));
    // display potential errors
    connect(itsBabyphone, SIGNAL(notificationError()),
            this, SLOT(showNotificationError()));
}


/*!
  The destructor initiates a save of the application settings. Then it frees
  class instances as needed.
*/
MainWindow::~MainWindow()
{
    // save settings
    itsSettings->Save();

    delete ui;
}


void MainWindow::setupGui()
{
    // setup GUI
    ui->setupUi(this);

    // set rotation lock
    if (itsSettings->itsDisableAutoRotate)
        setOrientation(ScreenOrientationLockLandscape);
    else
        setOrientation(ScreenOrientationAuto);
    // the horizontal layout is the pre-defined layout of the form classes
    // we need to set this, even if we start in portrait mode
    itsHorizontalLayout = true;
    itsIsScreenOff = false;

    // setup default UI values
    ui->label_contact->setText(itsSettings->itsContact.GetDisplayString());

    // setup graphs
    itsAudioLevelGraphics = new AudioLevelGraphicsScene(ui->graphicsViewLevel, itsSettings, this);
    connect(ui->graphicsViewLevel, SIGNAL(sizeChanged(QGraphicsView*)),
            itsAudioLevelGraphics, SLOT(UpdateSize(QGraphicsView*)));
    itsAudioCounterGraphics = new AudioLevelGraphicsScene(ui->graphicsViewCounter, itsSettings, this);
    connect(ui->graphicsViewCounter, SIGNAL(sizeChanged(QGraphicsView*)),
            itsAudioCounterGraphics, SLOT(UpdateSize(QGraphicsView*)));
}


/*!
  resizeEvent takes care for layout switches (horizonatal/vertical).
*/
void MainWindow::resizeEvent(QResizeEvent *)
{
    if ( (height() > width()) && itsHorizontalLayout) {
        // switch to vertical layout
        qDebug() << "switching to vertical layout";
        itsHorizontalLayout = false;

        // adjust phone number area
        ui->layout_phone->setDirection(QBoxLayout::TopToBottom);

        // adjust graph area
        ui->layout_graph->removeWidget(ui->graphicsViewLevel);
        ui->layout_graph->addWidget(ui->graphicsViewLevel, 1, 0);
        ui->layout_graph->removeWidget(ui->graphicsViewCounter);
        ui->layout_graph->addWidget(ui->graphicsViewCounter, 3, 0);
    }
    else if ((height() < width()) && !itsHorizontalLayout) {
        // switch to vertical layout
        qDebug() << "switching to horizontal layout";
        itsHorizontalLayout = true;

        // adjust phone number area
        ui->layout_phone->setDirection(QBoxLayout::LeftToRight);

        // adjust graph area
        ui->layout_graph->removeWidget(ui->graphicsViewLevel);
        ui->layout_graph->addWidget(ui->graphicsViewLevel, 0, 1);
        ui->layout_graph->removeWidget(ui->graphicsViewCounter);
        ui->layout_graph->addWidget(ui->graphicsViewCounter, 2, 1);
    }
}


/*!
  Sets the desired orientation mode (portrait/landscape/auto) of the application.
*/
void MainWindow::setOrientation(ScreenOrientation orientation)
{
    Qt::WidgetAttribute attribute;
    switch (orientation) {
#if QT_VERSION < 0x040702
    // Qt < 4.7.2 does not yet have the Qt::WA_*Orientation attributes
    case ScreenOrientationLockPortrait:
        attribute = static_cast<Qt::WidgetAttribute>(128);
        break;
    case ScreenOrientationLockLandscape:
        attribute = static_cast<Qt::WidgetAttribute>(129);
        break;
    default:
    case ScreenOrientationAuto:
        attribute = static_cast<Qt::WidgetAttribute>(130);
        break;
#else // QT_VERSION < 0x040702
    case ScreenOrientationLockPortrait:
        attribute = Qt::WA_LockPortraitOrientation;
        break;
    case ScreenOrientationLockLandscape:
        attribute = Qt::WA_LockLandscapeOrientation;
        break;
    default:
    case ScreenOrientationAuto:
        attribute = Qt::WA_AutoOrientation;
        break;
#endif // QT_VERSION < 0x040702
    };
    setAttribute(attribute, true);
}


/*!
  This user interface event trigger launches the settings dialog.
*/
void MainWindow::on_actionSettings_triggered()
{
    // show settings dialog
    SettingsDialog dialog(itsSettings, this);
    if (dialog.exec() == QDialog::Accepted)
    {
        dialog.SaveData();

        // update orientation lock, even if unchanged
        if (itsSettings->itsDisableAutoRotate)
            setOrientation(ScreenOrientationLockLandscape);
        else
            setOrientation(ScreenOrientationAuto);
    }
}


/*!
  This user interface event trigger displays an about message box.
*/
void MainWindow::on_actionAbout_triggered()
{
    QMessageBox msgBox;
    msgBox.setWindowTitle("Babyphone " + itsSettings->VERSION);
    msgBox.setText(tr("Babyphone is a baby monitoring application that measures the audio noise level.\nIf the volume exceeds a configurable threshold the phone initiates a call to a predefined phone number (the parents)."
                      "\n\n(c) 2011, Roman Morawek"
                      "\nhttp://babyphone.garage.maemo.org"));
    msgBox.exec();
}


/*!
  This user interface event trigger updates the value of the parent's phone
  number.
*/
void MainWindow::on_pushButton_enterNumber_clicked()
{
    bool resultOk;
    QString text = QInputDialog::getText(this,
                         tr("Babyphone Settings"),
                         tr("Please enter the parent's phone number:"),
                         QLineEdit::Normal,
                         itsSettings->itsContact.itsPhoneNumber,
                         &resultOk
                    );
    if (resultOk) {
        // update settings
        itsSettings->itsContact.SetNumber(text);
        // update GUI
        ui->label_contact->setText(itsSettings->itsContact.GetDisplayString());
    }
}


/*!
  on_pushButton_contacts_clicked is the event handler to start address book
  contact selection.
*/
void MainWindow::on_pushButton_contacts_clicked()
{
    if (ContactChooser::contactChooser(&itsSettings->itsContact)) {
        // update GUI
        ui->label_contact->setText(itsSettings->itsContact.GetDisplayString());
    }
}


/*!
  This user interface event trigger reflects the main application button that
  switches the application to the active mode or disables it respectively.
  Before switching to active mode it checks the phone number for a non-empty
  entry and aborts with en error pop-up if empty.
  on_pushButton_clicked also changes the enabled attribute of several GUI
  controls depending on the application state.
*/
void MainWindow::on_pushButton_clicked()
{
    // stop the potentially running activation delay counter
    itsActivationDelayTimer.stop();

    // did we switch ON or OFF?
    if (ui->pushButton->isChecked() == false) {
        // switch OFF
        deactivateMonitor();
    }
    else {
        // switch ON

        // check whether we have a valid phone number
        if (itsSettings->itsContact.HasValidNumber()) {
            // activate it
            activateMonitor();
        }
        else {
            // incomplete settings: no valid phone number
            // open dialog to warn user
            QMessageBox::warning(this, tr("Babyphone"),
                  tr("No valid parent's phone number set. Please adjust the application settings."));

            // reset the button color again
            ui->pushButton->setChecked(false);
        }
    }
}


/*!
  activateMonitor instantiates the UserNotifier and starts the activation
  timer. Also it updates the user interface items properties to the active
  status.
*/
void MainWindow::activateMonitor()
{
    // update GUI
    ui->pushButton->setText(tr("Active - Waiting for activation delay"));
    ui->pushButton_contacts->setEnabled(false);
    ui->pushButton_enterNumber->setEnabled(false);
    ui->menuSettings->setEnabled(false);
    if (itsSettings->itsDisableGraphs) {
        ui->graphicsViewCounter->setEnabled(false);
        ui->graphicsViewLevel->setEnabled(false);
        itsAudioLevelGraphics->Clear();
        itsAudioCounterGraphics->Clear();
    }

    // start activation timeout
    itsActivationDelayTimer.start(itsSettings->itsActivationDelay*1000);

    // inform state machine on new state
    itsBabyphone->setState(Babyphone::STATE_INACTIVE_ON);
}


/*!
  deactivateMonitor deletes the UserNotifier instance and switches to monitor
  mode. If demanded it displays the call statistics. Also it updates the user
  interface items properties to the inactive status.
*/
void MainWindow::deactivateMonitor()
{
    // switch OFF
    itsBabyphone->setState(Babyphone::STATE_OFF);

    // if demanded, show call statistics
    if (itsSettings->itsShowStatistics) {
        // show statistics
        QString statistics = itsBabyphone->getStatistics();
        QMessageBox::information(this, tr("Call Notification Statistics"), statistics);
    }

    // update GUI
    ui->pushButton->setText(tr("Inactive - Click to start"));
    ui->pushButton_contacts->setEnabled(true);
    ui->pushButton_enterNumber->setEnabled(true);
    ui->menuSettings->setEnabled(true);
    if (itsSettings->itsDisableGraphs) {
        ui->graphicsViewCounter->setEnabled(true);
        ui->graphicsViewLevel->setEnabled(true);
    }
}


void MainWindow::newCallStatus(bool finish, bool selfInitiated)
{
    ui->pushButton->setEnabled(finish);

    // check for reactivation delay
    if (finish && selfInitiated) {
        itsBabyphone->setState(Babyphone::STATE_INACTIVE_ON);

        // start activation timeout
        ui->pushButton->setText(tr("Active - Waiting for reactivation delay"));
        itsActivationDelayTimer.start(itsSettings->itsRecallTimer*1000);
    }
}


/*!
  activationTimerExpired gets called as the activation actually starts.
  It just modifies the button's label.
*/
void MainWindow::activationTimerExpired()
{
    ui->pushButton->setText(tr("Active - Click to Stop"));
    itsBabyphone->setState(Babyphone::STATE_ACTIVE_ON);
}


/*!
  bringWindowToFront puts the application window to full screen foreground.
*/
void MainWindow::bringWindowToFront()
{
    // bring the application back to focus
    activateWindow();
    raise();
}


/*!
  newAudioData updates the audio graphs.
*/
void MainWindow::newAudioData(int counter, int value)
{
    // update GUI if inactive or if set to always update
    if (!itsIsScreenOff) {
        itsAudioLevelGraphics->AddValue(value);
        itsAudioCounterGraphics->AddValue(counter);
    }
}


/*!
  showNotificationError displays the error message if the user notification
  failed.
*/
void MainWindow::showNotificationError() const
{
    // determine proper error message
    QString errorMsg;
    if (itsSettings->itsUserNotifyScript.isEmpty())
        errorMsg = tr("Could not establish phone call. Please check the phone number settings.");
    else
        errorMsg = QString(tr("Could not start user script %1.")
                           .arg(itsSettings->itsUserNotifyScript));

    // create message box
    QMessageBox *msgBox = new QMessageBox(QMessageBox::Critical,
                             tr("Babyphone"), errorMsg);

    // the message box needs a timeout to keep the application active for retry
    // no need to delete the timer, this is done together with the message box
    QTimer *msgBoxCloseTimer = new QTimer(msgBox);
    msgBoxCloseTimer->setInterval(itsSettings->MSG_BOX_TIMEOUT);
    msgBoxCloseTimer->setSingleShot(true);
    connect(msgBoxCloseTimer, SIGNAL(timeout()), msgBox, SLOT(accept())); // or reject
    msgBoxCloseTimer->start();

    // now we are ready to display the message box, the safety timer runs
    msgBox->exec();

    // delete message box, and with this also the timer
    delete msgBox;
}


/*!
  displayDimmed updates the itsIsScreenOff flag based on DBus messages.
  Consider that itsIsScreenOff will be true even if the application is in
  background but the screen is unlocked.
*/
void MainWindow::displayDimmed(const QDBusMessage &msg)
{
    QString value = msg.arguments()[0].toString();
    if (value == "off") {
        itsIsScreenOff = true;
        // empty graphs to be ready after next screen unlock
        itsAudioLevelGraphics->Clear();
        itsAudioCounterGraphics->Clear();
    }
    else if (value == "on") {
        itsIsScreenOff = false;
    }
}


/*!
  showFirstRunInfo displays a popup at first run with some basic information.
*/
void MainWindow::showFirstRunInfo() const
{
    // setup message
    QDBusMessage msg = QDBusMessage::createMethodCall(
            "org.freedesktop.Notifications", // --dest
            "/org/freedesktop/Notifications", // destination object path
            "org.freedesktop.Notifications", // message name (w/o method)
            "SystemNoteDialog" // method
        );
    QList<QVariant> args;
    args.append("Welcome to Babyphone!\nPlease consider: Since this application performs audio monitoring and analysis even when the device is locked the battery usage will be higher while babyphone is running.");
    args.append(static_cast<quint32>(0));
    args.append("OK");
    msg.setArguments(args);

    // display message
    QDBusMessage reply = QDBusConnection::systemBus().call(msg);
    if (reply.type() == QDBusMessage::ErrorMessage) {
        qWarning() << "Could not display information message.";
    }
}
