#include "callmonitor.h"

// the DBUS phone call handling as implemented only works in Maemo 5 systems
#ifndef Q_WS_MAEMO_5
  #error "DBUS user notification is only supported for Maemo 5"
#endif
#include <QtDBus>


// taken from telephony-maemo.c
#define CSD_CALL_STATUS_COMING          2
#define CSD_CALL_STATUS_MT_ALERTING     5



/*!
  receiveCall gets called on an incoming phone call and handles this.
  The call handling equals a drop or taking of the call if configured in the
  settings. Otherwise the event is ignored.
*/
CallMonitor::CallMonitor(Settings *settings, QObject *parent) :
    QObject(parent), itsSettings(settings)
{
    // setup variables
    // we assume that at application startup no call is pending
    // this may be wrong, but has only limited negative effects
    itsCallPending = false;
    itsTakeNextCall = false;

    // setup call timer
    itsCallTimer = new QTimer(this);
    itsCallTimer->setSingleShot(true);
    connect(itsCallTimer, SIGNAL(timeout()), this, SLOT(callTimer()));

    // register to receive incoming calls
    bool result = QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                          "/com/nokia/csd/call", "com.nokia.csd.Call", "Coming",
                          this, SLOT(receiveCall(const QDBusMessage&)));
    if (result == false)
        qWarning() << tr("Cannot connect to incoming calls: ") << QDBusConnection::systemBus().lastError();


    // register to receive call status notification
    result = QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                     "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                     "CallStatus", this, SLOT(callStatusUpdate(const QDBusMessage&)));
    if (result == false)
        qWarning() << tr("Cannot connect to call establishment notifications: ") << QDBusConnection::systemBus().lastError();


    // register to receive call establishment notifications
    result = QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                     "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance",
                     "AudioConnect", this, SLOT(callEstablished(const QDBusMessage&)));
    if (result == false)
        qCritical() << tr("Cannot connect to call establishment notifications: ") << QDBusConnection::systemBus().lastError();
}


/*!
  receiveCall gets called on an incoming phone call and handles this.
  The call handling equals a drop or taking of the call if configured in the
  settings. Otherwise the event is ignored.
*/
void CallMonitor::receiveCall(const QDBusMessage &msg)
{
    QList<QVariant> lst = msg.arguments();
    QString caller = lst[1].toString();
    qDebug() << tr("Receive call from ") << caller;

    // per default, we do not take the call
    itsTakeNextCall = false;

    if (itsSettings->itsHandleIncomingCalls) {
        // handle incoming calls
        // take the call if it is from the parent's phone and no call is pending
        if ( (itsSettings->itsContact.IsNumberMatching(caller)) && !itsCallPending ) {
            // we cannot answer the call right now, we need the phone to be in a proper call state
            itsTakeNextCall = true;
        }
        else {
            // drop the call
            if (dropCall()) {
                // notify parents on this event
                emit callReceived(caller);
            }
        }
    }
    else {
        // call handling is inactive
        // nevertheless, we signal the call
        emit callReceived(caller);
    }
}


/*!
  callStatusUpdate checks the call status event and potentially takes the call
  as it is ready.
*/
void CallMonitor::callStatusUpdate(const QDBusMessage &msg)
{
    int callStatus = msg.arguments()[0].toInt();

    if ( (itsTakeNextCall) && (callStatus >= CSD_CALL_STATUS_MT_ALERTING) ) {
        // now we are ready to take the call
        itsTakeNextCall = false;
        takeIncomingCall();
    }
}


/*!
  callEstablished get called as a phone call start or stop event is received.
  If a call is started it extends the call timeout. If it is finished it signals
  the end of notification.
*/
void CallMonitor::callEstablished(const QDBusMessage &msg)
{
    bool flag0 = msg.arguments()[0].toBool();
    bool flag1 = msg.arguments()[1].toBool();

    // is this the start or end of the call?
    if (flag0 && flag1) {
        // start of call
        // start the call safety timer if we took the call
        itsCallPending = true;
        emit callStatusChanged(true);
    }
    else if (!flag0 && !flag1) {
        // end of call
        // stop the call safety timer, which may (or may not) run
        itsCallTimer->stop();

        itsCallPending = false;
        emit callStatusChanged(false);
    }
}


/*!
  dropCall drops an incoming call using the DBus.
*/
bool CallMonitor::dropCall() const
{
    // setup proper DBUS message
    QDBusMessage msg = QDBusMessage::createMethodCall(
            "com.nokia.csd.Call",       // --dest
            "/com/nokia/csd/call",      // destination object path
            "com.nokia.csd.Call",       // message name (w/o method)
            "Release"                   // method
        );
    QDBusMessage reply = QDBusConnection::systemBus().call(msg);

    if (reply.type() == QDBusMessage::ErrorMessage) {
        qWarning() << tr("Call handling failed: ") << QDBusConnection::systemBus().lastError();
        return false;
    }
    qDebug() << tr("Call dropped");

    return true;
}


/*!
  takeIncomingCall answers an incoming call using the DBus.
*/
bool CallMonitor::takeIncomingCall()
{
    QDBusMessage msg = QDBusMessage::createMethodCall(
            "com.nokia.csd.Call",       // --dest
            "/com/nokia/csd/call/1",    // destination object path
            "com.nokia.csd.Call.Instance",  // message name (w/o method)
            "Answer"                    // method
        );
    QDBusMessage reply = QDBusConnection::systemBus().call(msg);

    if (reply.type() == QDBusMessage::ErrorMessage) {
        qWarning() << tr("Call handling failed: ") << QDBusConnection::systemBus().lastError();
        return false;
    }
    qDebug() << tr("Call taken");

    // start safety timer
    itsCallTimer->start(itsSettings->CALL_HOLD_TIMER);

    return true;
}


/*!
  callTimer gets triggered by a safety timer and drops the current call.
*/
void CallMonitor::callTimer()
{
    // as this safety timer expires, drop current call
    dropCall();
}
