/**
 * Copyright (C) 2009, 2010 Floriano Scioscia.
 *
 * This file is part of 100 Boxes.
 *
 * 100 Boxes is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * 100 Boxes is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with 100 Boxes. If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include "sound.h"

/* Data structures */
struct ss {
	gchar *name;
	GstElement *pipeline;
};

static void eos_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline);
static void stock_eos_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline);
static void error_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline);
static void free_pipeline(GstElement *pipeline);
static struct ss* find_stock_element(GList *stock, const gchar *name);
static void destroy_stock_element(struct ss* el);
gint stock_sound_cmp(gconstpointer a, gconstpointer b);

static void error_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline) {
	GError *err;
	gchar *debug;
	gst_message_parse_error(message, &err, &debug);
	g_printerr("Error: %s", debug);
	/* stop playback and free pipeline */
	gst_element_set_state(pipeline, GST_STATE_NULL);
	free_pipeline(pipeline);
}

static void eos_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline) {
	/* stop playback and free pipeline */
	gst_element_set_state(pipeline, GST_STATE_NULL);
	//g_debug("PLAYING FINISHED");
	free_pipeline(pipeline);
}

static void stock_eos_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline) {
	/* pause playback and rewind (seek to beginning) */
	//g_debug("PLAYING FINISHED");
	gst_element_set_state(pipeline, GST_STATE_PAUSED);
	gst_element_seek_simple(pipeline, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT, 0);
	//g_debug("STREAM SOUGHT BACK TO BEGINNING");
}

static void free_pipeline(GstElement *pipeline) {
	gst_object_unref(GST_OBJECT(pipeline));
	pipeline = NULL;
}

static void destroy_stock_element(struct ss* el) {
	gst_element_set_state(el->pipeline, GST_STATE_NULL);
	free_pipeline(el->pipeline);
	g_free(el->name);
	g_free(el);
}

static struct ss* find_stock_element(GList *stock, const gchar *name) {
	struct ss *template;
	GList *elem;

	if (stock != NULL) {
		template = (struct ss *)g_malloc(sizeof(struct ss));
		template->name = (gchar*)name;
		elem = g_list_find_custom(stock, template, stock_sound_cmp);
		g_free(template);
		if (elem == NULL) {
			return NULL;
		} else {
			return (struct ss*) elem->data;
		}
	} else {
		return NULL;
	}
}

void sound_init(int *argc, char **argv[]) {
	gst_init(argc, argv);
}

int play_sound(const gchar *path) {
	GstElement *pipeline;
	GstElement *filesrc;
	GstBus *bus;
	GError *error = NULL;

	/* setup pipeline and configure elements */
	pipeline = gst_parse_launch ("filesrc name=my_filesrc ! decodebin ! audioconvert ! audioresample ! autoaudiosink", &error);
	if (!pipeline) {
		g_printerr("Parse error: %s\n", error->message);
		return ERR_PIPELINE_SETUP_FAILED;
	}
	filesrc = gst_bin_get_by_name(GST_BIN(pipeline), "my_filesrc");
	if (!filesrc) {
		g_printerr("Parse error: no filesrc");
		free_pipeline(pipeline);
		return ERR_PIPELINE_SETUP_FAILED;
	}
	g_object_set(G_OBJECT(filesrc), "location", path, NULL);

	/* setup message handling */
	bus = gst_pipeline_get_bus(GST_PIPELINE (pipeline));
	gst_bus_add_signal_watch_full(bus, G_PRIORITY_HIGH);
	g_signal_connect(bus, "message::eos", (GCallback)eos_message_received, pipeline);
	g_signal_connect(bus, "message::error", (GCallback)error_message_received, pipeline);
	gst_object_unref(GST_OBJECT(bus));
	
	/* start playback */
	gst_element_set_state(pipeline, GST_STATE_PLAYING);
	//g_debug("PLAYING STARTED");
	return ERR_NONE;	
}

GList* add_stock_sound(GList *stock, const gchar *name, const gchar *path, gint *err) {
	struct ss *elem;
	GList *newstock;
	GstElement *pipeline;
	GstElement *filesrc;
	GstBus *bus;
	GError *error = NULL;

	/* setup pipeline and configure elements */
	pipeline = gst_parse_launch ("filesrc name=my_filesrc ! decodebin ! audioconvert ! audioresample ! autoaudiosink", &error);
	if (!pipeline) {
		g_printerr("Parse error: %s\n", error->message);
		*err = ERR_PIPELINE_SETUP_FAILED;
		return NULL;
	}
	filesrc = gst_bin_get_by_name(GST_BIN(pipeline), "my_filesrc");
	if (!filesrc) {
		g_printerr("Parse error: no filesrc");
		free_pipeline(pipeline);
		*err = ERR_PIPELINE_SETUP_FAILED;
		return NULL;
	}
	g_object_set(G_OBJECT(filesrc), "location", path, NULL);

	/* setup message handling */
	bus = gst_pipeline_get_bus(GST_PIPELINE (pipeline));
	gst_bus_add_signal_watch_full(bus, G_PRIORITY_HIGH);
	g_signal_connect(bus, "message::eos", (GCallback)stock_eos_message_received, pipeline);
	g_signal_connect(bus, "message::error", (GCallback)error_message_received, pipeline);
	gst_object_unref(GST_OBJECT(bus));
	
	elem = g_malloc(sizeof(struct ss));
	elem->name = g_malloc(sizeof(strlen(name)+1));
	strcpy(elem->name, name);
	elem->pipeline = pipeline;

	newstock = g_list_insert_sorted(stock, elem, stock_sound_cmp);
	*err = ERR_NONE;
	return newstock;
}

gint stock_sound_cmp(gconstpointer a, gconstpointer b) {
	struct ss *sa, *sb;
	sa = (struct ss*) a;
	sb = (struct ss*) b;
	return (gint)strcmp(sa->name, sb->name);
}

gint play_stock_sound(GList *stock, const gchar *name) {
	struct ss *found = find_stock_element(stock, name);
	if (found != NULL) {
		gst_element_set_state(found->pipeline, GST_STATE_PLAYING);
	}
}

void remove_stock_sound(GList *stock, const gchar *name) {
	struct ss *found = find_stock_element(stock, name);
	if (found != NULL) {
		destroy_stock_element(found);
	}
}

void clear_stock(GList *stock) {
	GList *current = g_list_first(stock);
	while (current != NULL) {
		destroy_stock_element((struct ss*)current->data);
		current = g_list_next(current);		
	}
	g_list_free(stock);
}
