/**
 *  Viito -- an image dictionary for sign language
 *
 *  Copyright (C) 2010  Aki Niemi <aki.niemi@iki.fi>
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QtGui>
#include "vsearchwidget.h"
#include "vimagewidget.h"
#include "vdatamodel.h"

VSearchWidget::VSearchWidget(QWidget *parent)
    : QWidget(parent)
{
    QVBoxLayout *vbox = new QVBoxLayout(this);

    mView = new QListView(this);

    mEdit = new QLineEdit(this);
    mEdit->setPlaceholderText(tr("Enter search term"));
    mEdit->setEchoMode(QLineEdit::Normal);

    vbox->addWidget(mView);
    vbox->addWidget(mEdit);

    mImageWidget = new VImageWidget(this);

    connect(mView, SIGNAL(clicked(const QModelIndex&)),
	    this, SLOT(imageClicked(const QModelIndex &)));
    connect(mImageWidget, SIGNAL(next()), this, SLOT(nextImage()));
    connect(mImageWidget, SIGNAL(previous()), this, SLOT(previousImage()));
    connect(mEdit, SIGNAL(textChanged(const QString&)),
	    this, SLOT(searchEdited(const QString &)));

    setWindowFlags(windowFlags() | Qt::Window);
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_Maemo5AutoOrientation, false);
    setAttribute(Qt::WA_Maemo5LandscapeOrientation);
}

void VSearchWidget::show()
{
    if (!mEdit->text().isEmpty()) {
	mEdit->selectAll();
	mEdit->setFocus();
    }
    QWidget::show();
}

void VSearchWidget::searchEdited(const QString &text)
{
    if (text.isEmpty()) {
	mModel->invalidate();
	return;
    }
    
    mModel->setFilterWildcard(QString("%1*").arg(text));
}

void VSearchWidget::imageClicked(const QModelIndex &index)
{
    setImage(index);
    mImageWidget->show();
}

void VSearchWidget::setModel(QStandardItemModel *model)
{
    mModel = new QSortFilterProxyModel(this);
    mModel->setSourceModel(model);
    mModel->setFilterCaseSensitivity(Qt::CaseInsensitive);
    mModel->setSortCaseSensitivity(Qt::CaseInsensitive);
    mModel->sort(0);
    mView->setModel(mModel);
}

void VSearchWidget::setImage(const QModelIndex &index, int offset)
{
    QModelIndex next;

    if (offset != 0) {
	next = index.sibling(index.row() + offset, 0);
	if (!next.isValid())
	    return;

	mView->scrollTo(next);
	mView->setCurrentIndex(next);
    } else {
	next = index;
    }

    mImageWidget->setWindowTitle(next.data().toString());
    mImageWidget->setImage(next.data(VImageItem::PathRole).toString(),
			   QString("\251 %1").arg(next.data(VImageItem::AuthorRole).toString()));
}

void VSearchWidget::nextImage()
{
    setImage(mView->currentIndex(), 1);
}

void VSearchWidget::previousImage()
{
    setImage(mView->currentIndex(), -1);
}

