//  © Copyright 2010 Reuben D'Netto (rdnetto+uremote@gmail.com)
//  This file is part of uRemote v1.0
//
//  uRemote is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  uRemote is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with uRemote.  If not, see <http://www.gnu.org/licenses/>.


#include "torrentdata.h"
#include <QString>
#include <QList>
#include <QMessageBox>
#include <cmath>

TorrentData::TorrentData(QStringList parts){
	//HASH (string)
	Hash = parts.at(0).mid(1, parts.at(0).length() - 2);
	//STATUS* (integer)
	Status = static_cast<TorrentStatus>(parts.at(1).toInt());
	//NAME (string)
	Name = parts.at(2).mid(1, parts.at(2).length() - 2);
	//SIZE (integer in bytes)
	Size = parts.at(3).toInt();
	//PERCENT PROGRESS (integer in per mils)
	Progress = parts.at(4).toDouble() / 1000l;
	//DOWNLOADED (integer in bytes)
	Downloaded = parts.at(5).toLong();
	//UPLOADED (integer in bytes)
	Uploaded = parts.at(6).toLong();
	//RATIO (integer in per mils)
	Ratio = parts.at(7).toDouble() / 1000l;
	//UPLOAD SPEED (integer in bytes per second)
	UpSpeed = parts.at(8).toInt();
	//DOWNLOAD SPEED (integer in bytes per second)
	DownSpeed = parts.at(9).toInt();
	//ETA (integer in seconds)
	ETA = parts.at(10).toInt();
	//LABEL (string)
	Label = parts.at(11).mid(1, parts.at(11).length() - 2);
	//PEERS CONNECTED (integer)
	PeersCon = parts.at(12).toInt();
	//PEERS IN SWARM (integer)
	PeersTotal = parts.at(13).toInt();
	//SEEDS CONNECTED (integer)
	SeedsCon = parts.at(14).toInt();
	//SEEDS IN SWARM (integer)
	SeedsTotal = parts.at(15).toInt();
	//AVAILABILITY (integer in 1/65535ths)
	Availability = parts.at(16).toDouble() / 65535l;
	//TORRENT QUEUE ORDER (integer)
	QueueOrder = parts.at(17).toInt();
	//REMAINING (integer in bytes)
	Remaining = parts.at(18).toLong();
}

const QString TorrentData::StatusStr(){return StatusStr(Status, Progress);}
const QString TorrentData::StatusStr(TorrentStatus Status, double Progress){
	//the following code is ported from the WebUI
	//return QString::number(Status, 2);
	//1100 1000 - queued down
	//1000 1001 - forced down
	//1100 1001 - down

	QString result = "";

	if(! (Status & TS_Loaded))
		return "Not Loaded";

	if(Status & TS_Started){
		if(Status & TS_Paused){
			result = "Paused";
		}else{
			result = (Progress == 1 ? "Seeding" : "Downloading");

			if(! (Status & TS_Queued) && !(Status & TS_Checking))
				result = "[F] " + result;
		} //end if

	}else{

		if(Status & TS_Checking){
			result = "Checking";

		}else{
			if(Status & TS_Error){
				result = "Error";
			}else{
				if(Status & TS_Queued){
					//B=[(A==1000)?"Status_Queued_Up":"Status_Queued_Down",lang[CONST.OV_FL_QUEUED]]
					result = (Progress == 1 ? "Queued Seed" : "Queued");
				} //end if
			} //end if
		} //end if
	} //end if


	if(result == "")
		result = (Progress == 1 ? "Finished" : "Stopped");

	return result; // + " " + QString::number(Status, 2);
}
const QString TorrentData::EtaStr(){
	if(ETA == -1) return QString::fromUtf8("∞");

	int sec,min,hour,day,week,year;
	sec = ETA % 60;
	min = ETA / 60;
	hour = ETA / 3600;
	day = ETA / (3600 * 24);
	week = ETA / (3600 * 24 * 7);
	year = ETA / (3600 * 24 * 365.25);

	QString result;
	int vals = 2;
	if(vals > 0 && year != 0){ result += QString::number(year) + "y "; vals -= 1; }
	if(vals > 0 && week != 0){ result += QString::number(week) + "w "; vals -= 1; }
	if(vals > 0 && day != 0){  result += QString::number(day)  + "d "; vals -= 1; }
	if(vals > 0 && hour != 0){ result += QString::number(hour) + "h "; vals -= 1; }
	if(vals > 0 && min != 0){  result += QString::number(min)  + "m "; vals -= 1; }
	if(vals > 0 && sec != 0){  result += QString::number(sec)  + "s "; vals -= 1; }
	return result;
}

const QString TorrentData::DownSpeedStr(){
	if(DownSpeed == 0)
		return "";
	else
		return FormatSize(DownSpeed) + "/s";
}
const QString TorrentData::UpSpeedStr(){
	if(UpSpeed == 0)
		return "";
	else
		return FormatSize(UpSpeed) + "/s";
}
const QString TorrentData::DownloadedStr(){
	return FormatSize(Downloaded);
}
const QString TorrentData::UploadedStr(){
	return FormatSize(Uploaded);
}
const QString TorrentData::SizeStr(){
	return FormatSize(Size);
}
const QString TorrentData::RemainingStr(){
	return FormatSize(Remaining);
}

QString TorrentData::FormatSize(long bytes){
	if(bytes == 0) return "0 B";

	double order = floor((log(bytes) / log(2)) / 10);
	if(order > 5) order = 5; //highest unit is PB
	double size = Round(bytes / pow(2, 10 * order), 3);

	switch(int(order)){
		case 0:
		return QString::number(size) + " B";
		case 1:
		return QString::number(size) + " KB";
		case 2:
		return QString::number(size) + " MB";
		case 3:
		return QString::number(size) + " GB";
		case 4:
		return QString::number(size) + " TB";
		case 5:
		return QString::number(size) + " PB";
		default:
		return "???";
	} //end switch
}

TorrentFile::TorrentFile(QStringList data){
	// FILE NAME (string),
	Filename = data.at(0).mid(1, data.at(0).length() - 2);
	// FILE SIZE (integer in bytes),
	Size = data.at(1).toLong();
	// DOWNLOADED (integer in bytes),
	Downloaded = data.at(2).toLong();
	// PRIORITY* (integer)
	Priority = static_cast<TFPriority>(data.at(3).toInt());
}
QString TorrentFile::SizeStr(){
	return TorrentData::FormatSize(Size);
}
QString TorrentFile::DownloadedStr(){
	return TorrentData::FormatSize(Downloaded);
}
QString TorrentFile::Progress(){
	return QString::number(round(double(Downloaded) * 10000l / double(Size)) / 100l) + "%";
}
QString TorrentFile::PriorityStr(){
	return PriorityStr(Priority);
}

QString TorrentFile::PriorityStr(TFPriority p){
	switch(p){
		case TFP_Skip:
		return "Skip";
		case TFP_Low:
		return "Low";
		case TFP_Normal:
		return "Normal";
		case TFP_High:
		return "High";
		default:
		return "";
	} //end switch
}
