//  © Copyright 2010 Reuben D'Netto (rdnetto+uremote@gmail.com)
//  This file is part of uRemote v1.0
//
//  uRemote is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  uRemote is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with uRemote.  If not, see <http://www.gnu.org/licenses/>.


#include "json.h"

QMap<QString, QString> JSON::ParseDictionary(QString data){
	QMap<QString, QString> result;
	if(data.length() == 0) return result;

	//strip surrounding {} and spaces
	data = data.mid(1, data.length() - 2);

	int marker = 0;
	int level = 0;


	for(int i = 0; i < data.length(); i += 1){
		switch(data.at(i).toAscii()){
		case '"':
			i = data.indexOf('"', i + 1);
			if(i == -1){
				qDebug("Error: Invalid data");
				return result;
			} //end if
		break;
		case '[':
			level += 1;
		break;
		case ']':
			level -= 1;
		break;
		case ',':
		if(level != 0) continue;

		AddEntry(result, data.mid(marker, i - marker));
		marker = i + 1;
		break;

		} //end switch
	} //end for

	//handle last element
	AddEntry(result, data.mid(marker, data.length() - marker));

	return result;
}

QStringList JSON::SplitArray(QString data){
	//Example forms:
	//[...,...,...],[...,...,...],[...,...,...]
	//[...],[...],[...]
	//Output: {"[...]", "[...]"}

	data = data.replace("\n", "");
	//qDebug(data.toLocal8Bit());
	//qDebug("=====================================================");
	if(data.isEmpty()) return QStringList();

	int marker = 0;
	int level = 0;
	QStringList result;

	for(int i = marker; i < data.length(); i += 1){
		switch(data.at(i).toAscii()){
		case '"':
			i = data.indexOf('"', i + 1);
		break;
		case '[':
			level += 1;
		break;
		case ']':
			level -= 1;
		break;
		case ',':
			if(level == 0){
				result.append(Trim(data.mid(marker, i - marker)));
				//qDebug(('{' + result.last() + '}').toLocal8Bit());
				marker = i + 1;
			} //end if
		break;

		} //end switch
	} //end for

	//handle last element
	result.append(Trim(data.mid(marker, data.length() - marker)));
	//qDebug(('{' + result.last() + '}').toLocal8Bit());

	//qDebug("=====================================================");
	return result;
}

void JSON::AddEntry(QMap<QString, QString>& dict, QString data){
	//entry will be of the following form:
	//"build":17414
	//"error": "Error"
	//"files": [...,...,...]
	//if(data.length() < 6000) qDebug(data.toLocal8Bit());
	QString key, value;
	data = data.replace("\n", "");

	int i = data.indexOf('"');
	key = data.mid(i + 1, data.indexOf('"', i + 1) - i - 1);

	i = data.indexOf(':') + 1;
	while(data.at(i).isSpace()) i += 1;

	if(data.at(i) == '"'){
		value = Trim(data.mid(i + 1, data.lastIndexOf('"') - i - 1));
	}else if(data.at(i) == '['){
		value = Trim(data.mid(i + 1, data.lastIndexOf(']') - i - 1));
	}else{
		value = Trim(data.mid(i));
	} //end if

	dict[key] = value;

	//qDebug(("Key: " + keys.last()).toLocal8Bit());
	//qDebug(("Value: " + values.last()).toLocal8Bit());
}

QStringList JSON::ParseLabels(QString data){
	QStringList result;

	foreach(QString entry, SplitArray(data)){
		entry = entry.mid(1, entry.length() - 2); //strip []
		QString label = SplitArray(entry).at(0);
		label = label.mid(1, label.length() - 2); //strip ""
		result.append(label);
	} //end foreach

	return result;
}

QStringList JSON::ParseHashes(QString data){
	data = data.mid(1, data.length() - 2); //strip []
	QStringList result = SplitArray(data);

	//strip ""
	for(int i = 0; i < result.length(); i += 1)
		result[i] = result[i].mid(1, result[i].length() - 2);

	return result;
}

QList<TorrentData> JSON::ParseTorrents(QString data){
	QList<TorrentData> result;

	foreach(QString tor, SplitArray(data)){
		tor = tor.mid(1, tor.length() - 2); //strip []
		QStringList elements = SplitArray(tor);
		result.append(TorrentData(elements));
	} //end foreach

	return result;
}

void JSON::ParseFiles(QList<TorrentData>& torrents, QMap<QString, QList<TorrentFile> >& filelists, QString data){
	QStringList rawData = SplitArray(data);
	QString hash = rawData.at(0);
	hash = hash.mid(1, hash.length() - 2); //strip ""

	data = rawData.at(1);
	data = data.mid(1, data.length() - 2); //strip []
	rawData = SplitArray(data);

	int index = -1;
	for(int i = 0; i < torrents.length(); i += 1){
		if(torrents.at(i).Hash == hash){
			index = i;
			break;
		} //end if
	} //end for

	if(index == -1) return;
	torrents[index].Files.clear();

	foreach(QString file, rawData){
		file = file.mid(1, file.length() - 2); //strip []
		torrents[index].Files.append(TorrentFile(SplitArray(file)));
	} //end foreach

	filelists[torrents[index].Hash] = torrents[index].Files;
}

QString JSON::Trim(QString data){
	if(data.length() == 0) return data;

	while(data.at(0).isSpace())
		data = data.mid(1);

	while(data.right(1).at(0).isSpace())
		data = data.mid(0, data.length() - 1);

	return data;
}
