/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "loginctrl.h"

/*! 
 * \brief LoginCtrl constructor.
 */
LoginCtrl::LoginCtrl(bool autologin, QGraphicsScene *scene)
    : _autologin(autologin)
{
    _scene = scene;

    _loginView = new LoginView();
    showView();

    connect(_loginView, SIGNAL(lockRotation(bool)), SIGNAL(lockRotation(bool)));
    connect(_loginView, SIGNAL(done()), SLOT(deleteLater()));
    connect(_loginView, SIGNAL(exitApp()), SLOT(_exit()));
    connect(_loginView, SIGNAL(loginAccount()), SLOT(_loginTwitter()));
    connect(_loginView, SIGNAL(abort()), SLOT(_abort()));
    connect(_loginView, SIGNAL(newAccount()), SLOT(_populateUserList()));
    connect(_loginView, SIGNAL(accRemoved()), SLOT(_populateUserList()));
    connect(Backend::getInstance(), SIGNAL(signedOn()), _loginView, SLOT(connected()));
    connect(Backend::getInstance(), SIGNAL(connectionError(QString)), this, SLOT(_connectionError(QString)));

    _populateUserList();
    _loadSettings();
}

/*! 
 * \brief LoginCtrl destructor.
 */
LoginCtrl::~LoginCtrl(void)
{
    if (_loginView) {
        delete _loginView;
    }
    emit done();
}

/*! 
 * \brief Show a dialog window with a error message
 * \param message the message in the dialog
 */
void LoginCtrl::_connectionError(QString message)
{
    // TODO: error handler
}

/*! 
 * \brief Creates the view and adds the scene
 */
void LoginCtrl::showView(void)
{
    _loginView->createUi();
    _scene->addItem(_loginView);
}

/*! 
 * \brief Try to login on the twitter.
 */
void LoginCtrl::_loginTwitter(void)
{
    // set ConsumerKey e ConsumerSecret
    Backend::getInstance()->setConsumerKey(OauthWizard::ConsumerKey);
    Backend::getInstance()->setConsumerSecret(OauthWizard::ConsumerSecret);

    // Get Selected user
    Account *user = AccountManager::instance()->getSelected();

    // And set them to connect
    Backend::getInstance()->setLogin(user->getUsername());
    Backend::getInstance()->setPassword(user->getTokenString());

    // Finally, try to connect
    Backend::getInstance()->verifyCredentials();
}

/*!
 * \brief Exits the application.
 */
void LoginCtrl::_exit(void)
{
    QApplication::quit();
}

/*!
 * \brief Loads username from a file if it exist.
 */
void LoginCtrl::_loadSettings(void)
{
    if(!_autologin)
        return;

    if(AccountManager::instance()->hasAutoLogin()) {
        int sel = AccountManager::instance()->autoLoginPosition();
        _loginView->usersList->setSelected(sel);
        _loginView->emitLoginAccount();
    }
}

/*!
 * \brief Aborts any connections in progress.
 */
void LoginCtrl::_abort()
{
    Backend::getInstance()->abort();
}

void LoginCtrl::_populateUserList(void)
{  
    _loginView->usersList->clearList();

    QList<Account>* accounts = AccountManager::instance()->getUsers();

    if (!accounts->size()) {
        User *newUser = new User;
        newUser->blockSignals(true);
        _loginView->usersList->addToList(newUser, 0);
    }
    else {
        for(int i = 0; i < accounts->size(); i++) {
            Account account = accounts->value(i);
            User *newUser = new User(&account);
            _loginView->usersList->addToList(newUser,i);
        }
    }
}
