/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
* aumiks 0.1
*/

/*
 * STK library (http://ccrma.stanford.edu/software/stk/index.html) was used as
 * a reference while creating this class.
 */

#pragma once

#include <ting/debug.hpp>

#include "config.hpp"

#include "FilterOneZero.hpp"
#include "FilterOnePole.hpp"
#include "DelayAllPass1.hpp"
#include "NoiseGen.hpp"


namespace aumiks{

class PluckedString{
	FilterOneZero loopFilter;
	FilterOnePole pluckFilter;
	DelayAllPass1 delayLine;
	real curLoopGain;
	real loopGain;
public:
	PluckedString(aumiks::real lowestFrequency){
		real delayLength = 44100 / lowestFrequency + 1;
		this->loopGain = 0.999;
		this->delayLine.Init(delayLength, 0.5 * delayLength);
		this->Clear();
	}

	void Clear(){
		this->delayLine.Clear();
		this->loopFilter.Clear();
		this->pluckFilter.Clear();
	}

	void SetFrequency(real frequency){
		ASSERT(frequency > 0)

		// Delay = length - approximate filter delay.
		real delay = (44100 / frequency) - 0.5;
		if(delay <= 0.5)//0.5 is the minimum delay allowed by DelayAllPass1 class
			delay = 0.5;
		else if(delay > this->delayLine.MaxDelay())
			delay = this->delayLine.MaxDelay();

		this->delayLine.SetDelay(delay);

		this->curLoopGain = 0.995 + (frequency * 0.000005);
		if(this->curLoopGain >= 1.0 )
			this->curLoopGain = 0.99999;
	}

	void SetLoopGain(real gain){
		ASSERT(gain > 0 && gain <= 1)
		this->loopGain = gain;
		this->curLoopGain = gain;
	}

	inline void SetLoopFilterCoefs(real a0, real a1){
		this->loopFilter.SetCoefs(a0, a1);
	}

	/**
	 * @brief excite the string, i.e. simulate plucking.
	 * This method fills the delay line with initial values which are
	 * generated by noise generator.
	 * @param amplitude - plucking amplitude, a value from [0, 1] interval.
	 */
	void Pluck(real amplitude){
		ASSERT(amplitude >= 0 && amplitude <= 1)

		this->curLoopGain = this->loopGain;

		this->pluckFilter.SetPole(0.970);

		//Fill delay with noise
		unsigned len = unsigned(this->delayLine.Delay());
		for(unsigned i = 0; i < len; ++i){
			this->delayLine.Tick(
					this->pluckFilter.Tick(amplitude * NoiseGen::Tick())
				);
		}
	}

	void Init(){
		this->curLoopGain = this->loopGain;
	}

	void Mute(real releaseGain = 0.8){
		this->curLoopGain = releaseGain;
	}

	inline aumiks::real Tick(aumiks::real input = 0){
		return this->delayLine.Tick(
				this->loopFilter.Tick(
						this->delayLine.LastOutValue() * this->curLoopGain
					) + input
			) + input;
	}
};

}//~namespace
