/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */


#include <cmath>
#include <gtkmm/dialog.h>
#include <gtkmm/main.h>
#include <cstdlib>

#include <ting/Exc.hpp>
#include <ting/math.hpp>
#include <ting/Array.hpp>
#include <ting/Ref.hpp>
#include <ting/Timer.hpp>
#include <ting/Thread.hpp>
#include <ting/utils.hpp>
#include <pugixml/pugixml.hpp>

#include "stdafx.hpp"
#include "Preferences.hpp"
#include "dialogs/OptionsDialog.hpp"
#include "KeyboardWindow.hpp"
#include "File.hpp"
#include "FSFile.hpp"
#include "MainWindow.hpp"
#include "MainThread.hpp"
#include "utils.hpp"



using namespace ting;



KeyboardWindow::KeyboardWindow(){

#ifndef M_NON_MAEMO_LINUX
	this->optionsMenuItem.signal_clicked().connect(
			sigc::mem_fun(*this, &KeyboardWindow::ShowOptionsDialog)
		);
#else
	this->optionsMenuItem.signal_activate().connect(
			sigc::mem_fun(*this, &KeyboardWindow::ShowOptionsDialog)
		);
#endif
	this->optionsMenuItem.set_sensitive(true);



	this->add_events(
			Gdk::BUTTON_PRESS_MASK |
			Gdk::BUTTON_RELEASE_MASK |
			Gdk::POINTER_MOTION_MASK |
			Gdk::BUTTON_MOTION_MASK
		);

	this->signal_button_press_event().connect(
			sigc::mem_fun(*this, &KeyboardWindow::OnButtonPressEvent)
		);
	this->signal_button_release_event().connect(
			sigc::mem_fun(*this, &KeyboardWindow::OnButtonReleaseEvent)
		);
	this->signal_motion_notify_event().connect(
			sigc::mem_fun(*this, &KeyboardWindow::OnMotionEvent)
		);

	//set CAN_FOCUS flag to be able to receive keyboard events
	this->set_flags(Gtk::CAN_FOCUS);

	//NOTE: if use simple connect() then Arrow Keys press events does not come,
	//this is the bug in Gtkmm. Thus, process arrow keys in OnKeyPress_Notify()
	//while process all other keys in OnKeyPress().
	this->signal_key_press_event().connect_notify(
			sigc::mem_fun(*this, &KeyboardWindow::OnKeyPress_Notify)
		);

	//set default timbre
	this->instr.ApplyTimbre(Preferences::Inst().GetKbTimbre());

	{
		KeyboardView *kw = new KeyboardView();
		this->add(*Gtk::manage(kw));
		kw->show();
	}
}



KeyboardWindow::KeyboardView::KeyboardView(){
	this->signal_expose_event().connect(
			sigc::mem_fun(*this, &KeyboardView::OnExposeEvent)
		);
}



void KeyboardWindow::ShowOptionsDialog(){
//	TRACE(<< "KeyboardWindow::ShowOptionsDialog(): enter" << std::endl)
	OptionsDialog d(MainWindow::Inst(), false);
	d.SetTimbre(Preferences::Inst().GetKbTimbre());

	switch(d.run()){
		case Gtk::RESPONSE_OK: //OK
			Preferences::Inst().SetKbTimbre(d.GetTimbre());
			this->instr.ApplyTimbre(Preferences::Inst().GetKbTimbre());

			Preferences::Inst().SaveToFile();
			break;
		default: //Cancel and all other cases
			break;
	}

	d.hide();
	MainWindow::Inst().queue_draw();//request redrawing entire window
//	TRACE(<< "KeyboardWindow::ShowOptionsDialog(): exit" << std::endl)
}



bool KeyboardWindow::KeyboardView::OnExposeEvent(GdkEventExpose* event){
//	TRACE(<< "KeyboardWindow::OnExposeEvent(): enter" << std::endl)

	Glib::RefPtr<Gdk::Window> window = this->get_window();
	if(!window)
		return true;

	Gtk::Allocation allocation = get_allocation();
	const int width = allocation.get_width();
	const int height = allocation.get_height();

	Cairo::RefPtr<Cairo::Context> c = window->create_cairo_context();

	if(event){
		// clip to the area indicated by the expose event so that we only
		// redraw the portion of the window that needs to be redrawn
		c->rectangle(event->area.x, event->area.y, event->area.width, event->area.height);
		c->clip();
	}

	c->set_line_width(1.0);
	c->set_source_rgb(0, 0, 0);//black
	c->select_font_face("Sans", Cairo::FONT_SLANT_NORMAL, Cairo::FONT_WEIGHT_NORMAL);
	c->set_font_size(50.0);

	for(unsigned i = 0; i < 3; ++i){
		c->save();
		c->translate(0, (i * height) / 3);
		KeyboardWindow::KeyboardView::DrawOctave(
				c,
				width,
				height / 3,
				Preferences::Inst().GetKbScrPos() + (3 - 1 - i) * 14
			);
		c->restore();
	}

//	TRACE(<< "KeyboardWindow::OnExposeEvent(): exit" << std::endl)

	return true;
}



static const char* notes[] = {
	"A", "B", "C", "D", "E", "F", "G"
};



//static
void KeyboardWindow::KeyboardView::DrawOctave(
		Cairo::RefPtr<Cairo::Context> c,
		unsigned w,
		unsigned h,
		unsigned baseScrPos
	)
{
	float height = float(h);
	float step = w / 14;

	//draw white background
	c->set_source_rgb(1, 1, 1);//white
	c->rectangle(0, 0, w, h);
	c->fill();

	float curX = - step;

	int n = baseScrPos % 14 - 1;
	if(n < 0)
		n += 14;

	for(unsigned i = 0; i < 15; ++i, ++n, curX += step){
		if(n >= 14)
			n -= 14;

		ASSERT(0 <= n && n < 14)

		if(n % 2 == 0){
			c->set_source_rgb(0, 0, 0);//black
			c->rectangle(curX, height / 2, step * 2, height / 2);

			c->save();
			c->set_identity_matrix();
			c->set_line_width(2);
			c->stroke();
			c->restore();

			Cairo::TextExtents extents;
			c->get_text_extents(notes[n / 2], extents);
			c->set_source_rgb(0, 0, 0);//black
			c->move_to(curX + step - (extents.width / 2) - extents.x_bearing, height * 7 / 8);
			c->show_text(notes[n / 2]);
		}else{
			if(n == 3 || n == 9){ // skip black keys for B-C and E-F
				c->set_source_rgb(0.5, 0.5, 0.5);//grey
			}else{
				c->set_source_rgb(0, 0, 0);//black
			}
			c->rectangle(curX + 1, 0, step * 2 - 1, height / 2);
			c->fill();
		}
	}

	//draw horizontal line delimiting octaves
//	c->move_to(0, h);
//	c->line_to(w, h);
//	c->save();
//	c->set_identity_matrix();
//	c->set_line_width(3);
//	c->stroke();
//	c->restore();

	//draw octave number
	c->set_source_rgb(0.5f, 0, 0);//dark red
	c->move_to(5, h - 5);
	char buf[33];
	snprintf(buf, sizeof(buf)-1, "%u", baseScrPos / 14);
	buf[32] = 0;
	c->show_text(buf);
}



struct MenuPlacer{
	int menuX, menuY;
	void onPopupMenuPosition(int& x, int& y, bool& pushIn){
		x = this->menuX;
		y = this->menuY;
		pushIn = true;
	}
};



//bool KeyboardWindow::OnKeyPress(GdkEventKey* event){
//	TRACE(<<"KeyboardWindow::OnKeyPress(): keyval = " << event->keyval << std::endl)
//	switch(event->keyval){
//		case GDK_BackSpace:
//		case GDK_Escape:
//			MainWindow::Inst().ShowMainMenuScreen();
//			break;
//		default:
//			return false;
//			break;
//	}
//	return true;
//}



void KeyboardWindow::OnKeyPress_Notify(GdkEventKey* event){
	TRACE(<<"OnKeyPress_Notify(): keyval = " << event->keyval << std::endl)
	switch(event->keyval){
		case GDK_Up:
			//DHighestOctave() + 1 is the number of octaves
			if(Preferences::Inst().GetKbScrPos() + 3 * 14 >= DHighestOctave() * 14){
				Preferences::Inst().SetKbScrPos(
						(DHighestOctave() + 1 - 3) * 14
					);
			}else{
				Preferences::Inst().SetKbScrPos(
						Preferences::Inst().GetKbScrPos() + 14
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Down:
			if(Preferences::Inst().GetKbScrPos() <= 14){
				Preferences::Inst().SetKbScrPos(0);
			}else{
				Preferences::Inst().SetKbScrPos(
						Preferences::Inst().GetKbScrPos() - 14
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Left:
			if(Preferences::Inst().GetKbScrPos() != 0){
				Preferences::Inst().SetKbScrPos(
						Preferences::Inst().GetKbScrPos() - 1
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Right:
			//octaves index starts from 0, here need the number of octaves,
			//i.e. DHighestOctave() + 1
			if(Preferences::Inst().GetKbScrPos() + 3 * 14 < 14 * (DHighestOctave() + 1)){
				Preferences::Inst().SetKbScrPos(
						Preferences::Inst().GetKbScrPos() + 1
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		default:
			//NOTE: no need to call OnKeyPress_Notify() handler
			//of the base class here. because it is connected to the signal
			//and will be called later, or already has been called.
			//this->BasicWindow::OnKeyPress_Notify(event);
			break;
	}
}



unsigned KeyboardWindow::CalcKeyOffset(real width, real height, real x, real y){
	real step = width / real(7);

	unsigned octHeight = unsigned(height / real(3) + real(1));

	ting::ClampBottom(y, real(0));
	ting::ClampTop(y, height);
	ASSERT(y >= real(0))

	ASSERT(y <= height)
	ASSERT(octHeight * 3 >= unsigned(height))
	//find the screen octave number of the touched octave (0, 1, 2)
	unsigned touchedOctave = 2 - unsigned(y) / octHeight;
	ASSERT(touchedOctave <= 2)

	int keyOffset = Preferences::Inst().GetKbScrPos() + 14 * touchedOctave;

	unsigned inOctaveBasePos = Preferences::Inst().GetKbScrPos() % 14;
	bool isWhiteKey = unsigned(y) % octHeight > octHeight / 2;

	if((inOctaveBasePos % 2 == 0 && isWhiteKey) ||
			(inOctaveBasePos % 2 == 1 && !isWhiteKey)
		)
	{
		keyOffset += int(x / step) * 2;
	}else{
		keyOffset += int((x + step / real(2)) / step) * 2 - 1;
	}

	if(keyOffset < 0){
		return 0;
	}

	return unsigned(keyOffset);
}



//A, A#, B, B#, C, C#, D, D#, E, E#, F, F#,  G, G#
unsigned keys[] = {
	0,  1, 2,  3, 3,  4, 5,  6, 7,  8, 8,  9, 10, 11
};



KeyboardWindow::FreqVol KeyboardWindow::CalcFreqAndVol(
		real width,
		real height,
		real x,
		real y
	)
{
	FreqVol ret;

	unsigned keyOffset = this->CalcKeyOffset(width, height, x, y);

	ret.keyNumber = keyOffset;

	unsigned inOctaveKeyOffset = keyOffset % 14;

	unsigned pitch = unsigned((keyOffset / 14) * 12 + keys[inOctaveKeyOffset]);

	ret.freq = PitchToFreq(pitch);

	if(inOctaveKeyOffset == 3 || inOctaveKeyOffset == 9){
		ret.vol = 0;
	}else{
		ret.vol = 0xff;//max volume
	}
	
	return ret;
}



bool KeyboardWindow::OnButtonPressEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< this->get_width() << " height = "<< this->get_height() << std::endl)

	FreqVol fv = this->CalcFreqAndVol(
			float(this->get_width()),
			float(this->get_height()),
			float(event->x),
			float(event->y)
		);

	this->curKeyOffset = fv.keyNumber;

	this->instr.Stop();

#ifndef M_NO_AUDIO
	this->instr.RenewChannel();
	this->instr.SetFrequency(fv.freq);
	this->instr.SetVolume(fv.vol);
	this->instr.Play();
#endif

	if(Preferences::Inst().GetBroadcast()){
		//send NOTE_ON music action event to main thread
		Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
		m->type = MusicActionMessage::NOTE_ON;
		m->timeStamp = ting::GetTicks();
		m->volume = fv.vol;
		m->freq = fv.freq;
		m->timbre = Preferences::Inst().GetKbTimbre();
		MainThread::Inst().PushMessage(m);
	}

	return true;
}



bool KeyboardWindow::OnButtonReleaseEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< window->get_width() << " height = "<< window->get_height() << std::endl)

	this->instr.Stop();

	if(Preferences::Inst().GetBroadcast()){
		//send NOTE_OFF music action event to main thread
		Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
		m->type = MusicActionMessage::NOTE_OFF;
		m->timeStamp = ting::GetTicks();
		m->timbre = Preferences::Inst().GetKbTimbre();
		MainThread::Inst().PushMessage(m);
	}

	return true;
}



bool KeyboardWindow::OnMotionEvent(GdkEventMotion* event){
	if(event->state & GDK_BUTTON1_MASK){

		if(this->instr.IsChannelValid()){
			FreqVol fv = this->CalcFreqAndVol(
					float(this->get_width()),
					float(this->get_height()),
					float(event->x),
					float(event->y)
				);

			if(this->curKeyOffset != fv.keyNumber){
				this->instr.Stop();
				this->instr.RenewChannel();

				this->instr.SetFrequency(fv.freq);
				this->instr.SetVolume(fv.vol);
				
				this->instr.Play();
				this->curKeyOffset = fv.keyNumber;

				if(Preferences::Inst().GetBroadcast()){
					{
						//send NOTE_OFF music action event to main thread
						Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
						m->type = MusicActionMessage::NOTE_OFF;
						m->timeStamp = ting::GetTicks();
						m->timbre = Preferences::Inst().GetKbTimbre();
						MainThread::Inst().PushMessage(m);
					}

					{
						//send NOTE_ON music action event to main thread
						Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
						m->type = MusicActionMessage::NOTE_ON;
						m->timeStamp = ting::GetTicks();
						m->volume = fv.vol;
						m->freq = fv.freq;
						m->timbre = Preferences::Inst().GetKbTimbre();
						MainThread::Inst().PushMessage(m);
					}
				}
			}else{
				this->instr.SetFrequency(fv.freq);
				this->instr.SetVolume(fv.vol);
			}
		}
	}
	return true;
}
