/* The MIT License:

Copyright (c) 2010 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */


#include <cstdlib>
#include <cmath>

#include <gtkmm/dialog.h>
#include <gtkmm/main.h>
#include <gtkmm/drawingarea.h>

#include <ting/Exc.hpp>
#include <ting/math.hpp>
#include <ting/Array.hpp>
#include <ting/Ref.hpp>
#include <ting/Timer.hpp>
#include <ting/Thread.hpp>
#include <ting/utils.hpp>
#include <pugixml/pugixml.hpp>

#include "stdafx.hpp"
#include "Preferences.hpp"
#include "dialogs/OptionsDialog.hpp"
#include "AnglephoneWindow.hpp"
#include "MainThread.hpp"
#include "MainWindow.hpp"
#include "utils.hpp"



using namespace ting;



AnglephoneWindow::AnglephoneWindow(){
	//options menu
	this->optionsMenuItem.
#ifndef M_NON_MAEMO_LINUX
			signal_clicked
#else
			signal_activate
#endif
			().connect(
					sigc::mem_fun(*this, &AnglephoneWindow::ShowOptionsDialog)
				);
	
	this->optionsMenuItem.set_sensitive(true);


	this->add_events(
			Gdk::BUTTON_PRESS_MASK |
			Gdk::BUTTON_RELEASE_MASK |
			Gdk::POINTER_MOTION_MASK |
			Gdk::BUTTON_MOTION_MASK
		);

	this->signal_button_press_event().connect(
			sigc::mem_fun(*this, &AnglephoneWindow::OnButtonPressEvent)
		);
	this->signal_button_release_event().connect(
			sigc::mem_fun(*this, &AnglephoneWindow::OnButtonReleaseEvent)
		);
	this->signal_motion_notify_event().connect(
			sigc::mem_fun(*this, &AnglephoneWindow::OnMotionEvent)
		);

	//set CAN_FOCUS flag to be able to receive keyboard events
	this->set_flags(Gtk::CAN_FOCUS);


	Gtk::Label *label = new Gtk::Label();
	this->add(*Gtk::manage(label));
	label->set_markup(
			"<span foreground=\"darkred\" size=\"larger\">Instructions</span>\n"
			"\n"
			"Make sound by tapping the screen.\n"
			"Sound loudness is controlled by horizontal postition of tap point.\n"
			"Sound pitch is controlled by tilting the device relative to vertical axis.\n"
		);

	this->show_all_children();//Show all window child widgets

	this->accelThread.Start();
}



AnglephoneWindow::~AnglephoneWindow(){
	this->accelThread.PushQuitMessage();
	this->accelThread.Join();
//		TRACE(<< "~ThereminWindow(): invoked" << std::endl)
}





class AnglephoneMusicEventMessage : public ting::Message{
	AnglephoneWindow::AccelThread *t;

	E_Timbre timbre;
	ting::u8 volume;
	bool isStop;
public:
	AnglephoneMusicEventMessage(
			AnglephoneWindow::AccelThread* thread,
			E_Timbre theTimbre = SINE_PLUS_ADSR,
			ting::u8 theVolume = 0xff,
			bool isStopEvent = false
		) :
			t(thread),
			timbre(theTimbre),
			volume(theVolume),
			isStop(isStopEvent)
	{
		ASSERT(this->t)
	}

	//override
	void Handle(){
		ASSERT(this->t)

		if(this->isStop){
			this->t->isPlaying = false;
			this->t->instr.Stop();
		}else{
			if(!this->t->isPlaying){
				this->t->isPlaying = true;
				this->t->instr.ApplyTimbre(this->timbre);
				this->t->instr.RenewChannel();
				this->t->isNeedToStartPlaying = true;
			}
			this->t->instr.SetVolume(this->volume);
		}
	}
};



#ifndef M_NON_MAEMO_LINUX
static const char *accelFilename = "/sys/class/i2c-adapter/i2c-3/3-001d/coord";
#endif



void AnglephoneWindow::AccelThread::Run(){
//	TRACE_ALWAYS(<< "AnglephoneWindow::AccelThread::Run(): enter" << std::endl)

	while(!this->quitFlag){
		if(this->isPlaying){
			while(ting::Ptr<ting::Message> m = this->queue.PeekMsg()){
				m->Handle();
			}
		}else{
			this->queue.GetMsg()->Handle();
		}

#ifndef M_NON_MAEMO_LINUX
		//read accelerometer measurements
		int x, y, z;
		{
			FILE *fd = fopen(accelFilename, "r");
			if(fd == 0){
				TRACE_ALWAYS(<< "AnglephoneWindow::AccelThread::Run(): fopen failed" << std::endl)
				continue;
			}

			int rs = fscanf(fd, "%i %i %i ", &x, &y, &z);
			if(rs != 3){
				TRACE_ALWAYS(<< "AnglephoneWindow::AccelThread::Run(): could not read info from accelerometer" << std::endl)
				fclose(fd);
				continue;
			}
			fclose(fd);
		}

		this->instr.SetFrequency(
				CalcFreq(
						440.0f,
						float(::atan2(x, -z) / (ting::DPi<float>() / 2))
					)
			);
#else
		this->instr.SetFrequency(440.0f);
#endif

		if(this->isNeedToStartPlaying){
			this->isNeedToStartPlaying = false;
			this->instr.Play();
		}

//		TRACE_ALWAYS(<< "Accel = " << x << " " << y << " " << z << std::endl)
//		TRACE_ALWAYS(<< "angle = " << a << std::endl)
	}

	this->instr.Stop();

//	TRACE_ALWAYS(<< "AnglephoneWindow::AccelThread::Run(): exit" << std::endl)
}



void AnglephoneWindow::ShowOptionsDialog(){
//	TRACE(<< "AnglephoneWindow::ShowOptionsDialog(): enter" << std::endl)
	OptionsDialog d(MainWindow::Inst(), false);
	d.SetTimbre(Preferences::Inst().GetAngTimbre());

	switch(d.run()){
		case Gtk::RESPONSE_OK: //OK
			Preferences::Inst().SetAngTimbre(d.GetTimbre());
			Preferences::Inst().SaveToFile();
			break;
		default: //Cancel and all other cases
			break;
	}

	d.hide();
	MainWindow::Inst().queue_draw();//request redrawing entire window
//	TRACE(<< "AnglephoneWindow::ShowOptionsDialog(): exit" << std::endl)
}



//static
ting::u8 AnglephoneWindow::CalcVol(real y, real height){
	real factor = y / height;
	//quadric dependency on factor allows more precise volume control by finger
	return ting::u8(s32(real(0xff) * factor * factor));
}



bool AnglephoneWindow::OnButtonPressEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< this->get_width() << " height = "<< this->get_height() << std::endl)

	{
		ting::Ptr<ting::Message> m(new AnglephoneMusicEventMessage(
				&this->accelThread,
				Preferences::Inst().GetAngTimbre(),
				AnglephoneWindow::CalcVol(event->x, this->get_width())
			));
		this->accelThread.PushMessage(m);
	}

	//TODO:
//	if(Preferences::Inst().GetBroadcast()){
//		//send NOTE_ON music action event to main thread
//		Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
//		m->type = MusicActionMessage::NOTE_ON;
//		m->timeStamp = ting::GetTicks();
//		m->volume = fv.vol;
//		m->freq = fv.freq;
		//TODO: timbre
//		m->timbre = Preferences::Inst().GetAngTimbre();
//		MainThread::Inst().PushMessage(m);
//	}

	return true;
}



bool AnglephoneWindow::OnButtonReleaseEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< window->get_width() << " height = "<< window->get_height() << std::endl)

	{
		ting::Ptr<ting::Message> m(new AnglephoneMusicEventMessage(
				&this->accelThread,
				THEREMIN, //ignored
				0, //ignored
				true
			));
		this->accelThread.PushMessage(m);
	}

//	if(Preferences::Inst().GetBroadcast()){
//		//send NOTE_OFF music action event to main thread
//		Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
//		m->type = MusicActionMessage::NOTE_OFF;
//		m->timeStamp = ting::GetTicks();
//		//TODO: timbre
//		m->timbre = Preferences::Inst().GetTimbre();
//		MainThread::Inst().PushMessage(m);
//	}

	return true;
}



bool AnglephoneWindow::OnMotionEvent(GdkEventMotion* event){
	if(event->state & GDK_BUTTON1_MASK){
		{
			ting::Ptr<ting::Message> m(new AnglephoneMusicEventMessage(
					&this->accelThread,
					Preferences::Inst().GetAngTimbre(),
					AnglephoneWindow::CalcVol(event->x, this->get_width())
				));
			this->accelThread.PushMessage(m);
		}
//		if(this->instr.IsChannelValid()){
//			if(Preferences::Inst().GetBroadcast()){
//				//send NOTE_ON music action event to main thread
//				Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
//				m->type = MusicActionMessage::NOTE_ON;
//				m->timeStamp = ting::GetTicks();
//				m->volume = fv.vol;
//				m->freq = fv.freq;
//				//TODO: timbre
//				m->timbre = Preferences::Inst().GetTimbre();
//				MainThread::Inst().PushMessage(m);
//			}
//		}
	}
	return true;
}



