#include <QFont>
#include <QPainter>
#include <QListIterator>
#include <QPen>
#include <QDate>
#include <QRect>

#include "expensesgraph.h"
#include "expense.h"

#define Y_LABEL_HEIGHT 30
#define X_LABEL_WIDTH 30
#define X_LABEL_HEIGHT 15

#define Y_DIVISIONS 10

ExpensesGraph::ExpensesGraph(QWidget *parent)
        : QWidget(parent)
{
        _padding = 10;
        _accumulated = true;
        _model = NULL;
        yLabelWidth = 0;
}

void ExpensesGraph::setPadding(int padding)
{
        _padding = padding;
        repaint();
}

void ExpensesGraph::setModel(ExpensesGraphModel *model)
{
        _model = model;
}

void ExpensesGraph::setAccumulated(bool accumulated)
{
        _accumulated = accumulated;
        repaint();
}

bool ExpensesGraph::accumulated()
{
        return _accumulated;
}

void ExpensesGraph::changeView()
{
        setAccumulated(!_accumulated);
}

QSize ExpensesGraph::minimumSizeHint() const
{
        return QSize(200, 100);
}

QSize ExpensesGraph::sizeHint() const
{
        return QSize(400, 200);
}

void ExpensesGraph::paintEvent (QPaintEvent *)
{
        QPainter painter(this);
        QPen pen;
        int aux;

        pen.setColor(Qt::white);
        painter.setPen(pen);
        QFont font = painter.font();
        font.setPointSize(10);
        painter.setFont(font);
        painter.fillRect(0, 0, width(), height(), QColor(Qt::black));

        if (!_model) {
                /* default scene start, end and width */
                xStart = 2 * _padding;
                xEnd = width() - _padding;
                sceneWidth = xEnd - xStart;

                /* default scene start, end and height */
                /* watch out with the reversed Y */
                yStart = height() - (_padding + X_LABEL_HEIGHT + _padding);
                yEnd = _padding;
                sceneHeight = yStart - yEnd;

                drawScene(painter);
        } else {
                /* check that we have more than one value to draw */
                if (_model->xDivisions() <= 1) {
                        /* default scene start, end and width */
                        xStart = 2 * _padding;
                        xEnd = width() - _padding;
                        sceneWidth = xEnd - xStart;

                        /* default scene start, end and height */
                        /* watch out with the reversed Y */
                        yStart = height() - (_padding + X_LABEL_HEIGHT + _padding);
                        yEnd = _padding;
                        sceneHeight = yStart - yEnd;

                        drawScene(painter);

                        painter.drawText(xStart,
                                         yEnd,
                                         sceneWidth,
                                         sceneHeight,
                                         Qt::AlignCenter,
                                         tr("Can't draw graph with only one value"));

                } else {

                        /* calculate max Y value */
                        if (_accumulated) {
                                if (_model->budget() >= _model->totalExpense()) {
                                        yMaxValue = _model->budget();
                                } else {
                                        aux = _model->totalExpense() / 100;
                                        yMaxValue = (aux + 1) * 100;
                                }
                        } else {
                                yMaxValue = qMax(_model->maxExpense(), _model->maxBudget());
                                yMaxValue = ((yMaxValue / 100) + 1) * 100;
                        }

                        QRect rect = painter.boundingRect(0, 0, 400, 400,
                                                          Qt::AlignRight | Qt::AlignVCenter,
                                                          QString("%1").arg(yMaxValue));
                        yLabelWidth = rect.width();

                        /* calculate scene start, end and width */
                        xStart = _padding + yLabelWidth + _padding;
                        xEnd = width() - _padding;
                        sceneWidth = xEnd - xStart;

                        /* calculate scene start, end and height */
                        yStart = height() - (_padding + X_LABEL_HEIGHT + _padding);
                        yEnd = _padding;
                        sceneHeight = yStart - yEnd;

                        /* labels in x axis */
                        xStepsPerLabel = _model->xDivisions() / qMin(sceneWidth / X_LABEL_WIDTH, _model->xDivisions());
                        /* labels in y axis. not more than 10 */
                        yStepsPerLabel = Y_DIVISIONS / qMin(sceneHeight / Y_LABEL_HEIGHT, Y_DIVISIONS);

                        yScale = sceneHeight / (double)yMaxValue;

                        xStep = sceneWidth / (double)(_model->xDivisions()-1);
                        yStep = sceneHeight / (double)Y_DIVISIONS;

                        drawScene(painter);
                        painter.setRenderHint(QPainter::Antialiasing);
                        drawGrid(painter);
                        drawReference(painter);
                        drawExpenses(painter);
                }
        }
}

void ExpensesGraph::drawScene (QPainter &painter)
{
        QPen pen;

        painter.save();
        painter.drawLine(xStart, yStart, xStart, yEnd);
        painter.drawLine(xStart, yStart, xEnd, yStart);

        painter.restore ();
}

void ExpensesGraph::drawGrid (QPainter &painter)
{
        int i;
        double y, x;
        double yLabelValue, yInc;
        QPen pen;

        painter.save();

        pen.setStyle(Qt::DotLine);
        pen.setColor(Qt::gray);
        pen.setWidth(1);
        painter.setPen(pen);

        /* vertical divisions  and y labels*/
        y = yStart;
        yLabelValue = 0;
        yInc = yMaxValue / (double)Y_DIVISIONS;
        for (i = 0; i <= Y_DIVISIONS; i+=yStepsPerLabel) {
                y -= yStepsPerLabel * yStep;
                yLabelValue += yStepsPerLabel * yInc;
                if (i != Y_DIVISIONS - 1) {
                        painter.drawLine (xStart + 2, y, xEnd, y);
                }
                painter.setPen(Qt::white);
                painter.drawText (_padding, y - (Y_LABEL_HEIGHT / 2), yLabelWidth, Y_LABEL_HEIGHT, 
                                  Qt::AlignRight | Qt::AlignVCenter,
                                  QString("%1").arg(qRound(yLabelValue)));
                painter.setPen(pen);
        }
        /* horizontal divisions and x labels*/
        painter.setPen(Qt::white);
        painter.drawText (xStart - (X_LABEL_WIDTH / 2),
                          yStart + _padding,
                          X_LABEL_WIDTH,
                          X_LABEL_HEIGHT,
                          Qt::AlignCenter,
                          QString("%1").arg(_model->label(0)));
        painter.setPen(pen);
        x = xStart + (xStep *xStepsPerLabel);
        for (i = xStepsPerLabel + 1; i<=_model->xDivisions(); i+=xStepsPerLabel) {
                if (i != _model->xDivisions()) {
                        painter.drawLine (qRound(x), yStart, qRound(x), yEnd);
                }
                painter.setPen(Qt::white);
                painter.drawText (qRound(x) - (X_LABEL_WIDTH / 2),
                                  yStart + _padding,
                                  X_LABEL_WIDTH,
                                  X_LABEL_HEIGHT,
                                  Qt::AlignCenter,
                                  QString("%1").arg(_model->label(i-1)));
                painter.setPen(pen);
                x += xStep * xStepsPerLabel;
        }

        /* current */
        if (_model->isCurrent()) {
                i = QDate::currentDate().day() - 1;
                painter.setPen (Qt::yellow);
                painter.drawLine (qRound(xStart + (i * xStep)),
                                  yStart,
                                  qRound(xStart + (i * xStep)),
                                  yEnd);
        }


        painter.restore();
}

void ExpensesGraph::drawReference (QPainter &painter)
{
        QPainterPath path;
        int i;
        double x, y;

        x = xStart + 1;
        y = yStart;
        path.moveTo (x, y);

        for (i=0; i<_model->xDivisions(); i++) {
                if (_accumulated) {
                        y -= _model->reference(i) * yScale;
                } else {
                        y = yStart - (_model->reference(i) * yScale);
                }
                if (i == 0) {
                        path.moveTo (x, y);
                } else {
                        path.lineTo (x, y);
                }
                x += xStep;
        }

        QPen pen;
        painter.save();
        pen.setColor(Qt::green);
        painter.setPen(pen);
        painter.drawPath(path);
        painter.restore();
}

void ExpensesGraph::drawExpenses (QPainter &painter)
{
        QPainterPath path;
        int i;
        double x, y;

        x = xStart + 1;
        y = yStart;
        path.moveTo (x, y);

        for (i=0; i<_model->xDivisions(); i++) {
                if (_accumulated) {
                        y -= _model->value(i) * yScale;
                } else {
                        y = yStart - (_model->value(i) * yScale);
                }
                if (i == 0) {
                        path.moveTo (x, y);
                } else {
                        path.lineTo (x, y);
                }
                x += xStep;
        }

        QPen pen;
        painter.save();
        pen.setColor(Qt::red);
        pen.setWidth(2);
        painter.setPen(pen);
        painter.drawPath(path);
        painter.restore();
}
