#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QDate>
#include <QAction>
#include <QMenuBar>
#include <QMessageBox>
#include "mainwindowdesktop.h"

MainWindowDesktop::MainWindowDesktop(QWidget *parent)
        : MainWindow(parent)
{
        month = NULL;

        setWindowTitle(tr("Account: none selected"));

        QVBoxLayout *vBox = new QVBoxLayout(this);

        QMenuBar *menubar = new QMenuBar();
        menubar->addAction(tr("Accounts"), this, SIGNAL(accountsSelected()));
        menubar->addAction(tr("Items"), this, SIGNAL(conceptsSelected()));
        budgetAction = menubar->addAction(tr("Budget"), this, SIGNAL(budgetSelected()));
        expensesAction = menubar->addAction(tr("Expenses"), this, SIGNAL(expensesSelected()));
        scheduledAction = menubar->addAction(tr("Scheduled"), this, SIGNAL(scheduledSelected()));
        summaryAction = menubar->addAction(tr("Summary"), this, SIGNAL(summarySelected()));
        budgetAction->setEnabled(false);
        expensesAction->setEnabled(false);
        scheduledAction->setEnabled(false);
        summaryAction->setEnabled(false);
        vBox->addWidget(menubar, 0);

        QHBoxLayout *hBox = new QHBoxLayout();
        vBox->addLayout(hBox);

        previous = new QPushButton("<");
        hBox->addWidget(previous, 0, Qt::AlignLeft);
        connect(previous, SIGNAL(clicked()), this, SIGNAL(previousMonth()));

        dateLabel = new QLabel("");
        hBox->addWidget(dateLabel, 0, Qt::AlignCenter);

        next = new QPushButton(">");
        hBox->addWidget(next, 0, Qt::AlignRight);
        connect(next, SIGNAL(clicked()), this, SIGNAL(nextMonth()));

        graph = new ExpensesGraph();
        vBox->addWidget(graph, 1);

        QHBoxLayout *footer = new QHBoxLayout();
        vBox->addLayout(footer, 0);

        footer->addWidget(new QLabel(tr("Budget")), 0, Qt::AlignRight | Qt::AlignVCenter);
        budgetLabel = new QLabel("");
        footer->addWidget(budgetLabel, 1, Qt::AlignLeft | Qt::AlignVCenter);

        footer->addWidget(new QLabel(tr("Expenses")), 0, Qt::AlignRight | Qt::AlignVCenter);
        totalLabel = new QLabel("");
        footer->addWidget(totalLabel, 1, Qt::AlignLeft | Qt::AlignVCenter);

        footer->addWidget(new QLabel(tr("Difference")), 0, Qt::AlignRight | Qt::AlignVCenter);
        savedLabel = new QLabel("");
        footer->addWidget(savedLabel, 1, Qt::AlignLeft | Qt::AlignVCenter);
}

QSize MainWindowDesktop::minimumSizeHint() const
{
        return QSize(800, 500);
}

QSize MainWindowDesktop::sizeHint() const
{
        return QSize(800, 500);
}

void MainWindowDesktop::setMonthData(MonthData *data)
{
        month = data;
        /* set data to the graph as well */
        graph->setModel(data);
        connect(month, SIGNAL(expensesChanged()), this, SLOT(expensesChanged()));
        connect(month, SIGNAL(budgetChanged()), this, SLOT(budgetChanged()));

        updateAll();

        budgetAction->setEnabled(true);
        expensesAction->setEnabled(true);
        scheduledAction->setEnabled(true);
        summaryAction->setEnabled(true);
}

void MainWindowDesktop::updateAccountLabel()
{
        setWindowTitle(tr("Account: %1").arg(month->account()->name()));
}

void MainWindowDesktop::expensesChanged()
{
        updateTotalLabel();
        updateSavedLabel();
        updateGraph();
}

void MainWindowDesktop::budgetChanged()
{
        updateBudgetLabel();
        updateSavedLabel();
        updateGraph();
}

void MainWindowDesktop::updateAll()
{
        updateAccountLabel();
        updateDateLabel();
        updateBudgetLabel();
        updateTotalLabel();
        updateSavedLabel();
        updateGraph();
}

void MainWindowDesktop::updateDateLabel()
{
        QString label = QString("<b>%1 %2</b>").arg(QDate::longMonthName(month->month())).arg(month->year());
        if (graph->accumulated()) {
                label.append(QString("   (%1)").arg(tr("accumulated view")));
        } else {
                label.append(QString("   (%1)").arg(tr("daily view")));
        }
        dateLabel->setText(label);
}

void MainWindowDesktop::updateBudgetLabel()
{
        budgetLabel->setNum(month->budget());
}

void MainWindowDesktop::updateTotalLabel()
{
        totalLabel->setNum(month->totalExpense());
}

void MainWindowDesktop::updateSavedLabel()
{
        double diff;

        diff = month->budget() - month->totalExpense();

        if (diff >= 0) {
                savedLabel->setText(QString("<font color='green'><b>+%1</b></font>").arg(diff));
        } else {
                savedLabel->setText(QString("<font color='red'><b>%1</b></font>").arg(diff));
        }
}

void MainWindowDesktop::updateGraph()
{
        graph->repaint();
}

void MainWindowDesktop::mousePressEvent(QMouseEvent *event)
{
        if (month != NULL) {
                graph->changeView();
                updateDateLabel();
        }
}

void MainWindowDesktop::message(const QString& text)
{
        QMessageBox msgBox(this);
        msgBox.setText(text);
        msgBox.exec();
}
