/**
    File: picasa_xml.c
    
    Author: Tero Niemel

*/


#include <glib.h>
#include <string.h>
#include <osso-log.h>
#include <sharing-plugin-interface.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#ifdef WITH_ALBUM_SUPPORT
#include <sharing-service-option.h>
#endif /* WITH_ALBUM_SUPPORT */

#include "picasa_xml.h"


/* defines for xml schemas */
#define XMLNS         "http://www.w3.org/2005/Atom"
#define XMLNS_MEDIA   "http://search.yahoo.com/mrss/"
#define XMLNS_GEORSS  "http://www.georss.org/georss"
#define XMLNS_GML     "http://www.opengis.net/gml"
#define SCHEME        "http://schemas.google.com/g/2005#kind"
#define TERM_PHOTO    "http://schemas.google.com/photos/2007#photo"
#define TERM_TAG      "http://schemas.google.com/photos/2007#tag"


/*
  PRIVATE functions
*/


#ifdef WITH_ALBUM_SUPPORT
/**
   picasa_xml_parse_entry
*/
static void
picasa_xml_parse_entry (xmlNodePtr entry, GSList **list) { }

#ifdef DEBUG
  gchar *fn = "picasa_xml_parse_entry()";
#endif

  xmlNodePtr node = entry->children;
  xmlNodePtr tmp = NULL;
  gchar *title = NULL;
  gchar *id = NULL;
  SharingServiceOptionValue *value = NULL;

  while (node != NULL) {

    if (xmlStrcmp (BAD_CAST "title", node->name) == 0) {
      tmp = node->children;
      title = (gchar*)tmp->content;
    }
    if (xmlStrcmp (BAD_CAST "id", node->name) == 0) {
      tmp = node->children;
      id = (gchar*)tmp->content;
    }

    node = node->next;
  }

  /* if title or id empty, just don't save */
  if (title != NULL && id != NULL) {
    ULOG_DEBUG_L ("%s: found album (title = %s ; id = %s", fn, title, id);
    value = sharing_service_option_value_new (id, title, NULL);
    *list = g_slist_append (*list, value);
  }
  
}
#endif /* WITH_ALBUM_SUPPORT */


/*
  PUBLIC functions
*/

/**
   picasa_xml_parse_upload_response
*/
gchar*
picasa_xml_parse_upload_response (gchar *body) {

#ifdef DEBUG
  gchar *fn = "picasa_xml_parse_upload_response()";
#endif

  xmlNodePtr root = NULL;
  xmlNodePtr tmp = NULL;
  gchar **href = NULL;
  gchar *photo_id = NULL;

  ULOG_DEBUG_L ("%s: entered", fn);

  xmlDoc *doc = xmlReadMemory (body, strlen (body),
			       NULL, NULL, 0);
  
  if (!doc) {
    ULOG_WARN_L ("%s: failed to create doc", fn);
    return NULL;
  }

  root = xmlDocGetRootElement (doc);
  ULOG_DEBUG_L ("%s: found root node %s", fn, (gchar*)root->name);

  tmp = root->children;
  while (tmp != NULL) {

    /* if link found */
    if (xmlStrcmp (BAD_CAST "link", tmp->name) == 0) {

      
      if (xmlStrcmp (xmlGetProp (tmp, BAD_CAST "rel"), 
		     BAD_CAST "edit") == 0) {
	
	ULOG_DEBUG_L ("%s: found link edit", fn);
	
	href = g_strsplit ((gchar*)xmlGetProp(tmp, BAD_CAST "href"), "?", 2);
	photo_id = g_strdup (href[0]);
	g_strfreev (href);
	break;
      }
    }
    
    tmp = tmp->next;
  }

  xmlFreeDoc (doc);

  return photo_id;
}


/**
   picasa_xml_parse_gps_response
*/
gchar*
picasa_xml_parse_gps_response (gchar *body) {

#ifdef DEBUG
  gchar *fn = "picasa_xml_parse_gps_response()";
#endif

  xmlNodePtr root = NULL;
  xmlNodePtr tmp = NULL;
  gchar *gps = NULL;
  gchar **coordinates = NULL;
  gint count = 0;

  ULOG_DEBUG_L ("%s: entered", fn);

  xmlDoc *doc = xmlReadMemory (body, strlen (body),
			       NULL, NULL, 0);
  
  if (!doc) {
    ULOG_WARN_L ("%s: failed to create doc", fn);
    return NULL;
  }

  root = xmlDocGetRootElement (doc);
  ULOG_DEBUG_L ("%s: found root node %s", fn, (gchar*)root->name);

  /* count how many Placemarks found */
  tmp = root->children; /* response element */
  tmp = tmp->children;
  while (tmp != NULL) {
    if (xmlStrcmp (BAD_CAST "Placemark", tmp->name) == 0) {
      count++;
    }    
    tmp = tmp->next;
  }

  /* if more than one found exit */
  if (count != 1) {
    ULOG_WARN_L ("%s: wrong amount of placemark found", fn);
    xmlFreeDoc (doc);
    return NULL;
  }

  /* get the coordinates */
  tmp = root->children;
  tmp = tmp->children;
  while (tmp != NULL) {

    if (xmlStrcmp (BAD_CAST "Placemark", tmp->name) == 0) {
      tmp = tmp->children;
    }    
    if (xmlStrcmp (BAD_CAST "Point", tmp->name) == 0) {
      tmp = tmp->children;
    }
    if (xmlStrcmp (BAD_CAST "coordinates", tmp->name) == 0) {
      tmp = tmp->children;
      ULOG_DEBUG_L ("%s: found coordinates = %s", fn, tmp->content);
      coordinates = g_strsplit_set ((gchar*)tmp->content, ",", 0);
      gps = g_strdup_printf("%s %s", coordinates[1], coordinates[0]);
      g_strfreev (coordinates);
    }        

    tmp = tmp->next;
  }

  xmlFreeDoc (doc);

  ULOG_DEBUG_L ("%s: leaving", fn);

  return gps;
}

#ifdef WITH_ALBUM_SUPPORT
/**
   picasa_xml_parse_album_response
*/
GSList*
picasa_xml_parse_album_response (gchar *body) {

#ifdef DEBUG
  gchar *fn = "picasa_xml_parse_album_response()";
#endif

  xmlNodePtr root = NULL;
  xmlNodePtr tmp = NULL;
  GSList *albums = NULL;

  xmlDoc *doc = xmlReadMemory (body, strlen (body),
			       NULL, NULL, 0);

  if (!doc) {
    ULOG_WARN_L ("%s: failed to create doc", fn);
    return NULL;
  }

  root = xmlDocGetRootElement (doc);
  ULOG_DEBUG_L ("%s: found root node %s", fn, (gchar*)root->name);

  if (xmlStrcmp (BAD_CAST "feed", root->name) != 0) {
    ULOG_WARN_L ("%s: invalid root node name", fn);
    return NULL;
  }

  /* look for all entries */
  tmp = root->children;
  while (tmp != NULL) {

    /* if entry found */
    if (xmlStrcmp (BAD_CAST "entry", tmp->name) == 0) {
      picasa_xml_parse_entry (tmp, &albums);
    }

    tmp = tmp->next;
  }
  
  xmlFreeDoc (doc);

  return albums;
}
#endif /* WITH_ALBUM_SUPPORT */

/**
   picasa_xml_create_metadata_entry
*/
gchar*
picasa_xml_create_metadata_entry (gchar *title,
				  gchar *summary) {

#ifdef DEBUG
  gchar *fn = "picasa_xml_create_metadata_entry()";
#endif
  
  gchar *entry = NULL;
  gint size = 0;

  /*
    <entry xmlns='http://www.w3.org/2005/Atom'>
      <title>plz-to-love-realcat.jpg</title>
      <summary>Real cat wants attention too.</summary>
      <category scheme="http://schemas.google.com/g/2005#kind"
                term="http://schemas.google.com/photos/2007#photo"/>
    </entry>
  */

  if (title == NULL && summary == NULL) {
    return NULL;
  }
  
  xmlDocPtr doc = xmlNewDoc (BAD_CAST "1.0");

  if (doc != NULL) {

    ULOG_DEBUG_L ("%s: creating xml", fn);
  
    xmlNodePtr root = xmlNewNode (NULL, BAD_CAST "entry");
    xmlNewProp (root, BAD_CAST "xmlns", BAD_CAST XMLNS);
    xmlDocSetRootElement (doc, root);
    
    xmlNewChild (root, NULL, BAD_CAST "title", 
		 BAD_CAST title);
    xmlNewChild (root, NULL, BAD_CAST "summary",
		 BAD_CAST summary);
    
    xmlNodePtr node_category = 
      xmlNewChild (root, NULL, BAD_CAST "category", NULL);
    xmlNewProp (node_category, BAD_CAST "scheme", BAD_CAST SCHEME);
    xmlNewProp (node_category, BAD_CAST "term", BAD_CAST TERM_PHOTO);

    xmlDocDumpFormatMemoryEnc (doc, (xmlChar**)&entry, &size, "UTF-8", 1);
    
    xmlFreeDoc (doc);

  } else {
    ULOG_WARN_L ("%s: failed to create new xmlDoc", fn);
  }
  
  ULOG_DEBUG_L ("%s: entry = %s", fn, entry);

  return entry;
}


/**
   picasa_xml_create_tag_entry
*/
gchar*
picasa_xml_create_tag_entry (gchar *tag) {

#ifdef DEBUG
  gchar *fn = "picasa_xml_create_tag_entry()";
#endif
  
  gchar *entry = NULL;
  gint size = 0;

  /*
    <entry xmlns='http://www.w3.org/2005/Atom'>
      <title>tag</title>
      <category scheme="http://schemas.google.com/g/2005#kind"
                term="http://schemas.google.com/photos/2007#tag"/>
    </entry>
  */

  if (tag == NULL) {
    ULOG_WARN_L ("%s: empty tag", fn);
    return NULL;
  }
  
  xmlDocPtr doc = xmlNewDoc (BAD_CAST "1.0");

  if (doc != NULL) {

    ULOG_DEBUG_L ("%s: creating xml", fn);
  
    xmlNodePtr root = xmlNewNode (NULL, BAD_CAST "entry");
    xmlNewProp (root, BAD_CAST "xmlns", BAD_CAST XMLNS);
    xmlDocSetRootElement (doc, root);
    
    xmlNewChild (root, NULL, BAD_CAST "title", 
		 BAD_CAST tag);
    
    xmlNodePtr node_category = 
      xmlNewChild (root, NULL, BAD_CAST "category", NULL);
    xmlNewProp (node_category, BAD_CAST "scheme", BAD_CAST SCHEME);
    xmlNewProp (node_category, BAD_CAST "term", BAD_CAST TERM_TAG);

    xmlDocDumpFormatMemoryEnc (doc, (xmlChar**)&entry, &size, "UTF-8", 1);
    
    xmlFreeDoc (doc);

  } else {
    ULOG_WARN_L ("%s: failed to create new xmlDoc", fn);
  }
  
  ULOG_DEBUG_L ("%s: entry = %s", fn, entry);

  return entry;
}

/**
   picasa_xml_create_geotag_entry
*/
gchar*
picasa_xml_create_gpstag_entry (gchar *tag) {

#ifdef DEBUG
  gchar *fn = "picasa_xml_create_gpstag_entry()";
#endif
  
  gchar *entry = NULL;
  gint size = 0;
  
  /*
  <entry xmlns='http://www.w3.org/2005/Atom'
         xmlns:georss="http://www.georss.org/georss"
         xmlns:gml="http://www.opengis.net/gml">
    <georss:where>
      <gml:Point>
	<gml:pos>35.669998 139.770004</gml:pos>
      </gml:Point>
    </georss:where>
    <category scheme="http://schemas.google.com/g/2005#kind"
              term="http://schemas.google.com/photos/2007#photo/>
  </entry>
  */

  if (tag == NULL) {
    ULOG_WARN_L ("%s: empty tag", fn);
    return NULL;
  }

  xmlDocPtr doc = xmlNewDoc (BAD_CAST "1.0");

  if (doc != NULL) {

    ULOG_DEBUG_L ("%s: creating xml", fn);
  
    xmlNodePtr root = xmlNewNode (NULL, BAD_CAST "entry");
    xmlNewProp (root, BAD_CAST "xmlns", BAD_CAST XMLNS);
    xmlNewProp (root, BAD_CAST "xmlns:georss", BAD_CAST XMLNS_GEORSS);
    xmlNewProp (root, BAD_CAST "xmlns:gml", BAD_CAST XMLNS_GML);
    xmlDocSetRootElement (doc, root);

    xmlNewChild (root, NULL, BAD_CAST "title", NULL);
    
    xmlNodePtr node_georss = 
      xmlNewChild (root, NULL, BAD_CAST "georss:where", NULL);
    xmlNodePtr node_gml = 
      xmlNewChild (node_georss, NULL, BAD_CAST "gml:Point", NULL);
    xmlNewChild (node_gml, NULL, BAD_CAST "gml:pos", BAD_CAST tag);
    
    xmlNodePtr node_category = 
      xmlNewChild (root, NULL, BAD_CAST "category", NULL);
    xmlNewProp (node_category, BAD_CAST "scheme", BAD_CAST SCHEME);
    xmlNewProp (node_category, BAD_CAST "term", BAD_CAST TERM_TAG);

    xmlDocDumpFormatMemoryEnc (doc, (xmlChar**)&entry, &size, "UTF-8", 1);
    
    xmlFreeDoc (doc);
    g_free (tag);

  } else {
    ULOG_WARN_L ("%s: failed to create new xmlDoc", fn);
  }
  
  ULOG_DEBUG_L ("%s: entry = %s", fn, entry);

  return entry;
}

/**
   picasa_xml_create_media_entry
*/
gchar*
picasa_xml_create_media_entry (gchar *tags,
			       gchar *gps) {

#ifdef DEBUG
  gchar *fn = "picasa_xml_create_media_entry()";
#endif
  
  gchar *entry = NULL;
  gint size = 0;
  
  /*
  <entry xmlns='http://www.w3.org/2005/Atom'
         xmlns:media="http://search.yahoo.com/mrss"
         xmlns:georss="http://www.georss.org/georss"
         xmlns:gml="http://www.opengis.net/gml">
    <media:group>
      <media:keywords>tag, tag2 ...</media:keywords>
    </media:group>
    <georss:where>
      <gml:Point>
	<gml:pos>35.669998 139.770004</gml:pos>
      </gml:Point>
    </georss:where>
    <category scheme="http://schemas.google.com/g/2005#kind"
              term="http://schemas.google.com/photos/2007#photo/>
  </entry>
  */

  if (tags == NULL && gps == NULL) {
    ULOG_WARN_L ("%s: parameters NULL", fn);
    return NULL;
  }

  ULOG_DEBUG_L ("%s: tags = %s ; gps = %s", fn, tags, gps);

  xmlDocPtr doc = xmlNewDoc (BAD_CAST "1.0");

  if (doc != NULL) {

    ULOG_DEBUG_L ("%s: creating xml", fn);
  
    xmlNodePtr root = xmlNewNode (NULL, BAD_CAST "entry");
    xmlNewProp (root, BAD_CAST "xmlns", BAD_CAST XMLNS);
    xmlNewProp (root, BAD_CAST "xmlns:media", BAD_CAST XMLNS_MEDIA);
    xmlNewProp (root, BAD_CAST "xmlns:georss", BAD_CAST XMLNS_GEORSS);
    xmlNewProp (root, BAD_CAST "xmlns:gml", BAD_CAST XMLNS_GML);
    xmlDocSetRootElement (doc, root);

    xmlNodePtr node_media = 
      xmlNewChild (root, NULL, BAD_CAST "media:group", NULL);
    xmlNewChild (node_media, NULL, BAD_CAST "media:keywords", BAD_CAST tags);
    
    xmlNodePtr node_georss = 
      xmlNewChild (root, NULL, BAD_CAST "georss:where", NULL);
    xmlNodePtr node_gml = 
      xmlNewChild (node_georss, NULL, BAD_CAST "gml:Point", NULL);
    xmlNewChild (node_gml, NULL, BAD_CAST "gml:pos", BAD_CAST gps);
    
    xmlNodePtr node_category = 
      xmlNewChild (root, NULL, BAD_CAST "category", NULL);
    xmlNewProp (node_category, BAD_CAST "scheme", BAD_CAST SCHEME);
    xmlNewProp (node_category, BAD_CAST "term", BAD_CAST TERM_TAG);

    xmlDocDumpFormatMemoryEnc (doc, (xmlChar**)&entry, &size, "UTF-8", 1);
    
    xmlFreeDoc (doc);

  } else {
    ULOG_WARN_L ("%s: failed to create new xmlDoc", fn);
  }
  
  ULOG_DEBUG_L ("%s: entry = %s", fn, entry);

  return entry;
}

/* EOF */
