/*
 *  scout Maemo 5 calendar, contact and conversations search tool
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <CComponent.h>
#include <CCalendar.h>
#include <CJournal.h>
#include <Common.h>
#include <CTodo.h>
#include <CEvent.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <locale.h>
#include <gconf/gconf-client.h>	
#include <CalInterface.h>

#include "calendar-event-preview.hpp"
#define GCONF_TIME_FORMAT_PATH "/apps/clock/time-format"
typedef struct _calendar_open_data {
  osso_context_t* osso;
  int time_stamp;
} calendar_open_data;


static void
_open_calendar(GtkButton* button, gpointer user_data)
{
  calendar_open_data* data = (calendar_open_data*)user_data;
  int type = 3;
  const char* text = "DUMMY";  
  osso_rpc_run(data->osso,
	       CALENDAR_SERVICE, 
	       CALENDAR_PATH, 
	       CALENDAR_INTERFACE,
	       CALENDAR_LAUNCH_METHOD,
	       NULL,
	       DBUS_TYPE_UINT32, type,
	       DBUS_TYPE_INT32, data->time_stamp, 
	       DBUS_TYPE_STRING, text,
	       DBUS_TYPE_INVALID);
}

static void
delete_cal_open_data(GtkWidget* widget,
		     GdkEvent* event,
		     gpointer user_data)
{
  g_free(user_data);
}

gboolean
_has_24hr_timeformat()
{
  GConfClient* client = NULL;
  gboolean is24Format = TRUE;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client)) 
    return is24Format;

  is24Format =   gconf_client_get_bool(client, GCONF_TIME_FORMAT_PATH, NULL);
  g_object_unref(client);
  return is24Format;
}


gchar*
_format_time_stamp(time_t ttime_stamp)
{
  gchar* time_string = (gchar*)g_malloc0(256);
  struct tm* time_stamp;
  time_stamp = localtime(&ttime_stamp);
  if(_has_24hr_timeformat())
  {
    gchar* full_time_format = g_strdup_printf("%s", 
					      dgettext("hildon-libs", "wdgt_va_24h_time"));
    strftime(time_string, 256, full_time_format, time_stamp);
    g_free(full_time_format);
  }
  else
  {
    gchar* full_time_format = g_strdup_printf("%s", 
					      (time_stamp->tm_hour > 11) ? 
					      dgettext("hildon-libs", "wdgt_va_12h_time_pm") :
					      dgettext("hildon-libs", "wdgt_va_12h_time_am"));
    strftime(time_string, 256, full_time_format, time_stamp);    
    g_free(full_time_format);
  }
  return time_string;
}

gchar*
_format_date_time_stamp(time_t ttime_stamp)
{
  gchar* time_string = (gchar*)g_malloc0(256);
  struct tm* time_stamp;
  time_stamp = localtime(&ttime_stamp);
  if(_has_24hr_timeformat())
  {
    gchar* full_time_format = g_strdup_printf("%s | %s", 
					      dgettext("hildon-libs", "wdgt_va_date"),
					      dgettext("hildon-libs", "wdgt_va_24h_time"));
    strftime(time_string, 256, full_time_format, time_stamp);
    g_free(full_time_format);
  }
  else
  {
    gchar* full_time_format = g_strdup_printf("%s | %s", 
					      dgettext("hildon-libs", "wdgt_va_date"),
					      (time_stamp->tm_hour > 11) ? 
					      dgettext("hildon-libs", "wdgt_va_12h_time_pm") :
					      dgettext("hildon-libs", "wdgt_va_12h_time_am"));
    strftime(time_string, 256, full_time_format, time_stamp);    
    g_free(full_time_format);
  }
  return time_string;
}

gchar*
_format_start_end_time_stamp(time_t start_time,
			     time_t end_time)
{
  gchar* time_string = (gchar*)g_malloc0(256);
  struct tm end_time_date;
  struct tm start_time_date;

  struct tm* time_stamp = localtime(&end_time);
  memcpy(&end_time_date, time_stamp, sizeof (struct tm));
  time_stamp = localtime(&start_time);
  memcpy(&start_time_date, time_stamp, sizeof (struct tm));

  if(start_time_date.tm_mday == end_time_date.tm_mday &&
     start_time_date.tm_mon == end_time_date.tm_mon &&
     start_time_date.tm_year == end_time_date.tm_year)
  {
    gchar* start_time_str = _format_date_time_stamp(start_time);
    gchar* end_time_str = _format_time_stamp(end_time);
    gchar* full = g_strdup_printf("%s - %s", start_time_str, end_time_str);
    g_free(start_time_str);
    g_free(end_time_str);
    return full;
  }
  else
  {
    gchar* start_time_str = _format_date_time_stamp(start_time);
    gchar* end_time_str = _format_date_time_stamp(end_time);
    gchar* full = g_strdup_printf("%s - %s", start_time_str, end_time_str);
    g_free(start_time_str);
    g_free(end_time_str);
    return full;
  }
}

void
open_calendar_event_window(osso_context_t* osso, int calendar_id, const gchar* event_id)
{
  int error = 0;  
  CCalendar* calendar = CMulticalendar::MCInstance()->getCalendarById(calendar_id, error);
  if(calendar)
  {
    CEvent* event = calendar->getEvent(event_id, error);
    if(event)
    {
      GtkWidget* window;
      GtkWidget* summary_text_view;
      GtkWidget* location_text_view;
      GtkWidget* description_text_view;
      GtkWidget* open_calendar_button;
      GtkWidget* start_time_label;
      window = hildon_stackable_window_new();
      summary_text_view = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
      location_text_view = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
      description_text_view = hildon_text_view_new();

      gchar* time_formatted = _format_start_end_time_stamp(event->getDateStart(),
							   event->getDateEnd());
      
      gchar* start_time_text = NULL;
      if(event->getStatus()==1)
	start_time_text = g_strdup_printf("%s %s (%s)", dgettext("calendar", "cal_fi_date"), time_formatted, dgettext("calendar", "cal_fi_done"));
      else
      	start_time_text = g_strdup_printf("%s %s", dgettext("calendar", "cal_fi_date"), time_formatted);
      start_time_label = gtk_label_new(start_time_text);
      g_free(start_time_text);
      g_free(time_formatted);
      open_calendar_button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
						    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						    dgettext("calendar", "cal_ap_name"), 
						    NULL);

      calendar_open_data* cal_open_data = g_new0(calendar_open_data,1);
      cal_open_data->osso = osso;
      cal_open_data->time_stamp = event->getDateStart();

      g_signal_connect(open_calendar_button, "clicked", G_CALLBACK(_open_calendar), cal_open_data);
      g_signal_connect(window, "destroy-event", G_CALLBACK(delete_cal_open_data), cal_open_data);
      hildon_entry_set_text(HILDON_ENTRY(summary_text_view),
			    event->getSummary().c_str());
      hildon_entry_set_text(HILDON_ENTRY(location_text_view),
			    event->getLocation().c_str());
      gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(description_text_view), GTK_WRAP_CHAR);
      gtk_text_view_set_editable(GTK_TEXT_VIEW(description_text_view), FALSE);
      GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(description_text_view));
      gtk_text_buffer_insert_at_cursor(buffer,
				       event->getDescription().c_str(),
				       event->getDescription().size());

      GtkWidget* vbox = gtk_vbox_new(FALSE, 3);
      gtk_box_pack_start(GTK_BOX(vbox), summary_text_view, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), description_text_view, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), start_time_label, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), location_text_view, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), open_calendar_button, FALSE, FALSE, 0);
      gtk_container_add(GTK_CONTAINER(window), vbox);
      gtk_window_set_title(GTK_WINDOW(window), dgettext("calendar", "cal_ti_event"));
      gtk_widget_show_all(window);
      delete event;
    }
    delete calendar;
  }
}

void
open_calendar_todo_window(osso_context_t* osso, int calendar_id, const gchar* event_id)
{
  int error = 0;  
  CCalendar* calendar = CMulticalendar::MCInstance()->getCalendarById(calendar_id, error);
  if(calendar)
  {
    CTodo* todo = calendar->getTodo(event_id, error);
    if(todo)
    {
      GtkWidget* window;
      GtkWidget* text_view;
      GtkWidget* open_calendar_button;
      GtkWidget* start_time_label;
      window = hildon_stackable_window_new();
      text_view = hildon_text_view_new();
      gchar* time_formatted = _format_date_time_stamp(todo->getDateStart());
      gchar* start_time_text = NULL;
      if(todo->getStatus()==1)
	start_time_text = g_strdup_printf("%s %s (%s)", dgettext("calendar", "cal_fi_date"), time_formatted, dgettext("calendar", "cal_fi_done"));
      else
      	start_time_text = g_strdup_printf("%s %s", dgettext("calendar", "cal_fi_date"), time_formatted);
      start_time_label = gtk_label_new(start_time_text);
      g_free(start_time_text);
      g_free(time_formatted);
      open_calendar_button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
						    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						    dgettext("calendar", "cal_ap_name"), 
						    NULL);

      calendar_open_data* cal_open_data = g_new0(calendar_open_data,1);
      cal_open_data->osso = osso;
      cal_open_data->time_stamp = todo->getDateStart();

      g_signal_connect(open_calendar_button, "clicked", G_CALLBACK(_open_calendar), cal_open_data);
      g_signal_connect(window, "destroy-event", G_CALLBACK(delete_cal_open_data), cal_open_data);
      gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_CHAR);
      gtk_text_view_set_editable(GTK_TEXT_VIEW(text_view), FALSE);
      GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text_view));
      GtkWidget* vbox = gtk_vbox_new(FALSE, 3);
      
      gtk_text_buffer_insert_at_cursor(buffer,
				       todo->getSummary().c_str(),
				       todo->getSummary().size());
      gtk_box_pack_start(GTK_BOX(vbox), text_view, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), start_time_label, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), open_calendar_button, FALSE, FALSE, 0);
      gtk_container_add(GTK_CONTAINER(window), vbox);
      gtk_window_set_title(GTK_WINDOW(window), dgettext("calendar", "cal_ti_tasks"));
      gtk_widget_show_all(window);
      delete todo;
    }
    delete calendar;
  }
}

void
open_calendar_journal_window(osso_context_t* osso, int calendar_id, const gchar* event_id)
{
  int error = 0;  
  CCalendar* calendar = CMulticalendar::MCInstance()->getCalendarById(calendar_id, error);
  if(calendar)
  {
    CJournal* journal = calendar->getJournal(event_id, error);
    if(journal)
    {
      GtkWidget* window;
      GtkWidget* text_view;
      GtkWidget* open_calendar_button;
      GtkWidget* last_modified_label;
      window = hildon_stackable_window_new();
      text_view = hildon_text_view_new();
      gchar* time_formatted = _format_date_time_stamp(journal->getLastModified());
      gchar* last_modified_text = g_strdup_printf(dgettext("calendar", "cal_va_note_saved"), time_formatted);
      last_modified_label = gtk_label_new(last_modified_text);
      g_free(last_modified_text);
      g_free(time_formatted);
      open_calendar_button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
						    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						    dgettext("calendar", "cal_ap_name"), 
						    NULL);

      calendar_open_data* cal_open_data = g_new0(calendar_open_data,1);
      cal_open_data->osso = osso;
      cal_open_data->time_stamp = journal->getDateStart();

      g_signal_connect(open_calendar_button, "clicked", G_CALLBACK(_open_calendar), cal_open_data);
      g_signal_connect(window, "destroy-event", G_CALLBACK(delete_cal_open_data), cal_open_data);
      gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_CHAR);
      gtk_text_view_set_editable(GTK_TEXT_VIEW(text_view), FALSE);
      GtkTextBuffer* buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text_view));
      GtkWidget* vbox = gtk_vbox_new(FALSE, 3);
      
      gtk_text_buffer_insert_at_cursor(buffer,
				       journal->getSummary().c_str(),
				       journal->getSummary().size());
      gtk_box_pack_start(GTK_BOX(vbox), last_modified_label, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), text_view, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), open_calendar_button, FALSE, FALSE, 0);
      gtk_container_add(GTK_CONTAINER(window), vbox);
      gtk_window_set_title(GTK_WINDOW(window), dgettext("calendar", "cal_ti_notes"));
      gtk_widget_show_all(window);
      delete journal;
    }
    delete calendar;
  }
}
