#ifndef ENGINE_H
#define ENGINE_H

#include <QVector>

#include "reversidefs.h"

class ReversiGame;

// Connect a move with its value.
class MoveAndValue
{
public:
    MoveAndValue(int x = 0, int y = 0, int value = 0) { setXYV(x, y, value); }
    void  setXYV(int x, int y, int value) { x_ = x; y_ = y; value_ = value; }
public:
    int  x_;
    int  y_;
    int  value_;
}; //MoveAndValue

/* This class keeps track of the score for both colors.  Such a score
 * could be either the number of pieces, the score from the evaluation
 * function or anything similar.
 */
class Score {
public:
    Score()
    {
        score_[White] = 0;
        score_[Black] = 0;
    }

    uint score(ChipColor color) const     { return score_[color]; }

    void set(ChipColor color, uint score) { score_[color] = score; }
    void inc(ChipColor color)             { score_[color]++; }
    void dec(ChipColor color)             { score_[color]--; }
    void add(ChipColor color, uint s)     { score_[color] += s; }
    void sub(ChipColor color, uint s)     { score_[color] -= s; }

private:
    uint  score_[2];
};

class SquareStackEntry;

// SquareStackEntry and SquareStack are used during search to keep track of turned pieces.
class SquareStackEntry
{
public:
    SquareStackEntry() {setXY(0,0); }
    void setXY(int x, int y) {x_ = x; y_ = y;};

public:
    int  x_;
    int  y_;

    friend class SquareStack;
};

class SquareStack
{
public:
  SquareStack(int size = 0) { init(size); }

  void              resize(int size) {squarestack_.resize(size); }
  void              init(int size);
  SquareStackEntry  pop() {  return squarestack_[--top_];  }
  void              push(int x, int y) {
                        squarestack_[top_].x_ = x;
                        squarestack_[top_++].y_ = y;
                    }

private:
  QVector<SquareStackEntry> squarestack_;
  int                       top_;
};


class Engine
{
public:
    Engine(int strength = 1, int seed = 0);
    void init(int strength = 1, int seed = 0);
    ReversiPos computeMove(const ReversiGame& game);
    ReversiPos computeFirstMove();
    void setStrength(int strength) { strength_ = strength; }

private:
    void    setupBcBoard();
    void    setupBits();
    int     calcBcScore(ChipColor color);
    quint64 computeOccupiedBits(ChipColor color);
    int     computeMove2(int xplay, int yplay, ChipColor color, int level,
                    int      cutoffval,
                    quint64  colorbits, quint64 opponentbits);
    int     evaluatePosition(ChipColor color);
    int     tryAllMoves(ChipColor opponent, int level, int cutoffval, quint64  opponentbits, quint64 colorbits);

private:
    int         strength_;
    ChipColor   board_[FIELDSIZE + 2][FIELDSIZE + 2];
    int         bc_board_[FIELDSIZE + 1][FIELDSIZE + 1];
    bool        exhaustive_;
    bool        competitive_;
    int         depth_;
    int         coeff_;
    int         nodes_searched_;
    SquareStack squarestack_;

    Score       score_;
    Score       bc_score_;

    quint64      coord_bit_[FIELDSIZE + 1][FIELDSIZE + 1];
    quint64      neighbor_bits_[FIELDSIZE + 1][FIELDSIZE + 1];
};

#endif // ENGINE_H
