//TrackView.cpp: Implementation of the TrackView class (view used for  playlists / search results).

#include "TrackView.h"

TrackView::TrackView(Spotify::Session* pSession, QWidget* parent) :
        QMainWindow(parent)
{
        //constructs the UI track view

#ifdef Q_WS_MAEMO_5
        setAttribute(Qt::WA_Maemo5StackedWindow); //window is part of window stack
#endif

        m_pSession = pSession; //set the Spotify session instance
        connect(m_pSession,SIGNAL(albumBrowserReady(Spotify::AlbumBrowser*,qint32)),SLOT(OnAlbumBrowseReady(Spotify::AlbumBrowser*,qint32)));
        m_CallerID = TRACKVIEW_ID;
        m_lastTimeTrackActivated.start();

        //init the track listview instance
        m_pTrackListView = new CListView(this);
        m_pTrackListModel = new QStandardItemModel(this);
        m_pTrackListDelegate = new ListViewDelegate(this);
        m_pTrackListDelegate->setSubHeaderFontPointSize(14);

        m_pTrackListView->setItemDelegate(m_pTrackListDelegate); //connect the delegate to view
        m_pTrackListView->setModel(m_pTrackListModel); //connect the model to view.
        connect(m_pTrackListView,SIGNAL(activated(QModelIndex)),SLOT(OnTrackActivated(QModelIndex)));

        setupContextMenu(); //set up the listview context menu

        setupMenubar(); //set up the view menubar

        m_Layout.addWidget(m_pTrackListView);

        m_MainLayout.setLayout(&m_Layout);

	setCentralWidget(&m_MainLayout);
}


TrackView::~TrackView()
{
}

void TrackView::setupContextMenu()
{
    //set up the listview context menu

    m_pTrackListView->setContextMenuPolicy(Qt::ActionsContextMenu); //actions context menu associated with listview

    //set up the listview contextmenu options
    QAction* pPlayAlbumAct = new QAction(tr("Play Album"),this);
    connect(pPlayAlbumAct,SIGNAL(triggered()),SLOT(OnPlayAlbumAction()));
    m_pTrackListView->addAction(pPlayAlbumAct);
    QAction* pViewAlbumAct = new QAction(tr("Browse Album"),this);
    connect(pViewAlbumAct,SIGNAL(triggered()),SLOT(OnViewAlbumAction()));
    m_pTrackListView->addAction(pViewAlbumAct);
    QAction* pViewArtistAct = new QAction(tr("Browse Artist"),this);
    connect(pViewArtistAct,SIGNAL(triggered()),SLOT(OnViewArtistAction()));
    m_pTrackListView->addAction(pViewArtistAct);
}

void TrackView::setupMenubar()
 {
     //set up the view menubar

    //set up the the track view menu
    m_pSortByAct = new PickSelectorAction(this);
    m_pSortByAct->setText(tr("Sort tracks by"));

    m_PlaylistSortByList << tr("Playlist order") << tr("Most popular") << tr("Track name (ascending)") << tr("Track name (descending)") << tr("Artist name (ascending)")
               << tr("Artist name (descending)") << tr("Album name (ascending)") << tr("Album name (descending)");

    m_SearchSortByList << tr("Artist, album, track (ascending)") << tr("Most popular") << tr("Track name (ascending)") << tr("Track name (descending)") << tr("Artist name (ascending)")
               << tr("Artist name (descending)") << tr("Album name (ascending)") << tr("Album name (descending)");

    connect(m_pSortByAct,SIGNAL(selectionChanged(int)),SLOT(OnSortingSelectionChanged(int)));
    menuBar()->addAction(m_pSortByAct);

#ifdef Q_WS_MAEMO_5
    menuBar()->addAction(tr("FM Transmitter"));
#endif

    connect(menuBar(),SIGNAL(triggered(QAction*)),SLOT(OnMenubarAction(QAction*)));
 }

Spotify::Track* TrackView::getTrackFromModelIndex(QModelIndex index)
{
    //get track instance pointer from model index. Returns NULL if index is invalid.

    Spotify::Track* track = NULL;

    if(index.isValid())
    {
        int trackPos = index.data(Qt::UserRole).toInt();

        if(m_pPlaylist) //track from playlist
            track = m_pPlaylist->getTrack(trackPos);
        else //search result
        {
            track = m_TrackList.at(trackPos);
        }
    }

    return track;
}

Spotify::Playlist* TrackView::getActivePlaylist()
{
    //returns pointer to active playlist
    return m_pPlaylist;
}

void TrackView::OnPlayAlbumAction()
{
    //request to play the album of currently selected track

    Spotify::Track* track = getTrackFromModelIndex(m_pTrackListView->currentIndex());

    if(track)
    {
        if(track->getAlbum())
        {
            if(m_pSession->browse(track->getAlbum(),&m_CallerID)) //request to load album tracks (browse album)
            {
                m_pTrackListView->setDisabled(true);
#ifdef Q_WS_MAEMO_5
                QMaemo5InformationBox::information(this,tr("Loading album tracks"));
                setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true);
#else
                Banner::information(this,tr("Loading album tracks"));
#endif

            }
        }
    }
}

void TrackView::OnViewArtistAction()
{
    //Request to show Artist View for currently selected artist.

    Spotify::Track* track = getTrackFromModelIndex(m_pTrackListView->currentIndex());

    if(track)
    {
        if(track->getArtist())
        {
            emit showArtistView(track->getArtist());
        }
    }
}

void TrackView::OnAlbumBrowseReady(Spotify::AlbumBrowser* browser, qint32 callerID)
{
    //notification that album browse request completed

    Spotify::Error error;

    if(callerID==m_CallerID)
    {
        if(browser)
        {
            error = sp_albumbrowse_error(browser->getAlbumBrowser());
            if(error.isError())
            {
                //error occured (album browse request failed)
#ifdef Q_WS_MAEMO_5
                QMaemo5InformationBox::information(this,tr("Fetching of album tracks failed: ") + error.toString());
#else
                Banner::information(this,tr("Fetching of album tracks failed: ") + error.toString());
#endif

            }
            else
            {
                if(!browser->load())
                {
#ifdef Q_WS_MAEMO_5
                    QMaemo5InformationBox::information(this,tr("Failed to load tracks!"));
#else
                    Banner::information(this,tr("Failed to load tracks!"));
#endif
                }
                browser->sortByDiscIndex(); //sort tracks in album order
                emit playQueueSet(browser->getTracks(false),0); //only add available (playable) album tracks to the playqueue
            }
        }
    }
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
#endif
    m_pTrackListView->setDisabled(false);
}

void TrackView::OnViewAlbumAction()
{
    //Request to show Album View for currently selected album.

    Spotify::Track* track = getTrackFromModelIndex(m_pTrackListView->currentIndex());

    if(track)
    {
        if(track->getAlbum())
        {
            emit showAlbumView(track->getAlbum());
        }
    }
}

void TrackView::OnMenubarAction(QAction* action)
{

    if(action->text()==tr("FM Transmitter"))
    { //request to display the FM Transmitter control panel dialog (plugin)
        emit showFMTransmitterDialog();
    }
}

void TrackView::OnSortingSelectionChanged(int currSelection)
{
    //sorting selection has changed (new selection is currSelection)

    if(m_pPlaylist)
    {
        //playlist mode
        if(currSelection==ORIGINAL)
            m_pPlaylist->restoreOrignalCollection();
        else if(currSelection==TRACK_ASC)
            m_pPlaylist->sortByTracks(true);
        else if(currSelection==TRACK_DESC)
            m_pPlaylist->sortByTracks(false);
        else if(currSelection==ARTIST_ASC)
            m_pPlaylist->sortByArtists(true);
        else if(currSelection==ARTIST_DESC)
            m_pPlaylist->sortByArtists(false);
        else if(currSelection==ALBUM_ASC)
            m_pPlaylist->sortByAlbums(true);
        else if(currSelection==ALBUM_DESC)
            m_pPlaylist->sortByAlbums(false);
        else if(currSelection==MOST_POPULAR)
            m_pPlaylist->sortByPopularity();
        addTracks(m_pPlaylist);
    }
    else
    {
        //search view mode
        if(currSelection==ORIGINAL)
            qSort(m_TrackList.begin(),m_TrackList.end(),trackLessThanCaseInsensitive);
        else if(currSelection==TRACK_ASC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByNameAsc);
        else if(currSelection==TRACK_DESC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByNameDesc);
        else if(currSelection==ARTIST_ASC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByArtistNameAsc);
        else if(currSelection==ARTIST_DESC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByArtistNameDesc);
        else if(currSelection==ALBUM_ASC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByAlbumNameAsc);
        else if(currSelection==ALBUM_DESC)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByAlbumNameDesc);
        else if(currSelection==MOST_POPULAR)
            qSort(m_TrackList.begin(),m_TrackList.end(),tracksByPopularityDesc);
        addTracks(m_TrackList);
    }

}

void TrackView::setPlaylist(Spotify::Playlist* pPlaylist)
{
    //set the playlist for which tracks should be displayed

	if(pPlaylist)
        {
            m_pPlaylist = pPlaylist;

            m_pSortByAct->populateListModel(m_PlaylistSortByList); //populate sorting alternatives

            setWindowTitle(pPlaylist->getName());

            m_pSortByAct->setCurrentSelection(0); //set sorting to Original order

            m_pPlaylist->restoreOrignalCollection();

            addTracks(m_pPlaylist); //add track data to track list
	}
}

void TrackView::setSearchResults(Spotify::Search* pSearch)
{
    //Set the search results to be displayed

    m_pPlaylist = NULL;

    if(pSearch)
    {
        setWindowTitle(tr("Search results: ") + pSearch->getSearchQuery());

        m_pSortByAct->populateListModel(m_SearchSortByList); //populate sorting alternatives

        m_TrackList = pSearch->getTracks(false); //only return available tracks
        m_pSortByAct->setCurrentSelection(0); //set sorting to Original order
	//sort track instance first by artist, then by album, and finally by track
	qSort(m_TrackList.begin(),m_TrackList.end(),trackLessThanCaseInsensitive); 
        addTracks(m_TrackList);
    }
}


void TrackView::OnTrackActivated(QModelIndex trackIndex)
{
    //A track has been activated (e.g., selected) in the track list

    Q_UNUSED(trackIndex)

    if(m_lastTimeTrackActivated.restart()>=700) //ignore successive (rapid) activations (within 700msecs since last activation)
    {
        m_pTrackListView->setEnabled(false); //disable table temporarily
#ifdef Q_WS_MAEMO_5
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
#endif
        setPlayQueue();
#ifdef Q_WS_MAEMO_5
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
#endif
        m_pTrackListView->setEnabled(true); //re-enable the list view
    }
}


void TrackView::setPlayQueue()
{
    //Assign new playqueue

    QList<Spotify::Track*> PlayQueue;

    if(m_pPlaylist) //playlist view
    {
        PlayQueue = m_pPlaylist->getTracks(false);
        emit playQueueSet(PlayQueue,m_pTrackListView->currentIndex().row()); //signal that playqueue has been set / updated
    }
    else //search results view
    { //add all tracks including (and following) the selected track for search results
        for(int pos = m_pTrackListView->currentIndex().row(); pos < m_pTrackListModel->rowCount(); pos++)
        {
            Spotify::Track* track = m_TrackList.at(pos);
            if(track)
            {
                PlayQueue.append(track);
            }
        }
        emit playQueueSet(PlayQueue,0); //signal that playqueue has been set / updated
    }
}


void TrackView::addTracks(Spotify::Playlist* playlist)
{
    //add playlist tracks to main listview

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
#endif

    m_pTrackListModel->clear();

    if(playlist)
    {
        for(int trackPos = 0; trackPos < playlist->getNumTracks(); trackPos++)
        {
            Spotify::Track* currTrack = playlist->getTrack(trackPos);
            if(currTrack)
            {
                if(currTrack->isAvailable()) //track is available (playable)
                {
                    addTrackListModelItem(currTrack,trackPos);
                }
            }
        }
    }

    if(m_pTrackListModel->rowCount()>0)
    { //scroll to top, no current / selected row by default
        m_pTrackListView->scrollToTop();
        QModelIndex index = m_pTrackListModel->index(-1,-1);
        m_pTrackListView->setCurrentIndex(index);
    }
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
#endif
}

void TrackView::addTrackListModelItem(Spotify::Track* track, qint32 pos)
{
    //add track item to the track list item model

    QString strSubHeader;
    QPixmap iconMap;

    iconMap.load("/opt/qspot/images/icon-type-song-color.png"); //load song (track) type icon from file
    QIcon trackTypeIcon(iconMap);
    iconMap.load("/opt/qspot/images/icon-type-song-color-offline.png"); //load song (track) offline type icon from file
    QIcon trackTypeOfflineIcon(iconMap);
    iconMap.load("/opt/qspot/images/icon-type-song-starred.png"); //load song (track) type icon from file
    QIcon trackTypeStarredIcon(iconMap);
    iconMap.load("/opt/qspot/images/icon-type-song-starred-offline.png"); //load song (track) type icon from file
    QIcon trackTypeStarredOfflineIcon(iconMap);

    //appends new track item to the track list model
    QStandardItem *trackItem = new QStandardItem();
    trackItem->setEditable(false);
    strSubHeader = "";
    if(track->isStarred()) //starred track
    {
        if(!track->isOffline()) //track not available offline
            trackItem->setData(trackTypeStarredIcon,ListViewDelegate::iconRole);
         else //track available offline
            trackItem->setData(trackTypeStarredOfflineIcon,ListViewDelegate::iconRole);
    }
    else
    {
        if(!track->isOffline()) //track not available offline
            trackItem->setData(trackTypeIcon,ListViewDelegate::iconRole);
         else //track available offline
            trackItem->setData(trackTypeOfflineIcon,ListViewDelegate::iconRole);
    }
    trackItem->setData(track->getName(),ListViewDelegate::headerRole);
    trackItem->setData(track->getDurationAsString(),ListViewDelegate::subHeaderRightJustifyRole);
    trackItem->setData(pos,Qt::UserRole); //associate position in track instances list with list item */
    if(track->getArtist())
    {
        strSubHeader = track->getArtist()->getName();
    }
    if(track->getAlbum())
    {
        strSubHeader += " | ";
        strSubHeader += track->getAlbum()->getName();
    }
    trackItem->setData(strSubHeader,ListViewDelegate::subHeaderLeftJustifyRole);
    m_pTrackListModel->appendRow(trackItem);
}

void TrackView::addTracks(const QList<Spotify::Track*>& trackList)
{
    //add tracks from specified list to main listview.
    //Overriden method.

    qint32 pos = 0;

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
#endif

    m_pTrackListModel->clear();

     for(int trackPos = 0; trackPos < trackList.size(); trackPos++)
     {
         Spotify::Track* currTrack = trackList.at(trackPos);
         if(currTrack)
         {
             if(currTrack->isAvailable())
             {
                 addTrackListModelItem(currTrack,pos);
                 pos++;
             }
         }
     }

     if(m_pTrackListModel->rowCount()>0)
     { //scroll to top, no current / selected row by default
         m_pTrackListView->scrollToTop();
         QModelIndex index = m_pTrackListModel->index(-1,-1);
         m_pTrackListView->setCurrentIndex(index);
     }
#ifdef Q_WS_MAEMO_5
     setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
#endif
}

void TrackView::keyPressEvent(QKeyEvent *evt)
{
    //Overriden keyPressEvent handler

    if(evt->key()==Qt::Key_Left)
    {
        close(); //hide the view
        evt->accept();
    }
    else if(evt->key()==Qt::Key_M)
    {
        //set sorting order to most popular
        if(m_pSortByAct->getCurrentSelection()!=1)
        {
            m_pSortByAct->setCurrentSelection(1); //set sorting to Most popular
            if(m_pPlaylist)
            { //playlist view
                m_pPlaylist->sortByPopularity();
                addTracks(m_pPlaylist);
            }
            else
            { //search results view
                qSort(m_TrackList.begin(),m_TrackList.end(),tracksByPopularityDesc);
                addTracks(m_TrackList);
            }
        }
    }
    else
    {
        QMainWindow::keyPressEvent(evt);
    }
}

