#include "link.h"

namespace Spotify
{
    Link::Link():
            m_link( 0 ),
            m_type( Spotify::Link::INVALID )
    {
    }

    Link::Link( const QString& link ):
            m_link( 0 ),
            m_type( Spotify::Link::INVALID ),
            m_uri( link )
    {
        m_link = sp_link_create_from_string( link.toUtf8().data() );
        if( m_link )
        sp_link_add_ref( m_link );
    }

    Link::~Link()
    {
        if( m_link )
            sp_link_release( m_link );
    }

    QString Link::toString() const
    {
        if( !m_uri.isEmpty() )
            return m_uri;
        else if( m_link )
        {
            int size = 256;
            char spotify_uri[ 256 ];
            size = sp_link_as_string( m_link, spotify_uri, size );
            if( size > 0 )
                m_uri.fromUtf8( spotify_uri, size );
            return m_uri;
        }
        else
            return m_uri;
    }

    Spotify::Link::Type Link::type() const
    {
        if( m_link )
        {
            sp_linktype type = sp_link_type( m_link );
            if( type == SP_LINKTYPE_TRACK ) return TRACK;
            else if( type == SP_LINKTYPE_ALBUM ) return ALBUM;
            else if( type == SP_LINKTYPE_ARTIST ) return ARTIST;
            else if( type == SP_LINKTYPE_SEARCH ) return SEARCH;
            else if( type == SP_LINKTYPE_PLAYLIST ) return PLAYLIST;
            else return INVALID;
        }
        return INVALID;
    }

    Spotify::Link Link::fromTrack( const Spotify::Track* track )
    {
        if( track )
        {
            sp_track* t = track->getTrack();
            if( t )
            {
                sp_link* l = sp_link_create_from_track( t, 0 );
                if( l )
                {
                    int size = 256;
                    char spotify_uri[ 256 ];
                    size = sp_link_as_string( l, spotify_uri, size );
                    if( size > 0 )
                    {
                        return Spotify::Link( spotify_uri );
                    }
                }
            }
        }
        return Spotify::Link();
    }

    Spotify::Link Link::fromPlaylist( const Spotify::Playlist* playlist )
    {
        if( playlist )
        {
            sp_playlist* p = playlist->getPlaylist();
            if( p )
            {
                sp_link* l = sp_link_create_from_playlist( p );
                if( l )
                {
                    int size = 256;
                    char spotify_uri[ 256 ];
                    size = sp_link_as_string( l, spotify_uri, size );
                    if( size > 0 )
                    {
                        return Spotify::Link( spotify_uri );
                    }
                }
            }
        }
        return Spotify::Link();
    }

    Spotify::Link Link::fromAlbum( const Spotify::Album* album )
    {
        if( album )
        {
            sp_album* a = album->getAlbum();
            if( a )
            {
                sp_link* l = sp_link_create_from_album( a );
                if( l )
                {
                    int size = 256;
                    char spotify_uri[ 256 ];
                    size = sp_link_as_string( l, spotify_uri, size );
                    if( size > 0 )
                    {
                        return Spotify::Link( spotify_uri );
                    }
                }
            }
        }
        return Spotify::Link();
    }

    Spotify::Link Link::fromArtist( const Spotify::Artist* artist )
    {
        if( artist )
        {
            sp_artist* a = artist->getArtist();
            if( a )
            {
                sp_link* l = sp_link_create_from_artist( a );
                if( l )
                {
                    int size = 256;
                    char spotify_uri[ 256 ];
                    size = sp_link_as_string( l, spotify_uri, size );
                    if( size > 0 )
                    {
                        return Spotify::Link( spotify_uri );
                    }
                }
            }
        }
        return Spotify::Link();
    }
} // namespace Spotify
