# This script configures PyQtMobility.
#
# Copyright (c) 2010 Riverbank Computing Limited.
# 
# This file is part of PyQtMobility.
# 
# This file may be used under the terms of the GNU General Public License
# v2 or v3 as published by the Free Software Foundation which can be found in
# the files LICENSE-GPL2.txt and LICENSE-GPL3.txt included in this package.
# In addition, as a special exception, Riverbank gives you certain additional
# rights. These rights are described in the Riverbank GPL Exception, which
# can be found in the file GPL-Exception.txt in this package.
# 
# This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
# WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.


import sys
import os
import subprocess
import glob
import optparse
import shutil


# Import SIP's configuration module so that we have access to the error
# reporting.  Then try and import the configuration modules for PyQt4.
try:
    import sipconfig
except ImportError:
    sys.stderr.write("Unable to import sipconfig.  Please make sure SIP is installed.\n")
    sys.exit(1)

try:
    from PyQt4 import pyqtconfig
except ImportError:
    sipconfig.error("Unable to import PyQt4.pyqtconfig.  Please make sure PyQt is installed.")


# Initialise the globals.
sip_min_version = 0x040b00
pyqt_min_version = 0x040705

pyqt = pyqtconfig.Configuration()
src_dir = os.path.dirname(os.path.abspath(__file__))


def create_optparser():
    """ Create the parser for the command line. """

    def store_abspath(option, opt_str, value, parser):
        setattr(parser.values, option.dest, os.path.abspath(value))

    def store_abspath_dir(option, opt_str, value, parser):
        if not os.path.isdir(value):
            raise optparse.OptionValueError("'%s' is not a directory" % value)
        setattr(parser.values, option.dest, os.path.abspath(value))

    p = optparse.OptionParser(usage="python %prog [options]",
            version="0.1")

    p.add_option("-c", "--concatenate", action="store_true", default=False,
            dest="concat", help="concatenate the C++ source files")
    p.add_option("-d", "--destdir", action="callback",
            default=pyqt.pyqt_mod_dir, type="string", metavar="DIR",
            dest="qtmmoddir", callback=store_abspath, help="where the "
            "QtMobility package will be installed [default: %s]" %
            pyqt.pyqt_mod_dir)
    p.add_option("-e", "--enable", action="append", default=[],
            metavar="MODULE", dest="enabled", help="enable the specified "
            "MODULE [default: all QtMobility modules will be enabled]")
    p.add_option("-j", "--concatenate-split", type="int", default=1,
            metavar="N", dest="split", help="split the concatenated C++ "
            "source files into N pieces [default: 1]")
    p.add_option("-n", action="callback", default=pyqt.qt_inc_dir,
            type="string", metavar="DIR", dest="qtmincdir",
            callback=store_abspath_dir, help="the directory containing the "
            "QtMobility header file directories [default: "
            "%s]" % pyqt.qt_inc_dir)

    if sys.platform != 'win32':
        if sys.platform in ('linux2', 'darwin'):
            pip_default = True
            pip_default_str = "enabled"
        else:
            pip_default = False
            pip_default_str = "disabled"

        p.add_option("--protected-is-public", action="store_true",
                default=pip_default, dest="prot_is_public",
                help="enable building with 'protected' redefined as 'public' "
                        "[default: %s]" % pip_default_str)
        p.add_option("--protected-not-public", action="store_false",
                dest="prot_is_public",
                help="disable building with 'protected' redefined as 'public'")

    p.add_option("--no-docstrings", action="store_true", default=False,
            dest="no_docstrings", help="disable the generation of docstrings")
    p.add_option("-o", action="callback", default=pyqt.qt_lib_dir,
            type="string", metavar="DIR", dest="qtmlibdir",
            callback=store_abspath_dir, help="the directory containing the "
            "QtMobility libraries [default: %s]" % pyqt.qt_lib_dir)
    p.add_option("-r", "--trace", action="store_true", default=False,
            dest="tracing", help="build the QtMobility modules with tracing "
            "enabled")
    p.add_option("-u", "--debug", action="store_true", default=False,
            help="build the QtMobility modules with debugging symbols")
    p.add_option("-v", "--sipdir", action="callback",
            default=pyqt.pyqt_sip_dir, metavar="DIR", dest="qtmsipdir",
            callback=store_abspath, type="string", help="where the QtMobility "
            ".sip files will be installed [default: %s]" % pyqt.pyqt_sip_dir)

    return p


def inform_user(qtm_version_str):
    """ Tell the user the option values that are going to be used. """

    sipconfig.inform("This is the GPL version of PyQtMobility 0.1.")
    sipconfig.inform("QtMobility %s is being used." % qtm_version_str)
    sipconfig.inform("PyQt %s is being used." % pyqt.pyqt_version_str)
    sipconfig.inform("Qt v%s %s edition is being used." % (sipconfig.version_to_string(pyqt.qt_version), pyqt.qt_edition))
    sipconfig.inform("SIP %s is being used." % pyqt.sip_version_str)

    sipconfig.inform("The QtMobility package will be installed in %s." % opts.qtmmoddir)
    sipconfig.inform("The QtMobility .sip files will be installed in %s." % opts.qtmsipdir)

    if opts.no_docstrings:
        sipconfig.inform("The QtMobility modules are being built without generated docstrings.")
    else:
        sipconfig.inform("The QtMobility modules are being built with generated docstrings.")

    if opts.prot_is_public:
        sipconfig.inform("The QtMobility modules are being built with 'protected' redefined as 'public'.")


def check_qtmobility():
    """ See if QtMobility can be found and what its version is. """

    # Find the QtMobility header files.
    qtmglobal = os.path.join(opts.qtmincdir, 'qmobilityglobal.h')

    if not os.access(qtmglobal, os.F_OK):
        sipconfig.error("qmobilityglobal.h could not be found in %s. If QtMobility is installed then use the -n argument to explicitly specify the correct directory." % opts.qtmincdir)

    # Get the QtMobility version string.
    return sipconfig.read_version(qtmglobal, "QtMobility", "QTM_VERSION",
            "QTM_VERSION_STR")


def sip_flags():
    """ Return the SIP flags. """

    # Get the flags used for the main PyQt module.
    flags = pyqt.pyqt_sip_flags.split()

    # Add PyQt's .sip files to the search path.
    flags.append("-I")
    flags.append(pyqt.pyqt_sip_dir)

    return flags


def mk_clean_dir(name, clean=1):
    """ Create a clean (ie. empty) directory. """

    if clean:
        try:
            shutil.rmtree(name)
        except:
            pass

    try:
        os.makedirs(name)
    except:
        if clean:
            sipconfig.error("Unable to create the %s directory." % name)


def generate_code(mname):
    """ Generate the code for a QtMobility module. """

    sipconfig.inform("Generating the C++ source for the %s module..." % mname)

    mk_clean_dir(mname)

    # Build the SIP command line.  Keyword argument support is enabled.
    argv = ['"' + pyqt.sip_bin + '"', '-w', '-k']

    argv.extend(sip_flags())

    if not opts.no_docstrings:
        argv.append('-o');

    if opts.prot_is_public:
        argv.append('-P')

    if opts.concat:
        argv.append('-j')
        argv.append(str(opts.split))

    if opts.tracing:
        argv.append('-r')

    argv.append('-c')
    argv.append(mname)

    buildfile = os.path.join(mname, mname + '.sbf')
    argv.append('-b')
    argv.append(buildfile)

    argv.append('-I')
    argv.append(os.path.join(src_dir, 'sip'))

    # Add the name of the .sip file keeping in mind SIP assumes POSIX style
    # path separators.
    drive, path = os.path.splitdrive(src_dir)
    parts = path.split(os.pathsep)
    parts.extend(['sip', mname, mname + 'mod.sip'])
    argv.append(drive + '/'.join(parts))

    argv[0] = argv[0].strip('"')
    print argv

    os.system(' '.join([argv[0], '-V']))
    os.system(' '.join(argv))
   # print subprocess.Popen(argv, shell=True, bufsize=8192, stderr=subprocess.PIPE).stderr.read()

    print "Done"

    # Check the result.
    if not os.access(buildfile, os.F_OK):
        sipconfig.error("Unable to create the C++ code.")

    # Generate the Makefile.
    sipconfig.inform("Creating the Makefile for the %s module..." % mname)

    installs = []
    sipfiles = []

    for s in glob.glob('sip/%s/*.sip' % mname):
        sipfiles.append(os.path.join('..', 'sip', mname, os.path.basename(s)))

    installs.append([sipfiles, os.path.join(opts.qtmsipdir, 'QtMobility', mname)])

    # Set the PyQt dependencies.
    if mname in ('QtContacts', 'QtVersit'):
        qt = ['QtCore', 'QtGui']
    elif mname in ('QtBearer', 'QtSystemInfo'):
        qt = ['QtCore', 'QtNetwork']
    elif mname in ('QtMultimediaKit', ):
        qt = ['QtCore', 'QtGui', 'QtNetwork']
    else:
        qt = ['QtCore']

    makefile = sipconfig.SIPModuleMakefile(configuration=pyqt,
            build_file=mname + '.sbf', dir=mname,
            install_dir=os.path.join(opts.qtmmoddir, 'QtMobility'),
            installs=installs, debug=opts.debug,
            prot_is_public=opts.prot_is_public, qt=qt)

    makefile.extra_include_dirs.append(opts.qtmincdir)
    makefile.extra_include_dirs.append(os.path.join(opts.qtmincdir, mname))
    makefile.extra_lib_dirs.append(opts.qtmlibdir)
    makefile.extra_libs.append(mname)

    # The only internal dependency is QtVersit on QtContacts.
    if mname == 'QtVersit':
        makefile.extra_include_dirs.append(
                os.path.join(opts.qtmincdir, 'QtContacts'))
        makefile.extra_libs.append('QtContacts')

    makefile.generate()


def create_init_py(qtm_version, qtm_version_str):
    """ Create __init__.py from the template to include the QtMobility
    information.
    """

    sipconfig.inform("Generating __init__.py for the package...")

    src = open(os.path.join(src_dir, '__init__.py.in'))
    dst = open('__init__.py', 'w')

    for line in src:
        line = line.replace('@CFG_QTM_VERSION@', hex(qtm_version))
        line = line.replace('@CFG_QTM_VERSION_STR@', qtm_version_str)

        dst.write(line)

    src.close()
    dst.close()


def main(argv):
    """Create the configuration module module.

    argv is the list of command line arguments.
    """
    # Check SIP is new enough.
    if pyqt.sip_version < sip_min_version:
        sipconfig.error("This version of PyQtMobility requires SIP v%s or later" % sipconfig.version_to_string(sip_min_version))

    # Check PyQt is new enough.
    if pyqt.pyqt_version < pyqt_min_version:
        sipconfig.error("This version of PyQtMobility requires PyQt v%s or later" % sipconfig.version_to_string(pyqt_min_version))

    # Parse the command line.
    global opts

    p = create_optparser()
    opts, args = p.parse_args()

    if args:
        p.print_help()
        sys.exit(2)

    # Provide defaults for platform-specific options.
    if sys.platform == 'win32':
        opts.prot_is_public = False

    # Check for QtMobility.
    qtm_version, qtm_version_str = check_qtmobility()

    # Tell the user what's been found.
    inform_user(qtm_version_str)

    # Create __init__.py.
    create_init_py(qtm_version, qtm_version_str)

    # Generate the code.
    if len(opts.enabled) == 0:
        modules = ['QtBearer', 'QtContacts', 'QtLocation', 'QtMessaging',
                'QtMultimediaKit', 'QtPublishSubscribe', 'QtSensors',
                'QtServiceFramework', 'QtSystemInfo', 'QtVersit']
    else:
        modules = opts.enabled

    for mname in modules:
        generate_code(mname)

    # Create the top level Makefile.
    sipconfig.inform("Creating top level Makefile...")

    installs=[(['__init__.py'], os.path.join(opts.qtmmoddir, 'QtMobility'))]

    sipconfig.ParentMakefile(configuration=pyqt, subdirs=modules,
            installs=installs).generate()


###############################################################################
# The script starts here.
###############################################################################

if __name__ == "__main__":
    try:
        main(sys.argv)
    except SystemExit:
        raise
    except:
        sys.stderr.write(
"""An internal error occured.  Please report all the output from the program,
including the following traceback, to support@riverbankcomputing.com.
""")
        raise
