/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtgui_python.h"

#include "qmatrix_wrapper.h"

// Extra includes
#include <QPainterPath>
#include <qdatastream.h>
#include <qline.h>
#include <qmatrix.h>
#include <qpainterpath.h>
#include <qpoint.h>
#include <qpolygon.h>
#include <qrect.h>
#include <qregion.h>
#include <qvariant.h>

using namespace Shiboken;

static int
SbkQMatrix_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QMatrix* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 2 || numArgs == 3 || numArgs == 4 || numArgs == 5)
        goto SbkQMatrix_Init_TypeError;

    if (!PyArg_UnpackTuple(args, "QMatrix", 0, 6, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4]), &(pyargs[5])))
        return -1;

    if (numArgs == 0) {
        // QMatrix()
        cptr = new QMatrix();
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QMatrix & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QMatrix* cpp_arg0 = Shiboken::Converter<QMatrix* >::toCpp(pyargs[0]);
        // QMatrix(QMatrix)
        cptr = new QMatrix(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<Qt::Initialization >::isConvertible(pyargs[0])) {
        Qt::Initialization cpp_arg0 = Shiboken::Converter<Qt::Initialization >::toCpp(pyargs[0]);
        // QMatrix(Qt::Initialization)
        cptr = new QMatrix(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 6 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3]) && PyNumber_Check(pyargs[4]) && PyNumber_Check(pyargs[5])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
        qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
        qreal cpp_arg4 = Shiboken::Converter<qreal >::toCpp(pyargs[4]);
        qreal cpp_arg5 = Shiboken::Converter<qreal >::toCpp(pyargs[5]);
        // QMatrix(qreal,qreal,qreal,qreal,qreal,qreal)
        cptr = new QMatrix(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4, cpp_arg5);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrix_Init_TypeError;

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QMatrix >(), cptr)) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQMatrix_Init_TypeError;

    sbkSelf->validCppObject = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;

    SbkQMatrix_Init_TypeError:
        const char* overloads[] = {"", "PySide.QtCore.Qt.Initialization", "PySide.QtGui.QMatrix", "float, float, float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QMatrix", overloads);
        return -1;
}

static PyObject* SbkQMatrixFunc_det(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // det()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QMatrix::det();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_determinant(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // determinant()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QMatrix::determinant();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_dx(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // dx()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QMatrix::dx();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_dy(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // dy()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QMatrix::dy();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_inverted(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // inverted(bool*)const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    // Begin code injection
    
    // TEMPLATE - fix_bool* - START
    bool ok_;
    QMatrix retval_ = cppSelf->inverted(&ok_);
    py_result = Shiboken::makeTuple(retval_, ok_);
    // TEMPLATE - fix_bool* - END
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_isIdentity(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isIdentity()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QMatrix::isIdentity();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_isInvertible(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isInvertible()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QMatrix::isInvertible();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_m11(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // m11()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QMatrix::m11();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_m12(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // m12()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QMatrix::m12();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_m21(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // m21()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QMatrix::m21();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_m22(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // m22()const
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QMatrix::m22();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQMatrixFunc_map(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QPolygon & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QPolygon > cpp_arg0_auto_ptr;
        QPolygon* cpp_arg0 = Shiboken::Converter<QPolygon* >::toCpp(arg);
        if (!SbkQPolygon_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QPolygon >(cpp_arg0);
        // map(QPolygon)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QPolygon cpp_result = cppSelf->QMatrix::map(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygon >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QRegion & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QRegion > cpp_arg0_auto_ptr;
        QRegion* cpp_arg0 = Shiboken::Converter<QRegion* >::toCpp(arg);
        if (!SbkQRegion_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QRegion >(cpp_arg0);
        // map(QRegion)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QRegion cpp_result = cppSelf->QMatrix::map(*cpp_arg0);
        py_result = Shiboken::Converter<QRegion >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QPolygonF & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QPolygonF > cpp_arg0_auto_ptr;
        QPolygonF* cpp_arg0 = Shiboken::Converter<QPolygonF* >::toCpp(arg);
        if (!SbkQPolygonF_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QPolygonF >(cpp_arg0);
        // map(QPolygonF)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QPolygonF cpp_result = cppSelf->QMatrix::map(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QPoint & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QPoint* cpp_arg0 = Shiboken::Converter<QPoint* >::toCpp(arg);
        // map(QPoint)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QPoint cpp_result = cppSelf->QMatrix::map(*cpp_arg0);
        py_result = Shiboken::Converter<QPoint >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QPointF & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(arg);
        if (!SbkQPointF_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // map(QPointF)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QPointF cpp_result = cppSelf->QMatrix::map(*cpp_arg0);
        py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QPainterPath & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QPainterPath* cpp_arg0 = Shiboken::Converter<QPainterPath* >::toCpp(arg);
        // map(QPainterPath)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QPainterPath cpp_result = cppSelf->QMatrix::map(*cpp_arg0);
        py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QLine & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QLine* cpp_arg0 = Shiboken::Converter<QLine* >::toCpp(arg);
        // map(QLine)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QLine cpp_result = cppSelf->QMatrix::map(*cpp_arg0);
        py_result = Shiboken::Converter<QLine >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QLineF & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QLineF > cpp_arg0_auto_ptr;
        QLineF* cpp_arg0 = Shiboken::Converter<QLineF* >::toCpp(arg);
        if (!SbkQLineF_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QLineF >(cpp_arg0);
        // map(QLineF)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QLineF cpp_result = cppSelf->QMatrix::map(*cpp_arg0);
        py_result = Shiboken::Converter<QLineF >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc_map_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc_map_TypeError:
        const char* overloads[] = {"PySide.QtCore.QLine", "PySide.QtCore.QLineF", "PySide.QtGui.QPainterPath", "PySide.QtCore.QPoint", "PySide.QtCore.QPointF", "PySide.QtGui.QPolygon", "PySide.QtGui.QPolygonF", "PySide.QtGui.QRegion", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QMatrix.map", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc_mapRect(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QRect & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QRect* cpp_arg0 = Shiboken::Converter<QRect* >::toCpp(arg);
        // mapRect(QRect)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QRect cpp_result = cppSelf->QMatrix::mapRect(*cpp_arg0);
        py_result = Shiboken::Converter<QRect >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QRectF & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(arg);
        if (!SbkQRectF_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // mapRect(QRectF)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QRectF cpp_result = cppSelf->QMatrix::mapRect(*cpp_arg0);
        py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc_mapRect_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc_mapRect_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRect", "PySide.QtCore.QRectF", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QMatrix.mapRect", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc_mapToPolygon(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QRect & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QRect* cpp_arg0 = Shiboken::Converter<QRect* >::toCpp(arg);
        // mapToPolygon(QRect)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QPolygon cpp_result = cppSelf->QMatrix::mapToPolygon(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygon >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc_mapToPolygon_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc_mapToPolygon_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRect", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QMatrix.mapToPolygon", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc_reset(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // reset()
    QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
    cppSelf->QMatrix::reset();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject* SbkQMatrixFunc_rotate(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // rotate(qreal)
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QMatrix & cpp_result = cppSelf->QMatrix::rotate(cpp_arg0);
        py_result = Shiboken::Converter<QMatrix & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc_rotate_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc_rotate_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QMatrix.rotate", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc_scale(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "scale", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // scale(qreal,qreal)
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QMatrix & cpp_result = cppSelf->QMatrix::scale(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<QMatrix & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc_scale_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc_scale_TypeError:
        const char* overloads[] = {"float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QMatrix.scale", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc_setMatrix(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "setMatrix", 6, 6, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4]), &(pyargs[5])))
        return 0;

    if (numArgs == 6 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3]) && PyNumber_Check(pyargs[4]) && PyNumber_Check(pyargs[5])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
        qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
        qreal cpp_arg4 = Shiboken::Converter<qreal >::toCpp(pyargs[4]);
        qreal cpp_arg5 = Shiboken::Converter<qreal >::toCpp(pyargs[5]);
        // setMatrix(qreal,qreal,qreal,qreal,qreal,qreal)
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        cppSelf->QMatrix::setMatrix(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4, cpp_arg5);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc_setMatrix_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQMatrixFunc_setMatrix_TypeError:
        const char* overloads[] = {"float, float, float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QMatrix.setMatrix", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc_shear(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "shear", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // shear(qreal,qreal)
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QMatrix & cpp_result = cppSelf->QMatrix::shear(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<QMatrix & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc_shear_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc_shear_TypeError:
        const char* overloads[] = {"float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QMatrix.shear", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc_translate(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "translate", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // translate(qreal,qreal)
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QMatrix & cpp_result = cppSelf->QMatrix::translate(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<QMatrix & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc_translate_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc_translate_TypeError:
        const char* overloads[] = {"float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QMatrix.translate", overloads);
        return 0;
}

static PyMethodDef SbkQMatrix_methods[] = {
    {"det", (PyCFunction)SbkQMatrixFunc_det, METH_NOARGS},
    {"determinant", (PyCFunction)SbkQMatrixFunc_determinant, METH_NOARGS},
    {"dx", (PyCFunction)SbkQMatrixFunc_dx, METH_NOARGS},
    {"dy", (PyCFunction)SbkQMatrixFunc_dy, METH_NOARGS},
    {"inverted", (PyCFunction)SbkQMatrixFunc_inverted, METH_NOARGS},
    {"isIdentity", (PyCFunction)SbkQMatrixFunc_isIdentity, METH_NOARGS},
    {"isInvertible", (PyCFunction)SbkQMatrixFunc_isInvertible, METH_NOARGS},
    {"m11", (PyCFunction)SbkQMatrixFunc_m11, METH_NOARGS},
    {"m12", (PyCFunction)SbkQMatrixFunc_m12, METH_NOARGS},
    {"m21", (PyCFunction)SbkQMatrixFunc_m21, METH_NOARGS},
    {"m22", (PyCFunction)SbkQMatrixFunc_m22, METH_NOARGS},
    {"map", (PyCFunction)SbkQMatrixFunc_map, METH_O},
    {"mapRect", (PyCFunction)SbkQMatrixFunc_mapRect, METH_O},
    {"mapToPolygon", (PyCFunction)SbkQMatrixFunc_mapToPolygon, METH_O},
    {"reset", (PyCFunction)SbkQMatrixFunc_reset, METH_NOARGS},
    {"rotate", (PyCFunction)SbkQMatrixFunc_rotate, METH_O},
    {"scale", (PyCFunction)SbkQMatrixFunc_scale, METH_VARARGS},
    {"setMatrix", (PyCFunction)SbkQMatrixFunc_setMatrix, METH_VARARGS},
    {"shear", (PyCFunction)SbkQMatrixFunc_shear, METH_VARARGS},
    {"translate", (PyCFunction)SbkQMatrixFunc_translate, METH_VARARGS},
    {0} // Sentinel
};

static PyObject* SbkQMatrixFunc___mul__(PyObject* self, PyObject* arg)
{
    bool isReverse = SbkQMatrix_Check(arg) && !SbkQMatrix_Check(self);
    if (isReverse)
        std::swap(self, arg);

    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (!isReverse
        && SbkBaseWrapper_Check(arg)
        && !PyObject_TypeCheck(arg, self->ob_type)
        && PyObject_HasAttrString(arg, const_cast<char*>("__rmul__"))) {
        PyObject* revOpMethod = PyObject_GetAttrString(arg, const_cast<char*>("__rmul__"));
        if (revOpMethod && PyCallable_Check(revOpMethod)) {
            py_result = PyObject_CallFunction(revOpMethod, const_cast<char*>("O"), self);
            if (PyErr_Occurred() && (PyErr_ExceptionMatches(PyExc_NotImplementedError) || PyErr_ExceptionMatches(PyExc_AttributeError))) {
                PyErr_Clear();
                Py_XDECREF(py_result);
                py_result = 0;
            }
        }
        Py_XDECREF(revOpMethod);

    }
    // Do not enter here if other object has implemented a reverse operator.
    if (!py_result) {

    if (isReverse && Shiboken::Converter<QPoint & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QPoint* cpp_arg0 = Shiboken::Converter<QPoint* >::toCpp(arg);
        // operator*(QPoint,QMatrix) [reverse operator]
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QPoint cpp_result = (*cpp_arg0) * (*cppSelf);
        py_result = Shiboken::Converter<QPoint >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (isReverse && Shiboken::Converter<QPointF & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(arg);
        if (!SbkQPointF_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // operator*(QPointF,QMatrix) [reverse operator]
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QPointF cpp_result = (*cpp_arg0) * (*cppSelf);
        py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<QMatrix & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QMatrix* cpp_arg0 = Shiboken::Converter<QMatrix* >::toCpp(arg);
        // operator*(QMatrix)const
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QMatrix cpp_result = (*cppSelf) * (*cpp_arg0);
        py_result = Shiboken::Converter<QMatrix >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (isReverse && Shiboken::Converter<QLine & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QLine* cpp_arg0 = Shiboken::Converter<QLine* >::toCpp(arg);
        // operator*(QLine,QMatrix) [reverse operator]
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QLine cpp_result = (*cpp_arg0) * (*cppSelf);
        py_result = Shiboken::Converter<QLine >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (isReverse && Shiboken::Converter<QLineF & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QLineF > cpp_arg0_auto_ptr;
        QLineF* cpp_arg0 = Shiboken::Converter<QLineF* >::toCpp(arg);
        if (!SbkQLineF_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QLineF >(cpp_arg0);
        // operator*(QLineF,QMatrix) [reverse operator]
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QLineF cpp_result = (*cpp_arg0) * (*cppSelf);
        py_result = Shiboken::Converter<QLineF >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc___mul___TypeError;

    } // End of "if (!py_result)"


    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc___mul___TypeError:
        const char* overloads[] = {"PySide.QtCore.QLine", "PySide.QtCore.QLineF", "PySide.QtGui.QMatrix", "PySide.QtCore.QPoint", "PySide.QtCore.QPointF", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QMatrix.__mul__", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc___imul__(PyObject* self, PyObject* arg)
{
    bool isReverse = SbkQMatrix_Check(arg) && !SbkQMatrix_Check(self);
    if (isReverse)
        std::swap(self, arg);

    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (!isReverse && Shiboken::Converter<QMatrix & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QMatrix* cpp_arg0 = Shiboken::Converter<QMatrix* >::toCpp(arg);
        // operator*=(QMatrix)
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        (*cppSelf) *= (*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQMatrixFunc___imul___TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_INCREF(self);
    return self;

    SbkQMatrixFunc___imul___TypeError:
        const char* overloads[] = {"PySide.QtGui.QMatrix", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QMatrix.__imul__", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc___lshift__(PyObject* self, PyObject* arg)
{
    bool isReverse = SbkQMatrix_Check(arg) && !SbkQMatrix_Check(self);
    if (isReverse)
        std::swap(self, arg);

    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (!isReverse
        && SbkBaseWrapper_Check(arg)
        && !PyObject_TypeCheck(arg, self->ob_type)
        && PyObject_HasAttrString(arg, const_cast<char*>("__rlshift__"))) {
        PyObject* revOpMethod = PyObject_GetAttrString(arg, const_cast<char*>("__rlshift__"));
        if (revOpMethod && PyCallable_Check(revOpMethod)) {
            py_result = PyObject_CallFunction(revOpMethod, const_cast<char*>("O"), self);
            if (PyErr_Occurred() && (PyErr_ExceptionMatches(PyExc_NotImplementedError) || PyErr_ExceptionMatches(PyExc_AttributeError))) {
                PyErr_Clear();
                Py_XDECREF(py_result);
                py_result = 0;
            }
        }
        Py_XDECREF(revOpMethod);

    }
    // Do not enter here if other object has implemented a reverse operator.
    if (!py_result) {

    if (isReverse && Shiboken::Converter<QDataStream & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QDataStream* cpp_arg0 = Shiboken::Converter<QDataStream* >::toCpp(arg);
        // operator<<(QDataStream&,QMatrix) [reverse operator]
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QDataStream & cpp_result = (*cpp_arg0) << (*cppSelf);
        py_result = Shiboken::Converter<QDataStream & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, py_result);
    } else goto SbkQMatrixFunc___lshift___TypeError;

    } // End of "if (!py_result)"


    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc___lshift___TypeError:
        const char* overloads[] = {"PySide.QtCore.QDataStream", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QMatrix.__lshift__", overloads);
        return 0;
}

static PyObject* SbkQMatrixFunc___rshift__(PyObject* self, PyObject* arg)
{
    bool isReverse = SbkQMatrix_Check(arg) && !SbkQMatrix_Check(self);
    if (isReverse)
        std::swap(self, arg);

    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (!isReverse
        && SbkBaseWrapper_Check(arg)
        && !PyObject_TypeCheck(arg, self->ob_type)
        && PyObject_HasAttrString(arg, const_cast<char*>("__rrshift__"))) {
        PyObject* revOpMethod = PyObject_GetAttrString(arg, const_cast<char*>("__rrshift__"));
        if (revOpMethod && PyCallable_Check(revOpMethod)) {
            py_result = PyObject_CallFunction(revOpMethod, const_cast<char*>("O"), self);
            if (PyErr_Occurred() && (PyErr_ExceptionMatches(PyExc_NotImplementedError) || PyErr_ExceptionMatches(PyExc_AttributeError))) {
                PyErr_Clear();
                Py_XDECREF(py_result);
                py_result = 0;
            }
        }
        Py_XDECREF(revOpMethod);

    }
    // Do not enter here if other object has implemented a reverse operator.
    if (!py_result) {

    if (isReverse && Shiboken::Converter<QDataStream & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QDataStream* cpp_arg0 = Shiboken::Converter<QDataStream* >::toCpp(arg);
        // operator>>(QDataStream&,QMatrix&) [reverse operator]
        QMatrix* cppSelf = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);
        QDataStream & cpp_result = (*cpp_arg0) >> (*cppSelf);
        py_result = Shiboken::Converter<QDataStream & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, py_result);
    } else goto SbkQMatrixFunc___rshift___TypeError;

    } // End of "if (!py_result)"


    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQMatrixFunc___rshift___TypeError:
        const char* overloads[] = {"PySide.QtCore.QDataStream", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QMatrix.__rshift__", overloads);
        return 0;
}

// type has number operators
static PyNumberMethods SbkQMatrix_as_number = {
    /*nb_add*/                  (binaryfunc)0,
    /*nb_subtract*/             (binaryfunc)0,
    /*nb_multiply*/             (binaryfunc)SbkQMatrixFunc___mul__,
    /*nb_divide*/               (binaryfunc)0,
    /*nb_remainder*/            (binaryfunc)0,
    /*nb_divmod*/               0,
    /*nb_power*/                0,
    /*nb_negative*/             (unaryfunc)0,
    /*nb_positive*/             (unaryfunc)0,
    /*nb_absolute*/             0,
    /*nb_nonzero*/              0,
    /*nb_invert*/               (unaryfunc)0,
    /*nb_lshift*/               (binaryfunc)SbkQMatrixFunc___lshift__,
    /*nb_rshift*/               (binaryfunc)SbkQMatrixFunc___rshift__,
    /*nb_and*/                  (binaryfunc)0,
    /*nb_xor*/                  (binaryfunc)0,
    /*nb_or*/                   (binaryfunc)0,
    /*nb_coerce*/               0,
    /*nb_int*/                  0,
    /*nb_long*/                 0,
    /*nb_float*/                0,
    /*nb_oct*/                  0,
    /*nb_hex*/                  0,
    /*nb_inplace_add*/          (binaryfunc)0,
    /*nb_inplace_subtract*/     (binaryfunc)0,
    /*nb_inplace_multiply*/     (binaryfunc)SbkQMatrixFunc___imul__,
    /*nb_inplace_divide*/       (binaryfunc)0,
    /*nb_inplace_remainder*/    (binaryfunc)0,
    /*nb_inplace_power*/        0,
    /*nb_inplace_lshift*/       (binaryfunc)0,
    /*nb_inplace_rshift*/       (binaryfunc)0,
    /*nb_inplace_and*/          (binaryfunc)0,
    /*nb_inplace_xor*/          (binaryfunc)0,
    /*nb_inplace_or*/           (binaryfunc)0,
    /*nb_floor_divide*/         0,
    /*nb_true_divide*/          0,
    /*nb_inplace_floor_divide*/ 0,
    /*nb_inplace_true_divide*/  0,
    /*nb_index*/                0
};

// Rich comparison
static PyObject* SbkQMatrix_richcompare(PyObject* self, PyObject* other, int op)
{
    bool result = false;
    QMatrix& cpp_self = *Shiboken::Converter<QMatrix* >::toCpp((PyObject*)self);

    switch (op) {
        case Py_NE:
            if (Shiboken::Converter<QMatrix & >::isConvertible(other)) {
                // operator!=(const QMatrix & arg__1) const
                QMatrix* cpp_other = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QMatrix  >::toCpp(other) : (cpp_self != (*cpp_other));
            } else goto SbkQMatrix_RichComparison_TypeError;

            break;
        case Py_EQ:
            if (Shiboken::Converter<QMatrix & >::isConvertible(other)) {
                // operator==(const QMatrix & arg__1) const
                QMatrix* cpp_other = Shiboken::Converter<QMatrix* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QMatrix  >::toCpp(other) : (cpp_self == (*cpp_other));
            } else goto SbkQMatrix_RichComparison_TypeError;

            break;
        default:
            PyErr_SetString(PyExc_NotImplementedError, "operator not implemented.");
            return 0;
    }

    if (result)
        Py_RETURN_TRUE;
    SbkQMatrix_RichComparison_TypeError:
    Py_RETURN_FALSE;

}

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQMatrix_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtGui.QMatrix",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        &SbkQMatrix_as_number,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      SbkQMatrix_richcompare,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQMatrix_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQMatrix_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QMatrix >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};


PyAPI_FUNC(void) init_QMatrix(PyObject* module)
{
    SbkPySide_QtGuiTypes[SBK_QMATRIX_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQMatrix_Type);

    if (PyType_Ready((PyTypeObject*)&SbkQMatrix_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQMatrix_Type));
    PyModule_AddObject(module, "QMatrix",
        ((PyObject*)&SbkQMatrix_Type));

    Shiboken::TypeResolver::createValueTypeResolver<QMatrix >("QMatrix");
    Shiboken::TypeResolver::createValueTypeResolver<QMatrix >(typeid(QMatrix).name());
}


} // extern "C"

