/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtgui_python.h"

#include "qimageiohandler_wrapper.h"

// Extra includes
#include <QRect>
#include <qbytearray.h>
#include <qimage.h>
#include <qimageiohandler.h>
#include <qiodevice.h>
#include <qrect.h>
#include <qvariant.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QImageIOHandlerWrapper::QImageIOHandlerWrapper() : QImageIOHandler() {
    // ... middle
}

bool QImageIOHandlerWrapper::canRead() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "canRead"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QImageIOHandler.canRead()' not implemented.");
        return bool(0);
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.canRead", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

int QImageIOHandlerWrapper::currentImageNumber() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "currentImageNumber"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::currentImageNumber();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.currentImageNumber", SbkType<int >()->tp_name, py_result->ob_type->tp_name);
        return int(0);
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

QRect QImageIOHandlerWrapper::currentImageRect() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "currentImageRect"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::currentImageRect();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QRect();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QRect >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.currentImageRect", SbkType<QRect >()->tp_name, py_result->ob_type->tp_name);
        return QRect();
    }
    QRect cpp_result(Shiboken::Converter<QRect >::toCpp(py_result));
    return cpp_result;
}

int QImageIOHandlerWrapper::imageCount() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "imageCount"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::imageCount();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.imageCount", SbkType<int >()->tp_name, py_result->ob_type->tp_name);
        return int(0);
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

bool QImageIOHandlerWrapper::jumpToImage(int imageNumber)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "jumpToImage"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::jumpToImage(imageNumber);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(i)",
        imageNumber
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.jumpToImage", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QImageIOHandlerWrapper::jumpToNextImage()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "jumpToNextImage"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::jumpToNextImage();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.jumpToNextImage", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

int QImageIOHandlerWrapper::loopCount() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "loopCount"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::loopCount();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.loopCount", SbkType<int >()->tp_name, py_result->ob_type->tp_name);
        return int(0);
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

int QImageIOHandlerWrapper::nextImageDelay() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "nextImageDelay"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::nextImageDelay();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.nextImageDelay", SbkType<int >()->tp_name, py_result->ob_type->tp_name);
        return int(0);
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

QVariant QImageIOHandlerWrapper::option(QImageIOHandler::ImageOption option) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "option"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::option(option);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QImageIOHandler::ImageOption >::toPython(option)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QVariant();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QVariant >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.option", SbkType<QVariant >()->tp_name, py_result->ob_type->tp_name);
        return QVariant();
    }
    QVariant cpp_result(Shiboken::Converter<QVariant >::toCpp(py_result));
    return cpp_result;
}

bool QImageIOHandlerWrapper::read(QImage * image)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "read"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QImageIOHandler.read()' not implemented.");
        return bool(0);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QImage * >::toPython(image)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.read", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

void QImageIOHandlerWrapper::setOption(QImageIOHandler::ImageOption option, const QVariant & value)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "setOption"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::setOption(option, value);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<QImageIOHandler::ImageOption >::toPython(option),
        Shiboken::Converter<QVariant & >::toPython(value)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

bool QImageIOHandlerWrapper::supportsOption(QImageIOHandler::ImageOption option) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "supportsOption"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::supportsOption(option);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QImageIOHandler::ImageOption >::toPython(option)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.supportsOption", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QImageIOHandlerWrapper::write(const QImage & image)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "write"));
    if (py_override.isNull()) {
        return this->QImageIOHandler::write(image);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QImage & >::toPython(image)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QImageIOHandler.write", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

QImageIOHandlerWrapper::~QImageIOHandlerWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQImageIOHandler_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QImageIOHandlerWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);
    SbkBaseWrapperType* type = reinterpret_cast<SbkBaseWrapperType*>(self->ob_type);
    SbkBaseWrapperType* myType = reinterpret_cast<SbkBaseWrapperType*>(SbkPySide_QtGuiTypes[SBK_QIMAGEIOHANDLER_IDX]);
    if (type == myType) {
        PyErr_SetString(PyExc_NotImplementedError,
            "'QImageIOHandler' represents a C++ abstract class and cannot be instantiated");
        return -1;
    }

    // QImageIOHandler()
    cptr = new QImageIOHandlerWrapper();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QImageIOHandler >(), cptr)) {
        delete cptr;
        return -1;
    }
    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;
}

static PyObject* SbkQImageIOHandlerFunc_canRead(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // canRead()const
    if (SbkBaseWrapper_containsCppWrapper(self)) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QImageIOHandler.canRead()' not implemented.");
        return 0;
    }
    QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->canRead();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageIOHandlerFunc_currentImageNumber(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // currentImageNumber()const
    QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImageIOHandler::currentImageNumber();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageIOHandlerFunc_currentImageRect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // currentImageRect()const
    QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
    QRect cpp_result = cppSelf->QImageIOHandler::currentImageRect();
    py_result = Shiboken::Converter<QRect >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageIOHandlerFunc_device(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // device()const
    QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
    QIODevice * cpp_result = cppSelf->QImageIOHandler::device();
    py_result = Shiboken::Converter<QIODevice * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageIOHandlerFunc_format(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // format()const
    QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
    QByteArray cpp_result = cppSelf->QImageIOHandler::format();
    py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageIOHandlerFunc_imageCount(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // imageCount()const
    QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImageIOHandler::imageCount();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageIOHandlerFunc_jumpToImage(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // jumpToImage(int)
        QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QImageIOHandler::jumpToImage(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageIOHandlerFunc_jumpToImage_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageIOHandlerFunc_jumpToImage_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImageIOHandler.jumpToImage", overloads);
        return 0;
}

static PyObject* SbkQImageIOHandlerFunc_jumpToNextImage(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // jumpToNextImage()
    QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QImageIOHandler::jumpToNextImage();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageIOHandlerFunc_loopCount(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // loopCount()const
    QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImageIOHandler::loopCount();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageIOHandlerFunc_nextImageDelay(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // nextImageDelay()const
    QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QImageIOHandler::nextImageDelay();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQImageIOHandlerFunc_option(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QImageIOHandler::ImageOption >::isConvertible(arg)) {
        QImageIOHandler::ImageOption cpp_arg0 = Shiboken::Converter<QImageIOHandler::ImageOption >::toCpp(arg);
        // option(QImageIOHandler::ImageOption)const
        QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
        QVariant cpp_result = cppSelf->QImageIOHandler::option(cpp_arg0);
        py_result = Shiboken::Converter<QVariant >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageIOHandlerFunc_option_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageIOHandlerFunc_option_TypeError:
        const char* overloads[] = {"PySide.QtGui.QImageIOHandler.ImageOption", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImageIOHandler.option", overloads);
        return 0;
}

static PyObject* SbkQImageIOHandlerFunc_read(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QImage * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QImage* cpp_arg0 = Shiboken::Converter<QImage* >::toCpp(arg);
        // read(QImage*)
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QImageIOHandler.read()' not implemented.");
            return 0;
        }
        QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->read(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageIOHandlerFunc_read_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageIOHandlerFunc_read_TypeError:
        const char* overloads[] = {"PySide.QtGui.QImage", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImageIOHandler.read", overloads);
        return 0;
}

static PyObject* SbkQImageIOHandlerFunc_setDevice(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QIODevice * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QIODevice* cpp_arg0 = Shiboken::Converter<QIODevice* >::toCpp(arg);
        // setDevice(QIODevice*)
        QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
        cppSelf->QImageIOHandler::setDevice(cpp_arg0);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, arg);
    } else goto SbkQImageIOHandlerFunc_setDevice_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageIOHandlerFunc_setDevice_TypeError:
        const char* overloads[] = {"PySide.QtCore.QIODevice", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImageIOHandler.setDevice", overloads);
        return 0;
}

static PyObject* SbkQImageIOHandlerFunc_setFormat(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QByteArray & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(arg);
        if (!SbkQByteArray_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // setFormat(QByteArray)
        QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
        cppSelf->QImageIOHandler::setFormat(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageIOHandlerFunc_setFormat_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageIOHandlerFunc_setFormat_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImageIOHandler.setFormat", overloads);
        return 0;
}

static PyObject* SbkQImageIOHandlerFunc_setOption(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setOption", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QImageIOHandler::ImageOption >::isConvertible(pyargs[0]) && Shiboken::Converter<QVariant & >::isConvertible(pyargs[1])) {
        QImageIOHandler::ImageOption cpp_arg0 = Shiboken::Converter<QImageIOHandler::ImageOption >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QVariant > cpp_arg1_auto_ptr;
        QVariant* cpp_arg1 = Shiboken::Converter<QVariant* >::toCpp(pyargs[1]);
        if (!SbkQVariant_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QVariant >(cpp_arg1);
        // setOption(QImageIOHandler::ImageOption,QVariant)
        QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
        cppSelf->QImageIOHandler::setOption(cpp_arg0, *cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageIOHandlerFunc_setOption_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQImageIOHandlerFunc_setOption_TypeError:
        const char* overloads[] = {"PySide.QtGui.QImageIOHandler.ImageOption, PySide.QtCore.QVariant", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QImageIOHandler.setOption", overloads);
        return 0;
}

static PyObject* SbkQImageIOHandlerFunc_supportsOption(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QImageIOHandler::ImageOption >::isConvertible(arg)) {
        QImageIOHandler::ImageOption cpp_arg0 = Shiboken::Converter<QImageIOHandler::ImageOption >::toCpp(arg);
        // supportsOption(QImageIOHandler::ImageOption)const
        QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QImageIOHandler::supportsOption(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageIOHandlerFunc_supportsOption_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageIOHandlerFunc_supportsOption_TypeError:
        const char* overloads[] = {"PySide.QtGui.QImageIOHandler.ImageOption", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImageIOHandler.supportsOption", overloads);
        return 0;
}

static PyObject* SbkQImageIOHandlerFunc_write(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QImage & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QImage* cpp_arg0 = Shiboken::Converter<QImage* >::toCpp(arg);
        // write(QImage)
        QImageIOHandler* cppSelf = Shiboken::Converter<QImageIOHandler* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QImageIOHandler::write(*cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQImageIOHandlerFunc_write_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQImageIOHandlerFunc_write_TypeError:
        const char* overloads[] = {"PySide.QtGui.QImage", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QImageIOHandler.write", overloads);
        return 0;
}

static PyMethodDef SbkQImageIOHandler_methods[] = {
    {"canRead", (PyCFunction)SbkQImageIOHandlerFunc_canRead, METH_NOARGS},
    {"currentImageNumber", (PyCFunction)SbkQImageIOHandlerFunc_currentImageNumber, METH_NOARGS},
    {"currentImageRect", (PyCFunction)SbkQImageIOHandlerFunc_currentImageRect, METH_NOARGS},
    {"device", (PyCFunction)SbkQImageIOHandlerFunc_device, METH_NOARGS},
    {"format", (PyCFunction)SbkQImageIOHandlerFunc_format, METH_NOARGS},
    {"imageCount", (PyCFunction)SbkQImageIOHandlerFunc_imageCount, METH_NOARGS},
    {"jumpToImage", (PyCFunction)SbkQImageIOHandlerFunc_jumpToImage, METH_O},
    {"jumpToNextImage", (PyCFunction)SbkQImageIOHandlerFunc_jumpToNextImage, METH_NOARGS},
    {"loopCount", (PyCFunction)SbkQImageIOHandlerFunc_loopCount, METH_NOARGS},
    {"nextImageDelay", (PyCFunction)SbkQImageIOHandlerFunc_nextImageDelay, METH_NOARGS},
    {"option", (PyCFunction)SbkQImageIOHandlerFunc_option, METH_O},
    {"read", (PyCFunction)SbkQImageIOHandlerFunc_read, METH_O},
    {"setDevice", (PyCFunction)SbkQImageIOHandlerFunc_setDevice, METH_O},
    {"setFormat", (PyCFunction)SbkQImageIOHandlerFunc_setFormat, METH_O},
    {"setOption", (PyCFunction)SbkQImageIOHandlerFunc_setOption, METH_VARARGS},
    {"supportsOption", (PyCFunction)SbkQImageIOHandlerFunc_supportsOption, METH_O},
    {"write", (PyCFunction)SbkQImageIOHandlerFunc_write, METH_O},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQImageIOHandler_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtGui.QImageIOHandler",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQImageIOHandler_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQImageIOHandler_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QImageIOHandler >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static SbkBaseWrapperType* SbkQImageIOHandler_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkQImageIOHandler_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<QImageIOHandler*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}

static PyGetSetDef SbkPySide_QtGui_QImageIOHandler_ImageOption_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QImageIOHandler_ImageOption_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QImageIOHandler_ImageOption_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "ImageOption",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QImageIOHandler_ImageOption_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QImageIOHandler_ImageOption_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtGui_QImageIOHandler_ImageOption_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}


PyAPI_FUNC(void) init_QImageIOHandler(PyObject* module)
{
    SbkPySide_QtGuiTypes[SBK_QIMAGEIOHANDLER_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQImageIOHandler_Type);

    // Fill type discovery information
    SbkQImageIOHandler_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkQImageIOHandler_Type.type_discovery->addTypeDiscoveryFunction(&SbkQImageIOHandler_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkQImageIOHandler_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQImageIOHandler_Type));
    PyModule_AddObject(module, "QImageIOHandler",
        ((PyObject*)&SbkQImageIOHandler_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: ImageOption
    SbkPySide_QtGuiTypes[SBK_QIMAGEIOHANDLER_IMAGEOPTION_IDX] = &SbkPySide_QtGui_QImageIOHandler_ImageOption_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type);
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
            "ImageOption",((PyObject*)&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::Size, "Size");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "Size", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "Size", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::ClipRect, "ClipRect");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "ClipRect", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "ClipRect", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::Description, "Description");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "Description", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "Description", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::ScaledClipRect, "ScaledClipRect");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "ScaledClipRect", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "ScaledClipRect", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::ScaledSize, "ScaledSize");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "ScaledSize", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "ScaledSize", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::CompressionRatio, "CompressionRatio");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "CompressionRatio", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "CompressionRatio", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::Gamma, "Gamma");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "Gamma", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "Gamma", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::Quality, "Quality");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "Quality", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "Quality", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::Name, "Name");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "Name", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "Name", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::SubType, "SubType");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "SubType", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "SubType", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::IncrementalReading, "IncrementalReading");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "IncrementalReading", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "IncrementalReading", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::Endianness, "Endianness");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "Endianness", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "Endianness", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::Animation, "Animation");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "Animation", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "Animation", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::BackgroundColor, "BackgroundColor");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "BackgroundColor", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "BackgroundColor", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QImageIOHandler_ImageOption_Type,
        (long) QImageIOHandler::ImageFormat, "ImageFormat");
    PyDict_SetItemString(SbkQImageIOHandler_Type.super.ht_type.tp_dict,
        "ImageFormat", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QImageIOHandler_ImageOption_Type.tp_dict,
        "ImageFormat", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QImageIOHandler::ImageOption");

    Shiboken::TypeResolver::createObjectTypeResolver<QImageIOHandler >("QImageIOHandler*");
    Shiboken::TypeResolver::createObjectTypeResolver<QImageIOHandler >(typeid(QImageIOHandler).name());
}


} // extern "C"

