/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtcore_python.h"

#include "qtextcodec_wrapper.h"
#include "qtextcodec_converterstate_wrapper.h"

// Extra includes
#include <QList>
#include <QTextCodec>
#include <qbytearray.h>
#include <qchar.h>
#include <qstring.h>
#include <qtextcodec.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QTextCodecWrapper::QTextCodecWrapper() : QTextCodec() {
    // ... middle
}

QList<QByteArray > QTextCodecWrapper::aliases() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "aliases"));
    if (py_override.isNull()) {
        return this->QTextCodec::aliases();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QList<QByteArray >();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QList<QByteArray > >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QTextCodec.aliases", "list", py_result->ob_type->tp_name);
        return QList<QByteArray >();
    }
    QList<QByteArray > cpp_result(Shiboken::Converter<QList<QByteArray > >::toCpp(py_result));
    return cpp_result;
}

QByteArray QTextCodecWrapper::convertFromUnicode(const QChar * in, int length, QTextCodec::ConverterState * state) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "convertFromUnicode"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QTextCodec.convertFromUnicode()' not implemented.");
        return QByteArray();
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SiS)",
        Shiboken::Converter<QChar * >::toPython(in),
        length,
        Shiboken::Converter<QTextCodec::ConverterState * >::toPython(state)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QByteArray();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QByteArray >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QTextCodec.convertFromUnicode", SbkType<QByteArray >()->tp_name, py_result->ob_type->tp_name);
        return QByteArray();
    }
    QByteArray cpp_result(Shiboken::Converter<QByteArray >::toCpp(py_result));
    return cpp_result;
}

QString QTextCodecWrapper::convertToUnicode(const char * in, int length, QTextCodec::ConverterState * state) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "convertToUnicode"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QTextCodec.convertToUnicode()' not implemented.");
        return QString();
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(ziS)",
        in,
        length,
        Shiboken::Converter<QTextCodec::ConverterState * >::toPython(state)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QString();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QString >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QTextCodec.convertToUnicode", SbkType<QString >()->tp_name, py_result->ob_type->tp_name);
        return QString();
    }
    QString cpp_result(Shiboken::Converter<QString >::toCpp(py_result));
    return cpp_result;
}

int QTextCodecWrapper::mibEnum() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mibEnum"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QTextCodec.mibEnum()' not implemented.");
        return int(0);
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QTextCodec.mibEnum", SbkType<int >()->tp_name, py_result->ob_type->tp_name);
        return int(0);
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

QByteArray QTextCodecWrapper::name() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "name"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QTextCodec.name()' not implemented.");
        return QByteArray();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QByteArray();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QByteArray >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QTextCodec.name", SbkType<QByteArray >()->tp_name, py_result->ob_type->tp_name);
        return QByteArray();
    }
    QByteArray cpp_result(Shiboken::Converter<QByteArray >::toCpp(py_result));
    return cpp_result;
}

QTextCodecWrapper::~QTextCodecWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQTextCodec_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QTextCodecWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);
    SbkBaseWrapperType* type = reinterpret_cast<SbkBaseWrapperType*>(self->ob_type);
    SbkBaseWrapperType* myType = reinterpret_cast<SbkBaseWrapperType*>(SbkPySide_QtCoreTypes[SBK_QTEXTCODEC_IDX]);
    if (type == myType) {
        PyErr_SetString(PyExc_NotImplementedError,
            "'QTextCodec' represents a C++ abstract class and cannot be instantiated");
        return -1;
    }

    // QTextCodec()
    cptr = new QTextCodecWrapper();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QTextCodec >(), cptr)) {
        delete cptr;
        return -1;
    }
    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;
}

static PyObject* SbkQTextCodecFunc_aliases(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // aliases()const
    QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
    QList<QByteArray > cpp_result = cppSelf->QTextCodec::aliases();
    py_result = Shiboken::Converter<QList<QByteArray > >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_availableCodecs(PyObject* self)
{
    PyObject* py_result = 0;

    // availableCodecs()
    QList<QByteArray > cpp_result = QTextCodec::availableCodecs();
    py_result = Shiboken::Converter<QList<QByteArray > >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_availableMibs(PyObject* self)
{
    PyObject* py_result = 0;

    // availableMibs()
    QList<int > cpp_result = QTextCodec::availableMibs();
    py_result = Shiboken::Converter<QList<int > >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_canEncode(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QChar >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(arg);
        if (!SbkQChar_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        // canEncode(QChar)const
        QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QTextCodec::canEncode(*cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // canEncode(QString)const
        QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QTextCodec::canEncode(*cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQTextCodecFunc_canEncode_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQTextCodecFunc_canEncode_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar", "PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QTextCodec.canEncode", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_codecForCStrings(PyObject* self)
{
    PyObject* py_result = 0;

    // codecForCStrings()
    QTextCodec * cpp_result = QTextCodec::codecForCStrings();
    py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_codecForHtml(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "codecForHtml", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QByteArray & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(pyargs[0]);
        if (!SbkQByteArray_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        if (numArgs == 1) {
            // codecForHtml(QByteArray)
            QTextCodec * cpp_result = QTextCodec::codecForHtml(*cpp_arg0);
            py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (numArgs == 2 && Shiboken::Converter<QTextCodec * >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QTextCodec* cpp_arg1 = Shiboken::Converter<QTextCodec* >::toCpp(pyargs[1]);
            // codecForHtml(QByteArray,QTextCodec*)
            QTextCodec * cpp_result = QTextCodec::codecForHtml(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQTextCodecFunc_codecForHtml_TypeError;
    } else goto SbkQTextCodecFunc_codecForHtml_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQTextCodecFunc_codecForHtml_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray", "PySide.QtCore.QByteArray, PySide.QtCore.QTextCodec", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QTextCodec.codecForHtml", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_codecForLocale(PyObject* self)
{
    PyObject* py_result = 0;

    // codecForLocale()
    QTextCodec * cpp_result = QTextCodec::codecForLocale();
    py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_codecForMib(PyObject* self, PyObject* arg)
{
    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // codecForMib(int)
        QTextCodec * cpp_result = QTextCodec::codecForMib(cpp_arg0);
        py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQTextCodecFunc_codecForMib_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQTextCodecFunc_codecForMib_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QTextCodec.codecForMib", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_codecForName(PyObject* self, PyObject* arg)
{
    PyObject* py_result = 0;

    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // codecForName(const char*)
        QTextCodec * cpp_result = QTextCodec::codecForName(cpp_arg0);
        py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QByteArray & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(arg);
        if (!SbkQByteArray_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // codecForName(QByteArray)
        QTextCodec * cpp_result = QTextCodec::codecForName(*cpp_arg0);
        py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQTextCodecFunc_codecForName_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQTextCodecFunc_codecForName_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray", "str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QTextCodec.codecForName", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_codecForTr(PyObject* self)
{
    PyObject* py_result = 0;

    // codecForTr()
    QTextCodec * cpp_result = QTextCodec::codecForTr();
    py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_codecForUtfText(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "codecForUtfText", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QByteArray & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(pyargs[0]);
        if (!SbkQByteArray_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        if (numArgs == 1) {
            // codecForUtfText(QByteArray)
            QTextCodec * cpp_result = QTextCodec::codecForUtfText(*cpp_arg0);
            py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (numArgs == 2 && Shiboken::Converter<QTextCodec * >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QTextCodec* cpp_arg1 = Shiboken::Converter<QTextCodec* >::toCpp(pyargs[1]);
            // codecForUtfText(QByteArray,QTextCodec*)
            QTextCodec * cpp_result = QTextCodec::codecForUtfText(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QTextCodec * >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQTextCodecFunc_codecForUtfText_TypeError;
    } else goto SbkQTextCodecFunc_codecForUtfText_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQTextCodecFunc_codecForUtfText_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray", "PySide.QtCore.QByteArray, PySide.QtCore.QTextCodec", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QTextCodec.codecForUtfText", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_convertFromUnicode(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "convertFromUnicode", 3, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && Shiboken::Converter<QChar * >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1]) && Shiboken::Converter<QTextCodec::ConverterState * >::isConvertible(pyargs[2])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        if (Shiboken::cppObjectIsInvalid(pyargs[2]))
            return 0;
        QTextCodec::ConverterState* cpp_arg2 = Shiboken::Converter<QTextCodec::ConverterState* >::toCpp(pyargs[2]);
        // convertFromUnicode(const QChar*,int,QTextCodec::ConverterState*)const
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QTextCodec.convertFromUnicode()' not implemented.");
            return 0;
        }
        QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
        QByteArray cpp_result = cppSelf->convertFromUnicode(cpp_arg0, cpp_arg1, cpp_arg2);
        py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQTextCodecFunc_convertFromUnicode_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQTextCodecFunc_convertFromUnicode_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, int, PySide.QtCore.QTextCodec::ConverterState", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QTextCodec.convertFromUnicode", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_convertToUnicode(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "convertToUnicode", 3, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && Shiboken::Converter<const char * >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1]) && Shiboken::Converter<QTextCodec::ConverterState * >::isConvertible(pyargs[2])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        if (Shiboken::cppObjectIsInvalid(pyargs[2]))
            return 0;
        QTextCodec::ConverterState* cpp_arg2 = Shiboken::Converter<QTextCodec::ConverterState* >::toCpp(pyargs[2]);
        // convertToUnicode(const char*,int,QTextCodec::ConverterState*)const
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QTextCodec.convertToUnicode()' not implemented.");
            return 0;
        }
        QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->convertToUnicode(cpp_arg0, cpp_arg1, cpp_arg2);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQTextCodecFunc_convertToUnicode_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQTextCodecFunc_convertToUnicode_TypeError:
        const char* overloads[] = {"str, int, PySide.QtCore.QTextCodec::ConverterState", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QTextCodec.convertToUnicode", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_fromUnicode(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "fromUnicode", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<QChar * >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        if (numArgs == 2) {
            // fromUnicode(const QChar*,int,QTextCodec::ConverterState*)const
            QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
            QByteArray cpp_result = cppSelf->QTextCodec::fromUnicode(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QTextCodec::ConverterState * >::isConvertible(pyargs[2])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                return 0;
            QTextCodec::ConverterState* cpp_arg2 = Shiboken::Converter<QTextCodec::ConverterState* >::toCpp(pyargs[2]);
            // fromUnicode(const QChar*,int,QTextCodec::ConverterState*)const
            QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
            QByteArray cpp_result = cppSelf->QTextCodec::fromUnicode(cpp_arg0, cpp_arg1, cpp_arg2);
            py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQTextCodecFunc_fromUnicode_TypeError;
    } else if (numArgs == 1 && Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // fromUnicode(QString)const
        QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
        QByteArray cpp_result = cppSelf->QTextCodec::fromUnicode(*cpp_arg0);
        py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQTextCodecFunc_fromUnicode_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQTextCodecFunc_fromUnicode_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, int, PySide.QtCore.QTextCodec::ConverterState = None", "PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QTextCodec.fromUnicode", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_makeDecoder(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // makeDecoder()const
    QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
    QTextDecoder * cpp_result = cppSelf->QTextCodec::makeDecoder();
    py_result = Shiboken::Converter<QTextDecoder * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_makeEncoder(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // makeEncoder()const
    QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
    QTextEncoder * cpp_result = cppSelf->QTextCodec::makeEncoder();
    py_result = Shiboken::Converter<QTextEncoder * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_mibEnum(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // mibEnum()const
    if (SbkBaseWrapper_containsCppWrapper(self)) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QTextCodec.mibEnum()' not implemented.");
        return 0;
    }
    QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->mibEnum();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_name(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // name()const
    if (SbkBaseWrapper_containsCppWrapper(self)) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QTextCodec.name()' not implemented.");
        return 0;
    }
    QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
    QByteArray cpp_result = cppSelf->name();
    py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQTextCodecFunc_setCodecForCStrings(PyObject* self, PyObject* arg)
{

    if (Shiboken::Converter<QTextCodec * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QTextCodec* cpp_arg0 = Shiboken::Converter<QTextCodec* >::toCpp(arg);
        // setCodecForCStrings(QTextCodec*)
        QTextCodec::setCodecForCStrings(cpp_arg0);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, arg);
    } else goto SbkQTextCodecFunc_setCodecForCStrings_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQTextCodecFunc_setCodecForCStrings_TypeError:
        const char* overloads[] = {"PySide.QtCore.QTextCodec", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QTextCodec.setCodecForCStrings", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_setCodecForLocale(PyObject* self, PyObject* arg)
{

    if (Shiboken::Converter<QTextCodec * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QTextCodec* cpp_arg0 = Shiboken::Converter<QTextCodec* >::toCpp(arg);
        // setCodecForLocale(QTextCodec*)
        QTextCodec::setCodecForLocale(cpp_arg0);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, arg);
    } else goto SbkQTextCodecFunc_setCodecForLocale_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQTextCodecFunc_setCodecForLocale_TypeError:
        const char* overloads[] = {"PySide.QtCore.QTextCodec", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QTextCodec.setCodecForLocale", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_setCodecForTr(PyObject* self, PyObject* arg)
{

    if (Shiboken::Converter<QTextCodec * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QTextCodec* cpp_arg0 = Shiboken::Converter<QTextCodec* >::toCpp(arg);
        // setCodecForTr(QTextCodec*)
        QTextCodec::setCodecForTr(cpp_arg0);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, arg);
    } else goto SbkQTextCodecFunc_setCodecForTr_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQTextCodecFunc_setCodecForTr_TypeError:
        const char* overloads[] = {"PySide.QtCore.QTextCodec", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QTextCodec.setCodecForTr", overloads);
        return 0;
}

static PyObject* SbkQTextCodecFunc_toUnicode(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "toUnicode", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // toUnicode(const char*)const
            QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QTextCodec::toUnicode(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // toUnicode(const char*,int,QTextCodec::ConverterState*)const
                QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QTextCodec::toUnicode(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<QTextCodec::ConverterState * >::isConvertible(pyargs[2])) {
                if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                    return 0;
                QTextCodec::ConverterState* cpp_arg2 = Shiboken::Converter<QTextCodec::ConverterState* >::toCpp(pyargs[2]);
                // toUnicode(const char*,int,QTextCodec::ConverterState*)const
                QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QTextCodec::toUnicode(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQTextCodecFunc_toUnicode_TypeError;
        } else goto SbkQTextCodecFunc_toUnicode_TypeError;
    } else if (numArgs == 1 && Shiboken::Converter<QByteArray & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(pyargs[0]);
        if (!SbkQByteArray_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // toUnicode(QByteArray)const
        QTextCodec* cppSelf = Shiboken::Converter<QTextCodec* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->QTextCodec::toUnicode(*cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQTextCodecFunc_toUnicode_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQTextCodecFunc_toUnicode_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray", "str", "str, int, PySide.QtCore.QTextCodec::ConverterState = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QTextCodec.toUnicode", overloads);
        return 0;
}

static PyMethodDef SbkQTextCodec_methods[] = {
    {"aliases", (PyCFunction)SbkQTextCodecFunc_aliases, METH_NOARGS},
    {"availableCodecs", (PyCFunction)SbkQTextCodecFunc_availableCodecs, METH_NOARGS|METH_STATIC},
    {"availableMibs", (PyCFunction)SbkQTextCodecFunc_availableMibs, METH_NOARGS|METH_STATIC},
    {"canEncode", (PyCFunction)SbkQTextCodecFunc_canEncode, METH_O},
    {"codecForCStrings", (PyCFunction)SbkQTextCodecFunc_codecForCStrings, METH_NOARGS|METH_STATIC},
    {"codecForHtml", (PyCFunction)SbkQTextCodecFunc_codecForHtml, METH_VARARGS|METH_STATIC},
    {"codecForLocale", (PyCFunction)SbkQTextCodecFunc_codecForLocale, METH_NOARGS|METH_STATIC},
    {"codecForMib", (PyCFunction)SbkQTextCodecFunc_codecForMib, METH_O|METH_STATIC},
    {"codecForName", (PyCFunction)SbkQTextCodecFunc_codecForName, METH_O|METH_STATIC},
    {"codecForTr", (PyCFunction)SbkQTextCodecFunc_codecForTr, METH_NOARGS|METH_STATIC},
    {"codecForUtfText", (PyCFunction)SbkQTextCodecFunc_codecForUtfText, METH_VARARGS|METH_STATIC},
    {"convertFromUnicode", (PyCFunction)SbkQTextCodecFunc_convertFromUnicode, METH_VARARGS},
    {"convertToUnicode", (PyCFunction)SbkQTextCodecFunc_convertToUnicode, METH_VARARGS},
    {"fromUnicode", (PyCFunction)SbkQTextCodecFunc_fromUnicode, METH_VARARGS},
    {"makeDecoder", (PyCFunction)SbkQTextCodecFunc_makeDecoder, METH_NOARGS},
    {"makeEncoder", (PyCFunction)SbkQTextCodecFunc_makeEncoder, METH_NOARGS},
    {"mibEnum", (PyCFunction)SbkQTextCodecFunc_mibEnum, METH_NOARGS},
    {"name", (PyCFunction)SbkQTextCodecFunc_name, METH_NOARGS},
    {"setCodecForCStrings", (PyCFunction)SbkQTextCodecFunc_setCodecForCStrings, METH_O|METH_STATIC},
    {"setCodecForLocale", (PyCFunction)SbkQTextCodecFunc_setCodecForLocale, METH_O|METH_STATIC},
    {"setCodecForTr", (PyCFunction)SbkQTextCodecFunc_setCodecForTr, METH_O|METH_STATIC},
    {"toUnicode", (PyCFunction)SbkQTextCodecFunc_toUnicode, METH_VARARGS},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQTextCodec_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtCore.QTextCodec",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQTextCodec_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQTextCodec_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QTextCodec >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static SbkBaseWrapperType* SbkQTextCodec_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkQTextCodec_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<QTextCodec*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}

PyObject*
SbkPySide_QtCore_QTextCodec_ConversionFlag___and__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QTextCodec::ConversionFlags >::toPython(
        ((QTextCodec::ConversionFlags) ((SbkEnumObject*)self)->ob_ival)
        & Shiboken::Converter< QTextCodec::ConversionFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtCore_QTextCodec_ConversionFlag___or__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QTextCodec::ConversionFlags >::toPython(
        ((QTextCodec::ConversionFlags) ((SbkEnumObject*)self)->ob_ival)
        | Shiboken::Converter< QTextCodec::ConversionFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtCore_QTextCodec_ConversionFlag___xor__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QTextCodec::ConversionFlags >::toPython(
        ((QTextCodec::ConversionFlags) ((SbkEnumObject*)self)->ob_ival)
        ^ Shiboken::Converter< QTextCodec::ConversionFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtCore_QTextCodec_ConversionFlag___invert__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QTextCodec::ConversionFlags >::toPython(
        ~ Shiboken::Converter< QTextCodec::ConversionFlags >::toCpp(self)
    );
}


static PyNumberMethods SbkPySide_QtCore_QTextCodec_ConversionFlag_as_number = {
    /*nb_add*/                  0,
    /*nb_subtract*/             0,
    /*nb_multiply*/             0,
    /*nb_divide*/               0,
    /*nb_remainder*/            0,
    /*nb_divmod*/               0,
    /*nb_power*/                0,
    /*nb_negative*/             0,
    /*nb_positive*/             0,
    /*nb_absolute*/             0,
    /*nb_nonzero*/              0,
    /*nb_invert*/               (unaryfunc)SbkPySide_QtCore_QTextCodec_ConversionFlag___invert__,
    /*nb_lshift*/               0,
    /*nb_rshift*/               0,
    /*nb_and*/                  (binaryfunc)SbkPySide_QtCore_QTextCodec_ConversionFlag___and__,
    /*nb_xor*/                  (binaryfunc)SbkPySide_QtCore_QTextCodec_ConversionFlag___xor__,
    /*nb_or*/                   (binaryfunc)SbkPySide_QtCore_QTextCodec_ConversionFlag___or__,
    /*nb_coerce*/               0,
    /*nb_int*/                  0,
    /*nb_long*/                 0,
    /*nb_float*/                0,
    /*nb_oct*/                  0,
    /*nb_hex*/                  0,
    /*nb_inplace_add*/          0,
    /*nb_inplace_subtract*/     0,
    /*nb_inplace_multiply*/     0,
    /*nb_inplace_divide*/       0,
    /*nb_inplace_remainder*/    0,
    /*nb_inplace_power*/        0,
    /*nb_inplace_lshift*/       0,
    /*nb_inplace_rshift*/       0,
    /*nb_inplace_and*/          0,
    /*nb_inplace_xor*/          0,
    /*nb_inplace_or*/           0,
    /*nb_floor_divide*/         0,
    /*nb_true_divide*/          0,
    /*nb_inplace_floor_divide*/ 0,
    /*nb_inplace_true_divide*/  0,
    /*nb_index*/                0
};


static PyGetSetDef SbkPySide_QtCore_QTextCodec_ConversionFlag_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QTextCodec_ConversionFlag_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QTextCodec_ConversionFlag_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "ConversionFlag",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        &SbkPySide_QtCore_QTextCodec_ConversionFlag_as_number,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QTextCodec_ConversionFlag_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QTextCodec_ConversionFlag_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtCore_QTextCodec_ConversionFlag_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QTextCodec_ConversionFlags_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QTextCodec_ConversionFlags_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "ConversionFlags",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             &SbkPySide_QtCore_QTextCodec_ConversionFlag_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QTextCodec_ConversionFlags_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtCore_QTextCodec_ConversionFlags_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    if (!PyType_IsSubtype(type, &SbkPySide_QtCore_QTextCodec_ConversionFlags_Type))
        return 0;

    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}



PyAPI_FUNC(void) init_QTextCodec(PyObject* module)
{
    SbkPySide_QtCoreTypes[SBK_QTEXTCODEC_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQTextCodec_Type);

    // Fill type discovery information
    SbkQTextCodec_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkQTextCodec_Type.type_discovery->addTypeDiscoveryFunction(&SbkQTextCodec_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkQTextCodec_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQTextCodec_Type));
    PyModule_AddObject(module, "QTextCodec",
        ((PyObject*)&SbkQTextCodec_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: ConversionFlag
    SbkPySide_QtCoreTypes[SBK_QTEXTCODEC_CONVERSIONFLAG_IDX] = &SbkPySide_QtCore_QTextCodec_ConversionFlag_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QTextCodec_ConversionFlag_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QTextCodec_ConversionFlag_Type);
    PyDict_SetItemString(SbkQTextCodec_Type.super.ht_type.tp_dict,
            "ConversionFlag",((PyObject*)&SbkPySide_QtCore_QTextCodec_ConversionFlag_Type));

    // init flags class: QFlags<QTextCodec::ConversionFlag>
    SbkPySide_QtCoreTypes[SBK_QFLAGS_QTEXTCODEC_CONVERSIONFLAG__IDX] = &SbkPySide_QtCore_QTextCodec_ConversionFlags_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QTextCodec_ConversionFlags_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QTextCodec_ConversionFlags_Type);
    PyDict_SetItemString(SbkQTextCodec_Type.super.ht_type.tp_dict,
            "ConversionFlags",((PyObject*)&SbkPySide_QtCore_QTextCodec_ConversionFlags_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QTextCodec_ConversionFlag_Type,
        (long) QTextCodec::DefaultConversion, "DefaultConversion");
    PyDict_SetItemString(SbkQTextCodec_Type.super.ht_type.tp_dict,
        "DefaultConversion", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QTextCodec_ConversionFlag_Type.tp_dict,
        "DefaultConversion", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QTextCodec_ConversionFlag_Type,
        (long) QTextCodec::ConvertInvalidToNull, "ConvertInvalidToNull");
    PyDict_SetItemString(SbkQTextCodec_Type.super.ht_type.tp_dict,
        "ConvertInvalidToNull", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QTextCodec_ConversionFlag_Type.tp_dict,
        "ConvertInvalidToNull", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QTextCodec_ConversionFlag_Type,
        (long) QTextCodec::IgnoreHeader, "IgnoreHeader");
    PyDict_SetItemString(SbkQTextCodec_Type.super.ht_type.tp_dict,
        "IgnoreHeader", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QTextCodec_ConversionFlag_Type.tp_dict,
        "IgnoreHeader", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QTextCodec_ConversionFlag_Type,
        (long) QTextCodec::FreeFunction, "FreeFunction");
    PyDict_SetItemString(SbkQTextCodec_Type.super.ht_type.tp_dict,
        "FreeFunction", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QTextCodec_ConversionFlag_Type.tp_dict,
        "FreeFunction", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QTextCodec::ConversionFlag");

    Shiboken::TypeResolver::createObjectTypeResolver<QTextCodec >("QTextCodec*");
    Shiboken::TypeResolver::createObjectTypeResolver<QTextCodec >(typeid(QTextCodec).name());
}


} // extern "C"

