/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtcore_python.h"

#include "qstring_wrapper.h"

// Extra includes
#include <QTextCodec>
#include <qbytearray.h>
#include <qchar.h>
#include <qregexp.h>
#include <qstring.h>
#include <qstringlist.h>

using namespace Shiboken;

// Begin code injection
// ========================================================================
// START of custom code block [file: glue/qstring_hashfunc.cpp]
static long QStringCustomHashFunction(const QString& str)
{
    QByteArray data = str.toUtf8();
    Shiboken::AutoDecRef unicodeObj(PyUnicode_DecodeUTF8(data.constData(), data.length(), 0));
    return unicodeObj->ob_type->tp_hash(unicodeObj);
}

// END of custom code block [file: glue/qstring_hashfunc.cpp]
// ========================================================================

// End of code injection
// Begin code injection
// ========================================================================
// START of custom code block [file: glue/qstring_bufferprotocol.cpp]

#if PY_VERSION_HEX < 0x03000000

// QByteArray buffer protocol functions
// see: http://www.python.org/dev/peps/pep-3118/

extern "C" {

static Py_ssize_t SbkQString_segcountproc(PyObject* self, Py_ssize_t* lenp)
{
    if (lenp)
        *lenp = self->ob_type->tp_as_sequence->sq_length(self);
    return 1;
}

static Py_ssize_t SbkQString_readbufferproc(PyObject* self, Py_ssize_t segment, char** ptrptr)
{
    if (segment || Shiboken::cppObjectIsInvalid(self))
        return -1;

    QString* cppSelf = Converter<QString*>::toCpp(self);
    QByteArray decodedData = cppSelf->toLocal8Bit();
    Shiboken::AutoDecRef decodedString(PyString_FromStringAndSize(decodedData.constData(), decodedData.size()));

    // delete __encodedStr attr if it exists
    Shiboken::AutoDecRef attrName(PyString_FromStringAndSize("__encodedStr", sizeof("__encodedStr")-1));
    if (PyObject_HasAttr(self, attrName))
        PyObject_DelAttr(self, attrName);
    PyObject_SetAttr(self, attrName, decodedString);
    *ptrptr = PyString_AS_STRING(decodedString.object());
    return decodedData.size();
}

PyBufferProcs SbkQStringBufferProc = {
    /*bf_getreadbuffer*/  0,
    /*bf_getwritebuffer*/ 0,
    /*bf_getsegcount*/    &SbkQString_segcountproc,
    /*bf_getcharbuffer*/  &SbkQString_readbufferproc
};

}

#endif

// END of custom code block [file: glue/qstring_bufferprotocol.cpp]
// ========================================================================

// End of code injection

static int
SbkQString_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QString* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "QString", 0, 2, &(pyargs[0]), &(pyargs[1])))
        return -1;

    if (numArgs == 0) {
        // QString()
        cptr = new QString();
        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(pyargs[0])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (numArgs == 2 && Shiboken::Converter<Qt::Initialization >::isConvertible(pyargs[1])) {
            Qt::Initialization cpp_arg1 = Shiboken::Converter<Qt::Initialization >::toCpp(pyargs[1]);
            // QString(int,Qt::Initialization)
            cptr = new QString(cpp_arg0, cpp_arg1);
            //CppGenerator::writeParentChildManagement
        } else if (numArgs == 2 && Shiboken::Converter<QChar >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return -1;
            std::auto_ptr<QChar > cpp_arg1_auto_ptr;
            QChar* cpp_arg1 = Shiboken::Converter<QChar* >::toCpp(pyargs[1]);
            if (!SbkQChar_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QChar >(cpp_arg1);
            // QString(int,QChar)
            cptr = new QString(cpp_arg0, *cpp_arg1);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQString_Init_TypeError;
    } else if (numArgs == 1 && Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        // QString(const char*)
        cptr = new QString(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[0]);
        // QString(QLatin1String)
        cptr = new QString(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QByteArray & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(pyargs[0]);
        if (!SbkQByteArray_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // QString(QByteArray)
        cptr = new QString(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QChar >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        // QString(QChar)
        cptr = new QString(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // QString(QString)
        cptr = new QString(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQString_Init_TypeError;

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QString >(), cptr)) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQString_Init_TypeError;

    sbkSelf->validCppObject = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;

    SbkQString_Init_TypeError:
        const char* overloads[] = {"", "PySide.QtCore.QChar", "PySide.QtCore.QByteArray", "PySide.QtCore.QLatin1String", "PySide.QtCore.QString", "str", "int, PySide.QtCore.QChar", "int, PySide.QtCore.Qt.Initialization", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString", overloads);
        return -1;
}

static PyObject* SbkQStringFunc___str__(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // __str__()
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    (void)cppSelf; // avoid warnings about unused variables
    // Begin code injection
    
    QByteArray utf8 = cppSelf->toUtf8();
    py_result = PyUnicode_DecodeUTF8(utf8.data(), utf8.count(), 0);
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_append(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // append(const char*)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::append(cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QStringRef & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QStringRef* cpp_arg0 = Shiboken::Converter<QStringRef* >::toCpp(arg);
        // append(QStringRef)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::append(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QLatin1String & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(arg);
        // append(QLatin1String)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::append(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QByteArray & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(arg);
        if (!SbkQByteArray_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // append(QByteArray)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::append(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QChar >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(arg);
        if (!SbkQChar_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        // append(QChar)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::append(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // append(QString)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::append(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_append_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_append_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar", "PySide.QtCore.QByteArray", "PySide.QtCore.QLatin1String", "PySide.QtCore.QString", "PySide.QtCore.QStringRef", "str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.append", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_arg(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0, 0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "arg", 1, 9, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4]), &(pyargs[5]), &(pyargs[6]), &(pyargs[7]), &(pyargs[8])))
        return 0;

    if (PyFloat_Check(pyargs[0])) {
        double cpp_arg0 = Shiboken::Converter<double >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // arg(double,int,char,int,QChar)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QString::arg(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // arg(double,int,char,int,QChar)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QString::arg(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<char >::isConvertible(pyargs[2])) {
                char cpp_arg2 = Shiboken::Converter<char >::toCpp(pyargs[2]);
                if (numArgs == 3) {
                    // arg(double,int,char,int,QChar)const
                    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                    QString cpp_result = cppSelf->QString::arg(cpp_arg0, cpp_arg1, cpp_arg2);
                    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                    //CppGenerator::writeParentChildManagement
                } else if (PyNumber_Check(pyargs[3])) {
                    int cpp_arg3 = Shiboken::Converter<int >::toCpp(pyargs[3]);
                    if (numArgs == 4) {
                        // arg(double,int,char,int,QChar)const
                        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                        QString cpp_result = cppSelf->QString::arg(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
                        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                        //CppGenerator::writeParentChildManagement
                    } else if (Shiboken::Converter<QChar & >::isConvertible(pyargs[4])) {
                        if (Shiboken::cppObjectIsInvalid(pyargs[4]))
                            return 0;
                        std::auto_ptr<QChar > cpp_arg4_auto_ptr;
                        QChar* cpp_arg4 = Shiboken::Converter<QChar* >::toCpp(pyargs[4]);
                        if (!SbkQChar_Check(pyargs[4]))
                            cpp_arg4_auto_ptr = std::auto_ptr<QChar >(cpp_arg4);
                        // arg(double,int,char,int,QChar)const
                        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                        QString cpp_result = cppSelf->QString::arg(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, *cpp_arg4);
                        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                        //CppGenerator::writeParentChildManagement
                    } else goto SbkQStringFunc_arg_TypeError;
                } else goto SbkQStringFunc_arg_TypeError;
            } else goto SbkQStringFunc_arg_TypeError;
        } else goto SbkQStringFunc_arg_TypeError;
    } else if (PyNumber_Check(pyargs[0])) {
        long cpp_arg0 = Shiboken::Converter<long >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // arg(long,int,int,QChar)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QString::arg(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // arg(long,int,int,QChar)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QString::arg(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (PyNumber_Check(pyargs[2])) {
                int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
                if (numArgs == 3) {
                    // arg(long,int,int,QChar)const
                    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                    QString cpp_result = cppSelf->QString::arg(cpp_arg0, cpp_arg1, cpp_arg2);
                    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                    //CppGenerator::writeParentChildManagement
                } else if (Shiboken::Converter<QChar & >::isConvertible(pyargs[3])) {
                    if (Shiboken::cppObjectIsInvalid(pyargs[3]))
                        return 0;
                    std::auto_ptr<QChar > cpp_arg3_auto_ptr;
                    QChar* cpp_arg3 = Shiboken::Converter<QChar* >::toCpp(pyargs[3]);
                    if (!SbkQChar_Check(pyargs[3]))
                        cpp_arg3_auto_ptr = std::auto_ptr<QChar >(cpp_arg3);
                    // arg(long,int,int,QChar)const
                    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                    QString cpp_result = cppSelf->QString::arg(cpp_arg0, cpp_arg1, cpp_arg2, *cpp_arg3);
                    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                    //CppGenerator::writeParentChildManagement
                } else goto SbkQStringFunc_arg_TypeError;
            } else goto SbkQStringFunc_arg_TypeError;
        } else goto SbkQStringFunc_arg_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // arg(QString,int,QChar)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QString::arg(*cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QString > cpp_arg1_auto_ptr;
            QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
            if (!SbkQString_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
            if (numArgs == 2) {
                // arg(QString,QString)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QString::arg(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[2])) {
                if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                    return 0;
                std::auto_ptr<QString > cpp_arg2_auto_ptr;
                QString* cpp_arg2 = Shiboken::Converter<QString* >::toCpp(pyargs[2]);
                if (!SbkQString_Check(pyargs[2]))
                    cpp_arg2_auto_ptr = std::auto_ptr<QString >(cpp_arg2);
                if (numArgs == 3) {
                    // arg(QString,QString,QString)const
                    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                    QString cpp_result = cppSelf->QString::arg(*cpp_arg0, *cpp_arg1, *cpp_arg2);
                    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                    //CppGenerator::writeParentChildManagement
                } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[3])) {
                    if (Shiboken::cppObjectIsInvalid(pyargs[3]))
                        return 0;
                    std::auto_ptr<QString > cpp_arg3_auto_ptr;
                    QString* cpp_arg3 = Shiboken::Converter<QString* >::toCpp(pyargs[3]);
                    if (!SbkQString_Check(pyargs[3]))
                        cpp_arg3_auto_ptr = std::auto_ptr<QString >(cpp_arg3);
                    if (numArgs == 4) {
                        // arg(QString,QString,QString,QString)const
                        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                        QString cpp_result = cppSelf->QString::arg(*cpp_arg0, *cpp_arg1, *cpp_arg2, *cpp_arg3);
                        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                        //CppGenerator::writeParentChildManagement
                    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[4])) {
                        if (Shiboken::cppObjectIsInvalid(pyargs[4]))
                            return 0;
                        std::auto_ptr<QString > cpp_arg4_auto_ptr;
                        QString* cpp_arg4 = Shiboken::Converter<QString* >::toCpp(pyargs[4]);
                        if (!SbkQString_Check(pyargs[4]))
                            cpp_arg4_auto_ptr = std::auto_ptr<QString >(cpp_arg4);
                        if (numArgs == 5) {
                            // arg(QString,QString,QString,QString,QString)const
                            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                            QString cpp_result = cppSelf->QString::arg(*cpp_arg0, *cpp_arg1, *cpp_arg2, *cpp_arg3, *cpp_arg4);
                            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                            //CppGenerator::writeParentChildManagement
                        } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[5])) {
                            if (Shiboken::cppObjectIsInvalid(pyargs[5]))
                                return 0;
                            std::auto_ptr<QString > cpp_arg5_auto_ptr;
                            QString* cpp_arg5 = Shiboken::Converter<QString* >::toCpp(pyargs[5]);
                            if (!SbkQString_Check(pyargs[5]))
                                cpp_arg5_auto_ptr = std::auto_ptr<QString >(cpp_arg5);
                            if (numArgs == 6) {
                                // arg(QString,QString,QString,QString,QString,QString)const
                                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                                QString cpp_result = cppSelf->QString::arg(*cpp_arg0, *cpp_arg1, *cpp_arg2, *cpp_arg3, *cpp_arg4, *cpp_arg5);
                                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                                //CppGenerator::writeParentChildManagement
                            } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[6])) {
                                if (Shiboken::cppObjectIsInvalid(pyargs[6]))
                                    return 0;
                                std::auto_ptr<QString > cpp_arg6_auto_ptr;
                                QString* cpp_arg6 = Shiboken::Converter<QString* >::toCpp(pyargs[6]);
                                if (!SbkQString_Check(pyargs[6]))
                                    cpp_arg6_auto_ptr = std::auto_ptr<QString >(cpp_arg6);
                                if (numArgs == 7) {
                                    // arg(QString,QString,QString,QString,QString,QString,QString)const
                                    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                                    QString cpp_result = cppSelf->QString::arg(*cpp_arg0, *cpp_arg1, *cpp_arg2, *cpp_arg3, *cpp_arg4, *cpp_arg5, *cpp_arg6);
                                    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                                    //CppGenerator::writeParentChildManagement
                                } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[7])) {
                                    if (Shiboken::cppObjectIsInvalid(pyargs[7]))
                                        return 0;
                                    std::auto_ptr<QString > cpp_arg7_auto_ptr;
                                    QString* cpp_arg7 = Shiboken::Converter<QString* >::toCpp(pyargs[7]);
                                    if (!SbkQString_Check(pyargs[7]))
                                        cpp_arg7_auto_ptr = std::auto_ptr<QString >(cpp_arg7);
                                    if (numArgs == 8) {
                                        // arg(QString,QString,QString,QString,QString,QString,QString,QString)const
                                        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                                        QString cpp_result = cppSelf->QString::arg(*cpp_arg0, *cpp_arg1, *cpp_arg2, *cpp_arg3, *cpp_arg4, *cpp_arg5, *cpp_arg6, *cpp_arg7);
                                        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                                        //CppGenerator::writeParentChildManagement
                                    } else if (numArgs == 9 && Shiboken::Converter<QString & >::isConvertible(pyargs[8])) {
                                        if (Shiboken::cppObjectIsInvalid(pyargs[8]))
                                            return 0;
                                        std::auto_ptr<QString > cpp_arg8_auto_ptr;
                                        QString* cpp_arg8 = Shiboken::Converter<QString* >::toCpp(pyargs[8]);
                                        if (!SbkQString_Check(pyargs[8]))
                                            cpp_arg8_auto_ptr = std::auto_ptr<QString >(cpp_arg8);
                                        // arg(QString,QString,QString,QString,QString,QString,QString,QString,QString)const
                                        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                                        QString cpp_result = cppSelf->QString::arg(*cpp_arg0, *cpp_arg1, *cpp_arg2, *cpp_arg3, *cpp_arg4, *cpp_arg5, *cpp_arg6, *cpp_arg7, *cpp_arg8);
                                        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                                        //CppGenerator::writeParentChildManagement
                                    } else goto SbkQStringFunc_arg_TypeError;
                                } else goto SbkQStringFunc_arg_TypeError;
                            } else goto SbkQStringFunc_arg_TypeError;
                        } else goto SbkQStringFunc_arg_TypeError;
                    } else goto SbkQStringFunc_arg_TypeError;
                } else goto SbkQStringFunc_arg_TypeError;
            } else goto SbkQStringFunc_arg_TypeError;
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // arg(QString,int,QChar)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QString::arg(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<QChar & >::isConvertible(pyargs[2])) {
                if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                    return 0;
                std::auto_ptr<QChar > cpp_arg2_auto_ptr;
                QChar* cpp_arg2 = Shiboken::Converter<QChar* >::toCpp(pyargs[2]);
                if (!SbkQChar_Check(pyargs[2]))
                    cpp_arg2_auto_ptr = std::auto_ptr<QChar >(cpp_arg2);
                // arg(QString,int,QChar)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QString::arg(*cpp_arg0, cpp_arg1, *cpp_arg2);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_arg_TypeError;
        } else goto SbkQStringFunc_arg_TypeError;
    } else goto SbkQStringFunc_arg_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_arg_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString, int = 0, PySide.QtCore.QChar = QLatin1Char(' ')", "PySide.QtCore.QString, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.QString", "float, int = 0, char = 'g', int = -1, PySide.QtCore.QChar = QLatin1Char(' ')", "long, int = 0, int = 10, PySide.QtCore.QChar = QLatin1Char(' ')", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.arg", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_at(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // at(int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QChar cpp_result = cppSelf->QString::at(cpp_arg0);
        py_result = Shiboken::Converter<QChar >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_at_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_at_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.at", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_capacity(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // capacity()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QString::capacity();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_chop(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // chop(int)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        cppSelf->QString::chop(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_chop_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQStringFunc_chop_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.chop", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_clear(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // clear()
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    cppSelf->QString::clear();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject* SbkQStringFunc_compare(PyObject* self, PyObject* args)
{
    if (self) {
        if (Shiboken::cppObjectIsInvalid(self))
            return 0;
    }

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "compare", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<QStringRef & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QStringRef* cpp_arg0 = Shiboken::Converter<QStringRef* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // compare(QStringRef,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::compare(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // compare(QStringRef,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::compare(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_compare_TypeError;
    } else if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // compare(QLatin1String,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::compare(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QString > cpp_arg1_auto_ptr;
            QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
            if (!SbkQString_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
            if (numArgs == 2) {
                // compare(QLatin1String,QString,Qt::CaseSensitivity)
                int cpp_result = QString::compare(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // compare(QLatin1String,QString,Qt::CaseSensitivity)
                int cpp_result = QString::compare(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_compare_TypeError;
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // compare(QLatin1String,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::compare(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_compare_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // compare(QString)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::compare(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QStringRef & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QStringRef* cpp_arg1 = Shiboken::Converter<QStringRef* >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // compare(QString,QStringRef,Qt::CaseSensitivity)
                int cpp_result = QString::compare(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // compare(QString,QStringRef,Qt::CaseSensitivity)
                int cpp_result = QString::compare(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_compare_TypeError;
        } else if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QLatin1String* cpp_arg1 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // compare(QString,QLatin1String,Qt::CaseSensitivity)
                int cpp_result = QString::compare(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // compare(QString,QLatin1String,Qt::CaseSensitivity)
                int cpp_result = QString::compare(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_compare_TypeError;
        } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QString > cpp_arg1_auto_ptr;
            QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
            if (!SbkQString_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
            if (numArgs == 2) {
                // compare(QString,QString)
                int cpp_result = QString::compare(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (numArgs == 3 && Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // compare(QString,QString,Qt::CaseSensitivity)
                int cpp_result = QString::compare(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_compare_TypeError;
        } else if (numArgs == 2 && Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // compare(QString,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::compare(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_compare_TypeError;
    } else goto SbkQStringFunc_compare_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_compare_TypeError:
        const char* overloads[] = {"PySide.QtCore.QLatin1String, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QLatin1String, PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity", "PySide.QtCore.QString, PySide.QtCore.QLatin1String, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QString, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity", "PySide.QtCore.QString, PySide.QtCore.QStringRef, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QStringRef, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.compare", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_contains(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "contains", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QRegExp & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QRegExp* cpp_arg0 = Shiboken::Converter<QRegExp* >::toCpp(pyargs[0]);
        // contains(QRegExp&)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QBool cpp_result = cppSelf->QString::contains(*cpp_arg0);
        py_result = Shiboken::Converter<QBool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QChar >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (numArgs == 1) {
            // contains(QChar,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QBool cpp_result = cppSelf->QString::contains(*cpp_arg0);
            py_result = Shiboken::Converter<QBool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // contains(QChar,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QBool cpp_result = cppSelf->QString::contains(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QBool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_contains_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // contains(QString,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QBool cpp_result = cppSelf->QString::contains(*cpp_arg0);
            py_result = Shiboken::Converter<QBool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // contains(QString,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QBool cpp_result = cppSelf->QString::contains(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QBool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_contains_TypeError;
    } else goto SbkQStringFunc_contains_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_contains_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QRegExp", "PySide.QtCore.QRegExp", "PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.contains", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_count(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "count", 0, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 0) {
        // count()const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        int cpp_result = cppSelf->QString::count();
        py_result = Shiboken::Converter<int >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QRegExp & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QRegExp* cpp_arg0 = Shiboken::Converter<QRegExp* >::toCpp(pyargs[0]);
        // count(QRegExp)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        int cpp_result = cppSelf->QString::count(*cpp_arg0);
        py_result = Shiboken::Converter<int >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QChar >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (numArgs == 1) {
            // count(QChar,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::count(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // count(QChar,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::count(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_count_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // count(QString,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::count(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // count(QString,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::count(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_count_TypeError;
    } else goto SbkQStringFunc_count_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_count_TypeError:
        const char* overloads[] = {"", "PySide.QtCore.QChar, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QRegExp", "PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.count", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_endsWith(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "endsWith", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // endsWith(QLatin1String,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::endsWith(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // endsWith(QLatin1String,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::endsWith(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_endsWith_TypeError;
    } else if (Shiboken::Converter<QChar & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (numArgs == 1) {
            // endsWith(QChar,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::endsWith(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // endsWith(QChar,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::endsWith(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_endsWith_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // endsWith(QString,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::endsWith(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // endsWith(QString,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::endsWith(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_endsWith_TypeError;
    } else goto SbkQStringFunc_endsWith_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_endsWith_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QLatin1String, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.endsWith", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_fill(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "fill", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QChar >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (numArgs == 1) {
            // fill(QChar,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::fill(*cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // fill(QChar,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::fill(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_fill_TypeError;
    } else goto SbkQStringFunc_fill_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_fill_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.fill", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_fromAscii(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "fromAscii", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // fromAscii(const char*,int)
            QString cpp_result = QString::fromAscii(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // fromAscii(const char*,int)
            QString cpp_result = QString::fromAscii(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_fromAscii_TypeError;
    } else goto SbkQStringFunc_fromAscii_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_fromAscii_TypeError:
        const char* overloads[] = {"str, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.fromAscii", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_fromLatin1(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "fromLatin1", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // fromLatin1(const char*,int)
            QString cpp_result = QString::fromLatin1(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // fromLatin1(const char*,int)
            QString cpp_result = QString::fromLatin1(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_fromLatin1_TypeError;
    } else goto SbkQStringFunc_fromLatin1_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_fromLatin1_TypeError:
        const char* overloads[] = {"str, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.fromLatin1", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_fromLocal8Bit(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "fromLocal8Bit", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // fromLocal8Bit(const char*,int)
            QString cpp_result = QString::fromLocal8Bit(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // fromLocal8Bit(const char*,int)
            QString cpp_result = QString::fromLocal8Bit(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_fromLocal8Bit_TypeError;
    } else goto SbkQStringFunc_fromLocal8Bit_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_fromLocal8Bit_TypeError:
        const char* overloads[] = {"str, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.fromLocal8Bit", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_fromUtf8(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "fromUtf8", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // fromUtf8(const char*,int)
            QString cpp_result = QString::fromUtf8(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // fromUtf8(const char*,int)
            QString cpp_result = QString::fromUtf8(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_fromUtf8_TypeError;
    } else goto SbkQStringFunc_fromUtf8_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_fromUtf8_TypeError:
        const char* overloads[] = {"str, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.fromUtf8", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_indexOf(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "indexOf", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // indexOf(QLatin1String,int,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::indexOf(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // indexOf(QLatin1String,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::indexOf(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // indexOf(QLatin1String,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::indexOf(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_indexOf_TypeError;
        } else goto SbkQStringFunc_indexOf_TypeError;
    } else if (Shiboken::Converter<QRegExp & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QRegExp* cpp_arg0 = Shiboken::Converter<QRegExp* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // indexOf(QRegExp&,int)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::indexOf(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // indexOf(QRegExp&,int)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::indexOf(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_indexOf_TypeError;
    } else if (Shiboken::Converter<QChar >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (numArgs == 1) {
            // indexOf(QChar,int,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::indexOf(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // indexOf(QChar,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::indexOf(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // indexOf(QChar,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::indexOf(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_indexOf_TypeError;
        } else goto SbkQStringFunc_indexOf_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // indexOf(QString,int,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::indexOf(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // indexOf(QString,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::indexOf(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // indexOf(QString,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::indexOf(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_indexOf_TypeError;
        } else goto SbkQStringFunc_indexOf_TypeError;
    } else goto SbkQStringFunc_indexOf_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_indexOf_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, int = 0, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QRegExp, int = 0", "PySide.QtCore.QLatin1String, int = 0, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QRegExp, int = 0", "PySide.QtCore.QString, int = 0, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.indexOf", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_insert(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "insert", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (PyNumber_Check(pyargs[0])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (numArgs == 2 && Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QLatin1String* cpp_arg1 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[1]);
            // insert(int,QLatin1String)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::insert(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (numArgs == 2 && Shiboken::Converter<QChar >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QChar > cpp_arg1_auto_ptr;
            QChar* cpp_arg1 = Shiboken::Converter<QChar* >::toCpp(pyargs[1]);
            if (!SbkQChar_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QChar >(cpp_arg1);
            // insert(int,QChar)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::insert(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (numArgs == 2 && Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QString > cpp_arg1_auto_ptr;
            QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
            if (!SbkQString_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
            // insert(int,QString)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::insert(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_insert_TypeError;
    } else goto SbkQStringFunc_insert_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_insert_TypeError:
        const char* overloads[] = {"int, PySide.QtCore.QChar", "int, PySide.QtCore.QLatin1String", "int, PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.insert", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_isEmpty(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isEmpty()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QString::isEmpty();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_isNull(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isNull()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QString::isNull();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_isRightToLeft(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isRightToLeft()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QString::isRightToLeft();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_isSimpleText(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isSimpleText()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QString::isSimpleText();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_lastIndexOf(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "lastIndexOf", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // lastIndexOf(QLatin1String,int,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // lastIndexOf(QLatin1String,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // lastIndexOf(QLatin1String,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_lastIndexOf_TypeError;
        } else goto SbkQStringFunc_lastIndexOf_TypeError;
    } else if (Shiboken::Converter<QRegExp & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QRegExp* cpp_arg0 = Shiboken::Converter<QRegExp* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // lastIndexOf(QRegExp&,int)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // lastIndexOf(QRegExp&,int)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_lastIndexOf_TypeError;
    } else if (Shiboken::Converter<QChar >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (numArgs == 1) {
            // lastIndexOf(QChar,int,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // lastIndexOf(QChar,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // lastIndexOf(QChar,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_lastIndexOf_TypeError;
        } else goto SbkQStringFunc_lastIndexOf_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // lastIndexOf(QString,int,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // lastIndexOf(QString,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // lastIndexOf(QString,int,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                int cpp_result = cppSelf->QString::lastIndexOf(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<int >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_lastIndexOf_TypeError;
        } else goto SbkQStringFunc_lastIndexOf_TypeError;
    } else goto SbkQStringFunc_lastIndexOf_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_lastIndexOf_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, int = -1, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QRegExp, int = -1", "PySide.QtCore.QLatin1String, int = -1, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QRegExp, int = -1", "PySide.QtCore.QString, int = -1, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.lastIndexOf", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_left(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // left(int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->QString::left(cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_left_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_left_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.left", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_leftJustified(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "leftJustified", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (PyNumber_Check(pyargs[0])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // leftJustified(int,QChar,bool)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QString::leftJustified(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QChar >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QChar > cpp_arg1_auto_ptr;
            QChar* cpp_arg1 = Shiboken::Converter<QChar* >::toCpp(pyargs[1]);
            if (!SbkQChar_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QChar >(cpp_arg1);
            if (numArgs == 2) {
                // leftJustified(int,QChar,bool)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QString::leftJustified(cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (PyNumber_Check(pyargs[2])) {
                bool cpp_arg2 = Shiboken::Converter<bool >::toCpp(pyargs[2]);
                // leftJustified(int,QChar,bool)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QString::leftJustified(cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_leftJustified_TypeError;
        } else goto SbkQStringFunc_leftJustified_TypeError;
    } else goto SbkQStringFunc_leftJustified_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_leftJustified_TypeError:
        const char* overloads[] = {"int, PySide.QtCore.QChar = QLatin1Char(' '), bool = false", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.leftJustified", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_leftRef(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // leftRef(int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QStringRef cpp_result = cppSelf->QString::leftRef(cpp_arg0);
        py_result = Shiboken::Converter<QStringRef >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_leftRef_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_leftRef_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.leftRef", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_length(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // length()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QString::length();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_localeAwareCompare(PyObject* self, PyObject* args)
{
    if (self) {
        if (Shiboken::cppObjectIsInvalid(self))
            return 0;
    }

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "localeAwareCompare", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 1 && Shiboken::Converter<QStringRef & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QStringRef* cpp_arg0 = Shiboken::Converter<QStringRef* >::toCpp(pyargs[0]);
        // localeAwareCompare(QStringRef)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        int cpp_result = cppSelf->QString::localeAwareCompare(*cpp_arg0);
        py_result = Shiboken::Converter<int >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // localeAwareCompare(QString)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            int cpp_result = cppSelf->QString::localeAwareCompare(*cpp_arg0);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (numArgs == 2 && Shiboken::Converter<QStringRef & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QStringRef* cpp_arg1 = Shiboken::Converter<QStringRef* >::toCpp(pyargs[1]);
            // localeAwareCompare(QString,QStringRef)
            int cpp_result = QString::localeAwareCompare(*cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (numArgs == 2 && Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QString > cpp_arg1_auto_ptr;
            QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
            if (!SbkQString_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
            // localeAwareCompare(QString,QString)
            int cpp_result = QString::localeAwareCompare(*cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<int >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_localeAwareCompare_TypeError;
    } else goto SbkQStringFunc_localeAwareCompare_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_localeAwareCompare_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QStringRef", "PySide.QtCore.QStringRef", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.localeAwareCompare", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_mid(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "mid", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (PyNumber_Check(pyargs[0])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // mid(int,int)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QString::mid(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // mid(int,int)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QString::mid(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_mid_TypeError;
    } else goto SbkQStringFunc_mid_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_mid_TypeError:
        const char* overloads[] = {"int, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.mid", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_midRef(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "midRef", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (PyNumber_Check(pyargs[0])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // midRef(int,int)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QStringRef cpp_result = cppSelf->QString::midRef(cpp_arg0);
            py_result = Shiboken::Converter<QStringRef >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // midRef(int,int)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QStringRef cpp_result = cppSelf->QString::midRef(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QStringRef >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_midRef_TypeError;
    } else goto SbkQStringFunc_midRef_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_midRef_TypeError:
        const char* overloads[] = {"int, int = -1", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.midRef", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_normalized(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "normalized", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QString::NormalizationForm >::isConvertible(pyargs[0])) {
        QString::NormalizationForm cpp_arg0 = Shiboken::Converter<QString::NormalizationForm >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // normalized(QString::NormalizationForm)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QString::normalized(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (numArgs == 2 && Shiboken::Converter<QChar::UnicodeVersion >::isConvertible(pyargs[1])) {
            QChar::UnicodeVersion cpp_arg1 = Shiboken::Converter<QChar::UnicodeVersion >::toCpp(pyargs[1]);
            // normalized(QString::NormalizationForm,QChar::UnicodeVersion)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QString::normalized(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_normalized_TypeError;
    } else goto SbkQStringFunc_normalized_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_normalized_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString.NormalizationForm", "PySide.QtCore.QString.NormalizationForm, PySide.QtCore.QChar.UnicodeVersion", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.normalized", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_number(PyObject* self, PyObject* args)
{
    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "number", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (PyFloat_Check(pyargs[0])) {
        double cpp_arg0 = Shiboken::Converter<double >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // number(double,char,int)
            QString cpp_result = QString::number(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<char >::isConvertible(pyargs[1])) {
            char cpp_arg1 = Shiboken::Converter<char >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // number(double,char,int)
                QString cpp_result = QString::number(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (PyNumber_Check(pyargs[2])) {
                int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
                // number(double,char,int)
                QString cpp_result = QString::number(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_number_TypeError;
        } else goto SbkQStringFunc_number_TypeError;
    } else if (PyNumber_Check(pyargs[0])) {
        long cpp_arg0 = Shiboken::Converter<long >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // number(long,int)
            QString cpp_result = QString::number(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // number(long,int)
            QString cpp_result = QString::number(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_number_TypeError;
    } else goto SbkQStringFunc_number_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_number_TypeError:
        const char* overloads[] = {"float, char = 'g', int = 6", "long, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.number", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_prepend(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // prepend(const char*)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::prepend(cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QLatin1String & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(arg);
        // prepend(QLatin1String)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::prepend(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QByteArray & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(arg);
        if (!SbkQByteArray_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // prepend(QByteArray)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::prepend(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QChar >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(arg);
        if (!SbkQChar_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        // prepend(QChar)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::prepend(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // prepend(QString)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::prepend(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_prepend_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_prepend_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar", "PySide.QtCore.QByteArray", "PySide.QtCore.QLatin1String", "PySide.QtCore.QString", "str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.prepend", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_push_back(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QChar >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(arg);
        if (!SbkQChar_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        // push_back(QChar)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        cppSelf->QString::push_back(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // push_back(QString)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        cppSelf->QString::push_back(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_push_back_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQStringFunc_push_back_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar", "PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.push_back", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_push_front(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QChar >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(arg);
        if (!SbkQChar_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        // push_front(QChar)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        cppSelf->QString::push_front(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // push_front(QString)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        cppSelf->QString::push_front(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_push_front_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQStringFunc_push_front_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar", "PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.push_front", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_remove(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "remove", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // remove(int,int)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::remove(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QRegExp & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QRegExp* cpp_arg0 = Shiboken::Converter<QRegExp* >::toCpp(pyargs[0]);
        // remove(QRegExp)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::remove(*cpp_arg0);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QChar >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (numArgs == 1) {
            // remove(QChar,Qt::CaseSensitivity)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::remove(*cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // remove(QChar,Qt::CaseSensitivity)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::remove(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_remove_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // remove(QString,Qt::CaseSensitivity)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::remove(*cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // remove(QString,Qt::CaseSensitivity)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::remove(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_remove_TypeError;
    } else goto SbkQStringFunc_remove_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_remove_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QRegExp", "PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "int, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.remove", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_repeated(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // repeated(int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->QString::repeated(cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_repeated_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_repeated_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.repeated", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_replace(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "replace", 2, 5, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4])))
        return 0;

    if (PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        if (Shiboken::Converter<QChar >::isConvertible(pyargs[2])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                return 0;
            std::auto_ptr<QChar > cpp_arg2_auto_ptr;
            QChar* cpp_arg2 = Shiboken::Converter<QChar* >::toCpp(pyargs[2]);
            if (!SbkQChar_Check(pyargs[2]))
                cpp_arg2_auto_ptr = std::auto_ptr<QChar >(cpp_arg2);
            if (numArgs == 3) {
                // replace(int,int,QChar)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(cpp_arg0, cpp_arg1, *cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (numArgs == 4 && PyNumber_Check(pyargs[3])) {
                int cpp_arg3 = Shiboken::Converter<int >::toCpp(pyargs[3]);
                // replace(int,int,const QChar*,int)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_replace_TypeError;
        } else if (numArgs == 3 && Shiboken::Converter<QString & >::isConvertible(pyargs[2])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                return 0;
            std::auto_ptr<QString > cpp_arg2_auto_ptr;
            QString* cpp_arg2 = Shiboken::Converter<QString* >::toCpp(pyargs[2]);
            if (!SbkQString_Check(pyargs[2]))
                cpp_arg2_auto_ptr = std::auto_ptr<QString >(cpp_arg2);
            // replace(int,int,QString)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::replace(cpp_arg0, cpp_arg1, *cpp_arg2);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_replace_TypeError;
    } else if (numArgs == 2 && Shiboken::Converter<QRegExp & >::isConvertible(pyargs[0]) && Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QRegExp* cpp_arg0 = Shiboken::Converter<QRegExp* >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QString > cpp_arg1_auto_ptr;
        QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
        if (!SbkQString_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
        // replace(QRegExp,QString)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1);
        py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[0]);
        if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QLatin1String* cpp_arg1 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // replace(QLatin1String,QLatin1String,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // replace(QLatin1String,QLatin1String,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_replace_TypeError;
        } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QString > cpp_arg1_auto_ptr;
            QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
            if (!SbkQString_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
            if (numArgs == 2) {
                // replace(QLatin1String,QString,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // replace(QLatin1String,QString,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_replace_TypeError;
        } else goto SbkQStringFunc_replace_TypeError;
    } else if (Shiboken::Converter<QChar >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (PyNumber_Check(pyargs[1]) && Shiboken::Converter<QChar * >::isConvertible(pyargs[2]) && PyNumber_Check(pyargs[3])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                return 0;
            std::auto_ptr<QChar > cpp_arg2_auto_ptr;
            QChar* cpp_arg2 = Shiboken::Converter<QChar* >::toCpp(pyargs[2]);
            if (!SbkQChar_Check(pyargs[2]))
                cpp_arg2_auto_ptr = std::auto_ptr<QChar >(cpp_arg2);
            int cpp_arg3 = Shiboken::Converter<int >::toCpp(pyargs[3]);
            if (numArgs == 4) {
                // replace(const QChar*,int,const QChar*,int,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[4])) {
                Qt::CaseSensitivity cpp_arg4 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[4]);
                // replace(const QChar*,int,const QChar*,int,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_replace_TypeError;
        } else if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QLatin1String* cpp_arg1 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // replace(QChar,QLatin1String,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // replace(QChar,QLatin1String,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_replace_TypeError;
        } else if (Shiboken::Converter<QChar >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QChar > cpp_arg1_auto_ptr;
            QChar* cpp_arg1 = Shiboken::Converter<QChar* >::toCpp(pyargs[1]);
            if (!SbkQChar_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QChar >(cpp_arg1);
            if (numArgs == 2) {
                // replace(QChar,QChar,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // replace(QChar,QChar,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_replace_TypeError;
        } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QString > cpp_arg1_auto_ptr;
            QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
            if (!SbkQString_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
            if (numArgs == 2) {
                // replace(QChar,QString,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // replace(QChar,QString,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_replace_TypeError;
        } else goto SbkQStringFunc_replace_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QLatin1String* cpp_arg1 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // replace(QString,QLatin1String,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // replace(QString,QLatin1String,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_replace_TypeError;
        } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QString > cpp_arg1_auto_ptr;
            QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
            if (!SbkQString_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
            if (numArgs == 2) {
                // replace(QString,QString,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // replace(QString,QString,Qt::CaseSensitivity)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::replace(*cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_replace_TypeError;
        } else goto SbkQStringFunc_replace_TypeError;
    } else goto SbkQStringFunc_replace_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_replace_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, PySide.QtCore.QChar, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QChar, PySide.QtCore.QLatin1String, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QChar, PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QChar, int, PySide.QtCore.QChar, int, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QLatin1String, PySide.QtCore.QLatin1String, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QLatin1String, PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QRegExp, PySide.QtCore.QString", "PySide.QtCore.QString, PySide.QtCore.QLatin1String, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QString, PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "int, int, PySide.QtCore.QChar", "int, int, PySide.QtCore.QChar, int", "int, int, PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.replace", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_reserve(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // reserve(int)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        cppSelf->QString::reserve(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_reserve_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQStringFunc_reserve_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.reserve", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_resize(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // resize(int)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        cppSelf->QString::resize(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_resize_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQStringFunc_resize_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.resize", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_right(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // right(int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->QString::right(cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_right_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_right_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.right", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_rightJustified(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "rightJustified", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (PyNumber_Check(pyargs[0])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // rightJustified(int,QChar,bool)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString cpp_result = cppSelf->QString::rightJustified(cpp_arg0);
            py_result = Shiboken::Converter<QString >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QChar >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QChar > cpp_arg1_auto_ptr;
            QChar* cpp_arg1 = Shiboken::Converter<QChar* >::toCpp(pyargs[1]);
            if (!SbkQChar_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QChar >(cpp_arg1);
            if (numArgs == 2) {
                // rightJustified(int,QChar,bool)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QString::rightJustified(cpp_arg0, *cpp_arg1);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (PyNumber_Check(pyargs[2])) {
                bool cpp_arg2 = Shiboken::Converter<bool >::toCpp(pyargs[2]);
                // rightJustified(int,QChar,bool)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString cpp_result = cppSelf->QString::rightJustified(cpp_arg0, *cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_rightJustified_TypeError;
        } else goto SbkQStringFunc_rightJustified_TypeError;
    } else goto SbkQStringFunc_rightJustified_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_rightJustified_TypeError:
        const char* overloads[] = {"int, PySide.QtCore.QChar = QLatin1Char(' '), bool = false", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.rightJustified", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_rightRef(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // rightRef(int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        QStringRef cpp_result = cppSelf->QString::rightRef(cpp_arg0);
        py_result = Shiboken::Converter<QStringRef >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_rightRef_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_rightRef_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.rightRef", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_setNum(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "setNum", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (PyNumber_Check(pyargs[0])) {
        ushort cpp_arg0 = Shiboken::Converter<ushort >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(ushort,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // setNum(ushort,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_setNum_TypeError;
    } else if (PyLong_Check(pyargs[0])) {
        ulong cpp_arg0 = Shiboken::Converter<ulong >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(ulong,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // setNum(ulong,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_setNum_TypeError;
    } else if (PyNumber_Check(pyargs[0])) {
        uint cpp_arg0 = Shiboken::Converter<uint >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(uint,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // setNum(uint,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_setNum_TypeError;
    } else if (PyLong_Check(pyargs[0])) {
        qulonglong cpp_arg0 = Shiboken::Converter<qulonglong >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(qulonglong,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // setNum(qulonglong,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_setNum_TypeError;
    } else if (PyLong_Check(pyargs[0])) {
        qlonglong cpp_arg0 = Shiboken::Converter<qlonglong >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(qlonglong,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // setNum(qlonglong,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_setNum_TypeError;
    } else if (PyFloat_Check(pyargs[0])) {
        float cpp_arg0 = Shiboken::Converter<float >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(float,char,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<char >::isConvertible(pyargs[1])) {
            char cpp_arg1 = Shiboken::Converter<char >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // setNum(float,char,int)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (PyNumber_Check(pyargs[2])) {
                int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
                // setNum(float,char,int)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_setNum_TypeError;
        } else goto SbkQStringFunc_setNum_TypeError;
    } else if (PyFloat_Check(pyargs[0])) {
        double cpp_arg0 = Shiboken::Converter<double >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(double,char,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<char >::isConvertible(pyargs[1])) {
            char cpp_arg1 = Shiboken::Converter<char >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // setNum(double,char,int)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (PyNumber_Check(pyargs[2])) {
                int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
                // setNum(double,char,int)
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_setNum_TypeError;
        } else goto SbkQStringFunc_setNum_TypeError;
    } else if (PyNumber_Check(pyargs[0])) {
        short cpp_arg0 = Shiboken::Converter<short >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(short,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // setNum(short,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_setNum_TypeError;
    } else if (PyNumber_Check(pyargs[0])) {
        long cpp_arg0 = Shiboken::Converter<long >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(long,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // setNum(long,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_setNum_TypeError;
    } else if (PyNumber_Check(pyargs[0])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setNum(int,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            // setNum(int,int)
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QString & cpp_result = cppSelf->QString::setNum(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QString & >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_setNum_TypeError;
    } else goto SbkQStringFunc_setNum_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_setNum_TypeError:
        const char* overloads[] = {"float, char = 'g', int = 6", "float, char = 'g', int = 6", "int, int = 10", "long, int = 10", "long long, int = 10", "unsigned long long, int = 10", "short, int = 10", "unsigned int, int = 10", "ulong, int = 10", "ushort, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.setNum", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_simplified(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // simplified()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QString::simplified();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_size(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // size()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QString::size();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_split(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "split", 1, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<QRegExp & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QRegExp* cpp_arg0 = Shiboken::Converter<QRegExp* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // split(QRegExp,QString::SplitBehavior)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QStringList cpp_result = cppSelf->QString::split(*cpp_arg0);
            py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QString::SplitBehavior >::isConvertible(pyargs[1])) {
            QString::SplitBehavior cpp_arg1 = Shiboken::Converter<QString::SplitBehavior >::toCpp(pyargs[1]);
            // split(QRegExp,QString::SplitBehavior)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QStringList cpp_result = cppSelf->QString::split(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_split_TypeError;
    } else if (Shiboken::Converter<QChar & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (numArgs == 1) {
            // split(QChar,QString::SplitBehavior,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QStringList cpp_result = cppSelf->QString::split(*cpp_arg0);
            py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QString::SplitBehavior >::isConvertible(pyargs[1])) {
            QString::SplitBehavior cpp_arg1 = Shiboken::Converter<QString::SplitBehavior >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // split(QChar,QString::SplitBehavior,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QStringList cpp_result = cppSelf->QString::split(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // split(QChar,QString::SplitBehavior,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QStringList cpp_result = cppSelf->QString::split(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_split_TypeError;
        } else goto SbkQStringFunc_split_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // split(QString,QString::SplitBehavior,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            QStringList cpp_result = cppSelf->QString::split(*cpp_arg0);
            py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QString::SplitBehavior >::isConvertible(pyargs[1])) {
            QString::SplitBehavior cpp_arg1 = Shiboken::Converter<QString::SplitBehavior >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // split(QString,QString::SplitBehavior,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QStringList cpp_result = cppSelf->QString::split(*cpp_arg0, cpp_arg1);
                py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[2])) {
                Qt::CaseSensitivity cpp_arg2 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[2]);
                // split(QString,QString::SplitBehavior,Qt::CaseSensitivity)const
                QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
                QStringList cpp_result = cppSelf->QString::split(*cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
                //CppGenerator::writeParentChildManagement
            } else goto SbkQStringFunc_split_TypeError;
        } else goto SbkQStringFunc_split_TypeError;
    } else goto SbkQStringFunc_split_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_split_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, PySide.QtCore.QString.SplitBehavior = KeepEmptyParts, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QRegExp, PySide.QtCore.QString.SplitBehavior = KeepEmptyParts", "PySide.QtCore.QString, PySide.QtCore.QString.SplitBehavior = KeepEmptyParts, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.split", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_squeeze(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // squeeze()
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    cppSelf->QString::squeeze();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject* SbkQStringFunc_startsWith(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "startsWith", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QLatin1String & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // startsWith(QLatin1String,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::startsWith(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // startsWith(QLatin1String,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::startsWith(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_startsWith_TypeError;
    } else if (Shiboken::Converter<QChar & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(pyargs[0]);
        if (!SbkQChar_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        if (numArgs == 1) {
            // startsWith(QChar,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::startsWith(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // startsWith(QChar,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::startsWith(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_startsWith_TypeError;
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // startsWith(QString,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::startsWith(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<Qt::CaseSensitivity >::isConvertible(pyargs[1])) {
            Qt::CaseSensitivity cpp_arg1 = Shiboken::Converter<Qt::CaseSensitivity >::toCpp(pyargs[1]);
            // startsWith(QString,Qt::CaseSensitivity)const
            QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QString::startsWith(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQStringFunc_startsWith_TypeError;
    } else goto SbkQStringFunc_startsWith_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_startsWith_TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QLatin1String, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", "PySide.QtCore.QString, PySide.QtCore.Qt.CaseSensitivity = Qt.CaseSensitive", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QString.startsWith", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_toAscii(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toAscii()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    QByteArray cpp_result = cppSelf->QString::toAscii();
    py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_toCaseFolded(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toCaseFolded()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QString::toCaseFolded();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_toDouble(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toDouble(bool*)const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    // Begin code injection
    
    // TEMPLATE - fix_bool* - START
    bool ok_;
    double retval_ = cppSelf->toDouble(&ok_);
    py_result = Shiboken::makeTuple(retval_, ok_);
    // TEMPLATE - fix_bool* - END
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_toFloat(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toFloat(bool*)const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    // Begin code injection
    
    // TEMPLATE - fix_bool* - START
    bool ok_;
    float retval_ = cppSelf->toFloat(&ok_);
    py_result = Shiboken::makeTuple(retval_, ok_);
    // TEMPLATE - fix_bool* - END
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_toInt(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toInt(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        int retval_ = cppSelf->toInt(&ok_, 10);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // toInt(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        int retval_ = cppSelf->toInt(&ok_, cpp_arg0);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_toInt_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_toInt_TypeError:
        const char* overloads[] = {"bool = 0, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.toInt", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_toLatin1(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toLatin1()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    QByteArray cpp_result = cppSelf->QString::toLatin1();
    py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_toLocal8Bit(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toLocal8Bit()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    QByteArray cpp_result = cppSelf->QString::toLocal8Bit();
    py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_toLong(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toLong(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        long retval_ = cppSelf->toLong(&ok_, 10);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // toLong(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        long retval_ = cppSelf->toLong(&ok_, cpp_arg0);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_toLong_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_toLong_TypeError:
        const char* overloads[] = {"bool = 0, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.toLong", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_toLongLong(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toLongLong(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        qlonglong retval_ = cppSelf->toLongLong(&ok_, 10);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // toLongLong(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        qlonglong retval_ = cppSelf->toLongLong(&ok_, cpp_arg0);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_toLongLong_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_toLongLong_TypeError:
        const char* overloads[] = {"bool = 0, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.toLongLong", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_toLower(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toLower()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QString::toLower();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_toShort(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toShort(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        short retval_ = cppSelf->toShort(&ok_, 10);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // toShort(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        short retval_ = cppSelf->toShort(&ok_, cpp_arg0);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_toShort_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_toShort_TypeError:
        const char* overloads[] = {"bool = 0, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.toShort", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_toUInt(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toUInt(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        uint retval_ = cppSelf->toUInt(&ok_, 10);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // toUInt(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        uint retval_ = cppSelf->toUInt(&ok_, cpp_arg0);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_toUInt_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_toUInt_TypeError:
        const char* overloads[] = {"bool = 0, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.toUInt", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_toULong(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toULong(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        ulong retval_ = cppSelf->toULong(&ok_, 10);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // toULong(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        ulong retval_ = cppSelf->toULong(&ok_, cpp_arg0);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_toULong_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_toULong_TypeError:
        const char* overloads[] = {"bool = 0, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.toULong", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_toULongLong(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toULongLong(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        qulonglong retval_ = cppSelf->toULongLong(&ok_, 10);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // toULongLong(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        qulonglong retval_ = cppSelf->toULongLong(&ok_, cpp_arg0);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_toULongLong_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_toULongLong_TypeError:
        const char* overloads[] = {"bool = 0, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.toULongLong", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_toUShort(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toUShort(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        ushort retval_ = cppSelf->toUShort(&ok_, 10);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // toUShort(bool*,int)const
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        // Begin code injection
        
        // TEMPLATE - fix_bool*,arg - START
        bool ok_;
        ushort retval_ = cppSelf->toUShort(&ok_, cpp_arg0);
        py_result = Shiboken::makeTuple(retval_, ok_);
        // TEMPLATE - fix_bool*,arg - END
        
        // End of code injection


        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_toUShort_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc_toUShort_TypeError:
        const char* overloads[] = {"bool = 0, int = 10", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.toUShort", overloads);
        return 0;
}

static PyObject* SbkQStringFunc_toUpper(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toUpper()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QString::toUpper();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_toUtf8(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toUtf8()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    QByteArray cpp_result = cppSelf->QString::toUtf8();
    py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_trimmed(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // trimmed()const
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QString::trimmed();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQStringFunc_truncate(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // truncate(int)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        cppSelf->QString::truncate(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc_truncate_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQStringFunc_truncate_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.truncate", overloads);
        return 0;
}

static PyMethodDef SbkQStringMethod_compare = {
    "compare", (PyCFunction)SbkQStringFunc_compare, METH_VARARGS|METH_STATIC
};

static PyMethodDef SbkQStringMethod_localeAwareCompare = {
    "localeAwareCompare", (PyCFunction)SbkQStringFunc_localeAwareCompare, METH_VARARGS|METH_STATIC
};

static PyMethodDef SbkQString_methods[] = {
    {"append", (PyCFunction)SbkQStringFunc_append, METH_O},
    {"arg", (PyCFunction)SbkQStringFunc_arg, METH_VARARGS},
    {"at", (PyCFunction)SbkQStringFunc_at, METH_O},
    {"capacity", (PyCFunction)SbkQStringFunc_capacity, METH_NOARGS},
    {"chop", (PyCFunction)SbkQStringFunc_chop, METH_O},
    {"clear", (PyCFunction)SbkQStringFunc_clear, METH_NOARGS},
    SbkQStringMethod_compare,
    {"contains", (PyCFunction)SbkQStringFunc_contains, METH_VARARGS},
    {"count", (PyCFunction)SbkQStringFunc_count, METH_VARARGS},
    {"endsWith", (PyCFunction)SbkQStringFunc_endsWith, METH_VARARGS},
    {"fill", (PyCFunction)SbkQStringFunc_fill, METH_VARARGS},
    {"fromAscii", (PyCFunction)SbkQStringFunc_fromAscii, METH_VARARGS|METH_STATIC},
    {"fromLatin1", (PyCFunction)SbkQStringFunc_fromLatin1, METH_VARARGS|METH_STATIC},
    {"fromLocal8Bit", (PyCFunction)SbkQStringFunc_fromLocal8Bit, METH_VARARGS|METH_STATIC},
    {"fromUtf8", (PyCFunction)SbkQStringFunc_fromUtf8, METH_VARARGS|METH_STATIC},
    {"indexOf", (PyCFunction)SbkQStringFunc_indexOf, METH_VARARGS},
    {"insert", (PyCFunction)SbkQStringFunc_insert, METH_VARARGS},
    {"isEmpty", (PyCFunction)SbkQStringFunc_isEmpty, METH_NOARGS},
    {"isNull", (PyCFunction)SbkQStringFunc_isNull, METH_NOARGS},
    {"isRightToLeft", (PyCFunction)SbkQStringFunc_isRightToLeft, METH_NOARGS},
    {"isSimpleText", (PyCFunction)SbkQStringFunc_isSimpleText, METH_NOARGS},
    {"lastIndexOf", (PyCFunction)SbkQStringFunc_lastIndexOf, METH_VARARGS},
    {"left", (PyCFunction)SbkQStringFunc_left, METH_O},
    {"leftJustified", (PyCFunction)SbkQStringFunc_leftJustified, METH_VARARGS},
    {"leftRef", (PyCFunction)SbkQStringFunc_leftRef, METH_O},
    {"length", (PyCFunction)SbkQStringFunc_length, METH_NOARGS},
    SbkQStringMethod_localeAwareCompare,
    {"mid", (PyCFunction)SbkQStringFunc_mid, METH_VARARGS},
    {"midRef", (PyCFunction)SbkQStringFunc_midRef, METH_VARARGS},
    {"normalized", (PyCFunction)SbkQStringFunc_normalized, METH_VARARGS},
    {"number", (PyCFunction)SbkQStringFunc_number, METH_VARARGS|METH_STATIC},
    {"prepend", (PyCFunction)SbkQStringFunc_prepend, METH_O},
    {"push_back", (PyCFunction)SbkQStringFunc_push_back, METH_O},
    {"push_front", (PyCFunction)SbkQStringFunc_push_front, METH_O},
    {"remove", (PyCFunction)SbkQStringFunc_remove, METH_VARARGS},
    {"repeated", (PyCFunction)SbkQStringFunc_repeated, METH_O},
    {"replace", (PyCFunction)SbkQStringFunc_replace, METH_VARARGS},
    {"reserve", (PyCFunction)SbkQStringFunc_reserve, METH_O},
    {"resize", (PyCFunction)SbkQStringFunc_resize, METH_O},
    {"right", (PyCFunction)SbkQStringFunc_right, METH_O},
    {"rightJustified", (PyCFunction)SbkQStringFunc_rightJustified, METH_VARARGS},
    {"rightRef", (PyCFunction)SbkQStringFunc_rightRef, METH_O},
    {"setNum", (PyCFunction)SbkQStringFunc_setNum, METH_VARARGS},
    {"simplified", (PyCFunction)SbkQStringFunc_simplified, METH_NOARGS},
    {"size", (PyCFunction)SbkQStringFunc_size, METH_NOARGS},
    {"split", (PyCFunction)SbkQStringFunc_split, METH_VARARGS},
    {"squeeze", (PyCFunction)SbkQStringFunc_squeeze, METH_NOARGS},
    {"startsWith", (PyCFunction)SbkQStringFunc_startsWith, METH_VARARGS},
    {"toAscii", (PyCFunction)SbkQStringFunc_toAscii, METH_NOARGS},
    {"toCaseFolded", (PyCFunction)SbkQStringFunc_toCaseFolded, METH_NOARGS},
    {"toDouble", (PyCFunction)SbkQStringFunc_toDouble, METH_NOARGS},
    {"toFloat", (PyCFunction)SbkQStringFunc_toFloat, METH_NOARGS},
    {"toInt", (PyCFunction)SbkQStringFunc_toInt, METH_NOARGS|METH_O},
    {"toLatin1", (PyCFunction)SbkQStringFunc_toLatin1, METH_NOARGS},
    {"toLocal8Bit", (PyCFunction)SbkQStringFunc_toLocal8Bit, METH_NOARGS},
    {"toLong", (PyCFunction)SbkQStringFunc_toLong, METH_NOARGS|METH_O},
    {"toLongLong", (PyCFunction)SbkQStringFunc_toLongLong, METH_NOARGS|METH_O},
    {"toLower", (PyCFunction)SbkQStringFunc_toLower, METH_NOARGS},
    {"toShort", (PyCFunction)SbkQStringFunc_toShort, METH_NOARGS|METH_O},
    {"toUInt", (PyCFunction)SbkQStringFunc_toUInt, METH_NOARGS|METH_O},
    {"toULong", (PyCFunction)SbkQStringFunc_toULong, METH_NOARGS|METH_O},
    {"toULongLong", (PyCFunction)SbkQStringFunc_toULongLong, METH_NOARGS|METH_O},
    {"toUShort", (PyCFunction)SbkQStringFunc_toUShort, METH_NOARGS|METH_O},
    {"toUpper", (PyCFunction)SbkQStringFunc_toUpper, METH_NOARGS},
    {"toUtf8", (PyCFunction)SbkQStringFunc_toUtf8, METH_NOARGS},
    {"trimmed", (PyCFunction)SbkQStringFunc_trimmed, METH_NOARGS},
    {"truncate", (PyCFunction)SbkQStringFunc_truncate, METH_O},
    {0} // Sentinel
};

static PyObject* SbkQString_getattro(PyObject* self, PyObject* name)
{
    if (self) {
        if (SbkBaseWrapper_instanceDict(self)) {
            PyObject* meth = PyDict_GetItem(SbkBaseWrapper_instanceDict(self), name);
            if (meth) {
                Py_INCREF(meth);
                return meth;
            }
        }
        const char* cname = PyString_AS_STRING(name);
        if (strcmp(cname, "compare") == 0)
            return PyCFunction_NewEx(&SbkQStringMethod_compare, self, 0);
        if (strcmp(cname, "localeAwareCompare") == 0)
            return PyCFunction_NewEx(&SbkQStringMethod_localeAwareCompare, self, 0);
    }
    return PyObject_GenericGetAttr(self, name);
}

static PyObject* SbkQStringFunc___add__(PyObject* self, PyObject* arg)
{
    bool isReverse = SbkQString_Check(arg) && !SbkQString_Check(self);
    if (isReverse)
        std::swap(self, arg);

    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (!isReverse
        && SbkBaseWrapper_Check(arg)
        && !PyObject_TypeCheck(arg, self->ob_type)
        && PyObject_HasAttrString(arg, const_cast<char*>("__radd__"))) {
        PyObject* revOpMethod = PyObject_GetAttrString(arg, const_cast<char*>("__radd__"));
        if (revOpMethod && PyCallable_Check(revOpMethod)) {
            py_result = PyObject_CallFunction(revOpMethod, const_cast<char*>("O"), self);
            if (PyErr_Occurred() && (PyErr_ExceptionMatches(PyExc_NotImplementedError) || PyErr_ExceptionMatches(PyExc_AttributeError))) {
                PyErr_Clear();
                Py_XDECREF(py_result);
                py_result = 0;
            }
        }
        Py_XDECREF(revOpMethod);

    }
    // Do not enter here if other object has implemented a reverse operator.
    if (!py_result) {

    if (!isReverse && Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // operator+(QString,const char*)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QString cpp_result = (*cppSelf) + cpp_arg0;
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (isReverse && Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // operator+(const char*,QString) [reverse operator]
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QString cpp_result = cpp_arg0 + (*cppSelf);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<QByteArray & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(arg);
        if (!SbkQByteArray_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // operator+(QString,QByteArray)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QString cpp_result = (*cppSelf) + (*cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (isReverse && Shiboken::Converter<QByteArray & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(arg);
        if (!SbkQByteArray_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // operator+(QByteArray,QString) [reverse operator]
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QString cpp_result = (*cpp_arg0) + (*cppSelf);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (isReverse && Shiboken::Converter<char >::isConvertible(arg)) {
        char cpp_arg0 = Shiboken::Converter<char >::toCpp(arg);
        // operator+(char,QString) [reverse operator]
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QString cpp_result = cpp_arg0 + (*cppSelf);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<char >::isConvertible(arg)) {
        char cpp_arg0 = Shiboken::Converter<char >::toCpp(arg);
        // operator+(QString,char)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QString cpp_result = (*cppSelf) + cpp_arg0;
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<QChar >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(arg);
        if (!SbkQChar_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        // operator+(QString,QChar)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QString cpp_result = (*cppSelf) + (*cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // operator+(QString,QString)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QString cpp_result = (*cppSelf) + (*cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (isReverse && Shiboken::Converter<QChar >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(arg);
        if (!SbkQChar_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        // operator+(QChar,QString) [reverse operator]
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        const QString cpp_result = (*cpp_arg0) + (*cppSelf);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc___add___TypeError;

    } // End of "if (!py_result)"


    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQStringFunc___add___TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar", "PySide.QtCore.QChar", "char", "char", "PySide.QtCore.QByteArray", "PySide.QtCore.QByteArray", "PySide.QtCore.QString", "str", "str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.__add__", overloads);
        return 0;
}

static PyObject* SbkQStringFunc___iadd__(PyObject* self, PyObject* arg)
{
    bool isReverse = SbkQString_Check(arg) && !SbkQString_Check(self);
    if (isReverse)
        std::swap(self, arg);

    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (!isReverse && Shiboken::Converter<QStringRef & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QStringRef* cpp_arg0 = Shiboken::Converter<QStringRef* >::toCpp(arg);
        // operator+=(QStringRef)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        (*cppSelf) += (*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<QLatin1String & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QLatin1String* cpp_arg0 = Shiboken::Converter<QLatin1String* >::toCpp(arg);
        // operator+=(QLatin1String)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        (*cppSelf) += (*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<QChar::SpecialCharacter >::isConvertible(arg)) {
        QChar::SpecialCharacter cpp_arg0 = Shiboken::Converter<QChar::SpecialCharacter >::toCpp(arg);
        // operator+=(QChar::SpecialCharacter)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        (*cppSelf) += cpp_arg0;
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // operator+=(const char*)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        (*cppSelf) += cpp_arg0;
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<QByteArray & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(arg);
        if (!SbkQByteArray_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // operator+=(QByteArray)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        (*cppSelf) += (*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<QChar >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QChar > cpp_arg0_auto_ptr;
        QChar* cpp_arg0 = Shiboken::Converter<QChar* >::toCpp(arg);
        if (!SbkQChar_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QChar >(cpp_arg0);
        // operator+=(QChar)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        (*cppSelf) += (*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (!isReverse && Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // operator+=(QString)
        QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
        (*cppSelf) += (*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQStringFunc___iadd___TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_INCREF(self);
    return self;

    SbkQStringFunc___iadd___TypeError:
        const char* overloads[] = {"PySide.QtCore.QChar", "PySide.QtCore.QChar.SpecialCharacter", "PySide.QtCore.QByteArray", "PySide.QtCore.QLatin1String", "PySide.QtCore.QString", "PySide.QtCore.QStringRef", "str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QString.__iadd__", overloads);
        return 0;
}

// type has number operators
static PyNumberMethods SbkQString_as_number = {
    /*nb_add*/                  (binaryfunc)SbkQStringFunc___add__,
    /*nb_subtract*/             (binaryfunc)0,
    /*nb_multiply*/             (binaryfunc)0,
    /*nb_divide*/               (binaryfunc)0,
    /*nb_remainder*/            (binaryfunc)0,
    /*nb_divmod*/               0,
    /*nb_power*/                0,
    /*nb_negative*/             (unaryfunc)0,
    /*nb_positive*/             (unaryfunc)0,
    /*nb_absolute*/             0,
    /*nb_nonzero*/              0,
    /*nb_invert*/               (unaryfunc)0,
    /*nb_lshift*/               (binaryfunc)0,
    /*nb_rshift*/               (binaryfunc)0,
    /*nb_and*/                  (binaryfunc)0,
    /*nb_xor*/                  (binaryfunc)0,
    /*nb_or*/                   (binaryfunc)0,
    /*nb_coerce*/               0,
    /*nb_int*/                  0,
    /*nb_long*/                 0,
    /*nb_float*/                0,
    /*nb_oct*/                  0,
    /*nb_hex*/                  0,
    /*nb_inplace_add*/          (binaryfunc)SbkQStringFunc___iadd__,
    /*nb_inplace_subtract*/     (binaryfunc)0,
    /*nb_inplace_multiply*/     (binaryfunc)0,
    /*nb_inplace_divide*/       (binaryfunc)0,
    /*nb_inplace_remainder*/    (binaryfunc)0,
    /*nb_inplace_power*/        0,
    /*nb_inplace_lshift*/       (binaryfunc)0,
    /*nb_inplace_rshift*/       (binaryfunc)0,
    /*nb_inplace_and*/          (binaryfunc)0,
    /*nb_inplace_xor*/          (binaryfunc)0,
    /*nb_inplace_or*/           (binaryfunc)0,
    /*nb_floor_divide*/         0,
    /*nb_true_divide*/          0,
    /*nb_inplace_floor_divide*/ 0,
    /*nb_inplace_true_divide*/  0,
    /*nb_index*/                0
};

Py_ssize_t SbkQStringFunc___len__(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    (void)cppSelf; // avoid warnings about unused variables
    // Begin code injection
    
    return cppSelf->count();
    
    // End of code injection
}

PyObject* SbkQStringFunc___getitem__(PyObject* self, Py_ssize_t _i)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    (void)cppSelf; // avoid warnings about unused variables
    // Begin code injection
    
    if (_i < 0 || _i >= cppSelf->size()) {
        PyErr_SetString(PyExc_IndexError, "index out of bounds");
        return 0;
    }
    return Shiboken::Converter<QString >::toPython(cppSelf->at(_i));
    
    // End of code injection
}

int SbkQStringFunc___setitem__(PyObject* self, Py_ssize_t _i, PyObject* _value)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    (void)cppSelf; // avoid warnings about unused variables
    // Begin code injection
    
    cppSelf->remove(_i, 1);
    PyObject* args = Py_BuildValue("(nO)", _i, _value);
    PyObject* result = SbkQStringFunc_insert(self, args);
    Py_DECREF(args);
    Py_XDECREF(result);
    return !result ? -1 : 0;
    
    // End of code injection
}

PyObject* SbkQStringFunc___getslice__(PyObject* self, Py_ssize_t _i1, Py_ssize_t _i2)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;
    QString* cppSelf = Shiboken::Converter<QString* >::toCpp((PyObject*)self);
    (void)cppSelf; // avoid warnings about unused variables
    // Begin code injection
    
    // TEMPLATE - get_slice - START
    QString* sequence;
    Py_ssize_t start, end;
    Py_ssize_t len = cppSelf->count();
    
    if (_i1 > len)
    start = len;
    else if (_i1 < 0)
    start = 0;
    else
    start = _i1;
    
    if (_i2 > len)
    end = len;
    else if (_i2 < 0)
    end = 0;
    else
    end = _i2;
    
    sequence = new QString();
    for (Py_ssize_t i = start; i < end; i++)
    sequence->append(cppSelf->at(i));
    
    return Shiboken::Converter<QString >::toPython(*sequence);
    // TEMPLATE - get_slice - END
    
    // End of code injection
}

static PySequenceMethods PyQString_as_sequence = {
    /*sq_length*/ &SbkQStringFunc___len__,
    /*sq_concat*/ 0,
    /*sq_repeat*/ 0,
    /*sq_item*/ &SbkQStringFunc___getitem__,
    /*sq_slice*/ &SbkQStringFunc___getslice__,
    /*sq_ass_item*/ &SbkQStringFunc___setitem__,
    /*sq_ass_slice*/ 0,
    /*sq_contains*/ 0,
    /*sq_inplace_concat*/ 0,
    /*sq_inplace_repeat*/ 0
};

// Rich comparison
static PyObject* SbkQString_richcompare(PyObject* self, PyObject* other, int op)
{
    bool result = false;
    QString& cpp_self = *Shiboken::Converter<QString* >::toCpp((PyObject*)self);

    switch (op) {
        case Py_NE:
            if (Shiboken::Converter<QByteArray & >::isConvertible(other)) {
                // operator!=(const QByteArray & s) const
                QByteArray* cpp_other = Shiboken::Converter<QByteArray* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QByteArray  >::toCpp(other) : (cpp_self != (*cpp_other));
            } else if (Shiboken::Converter<QLatin1String & >::isConvertible(other)) {
                // operator!=(const QLatin1String & s) const
                QLatin1String* cpp_other = Shiboken::Converter<QLatin1String* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QLatin1String  >::toCpp(other) : (cpp_self != (*cpp_other));
            } else if (Shiboken::Converter<QString & >::isConvertible(other)) {
                // operator!=(const QString & s) const
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QString  >::toCpp(other) : (cpp_self != (*cpp_other));
            } else if (Shiboken::Converter<QStringRef & >::isConvertible(other)) {
                // operator!=(const QStringRef & s2)
                QStringRef* cpp_other = Shiboken::Converter<QStringRef* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QStringRef  >::toCpp(other) : (cpp_self != (*cpp_other));
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator!=(const char * s) const
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self != cpp_other);
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator!=(const char * s1)
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self != cpp_other);
            } else if (Shiboken::Converter<QString >::isConvertible(other)) {
                if (Shiboken::cppObjectIsInvalid(other))
                    return 0;
                std::auto_ptr<QString > cpp_other_auto_ptr;
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp(other);
                if (!SbkQString_Check(other))
                    cpp_other_auto_ptr = std::auto_ptr<QString >(cpp_other);
                result = (cpp_self != (*cpp_other));
            } else goto SbkQString_RichComparison_TypeError;

            break;
        case Py_LT:
            if (Shiboken::Converter<QByteArray & >::isConvertible(other)) {
                // operator<(const QByteArray & s) const
                QByteArray* cpp_other = Shiboken::Converter<QByteArray* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QByteArray  >::toCpp(other) : (cpp_self < (*cpp_other));
            } else if (Shiboken::Converter<QLatin1String & >::isConvertible(other)) {
                // operator<(const QLatin1String & s) const
                QLatin1String* cpp_other = Shiboken::Converter<QLatin1String* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QLatin1String  >::toCpp(other) : (cpp_self < (*cpp_other));
            } else if (Shiboken::Converter<QString & >::isConvertible(other)) {
                // operator<(const QString & s) const
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QString  >::toCpp(other) : (cpp_self < (*cpp_other));
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator<(const char * s) const
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self < cpp_other);
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator<(const char * s1)
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self < cpp_other);
            } else if (Shiboken::Converter<QString >::isConvertible(other)) {
                if (Shiboken::cppObjectIsInvalid(other))
                    return 0;
                std::auto_ptr<QString > cpp_other_auto_ptr;
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp(other);
                if (!SbkQString_Check(other))
                    cpp_other_auto_ptr = std::auto_ptr<QString >(cpp_other);
                result = (cpp_self < (*cpp_other));
            } else goto SbkQString_RichComparison_TypeError;

            break;
        case Py_LE:
            if (Shiboken::Converter<QByteArray & >::isConvertible(other)) {
                // operator<=(const QByteArray & s) const
                QByteArray* cpp_other = Shiboken::Converter<QByteArray* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QByteArray  >::toCpp(other) : (cpp_self <= (*cpp_other));
            } else if (Shiboken::Converter<QLatin1String & >::isConvertible(other)) {
                // operator<=(const QLatin1String & s) const
                QLatin1String* cpp_other = Shiboken::Converter<QLatin1String* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QLatin1String  >::toCpp(other) : (cpp_self <= (*cpp_other));
            } else if (Shiboken::Converter<QString & >::isConvertible(other)) {
                // operator<=(const QString & s) const
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QString  >::toCpp(other) : (cpp_self <= (*cpp_other));
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator<=(const char * s1)
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self <= cpp_other);
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator<=(const char * s2) const
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self <= cpp_other);
            } else if (Shiboken::Converter<QString >::isConvertible(other)) {
                if (Shiboken::cppObjectIsInvalid(other))
                    return 0;
                std::auto_ptr<QString > cpp_other_auto_ptr;
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp(other);
                if (!SbkQString_Check(other))
                    cpp_other_auto_ptr = std::auto_ptr<QString >(cpp_other);
                result = (cpp_self <= (*cpp_other));
            } else goto SbkQString_RichComparison_TypeError;

            break;
        case Py_EQ:
            if (Shiboken::Converter<QByteArray & >::isConvertible(other)) {
                // operator==(const QByteArray & s) const
                QByteArray* cpp_other = Shiboken::Converter<QByteArray* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QByteArray  >::toCpp(other) : (cpp_self == (*cpp_other));
            } else if (Shiboken::Converter<QLatin1String & >::isConvertible(other)) {
                // operator==(const QLatin1String & s) const
                QLatin1String* cpp_other = Shiboken::Converter<QLatin1String* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QLatin1String  >::toCpp(other) : (cpp_self == (*cpp_other));
            } else if (Shiboken::Converter<QString & >::isConvertible(other)) {
                // operator==(const QString & s) const
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QString  >::toCpp(other) : (cpp_self == (*cpp_other));
            } else if (Shiboken::Converter<QStringRef & >::isConvertible(other)) {
                // operator==(const QStringRef & s2)
                QStringRef* cpp_other = Shiboken::Converter<QStringRef* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QStringRef  >::toCpp(other) : (cpp_self == (*cpp_other));
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator==(const char * s) const
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self == cpp_other);
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator==(const char * s1)
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self == cpp_other);
            } else if (Shiboken::Converter<QString >::isConvertible(other)) {
                if (Shiboken::cppObjectIsInvalid(other))
                    return 0;
                std::auto_ptr<QString > cpp_other_auto_ptr;
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp(other);
                if (!SbkQString_Check(other))
                    cpp_other_auto_ptr = std::auto_ptr<QString >(cpp_other);
                result = (cpp_self == (*cpp_other));
            } else goto SbkQString_RichComparison_TypeError;

            break;
        case Py_GT:
            if (Shiboken::Converter<QByteArray & >::isConvertible(other)) {
                // operator>(const QByteArray & s) const
                QByteArray* cpp_other = Shiboken::Converter<QByteArray* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QByteArray  >::toCpp(other) : (cpp_self > (*cpp_other));
            } else if (Shiboken::Converter<QLatin1String & >::isConvertible(other)) {
                // operator>(const QLatin1String & s) const
                QLatin1String* cpp_other = Shiboken::Converter<QLatin1String* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QLatin1String  >::toCpp(other) : (cpp_self > (*cpp_other));
            } else if (Shiboken::Converter<QString & >::isConvertible(other)) {
                // operator>(const QString & s) const
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QString  >::toCpp(other) : (cpp_self > (*cpp_other));
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator>(const char * s1)
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self > cpp_other);
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator>(const char * s2) const
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self > cpp_other);
            } else if (Shiboken::Converter<QString >::isConvertible(other)) {
                if (Shiboken::cppObjectIsInvalid(other))
                    return 0;
                std::auto_ptr<QString > cpp_other_auto_ptr;
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp(other);
                if (!SbkQString_Check(other))
                    cpp_other_auto_ptr = std::auto_ptr<QString >(cpp_other);
                result = (cpp_self > (*cpp_other));
            } else goto SbkQString_RichComparison_TypeError;

            break;
        case Py_GE:
            if (Shiboken::Converter<QByteArray & >::isConvertible(other)) {
                // operator>=(const QByteArray & s) const
                QByteArray* cpp_other = Shiboken::Converter<QByteArray* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QByteArray  >::toCpp(other) : (cpp_self >= (*cpp_other));
            } else if (Shiboken::Converter<QLatin1String & >::isConvertible(other)) {
                // operator>=(const QLatin1String & s) const
                QLatin1String* cpp_other = Shiboken::Converter<QLatin1String* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QLatin1String  >::toCpp(other) : (cpp_self >= (*cpp_other));
            } else if (Shiboken::Converter<QString & >::isConvertible(other)) {
                // operator>=(const QString & s) const
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QString  >::toCpp(other) : (cpp_self >= (*cpp_other));
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator>=(const char * s1)
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self >= cpp_other);
            } else if (Shiboken::Converter<const char * >::isConvertible(other)) {
                // operator>=(const char * s2) const
                const char * cpp_other = Shiboken::Converter<const char * >::toCpp(other);
                result = (cpp_self >= cpp_other);
            } else if (Shiboken::Converter<QString >::isConvertible(other)) {
                if (Shiboken::cppObjectIsInvalid(other))
                    return 0;
                std::auto_ptr<QString > cpp_other_auto_ptr;
                QString* cpp_other = Shiboken::Converter<QString* >::toCpp(other);
                if (!SbkQString_Check(other))
                    cpp_other_auto_ptr = std::auto_ptr<QString >(cpp_other);
                result = (cpp_self >= (*cpp_other));
            } else goto SbkQString_RichComparison_TypeError;

            break;
        default:
            PyErr_SetString(PyExc_NotImplementedError, "operator not implemented.");
            return 0;
    }

    if (result)
        Py_RETURN_TRUE;
    SbkQString_RichComparison_TypeError:
    Py_RETURN_FALSE;

}

extern "C"
{

static long SbkQString_HashFunc(PyObject* obj){
    return QStringCustomHashFunction(Shiboken::Converter<QString >::toCpp(obj));
}

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQString_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtCore.QString",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        &SbkQString_as_number,
    /*tp_as_sequence*/      &PyQString_as_sequence,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             &SbkQString_HashFunc,
    /*tp_call*/             0,
    /*tp_str*/              SbkQStringFunc___str__,
    /*tp_getattro*/         SbkQString_getattro,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      SbkQString_richcompare,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQString_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQString_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QString >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static PyGetSetDef SbkPySide_QtCore_QString_SplitBehavior_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QString_SplitBehavior_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QString_SplitBehavior_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "SplitBehavior",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QString_SplitBehavior_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QString_SplitBehavior_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtCore_QString_SplitBehavior_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

static PyGetSetDef SbkPySide_QtCore_QString_NormalizationForm_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QString_NormalizationForm_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QString_NormalizationForm_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "NormalizationForm",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QString_NormalizationForm_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QString_NormalizationForm_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtCore_QString_NormalizationForm_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

static PyGetSetDef SbkPySide_QtCore_QString_SectionFlag_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QString_SectionFlag_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QString_SectionFlag_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "SectionFlag",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QString_SectionFlag_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QString_SectionFlag_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtCore_QString_SectionFlag_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}


PyAPI_FUNC(void) init_QString(PyObject* module)
{
    SbkPySide_QtCoreTypes[SBK_QSTRING_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQString_Type);


    if (PyType_Ready((PyTypeObject*)&SbkQString_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQString_Type));
    PyModule_AddObject(module, "QString",
        ((PyObject*)&SbkQString_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: SplitBehavior
    SbkPySide_QtCoreTypes[SBK_QSTRING_SPLITBEHAVIOR_IDX] = &SbkPySide_QtCore_QString_SplitBehavior_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QString_SplitBehavior_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QString_SplitBehavior_Type);
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
            "SplitBehavior",((PyObject*)&SbkPySide_QtCore_QString_SplitBehavior_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_SplitBehavior_Type,
        (long) QString::KeepEmptyParts, "KeepEmptyParts");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "KeepEmptyParts", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_SplitBehavior_Type.tp_dict,
        "KeepEmptyParts", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_SplitBehavior_Type,
        (long) QString::SkipEmptyParts, "SkipEmptyParts");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "SkipEmptyParts", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_SplitBehavior_Type.tp_dict,
        "SkipEmptyParts", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QString::SplitBehavior");

    // init enum class: NormalizationForm
    SbkPySide_QtCoreTypes[SBK_QSTRING_NORMALIZATIONFORM_IDX] = &SbkPySide_QtCore_QString_NormalizationForm_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QString_NormalizationForm_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QString_NormalizationForm_Type);
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
            "NormalizationForm",((PyObject*)&SbkPySide_QtCore_QString_NormalizationForm_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_NormalizationForm_Type,
        (long) QString::NormalizationForm_D, "NormalizationForm_D");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "NormalizationForm_D", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_NormalizationForm_Type.tp_dict,
        "NormalizationForm_D", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_NormalizationForm_Type,
        (long) QString::NormalizationForm_C, "NormalizationForm_C");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "NormalizationForm_C", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_NormalizationForm_Type.tp_dict,
        "NormalizationForm_C", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_NormalizationForm_Type,
        (long) QString::NormalizationForm_KD, "NormalizationForm_KD");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "NormalizationForm_KD", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_NormalizationForm_Type.tp_dict,
        "NormalizationForm_KD", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_NormalizationForm_Type,
        (long) QString::NormalizationForm_KC, "NormalizationForm_KC");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "NormalizationForm_KC", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_NormalizationForm_Type.tp_dict,
        "NormalizationForm_KC", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QString::NormalizationForm");

    // init enum class: SectionFlag
    SbkPySide_QtCoreTypes[SBK_QSTRING_SECTIONFLAG_IDX] = &SbkPySide_QtCore_QString_SectionFlag_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QString_SectionFlag_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QString_SectionFlag_Type);
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
            "SectionFlag",((PyObject*)&SbkPySide_QtCore_QString_SectionFlag_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_SectionFlag_Type,
        (long) QString::SectionDefault, "SectionDefault");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "SectionDefault", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_SectionFlag_Type.tp_dict,
        "SectionDefault", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_SectionFlag_Type,
        (long) QString::SectionSkipEmpty, "SectionSkipEmpty");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "SectionSkipEmpty", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_SectionFlag_Type.tp_dict,
        "SectionSkipEmpty", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_SectionFlag_Type,
        (long) QString::SectionIncludeLeadingSep, "SectionIncludeLeadingSep");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "SectionIncludeLeadingSep", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_SectionFlag_Type.tp_dict,
        "SectionIncludeLeadingSep", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_SectionFlag_Type,
        (long) QString::SectionIncludeTrailingSep, "SectionIncludeTrailingSep");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "SectionIncludeTrailingSep", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_SectionFlag_Type.tp_dict,
        "SectionIncludeTrailingSep", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QString_SectionFlag_Type,
        (long) QString::SectionCaseInsensitiveSeps, "SectionCaseInsensitiveSeps");
    PyDict_SetItemString(SbkQString_Type.super.ht_type.tp_dict,
        "SectionCaseInsensitiveSeps", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QString_SectionFlag_Type.tp_dict,
        "SectionCaseInsensitiveSeps", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QString::SectionFlag");


    // Begin code injection
    
    #if PY_VERSION_HEX < 0x03000000
        Shiboken::SbkType<QString>()->tp_as_buffer = &SbkQStringBufferProc;
        Shiboken::SbkType<QString>()->tp_flags |= Py_TPFLAGS_HAVE_GETCHARBUFFER;
    #endif
    
    // End of code injection
    Shiboken::TypeResolver::createValueTypeResolver<QString >("QString");
    Shiboken::TypeResolver::createValueTypeResolver<QString >(typeid(QString).name());
}


} // extern "C"


