#include <QtCore>
#include <QPixmapCache>
#include "snapshotentity.h"
#include "pagemodel.h"

int PageModel::PageIdRole = Qt::UserRole + 1;

PageModel::PageModel (QObject *parent  ) : QAbstractItemModel(parent) {
    setSupportedDragActions(Qt::LinkAction);
    m_thumbnailSize = QSize(150,150);
}

PageModel::~PageModel () {
}

bool PageModel::open(int bookId){
    clear();

    m_record = PageEntity::pagesOfBook(bookId);
    /*
    qDebug() << metaObject()->className() << __func__ << bookId << m_record.size();
    foreach (PageEntity entity , m_record) {
        qDebug() << entity.id() << entity.bookId();
    }
    */

    emit layoutChanged();

    return true;
}

void PageModel::clear() {
    m_record.clear();

    emit layoutChanged();
}

void PageModel::append(int pageId){
    bool found = false;

    foreach (PageEntity entity , m_record) {
        if (entity.id() == pageId) {
            found = true;
        }
    }

    if (!found) {
        PageEntity entity;
        entity.load(Storage::instance(),pageId);
        m_record << entity;
    } else {
        QPixmapCache::remove(pixmapCacheKey(pageId));
    }

    emit layoutChanged();
}

void PageModel::remove(int pageId){
    bool found = false;

    for (int i = 0 ; i < m_record.size() ;i++){
        PageEntity entity = m_record.at(i);
        if (entity.id() == pageId) {
            found = true;
            m_record.removeAt(i);
            break;
        }
    }

    emit layoutChanged();
}

void PageModel::refresh(){
    return layoutChanged();
}

void PageModel::setThumbnailSize(const QSize &val){
    m_thumbnailSize = val;
}

int PageModel::rowCount(const QModelIndex &parent) const{
    int ret;
    if (parent.column() > 0) {
        ret = 0;
    } else {
        ret = m_record.size();
    }

    return ret;
}

int PageModel::columnCount(const QModelIndex &parent) const{
    int ret = 1;
    if (parent.column() > 0)
        ret = 0;

//	qDebug() << __func__  << ret;
    return ret;

}

QModelIndex PageModel::index(int row, int column, const QModelIndex &parent) const{
//	qDebug() << __func__ << row << column;
    if (parent.isValid()) // Nested is not supported
        return QModelIndex();
    return createIndex(row,column);
}

QVariant PageModel::data(const QModelIndex &index, int role) const{
    QVariant v;

    if (index.column() == 0 && index.row() < m_record.size() ){

        if (role == Qt::DisplayRole || role == Qt::EditRole) {
/*
            QString id;
            id.setNum(index.row());
            v = id;
*/
//            v = files.at(index.row()).absoluteFilePath();

        } else if (role == Qt::DecorationRole) {
            PageEntity page = m_record.at(index.row());

            QString key = pixmapCacheKey(page.id());
            QPixmap pixmap;

            if (!QPixmapCache::find(key,&pixmap)){
                SnapshotEntity entity;
                if (entity.find(page.id())){
                    QByteArray snapshot = entity.content();
                    pixmap.loadFromData(snapshot,"PNG");
//                    qDebug() << __func__ << "Origianl size" << pixmap.size();
                    QRect rect(0,0,m_thumbnailSize.width(),m_thumbnailSize.height());
                    pixmap = pixmap.copy(rect);
//                    qDebug() << __func__ << pixmap.size();
                }

                if (pixmap.isNull()) {
                    pixmap = QPixmap (m_thumbnailSize.width()  , m_thumbnailSize.height() );
                    pixmap.fill(QColor(193,193,193,255));
                }

                QPixmapCache::insert(key,pixmap);
            }

            v = pixmap;
        } else if (role == PageIdRole ) {
            PageEntity page = m_record.at(index.row());
            v = page.id();
//            qDebug() << __func__ << index.row() << v;
        }

    }

    return v;
}

QVariant PageModel::headerData(int section, Qt::Orientation orientation, int role) const{
    QVariant header;
    if (orientation == Qt::Horizontal) {
        if (role == Qt::DisplayRole) {
            switch (section) {
                case 0:
                    header = tr("Name");
                    break;
            }
        }
    } else {
        header = QAbstractItemModel::headerData(section, orientation, role);
    }

    return header;
}

QModelIndex PageModel::parent ( const QModelIndex & index ) const{
    Q_UNUSED(index);
//	qDebug() << __func__;
    return QModelIndex();
}

Qt::ItemFlags PageModel::flags ( const QModelIndex & index ) const{
    if (index.isValid())
         return Qt::ItemIsDragEnabled | Qt::ItemIsDropEnabled | Qt::ItemIsSelectable | Qt::ItemIsEnabled;
    return Qt::NoItemFlags	;
}

QString PageModel::pixmapCacheKey(int id) const {

    QString key = QString("page-model-%1-%2-%3")
                  .arg(m_thumbnailSize.width())
                  .arg(m_thumbnailSize.height())
                  .arg(id);

    return key;
}
