# -*- coding: latin-1 -*-
#
#This file is part of Outlaw Solitaire.
#
#Copyright (c) 2010, Toni Knsl
#All rights reserved.
#
#Redistribution and use in source and binary forms, with or without 
#modification, are permitted provided that the following conditions are met:
#
#    * Redistributions of source code must retain the above copyright notice, 
#      this list of conditions and the following disclaimer.
#    * Redistributions in binary form must reproduce the above copyright notice, 
#      this list of conditions and the following disclaimer in the documentation 
#      and/or other materials provided with the distribution.
#    * The name of the authors may not be used to endorse or promote products 
#      derived from this software without specific prior written permission.
#
#THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
#ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
#WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
#DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE 
#FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
#DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
#SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
#CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
#OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
#OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
import sys, os, random, math, pickle, time, pygame, gfxlib, fontlib, guilib
from constants import *

class HighscoreEntry():
    name = None
    points = None
    date = None

    def __init__(self):
        self.name = ''
        self.points = -1
        self.date = ''

    def compare(entry1, entry2):
        return cmp(entry2.points, entry1.points)

class Highscores():
    entries = None
    
    def __init__(self):
        self.entries = [0 for ct in range(9)]

        for ct in range(9):
            self.entries[ct] = HighscoreEntry()

    def sort_entries(self):
        self.entries.sort(HighscoreEntry.compare)

    def add_entry(self, name, points, date):
        self.sort_entries()
        self.entries[8].name = name
        self.entries[8].points = points
        self.entries[8].date = date
        self.sort_entries()

class Settings():
    def __init__(self, mainapp_ref):
        self.pointsystem = mainapp_ref.POINTS_ENGLISH
        self.deck_gfx_key = 'deck_01.png'
        self.background_gfx_key = 'background_blue.jpg'
        self.deckback_gfx_key = 'deck_back_01.png'

class SavedGame():
    def __init__(self):
        self.suit = [[0 for x_ct in range(5)]for y_ct in range(5)]
        self.value = [[0 for x_ct in range(5)]for y_ct in range(5)]
        
        self.dealtcard_suit = -1
        self.dealtcard_value = -1
        
        for y_ct in range(5):
            for x_ct in range(5):
                self.suit[x_ct][y_ct]=-1
                self.value[x_ct][y_ct]=-1

class BlankCard(guilib.PictureBox):
    def __init__(self):
        guilib.PictureBox.__init__(self, gfxlib.CreateSurface((CARDW,CARDH)))

class Card(BlankCard):
    def __init__(self, suit=-1, value=-1):
        BlankCard.__init__(self)

        self.suit = suit
        self.value = value
        self.dealt = False

    def copy_from(self, source):
        self.suit = source.suit
        self.value = source.value
        self.change_image(source.image)

class Deck:
    def __init__(self):        
        self.cards = []
        self.dealt_cards_ct = 0

        for suit_ct in range(4):
            for value_ct in range(13):
                self.cards.append(Card(suit_ct, value_ct))

    def reset(self):
        self.dealt_cards_ct = 0

        for tempcard in self.cards:
            tempcard.dealt = False

    def set_gfx(self, gfx):
        for tempcard in self.cards:
            tempcard.image.blit(gfx, (0,0), (tempcard.value*CARDW,tempcard.suit*CARDH,CARDW,CARDH))
            tempcard.image.set_colorkey(tempcard.image.get_at((0,0)), pygame.RLEACCEL)

    def get_card(self):
        if self.dealt_cards_ct < 52:
            tempcard = self.cards[random.randint(0,51)]
            while tempcard.dealt:
                tempcard = self.cards[random.randint(0,51)]
                
            tempcard.dealt = True
            self.dealt_cards_ct += 1
            
            return tempcard
        else:
            return None

    def get_specific_card(self, suit, value, mark_as_dealt=False):
        for tempcard in self.cards:
            if tempcard.suit == suit and tempcard.value == value:
                if mark_as_dealt:
                    tempcard.dealt = True
                    self.dealt_cards_ct += 1
                return tempcard

class DealtCardEventArea(guilib.Container):
    def __init__(self):
        guilib.Container.__init__(self)

        self.saved_mouseover = False
        self.saved_mousewasdown = False
        self.saved_clicked = False
        
        self.picturebox = guilib.PictureBox(gfxlib.CreateSurface((DEALTCARDEVENTAREAW,480)))
        self.picturebox.hide()
        self.add_widget(self.picturebox, 0, (DEALTCARDEVENTAREAOFFSET))

    def update(self):
        self.saved_mouseover = self.picturebox.mouseover
        self.saved_mousewasdown = self.picturebox.mousewasdown
        self.saved_clicked = self.picturebox.clicked

class DealtCard(guilib.Container):
    STATE_IDLE = 0
    STATE_MOVING_TO_MOUSE = 1
    STATE_MOVING_WITH_MOUSE = 2
    STATE_MOVING_TO_SHOE = 3
    STATE_MOVING_TO_PLAYFIELD = 4
        
    def __init__(self, mainapp_ref, dealtcardeventarea_ref, playfield_ref, shoe_ref, deck_ref):
        guilib.Container.__init__(self)

        self.mainapp_ref = mainapp_ref
        self.eventarea_ref = dealtcardeventarea_ref
        self.playfield_ref = playfield_ref
        self.shoe_ref = shoe_ref
        self.deck_ref = deck_ref

        self.mousedowntime = 0
        self.card = Card()
        self.add_widget(self.card, 1, (0,0))
        
        self.shadow = BlankCard()
        self.shadow.image.set_alpha(128, pygame.RLEACCEL)
        self.shadow.image.fill((0,0,0))
        self.shadow.hide()
        self.add_widget(self.shadow, 0, (DEALTCARDSHADOWOFFSET,DEALTCARDSHADOWOFFSET))

        self.set_center(self.shoe_ref.dealtcardholder.rect.center)        

        self.state = self.STATE_IDLE
        self.card.copy_from(self.deck_ref.get_card())
        
        self.targetcard = None

    def reset(self, suit=-1, value=-1):
        self.state = self.STATE_IDLE
        if suit > -1 and value > -1:
            self.card.copy_from(self.deck_ref.get_specific_card(suit, value, True))
        else:
            self.card.copy_from(self.deck_ref.get_card())
        self.targetcard = None
        self.show()
    
    def update(self):
        if self.state != self.STATE_MOVING_TO_PLAYFIELD:
            if self.eventarea_ref.saved_mousewasdown:
                if self.state != self.STATE_MOVING_WITH_MOUSE:
                    if self.mousedowntime > DEALTCARDMOUSEDOWNGUARD:
                        self.state = self.STATE_MOVING_TO_MOUSE
                        self.mousedowntime = 0
                    else:
                        self.mousedowntime += 1
            else:
                self.mousedowntime = 0
                if self.state == self.STATE_MOVING_WITH_MOUSE or self.state == self.STATE_MOVING_TO_MOUSE:
                    self.mouseoverplayfield = False
                    self.collisionpoint = (pygame.mouse.get_pos()[0], pygame.mouse.get_pos()[1]-CARDH/2)
                    for y_ct in range(5):
                        for x_ct in range(5):
                            if self.playfield_ref.cards[x_ct][y_ct].rect.collidepoint(self.collisionpoint):
                                if self.playfield_ref.cards[x_ct][y_ct].suit == -1:
                                    self.mouseoverplayfield = True
                                    self.targetcard = self.playfield_ref.cards[x_ct][y_ct]
                                    self.state = self.STATE_MOVING_TO_PLAYFIELD
                    if not self.mouseoverplayfield:
                        self.state = self.STATE_MOVING_TO_SHOE

        if self.state == self.STATE_IDLE:
            self.shadow_off()
        elif self.state == self.STATE_MOVING_TO_MOUSE:
            if self.move_to(pygame.Rect((pygame.mouse.get_pos()[0], pygame.mouse.get_pos()[1]-CARDH/2),(1,1))):
                self.state = self.STATE_MOVING_WITH_MOUSE
            self.dirty()
            self.shadow_on()
        elif self.state == self.STATE_MOVING_WITH_MOUSE:
            self.set_mid_bottom(pygame.mouse.get_pos())
            self.dirty()
            self.shadow_on()
        elif self.state == self.STATE_MOVING_TO_PLAYFIELD:
            if self.move_to(self.targetcard.rect):
                self.targetcard.copy_from(self.card)
                self.targetcard.dirty = True
                self.set_center(self.shoe_ref.dealtcardholder.rect.center)

                if self.deck_ref.dealt_cards_ct < 25:
                    self.card.copy_from(self.deck_ref.get_card())
                else:
                    self.hide()
                    self.mainapp_ref.change_state(self.mainapp_ref.STATE_END_GAME_AFTER_UPDATE)

                if not (self.deck_ref.dealt_cards_ct-1) % 8:
                    self.shoe_ref.lower_deck()
                    
                self.state = self.STATE_IDLE
                self.dirty()
            else:
                self.dirty()
                self.shadow_on()
        elif self.state == self.STATE_MOVING_TO_SHOE:
            if self.move_to(self.shoe_ref.dealtcardholder.rect):
                self.set_center(self.shoe_ref.dealtcardholder.rect.center)
                self.dirty()
                self.state = self.STATE_IDLE
            self.dirty()
            self.shadow_on()

    def shadow_on(self):
        if not self.shadow.visible:
            self.shadow.show()

    def shadow_off(self):
        if self.shadow.visible:
            self.shadow.hide()

    def move_to(self, target_rect):
        self.dx = target_rect.centerx - self.rect.centerx
        self.dy = target_rect.centery - self.rect.centery

        self.d = math.sqrt((math.pow(self.dx,2)+math.pow(self.dy,2)))

        if self.d <= DEALTCARDSPEED:
            self.speed = self.d/2
        else:
            self.speed = DEALTCARDSPEED
            
        if self.speed <= 2:
            self.speed = 2

        if self.d<=self.speed:
            return True
            
        self.angle=math.atan2(self.dy, self.dx)
        self.move((self.speed*math.cos(self.angle), self.speed*math.sin(self.angle)))

class Background(guilib.Container):
    def __init__(self):
        guilib.Container.__init__(self)  

        self.picturebox = guilib.PictureBox(gfxlib.CreateSurface((SCREENW, SCREENH)))
        self.add_widget(self.picturebox)

    def change_image(self, image):
        self.picturebox.change_image(image)

class ScoreDisplay(guilib.Container):
    def __init__(self, mainapp_ref):
        guilib.Container.__init__(self)

        self.mainapp_ref = mainapp_ref

        self.rowpoints = [0 for row_ct in range(5)]
        self.colpoints = [0 for col_ct in range(5)]

        self.back = guilib.PictureBox(gfxlib.CreateSurface((714,480)))
        self.add_widget(self.back, 0, (43, 0))

        self.newhighscoreshadow = guilib.Label('NEW HIGH SCORE', self.mainapp_ref.small_font)
        self.add_widget(self.newhighscoreshadow, 0, (589, 71))

        self.newhighscore = guilib.Label('NEW HIGH SCORE', self.mainapp_ref.small_white_font)
        self.add_widget(self.newhighscore, 1, (586, 68))

        self.total = guilib.Label('777$', self.mainapp_ref.big_white_font)
        self.total.image.fill((0,0,0,255))
        self.add_widget(self.total, 0, (640 ,138))

        self.tapscreentoenternameshadow = guilib.MultilineLabel(('TAP SCREEN','TO ENTER NAME'), self.mainapp_ref.small_font, 0, False, 'center')
        self.add_widget(self.tapscreentoenternameshadow, 0, (592, 212))

        self.tapscreentoentername = guilib.MultilineLabel(('TAP SCREEN','TO ENTER NAME'), self.mainapp_ref.small_white_font, 0, False, 'center')
        self.add_widget(self.tapscreentoentername, 1, (589, 209))        

        self.tapscreentocontinueshadow = guilib.MultilineLabel(('TAP SCREEN','FOR NEW GAME'), self.mainapp_ref.small_font, 0, False, 'center')
        self.add_widget(self.tapscreentocontinueshadow, 0, (594, 212))

        self.tapscreentocontinue = guilib.MultilineLabel(('TAP SCREEN','FOR NEW GAME'), self.mainapp_ref.small_white_font, 0, False, 'center')
        self.add_widget(self.tapscreentocontinue, 1, (591, 209))  

        for row_ct in range(5):
            self.rowpoints[row_ct] = guilib.Label('100', self.mainapp_ref.small_white_font)
            self.rowpoints[row_ct].image.fill((0,0,0,255))
            self.add_widget(self.rowpoints[row_ct], 0, (184, row_ct*(CARDH+PLAYFIELDPADY)+68))

        for col_ct in range(5):
            self.colpoints[col_ct] = guilib.Label('100', self.mainapp_ref.small_white_font)
            self.colpoints[col_ct].image.fill((0,0,0,255))
            self.add_widget(self.colpoints[col_ct], 0, ((col_ct*(CARDW+PLAYFIELDPADX))+231, 14))

        self.hide()

    def reset(self):
        for row_ct in range(5):
            self.rowpoints[row_ct].image.fill((0,0,0,255))

        for col_ct in range(5):
            self.colpoints[col_ct].image.fill((0,0,0,255))

        self.total.image.fill((0,0,0,255))            
          
        self.hide()

    def show_new_high_score(self):
        self.show()
        self.back.hide()
        self.tapscreentocontinueshadow.hide()
        self.tapscreentocontinue.hide()

    def show_tap_to_continue(self):
        self.show()
        self.back.hide()        
        self.newhighscoreshadow.hide()
        self.newhighscore.hide()
        self.tapscreentoenternameshadow.hide()
        self.tapscreentoentername.hide()

    def update(self):
        if self.back.clicked:
            self.mainapp_ref.change_state(self.mainapp_ref.STATE_EXIT_SCORE_DISPLAY)

class Playfield(guilib.Container):
    dealtcard_ref = None
    cards = None
    def __init__(self):
        guilib.Container.__init__(self)

        self.cards = [[0 for x_ct in range(5)]for y_ct in range(5)]

        self.card_holder_image = gfxlib.CreateSurface((CARDW,CARDH))
        self.card_holder_image.fill((255,255,255))
        self.card_holder_image.set_alpha(64, pygame.RLEACCEL)

        for y_ct in range(5):
            for x_ct in range(5):
                self.cards[x_ct][y_ct]=Card()
                self.cards[x_ct][y_ct].change_image(self.card_holder_image)
                self.add_widget(self.cards[x_ct][y_ct], 0, (((CARDW+PLAYFIELDPADX)*x_ct,(CARDH+PLAYFIELDPADY)*y_ct)))

        self.set_top_left((PLAYFIELDOFFSETX, PLAYFIELDOFFSETY))

    def reset(self):
        for y_ct in range(5):
            for x_ct in range(5):
                self.cards[x_ct][y_ct].change_image(self.card_holder_image)
                self.cards[x_ct][y_ct].suit = -1
                self.cards[x_ct][y_ct].value = -1

    def update(self):
         for y_ct in range(5):
            for x_ct in range(5):
                if self.cards[x_ct][y_ct].clicked:
                    if self.dealtcard_ref.state != self.dealtcard_ref.STATE_MOVING_WITH_MOUSE and self.cards[x_ct][y_ct].suit == -1:
                        self.dealtcard_ref.targetcard = self.cards[x_ct][y_ct]
                        self.dealtcard_ref.state = self.dealtcard_ref.STATE_MOVING_TO_PLAYFIELD

class Shoe(guilib.Container):
    def __init__(self):
        guilib.Container.__init__(self) 

        self.dealtcardholder = BlankCard()
        self.dealtcardholder.hide()
        
        self.add_widget(self.dealtcardholder, 0, (0, CARDH+SHOEPADY))

        self.deckbacks = [0 for x_ct in range(3)]
        for ct in range(3):       
            self.deckbacks[ct] = BlankCard()
            self.add_widget(self.deckbacks[ct], ct, (ct*2, ct*2))
        self.deck_height = 3

        self.set_top_left((SHOEOFFSETX, SHOEOFFSETY))

    def reset(self):
        self.deck_height = 3
        for temp_deckback in self.deckbacks:   
            temp_deckback.show()        

    def change_image(self, image):
        for temp_deckback in self.deckbacks:   
            temp_deckback.change_image(image)
            temp_deckback.image.set_colorkey(temp_deckback.image.get_at((0,0)), pygame.RLEACCEL)

    def lower_deck(self):
        self.deckbacks[self.deck_height-1].hide()
        self.deck_height -= 1

class LeftMenu(guilib.Container):
    STATE_IN = 0
    STATE_SLIDING_OUT = 1
    STATE_OUT = 2
    STATE_SLIDING_IN = 3
    STATE_YES_NO_SHOW = 4

    WANTED_STATE_POINTS = 0
    WANTED_STATE_HIGHSCORES = 1
    def __init__(self, mainapp_ref):
        guilib.Container.__init__(self)

        self.mainapp_ref = mainapp_ref

        self.temp_pointsystem = self.mainapp_ref.pointsystem

        self.back = guilib.PictureBox(gfxlib.LoadGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUBACKGROUNDGFXFILE))
        self.add_widget(self.back, 0)

        self.backshadow = guilib.PictureBox(gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUBACKGROUNDSHADOWGFXFILE))
        self.add_widget(self.backshadow, 0, LEFTMENUBACKGROUNDSHADOWOFFSET)

        self.slideoutinactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUSLIDEOUTINACTIVEGFXFILE)
        self.slideoutactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUSLIDEOUTACTIVEGFXFILE)
        self.slideout = guilib.ImageButton(self.slideoutactivegfx, self.slideoutinactivegfx)
        self.add_widget(self.slideout, 1, LEFTMENUSLIDEOUTOFFSET, LEFTMENUSLIDEOUTEVENTAREA, (3,0))

        self.slideininactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUSLIDEININACTIVEGFXFILE)
        self.slideinactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUSLIDEINACTIVEGFXFILE)
        self.slidein = guilib.ImageButton(self.slideinactivegfx, self.slideininactivegfx)
        self.add_widget(self.slidein, 1, LEFTMENUSLIDEINOFFSET, LEFTMENUSLIDEINEVENTAREA, (4,0))       

        self.newgame = guilib.TextButton2('NEW GAME', self.mainapp_ref.big_red_font, self.mainapp_ref.big_font)
        self.add_widget(self.newgame, 1, LEFTMENUNEWGAMEOFFSET)

        self.wanted = guilib.PictureBox(gfxlib.CreateSurface((310,365)))
        self.add_widget(self.wanted, 1, LEFTMENUWANTEDOFFSET)
        self.wanted.hide()        

        self.tapforhiscores = guilib.PictureBox(gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUTAPFORHISCORESGFXFILE))
        self.add_widget(self.tapforhiscores, 1, LEFTMENUTAPFOROFFSET)        

        self.tapforpoints = guilib.PictureBox(gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUTAPFORPOINTSGFXFILE))
        self.add_widget(self.tapforpoints, 1, LEFTMENUTAPFOROFFSET) 
        self.tapforpoints.hide()
        
        self.americantitle = guilib.Label(LEFTMENUAMERICANTITLE, self.mainapp_ref.big_font)
        self.add_widget(self.americantitle, 1, LEFTMENUAMERICANTITLEOFFSET)   

        self.englishtitle = guilib.Label(LEFTMENUENGLISHTITLE, self.mainapp_ref.big_font)
        self.add_widget(self.englishtitle, 1, LEFTMENUENGLISHTITLEOFFSET)
        
        self.handnames = guilib.MultilineLabel(LEFTMENUHANDNAMES, self.mainapp_ref.small_font)
        self.add_widget(self.handnames, 1, LEFTMENUHANDNAMESOFFSET)

        self.americanpoints = guilib.MultilineLabel(LEFTMENUAMERICANPOINTS, self.mainapp_ref.small_font, 0, False, 'right')
        self.add_widget(self.americanpoints, 1, LEFTMENUAMERICANPOINTSOFFSET)

        self.englishpoints = guilib.MultilineLabel(LEFTMENUENGLISHPOINTS, self.mainapp_ref.small_font, 0, False, 'right')
        self.add_widget(self.englishpoints, 1, LEFTMENUENGLISHPOINTSOFFSET)        

        if self.temp_pointsystem == self.mainapp_ref.POINTS_AMERICAN:
            self.englishtitle.hide()
            self.englishpoints.hide()
        elif self.temp_pointsystem == self.mainapp_ref.POINTS_ENGLISH:
            self.americantitle.hide()
            self.americanpoints.hide()
        else:
            raise RuntimeError('LeftMenu: __init__: Invalid point system!')

        self.highscoretitle = guilib.Label('MOST WANTED', self.mainapp_ref.big_font)
        self.add_widget(self.highscoretitle, 1, LEFTMENUHIGHSCORETITLEOFFSET)
        self.highscoretitle.hide()

        self.americanhighscorenamelist = ['WWWWW' for ct in range(9)]
        self.americanhighscorepointlist = ['777$' for ct in range(9)]
        self.americanhighscoredatelist = ['27.07.77' for ct in range(9)]

        self.englishhighscorenamelist = ['WWWWW' for ct in range(9)]
        self.englishhighscorepointlist = ['777$' for ct in range(9)]
        self.englishhighscoredatelist = ['27.07.77' for ct in range(9)]

        self.americanhighscorenames = guilib.MultilineLabel(self.americanhighscorenamelist, self.mainapp_ref.small_font)
        self.add_widget(self.americanhighscorenames, 1, LEFTMENUHIGHSCORENAMESOFFSET)
        self.americanhighscorenames.hide()

        self.englishhighscorenames = guilib.MultilineLabel(self.englishhighscorenamelist, self.mainapp_ref.small_font)
        self.add_widget(self.englishhighscorenames, 1, LEFTMENUHIGHSCORENAMESOFFSET)
        self.englishhighscorenames.hide()

        self.americanhighscorepoints = guilib.MultilineLabel(self.americanhighscorepointlist, self.mainapp_ref.small_font, 0, False, 'right')
        self.add_widget(self.americanhighscorepoints, 1, LEFTMENUHIGHSCOREPOINTSOFFSET)
        self.americanhighscorepoints.hide()

        self.englishhighscorepoints = guilib.MultilineLabel(self.englishhighscorepointlist, self.mainapp_ref.small_font, 0, False, 'right')
        self.add_widget(self.englishhighscorepoints, 1, LEFTMENUHIGHSCOREPOINTSOFFSET)
        self.englishhighscorepoints.hide()

        self.americanhighscoredates = guilib.MultilineLabel(self.americanhighscoredatelist, self.mainapp_ref.small_font, 0, False, 'right')
        self.add_widget(self.americanhighscoredates, 1, LEFTMENUHIGHSCOREDATESOFFSET)
        self.americanhighscoredates.hide()

        self.englishhighscoredates = guilib.MultilineLabel(self.englishhighscoredatelist, self.mainapp_ref.small_font, 0, False, 'right')
        self.add_widget(self.englishhighscoredates, 1, LEFTMENUHIGHSCOREDATESOFFSET)
        self.englishhighscoredates.hide()

        self.settingstitle = guilib.Label('SETTINGS', self.mainapp_ref.big_font)
        self.add_widget(self.settingstitle, 1, LEFTMENUSETTINGSTITLEOFFSET)

        self.settingssubtitle = guilib.Label('TAP TO CYCLE THROUGH CHOICES', self.mainapp_ref.small_font)
        self.add_widget(self.settingssubtitle, 1, LEFTMENUSETTINGSSUBTITLEOFFSET)

        self.carddecktitle = guilib.Label('CARDS & DECK', self.mainapp_ref.small_font)
        self.add_widget(self.carddecktitle, 1, LEFTMENUCARDDECKTITLEOFFSET)

        self.card1 = Card()
        self.card1.copy_from(self.mainapp_ref.deck.get_specific_card(2, 10))
        self.add_widget(self.card1, 1, LEFTMENUCARD1OFFSET)

        self.card2 = Card()
        self.card2.copy_from(self.mainapp_ref.deck.get_specific_card(3, 11))
        self.add_widget(self.card2, 1, LEFTMENUCARD2OFFSET)

        self.card3 = Card()
        self.card3.copy_from(self.mainapp_ref.deck.get_specific_card(0, 12))
        self.add_widget(self.card3, 1, LEFTMENUCARD3OFFSET)

        self.card4 = Card()
        self.card4.copy_from(self.mainapp_ref.deck.get_specific_card(1, 0))
        self.add_widget(self.card4, 1, LEFTMENUCARD4OFFSET)

        self.deckback = BlankCard()
        self.deckback.change_image(self.mainapp_ref.shoe.deckbacks[0].image)
        self.add_widget(self.deckback, 1, LEFTMENUDECKBACKOFFSET)

        self.backgroundtitle = guilib.Label('BACKGROUND', self.mainapp_ref.small_font)
        self.add_widget(self.backgroundtitle, 1, LEFTMENUBACKGROUNDTITLEOFFSET)         

        self.background = guilib.PictureBox(gfxlib.CreateSurface((314,60)))
        self.background.image.blit(self.mainapp_ref.background.picturebox.image, (0,0), (0,0,314,60))
        self.add_widget(self.background, 1, LEFTMENUBACKGROUNDOFFSET) 

        self.pointsystemtitle = guilib.Label('POINT SYSTEM', self.mainapp_ref.small_font)
        self.add_widget(self.pointsystemtitle, 1, LEFTMENUPOINTSYSTEMTITLEOFFSET)  

        self.americanflag = gfxlib.LoadGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUAMERICANFLAGGFXFILE)
        self.englishflag = gfxlib.LoadGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUENGLISHFLAGGFXFILE)
        if self.temp_pointsystem == self.mainapp_ref.POINTS_AMERICAN:
            self.pointsystem = guilib.PictureBox(self.americanflag)
        elif self.temp_pointsystem == self.mainapp_ref.POINTS_ENGLISH:
            self.pointsystem = guilib.PictureBox(self.englishflag)
        else:
            raise RuntimeError('LeftMenu: __init__: Invalid point system!')
        self.add_widget(self.pointsystem, 1, LEFTMENUPOINTSYSTEMOFFSET)
        
        self.update_highscores()        

        self.yesnoback = guilib.PictureBox(gfxlib.CreateSurface((SCREENW-84,SCREENH)))
        self.yesnoback.image.fill((0,0,0))
        self.yesnoback.image.set_alpha(224, pygame.RLEACCEL)
        self.add_widget(self.yesnoback, 2,(42,0))
        self.yesnoback.hide()

        self.yesnotext = guilib.MultilineLabel(('CHANGING POINT SYSTEM REQUIRES STARTING A NEW GAME.', 'START NEW GAME OR CONTINUE WITH EXISTING POINT SYSTEM?'), self.mainapp_ref.small_white_font, 0, False, 'center')
        self.add_widget(self.yesnotext, 3, ((SCREENW-self.yesnotext.rect.width)/2,100))
        self.yesnotext.hide()

        self.yes = guilib.TextButton2('NEW GAME', self.mainapp_ref.big_red_font, self.mainapp_ref.big_white_font)
        self.add_widget(self.yes, 3, (78,300))
        self.yes.hide()

        self.no = guilib.TextButton2('CONTINUE', self.mainapp_ref.big_red_font, self.mainapp_ref.big_white_font)
        self.add_widget(self.no, 3, (589,300))
        self.no.hide()

        self.set_top_left((-LEFTMENUSLIDEX, 0))

        self.state = self.STATE_IN
        self.wanted_state = self.WANTED_STATE_POINTS

    def update_highscores(self):
        for ct in range(9):
            self.americanhighscorenamelist[ct] = self.mainapp_ref.americanhighscores.entries[ct].name
            self.americanhighscorepointlist[ct] = str(self.mainapp_ref.americanhighscores.entries[ct].points)+'$'
            self.americanhighscoredatelist[ct] = self.mainapp_ref.americanhighscores.entries[ct].date

        self.americanhighscorenames.change_text(self.americanhighscorenamelist)
        self.americanhighscorepoints.change_text(self.americanhighscorepointlist)
        self.americanhighscoredates.change_text(self.americanhighscoredatelist)

        for ct in range(9):
            self.englishhighscorenamelist[ct] = self.mainapp_ref.englishhighscores.entries[ct].name
            self.englishhighscorepointlist[ct] = str(self.mainapp_ref.englishhighscores.entries[ct].points)+'$'
            self.englishhighscoredatelist[ct] = self.mainapp_ref.englishhighscores.entries[ct].date

        self.englishhighscorenames.change_text(self.englishhighscorenamelist)
        self.englishhighscorepoints.change_text(self.englishhighscorepointlist)
        self.englishhighscoredates.change_text(self.englishhighscoredatelist)

    def slide(self, targetx):
        self.d = abs(targetx - self.rect.left)

        if self.d <= LEFTMENUSPEED:
            speed = self.d/2
        else:
            speed = LEFTMENUSPEED
            
        if speed <= 2:
            speed = 2

        if self.d <= speed:
            return True

        if targetx < 0:
            speed = -speed

        self.move((speed,0))

    def update(self):
        if self.state == self.STATE_YES_NO_SHOW:
            if self.yes.clicked:
                self.yesnoback.hide()
                self.yesnotext.hide()
                self.yes.hide()
                self.no.hide()
                self.mainapp_ref.pointsystem = self.temp_pointsystem
                self.mainapp_ref.save_settings()
                self.slidein.show()
                self.state = self.STATE_SLIDING_IN
                self.mainapp_ref.change_state(self.mainapp_ref.STATE_NEW_GAME)
            elif self.no.clicked:
                self.yesnoback.hide()
                self.yesnotext.hide()
                self.yes.hide()
                self.no.hide()
                self.temp_pointsystem = self.mainapp_ref.pointsystem
                self.refresh_wanted_and_flag()
                self.mainapp_ref.save_settings()
                self.slidein.show()
                self.state = self.STATE_SLIDING_IN
                self.mainapp_ref.change_state(self.mainapp_ref.STATE_GAME)
        elif self.state == self.STATE_OUT:
            if self.slidein.clicked:
                if self.temp_pointsystem != self.mainapp_ref.pointsystem and self.mainapp_ref.deck.dealt_cards_ct > 1:
                    self.yesnoback.show()
                    self.yesnotext.show()
                    self.yes.show()
                    self.no.show()
                    self.slidein.hide()
                    self.state = self.STATE_YES_NO_SHOW
                else:
                    self.mainapp_ref.pointsystem = self.temp_pointsystem
                    self.mainapp_ref.save_settings()
                    self.state = self.STATE_SLIDING_IN
                    self.mainapp_ref.change_state(self.mainapp_ref.STATE_GAME)

            elif self.newgame.clicked:
                self.mainapp_ref.pointsystem = self.temp_pointsystem
                self.mainapp_ref.save_settings()
                self.state = self.STATE_SLIDING_IN
                self.mainapp_ref.change_state(self.mainapp_ref.STATE_NEW_GAME)
            elif self.wanted.clicked:
                if self.wanted_state == self.WANTED_STATE_POINTS:
                    self.wanted_state = self.WANTED_STATE_HIGHSCORES
                elif self.wanted_state == self.WANTED_STATE_HIGHSCORES:
                    self.wanted_state = self.WANTED_STATE_POINTS
                self.refresh_wanted_and_flag()
            elif self.pointsystem.clicked:
                if self.temp_pointsystem == self.mainapp_ref.POINTS_AMERICAN:
                    self.temp_pointsystem = self.mainapp_ref.POINTS_ENGLISH                      
                elif self.temp_pointsystem == self.mainapp_ref.POINTS_ENGLISH:
                    self.temp_pointsystem = self.mainapp_ref.POINTS_AMERICAN
                self.refresh_wanted_and_flag()                        
            elif self.background.clicked:
                tempgfx = self.mainapp_ref.backgroundgfxs.get_next_gfx()
                if tempgfx:
                    self.mainapp_ref.background.change_image(tempgfx)
                else:
                    tempgfx = self.mainapp_ref.backgroundgfxs.get_first_gfx()
                    self.mainapp_ref.background.change_image(tempgfx)
                self.background.image.blit(self.mainapp_ref.background.picturebox.image, (0,0), (0,0,314,60))
                self.background.dirty = True

            elif self.deckback.clicked:
                tempgfx = self.mainapp_ref.deckbackgfxs.get_next_gfx()
                if tempgfx:
                    self.mainapp_ref.shoe.change_image(tempgfx)
                else:
                    tempgfx = self.mainapp_ref.deckbackgfxs.get_first_gfx()
                    self.mainapp_ref.shoe.change_image(tempgfx)
                self.deckback.change_image(self.mainapp_ref.shoe.deckbacks[0].image)

            elif self.card1.clicked or self.card2.clicked or self.card3.clicked or self.card4.clicked:
                tempgfx = self.mainapp_ref.deckgfxs.get_next_gfx()
                if tempgfx:
                    self.mainapp_ref.deck.set_gfx(tempgfx)
                else:
                    tempgfx = self.mainapp_ref.deckgfxs.get_first_gfx()
                    self.mainapp_ref.deck.set_gfx(tempgfx)
                self.mainapp_ref.playfield.dirty()
                self.mainapp_ref.dealtcard.dirty()
                self.card1.dirty = True
                self.card2.dirty = True
                self.card3.dirty = True
                self.card4.dirty = True
        elif self.state == self.STATE_IN:
            if self.slideout.clicked:
                if self.state == self.STATE_IN:
                    self.state = self.STATE_SLIDING_OUT
                    self.temp_pointsystem = self.mainapp_ref.pointsystem
                    self.mainapp_ref.change_state(self.mainapp_ref.STATE_LEFT_MENU)            
        elif self.state == self.STATE_SLIDING_OUT:
            if self.slide(0):
                self.state = self.STATE_OUT
                self.set_top_left((0,0))
                
            self.dirty()
        elif self.state == self.STATE_SLIDING_IN:
            if self.slide(-LEFTMENUSLIDEX):
                self.state = self.STATE_IN
                self.set_top_left((-LEFTMENUSLIDEX,0))

            self.dirty()

    def refresh_wanted_and_flag(self):
        self.tapforpoints.hide()
        self.tapforhiscores.hide()
        
        self.highscoretitle.hide()

        self.handnames.hide()

        self.americantitle.hide()
        self.americanpoints.hide()        

        self.englishtitle.hide()
        self.englishpoints.hide()

        self.americanhighscorenames.hide()
        self.americanhighscorepoints.hide()
        self.americanhighscoredates.hide()

        self.englishhighscorenames.hide()
        self.englishhighscorepoints.hide()
        self.englishhighscoredates.hide()
        
        if self.temp_pointsystem == self.mainapp_ref.POINTS_ENGLISH:
            self.pointsystem.change_image(self.englishflag)
            if self.wanted_state == self.WANTED_STATE_POINTS:
                self.tapforhiscores.show()

                self.englishtitle.show()
                self.englishpoints.show()
                self.handnames.show()
            elif self.wanted_state == self.WANTED_STATE_HIGHSCORES:
                self.tapforpoints.show()
                
                self.highscoretitle.show()
                self.englishhighscorenames.show()
                self.englishhighscorepoints.show()
                self.englishhighscoredates.show()                        
        elif self.temp_pointsystem == self.mainapp_ref.POINTS_AMERICAN:
            self.pointsystem.change_image(self.americanflag)
            if self.wanted_state == self.WANTED_STATE_POINTS:
                self.tapforhiscores.show()

                self.americantitle.show()
                self.americanpoints.show()
                self.handnames.show()
            elif self.wanted_state == self.WANTED_STATE_HIGHSCORES:
                self.tapforpoints.show()

                self.highscoretitle.show()
                self.americanhighscorenames.show()
                self.americanhighscorepoints.show()
                self.americanhighscoredates.show()

                
class RightMenu(guilib.Container):
    def __init__(self, main_app_ref):
        guilib.Container.__init__(self)

        self.main_app_ref = main_app_ref

        self.closeinactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,RIGHTMENUGFXDIR), RIGHTMENUCLOSEINACTIVEGFXFILE)
        self.closeactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,RIGHTMENUGFXDIR), RIGHTMENUCLOSEACTIVEGFXFILE)
        self.close = guilib.ImageButton(self.closeactivegfx, self.closeinactivegfx)
        self.add_widget(self.close, 0, RIGHTMENUCLOSEOFFSET, RIGHTMENUCLOSEEVENTAREA)  

        self.helpinactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,RIGHTMENUGFXDIR), RIGHTMENUHELPINACTIVEGFXFILE)
        self.helpactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,RIGHTMENUGFXDIR), RIGHTMENUHELPACTIVEGFXFILE)
        self.help = guilib.ImageButton(self.helpactivegfx, self.helpinactivegfx)
        self.add_widget(self.help, 0, RIGHTMENUHELPOFFSET, RIGHTMENUHELPEVENTAREA)

        self.toggelinactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,RIGHTMENUGFXDIR), RIGHTMENUTOGGLEINACTIVEGFXFILE)
        self.toggleactivegfx = gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,RIGHTMENUGFXDIR), RIGHTMENUTOGGLEACTIVEGFXFILE)
        self.toggle = guilib.ImageButton(self.toggleactivegfx, self.toggelinactivegfx)
        self.add_widget(self.toggle, 0, RIGHTMENUTOGGLEOFFSET, RIGHTMENUTOGGLEEVENTAREA)      

        self.back = guilib.PictureBox(gfxlib.CreateSurface((SCREENW,SCREENH)))
        self.back.image.fill((0,0,0))
        self.back.image.set_alpha(224, pygame.RLEACCEL)
        self.add_widget(self.back, 1)
        self.back.hide()

        self.backtext = guilib.MultilineLabel(('GAME PAUSED.', 'TAP SCREEN TO RETURN TO GAME.'), self.main_app_ref.big_white_font, 0, False, 'center')
        self.add_widget(self.backtext, 3, ((WINDOWEDSCREENW-self.backtext.rect.width)/2, (WINDOWEDSCREENH-self.backtext.rect.height)/2))
        self.backtext.hide()

    def update(self):
        if self.main_app_ref.windowed:
            if self.back.clicked:
                self.main_app_ref.set_fullscreen_mode()
        else:
            if self.close.clicked:
                self.main_app_ref.change_state(self.main_app_ref.STATE_EXIT)
            elif self.help.clicked:
                if self.main_app_ref.helpdisplay.back.visible:
                    self.main_app_ref.leftmenu.slideout.show()
                    if self.main_app_ref.leftmenu.state != self.main_app_ref.leftmenu.STATE_YES_NO_SHOW:
                        self.main_app_ref.leftmenu.slidein.show()
                    self.main_app_ref.helpdisplay.hide()
                else:
                    self.main_app_ref.leftmenu.slideout.hide()
                    self.main_app_ref.leftmenu.slidein.hide()
                    self.main_app_ref.helpdisplay.show()
            elif self.toggle.clicked:
                self.main_app_ref.set_window_mode()

class RightMenuBack(guilib.Container):
    def __init__(self):
        guilib.Container.__init__(self)

        self.side = guilib.PictureBox(gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,RIGHTMENUGFXDIR), RIGHTMENUBACKGROUNDGFXFILE))
        self.add_widget(self.side, 0, (SCREENW-self.side.rect.width, 0))

class HelpDisplay(guilib.Container):
    def __init__(self, main_app_ref):
        guilib.Container.__init__(self)

        self.main_app_ref = main_app_ref            

        self.helptextlist = []
        self.helptextlist.append('OUTLAW SOLITAIRE IS A GAME OF POKER SQUARES. THE AIM OF THE')
        self.helptextlist.append('GAME IS TO CREATE BEST POSSIBLE POKER HANDS HORIZONTALLY AND')
        self.helptextlist.append('VERTICALLY BY PLACING 25 RANDOMLY DEALT CARDS TO A 5X5 GRID.')
        self.helptextlist.append('PLACE CARDS BY TAPPING THE PLAYFIELD GRID OR BY DRAGGING THE')
        self.helptextlist.append('CARDS TO THE PLAYFIELD.')
        self.helptextlist.append('IN-GAME MENU CAN BE ACCESSED BY TAPPING THE LEFT SIDE BAR')
        self.helptextlist.append('WHEN THE ARROW ICON IS VISIBLE. HAND VALUES AND HIGH SCORES')
        self.helptextlist.append('CAN BE CHECKED FROM THE MENU. ALSO SOME GRAPHICS AND POINT')
        self.helptextlist.append('SYSTEM CAN BE CHANGED VIA THE MENU.')
        self.helptextlist.append('BUTTONS ON THE RIGHT SIDE BAR CAN BE USED TO EXIT THE GAME,')
        self.helptextlist.append('DISPLAY THIS HELP AND TO ICONIFY THE GAME SCREEN.')
        self.helptextlist.append('CODE LICENSED UNDER 3-CLAUSE BSD LICENSE, GRAPHICS UNDER')
        self.helptextlist.append('CC-BY-SA V3.0. SEE COPYING FILE FOR MORE INFORMATION.')
        
        self.back = guilib.PictureBox(gfxlib.CreateSurface((SCREENW-84,SCREENH)))
        self.back.image.fill((0,0,0))
        self.back.image.set_alpha(224, pygame.RLEACCEL)
        self.add_widget(self.back, 0, (42,0))

        self.helptext = guilib.MultilineLabel(self.helptextlist, self.main_app_ref.small_white_font, 0, False, 'left')
        self.add_widget(self.helptext, 1, ((SCREENW-self.helptext.rect.width)/2, 0))

        self.exit = guilib.TextButton2('EXIT HELP', self.main_app_ref.big_red_font, self.main_app_ref.big_white_font)
        self.add_widget(self.exit, 1, ((SCREENW-self.exit.rect.width)/2, 421))

        self.hide()

    def update(self):
        if self.exit.clicked:
            self.main_app_ref.leftmenu.slideout.show()
            if self.main_app_ref.leftmenu.state != self.main_app_ref.leftmenu.STATE_YES_NO_SHOW:
                self.main_app_ref.leftmenu.slidein.show()
            self.hide()

class Splash(guilib.Container):    
    def __init__(self, mainapp_ref):
        guilib.Container.__init__(self)

        self.mainapp_ref = mainapp_ref        

        self.back = guilib.PictureBox(gfxlib.LoadGfx(os.path.join(GFXBASEDIR,SPLASHGFXDIR), SPLASHFILE))
        self.add_widget(self.back, 0)

        self.newgame = guilib.TextButton2("NEW GAME", self.mainapp_ref.big_red_font, self.mainapp_ref.big_font)
        self.add_widget(self.newgame, 1, SPLASHNEWGAMEOFFSET)

        self.continuegame = guilib.TextButton2("CONTINUE GAME", self.mainapp_ref.big_red_font, self.mainapp_ref.big_font)
        self.add_widget(self.continuegame, 1, SPLASHCONTINUEGAMEOFFSET)
        if self.mainapp_ref.savedgame_exists:
            pass
        else:
            self.continuegame.hide()

    def update(self):
        if self.newgame.clicked:
            self.hide()
            self.mainapp_ref.change_state(self.mainapp_ref.STATE_NEW_GAME)
        elif self.continuegame.clicked and self.mainapp_ref.savedgame_exists:
            self.hide()
            self.mainapp_ref.change_state(self.mainapp_ref.STATE_CONTINUE)

class HighscoreInput(guilib.Container):
    STATE_IN = 0
    STATE_SLIDING_OUT = 1
    STATE_OUT = 2
    STATE_SLIDING_IN = 3
    
    def __init__(self, mainapp_ref):
        guilib.Container.__init__(self)

        self.mainapp_ref = mainapp_ref  

        self.state = self.STATE_IN

        self.charbuttons = {}
        self.playername = ''

        self.back = guilib.PictureBox(gfxlib.LoadGfx(os.path.join(GFXBASEDIR,LEFTMENUGFXDIR), LEFTMENUBACKGROUNDGFXFILE))
        self.add_widget(self.back, 0)

        self.title = guilib.Label('WANTED', self.mainapp_ref.big_font)
        self.add_widget(self.title, 1, (65+95,23+20))

        self.silhouette = guilib.PictureBox(gfxlib.LoadAlphaGfx(os.path.join(GFXBASEDIR,HIGHSCOREINPUTGFXDIR), HIGHSCOREINPUTSILHOUETTEGFXFILE))
        self.add_widget(self.silhouette, 1, (171,132))

        self.subtitle = guilib.Label('DEAD OR ALIVE', self.mainapp_ref.small_font)
        self.add_widget(self.subtitle, 1, (65+79,48+23+20))

        self.entry = guilib.Label('WWWWW', self.mainapp_ref.big_font)
        self.entry.image.fill((0,0,0,0))
        self.add_widget(self.entry, 1, (147,265))

        self.score = guilib.Label('777$ REWARD', self.mainapp_ref.big_font)
        self.score.image.fill((0,0,0,0))
        self.add_widget(self.score, 1, (65+56,265+48+12))

        prev_center = 370
        char_ct = 0
        row_ct = 0
        for temp_char in FONTCHARACTERSET:
            if temp_char == ' ':
                pass
            else:                
                self.charbuttons[temp_char] = guilib.TextButton2(temp_char, self.mainapp_ref.big_red_font, self.mainapp_ref.big_white_font)
                self.add_widget(self.charbuttons[temp_char], 1, (prev_center+50-(self.charbuttons[temp_char].rect.width/2),30+(row_ct*50)), (50,50))
                prev_center = self.charbuttons[temp_char].rect.center[0]
                char_ct += 1
                if char_ct == 7:
                    prev_center = 370
                    char_ct = 0
                    row_ct += 1
         
        self.charbuttons[' '] = guilib.TextButton2('SPACE', self.mainapp_ref.big_red_font, self.mainapp_ref.big_white_font)
        self.add_widget(self.charbuttons[' '], 1, (524,30+(row_ct*50)))
        prev_center = self.charbuttons[' '].rect.center[0]

        self.charbuttons['DEL'] = guilib.TextButton2('DELETE', self.mainapp_ref.big_red_font, self.mainapp_ref.big_white_font)
        self.add_widget(self.charbuttons['DEL'], 1, (631,30+(row_ct*50)))       

        row_ct += 1
        self.done = guilib.TextButton2('DONE', self.mainapp_ref.big_red_font, self.mainapp_ref.big_font)
        self.add_widget(self.done, 1, (91, 420))

        self.set_top_left((-800, 0))

    def reset(self):
        pass

    def update(self):
        if self.state == self.STATE_SLIDING_OUT:
            if self.slide(0):
                self.state = self.STATE_OUT
                self.set_top_left((0,0))
            self.dirty()
        elif self.state == self.STATE_OUT:
            if self.done.clicked:
                self.state = self.STATE_SLIDING_IN
                self.mainapp_ref.change_state(self.mainapp_ref.STATE_SAVE_HIGH_SCORE)
            else:
                for temp_key in self.charbuttons.keys():
                    if self.charbuttons[temp_key].clicked:
                        if temp_key == 'DEL':
                            if len(self.playername) > 0:
                                self.playername = self.playername[0:-1]
                        else:
                            if len(self.playername) < 5:
                                self.playername += temp_key
                        self.entry.change_text(self.playername, 'center')
        elif self.state == self.STATE_SLIDING_IN:
            if self.slide(-800):
                self.state = self.STATE_IN
                self.set_top_left((-800,0))
            self.dirty()


    def slide_out(self):
        self.state = self.STATE_SLIDING_OUT

    def slide(self, targetx):
        self.d = abs(targetx - self.rect.left)

        if self.d <= LEFTMENUSPEED:
            speed = self.d/2
        else:
            speed = LEFTMENUSPEED
            
        if speed <= 2:
            speed = 2

        if self.d <= speed:
            return True

        if targetx < 0:
            speed = -speed

        self.move((speed,0))
                
class FpsCounter(guilib.Container):
    def __init__(self, font, clock):
        guilib.Container.__init__(self) 

        self.font = font
        self.clock = clock

        self.picturebox = guilib.PictureBox(gfxlib.CreateSurface((100, 10)))
        self.add_widget(self.picturebox)

    def update(self):
        if SHOWFPSCOUNTER:
            self.picturebox.change_image(self.font.render(str(int(self.clock.get_fps())), 0, (255,255,255)))

class MainApp:
    STATE_SPLASH = 0
    STATE_NEW_GAME = 1
    STATE_GAME = 2
    STATE_LEFT_MENU = 3
    STATE_END_GAME = 4
    STATE_END_GAME_AFTER_UPDATE = 5
    STATE_DISPLAY_SCORES = 6
    STATE_EXIT_SCORE_DISPLAY = 7
    STATE_INPUT_NAME = 8
    STATE_SAVE_HIGH_SCORE = 9
    STATE_EXIT = 10
    STATE_CONTINUE = 11

    POINTS_AMERICAN = 1
    POINTS_ENGLISH = 2

    def __init__(self, screen, screenflags):
        prev_time = pygame.time.get_ticks()
        self.screen = screen
        self.screenflags = screenflags
        self.clock = pygame.time.Clock()
        self.fps = 50
        self.sleep_timer = 0
        self.running = True
        if self.screenflags == pygame.FULLSCREEN:
            self.windowed = False
        else:
            self.windowed = True
        self.font = pygame.font.SysFont('', 16)
        self.state = self.STATE_SPLASH
        self.total_points = 0
        self.savedgame_exists = False
        #print 'first parts of init', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()        
        self.americanhighscores = Highscores()
        #print 'self.americanhighscores', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()        
        self.englishhighscores = Highscores()
        #print 'self.englishhighscores', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()        
        self.settings = Settings(self)
        #print 'self.settings', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()        
        self.savedgame = SavedGame()
        #print 'self.savedgame', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()        
        self.load_settings()
        #print 'self.load_settings', pygame.time.get_ticks() - prev_time
        
        self.pointsystem = self.settings.pointsystem

        prev_time = pygame.time.get_ticks()
        self.small_font = fontlib.BitmapFont(FONTCHARACTERSET, FONTDIR, SMALLFONTFILE)
        #print 'self.small_font', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.small_white_font = fontlib.BitmapFont(FONTCHARACTERSET, FONTDIR, SMALLWHITEFONTFILE)
        #print 'self.small_white_font', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()        
        self.big_font = fontlib.BitmapFont(FONTCHARACTERSET, FONTDIR, BIGFONTFILE)
        #print 'self.big_font', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()        
        self.big_white_font = fontlib.BitmapFont(FONTCHARACTERSET, FONTDIR, BIGWHITEFONTFILE)
        #print 'self.big_white_font', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()        
        self.big_red_font = fontlib.BitmapFont(FONTCHARACTERSET, FONTDIR, BIGREDFONTFILE)
        #print 'self.big_red_font', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.containers = guilib.ContainerGroup(4)
        #print 'self.containers', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.deckgfxs = gfxlib.GfxBank(os.path.join(GFXBASEDIR,DECKGFXDIR))
        #print 'self.deckgfxs', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()        
        self.deck = Deck()
        self.deck.set_gfx(self.deckgfxs.get_gfx(self.settings.deck_gfx_key))
        #print 'self.deck', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.backgroundgfxs = gfxlib.GfxBank(os.path.join(GFXBASEDIR,BACKGROUNDGFXDIR))
        #print 'self.backgroundgfxs', pygame.time.get_ticks() - prev_time
        
        prev_time = pygame.time.get_ticks()
        self.background = Background()
        self.background.change_image(self.backgroundgfxs.get_gfx(self.settings.background_gfx_key))
        self.containers.add_container(self.background, 0)
        #print 'self.background', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.deckbackgfxs = gfxlib.GfxBank(os.path.join(GFXBASEDIR,DECKBACKGFXDIR))
        #print 'self.deckbackgfxs', pygame.time.get_ticks() - prev_time
        
        prev_time = pygame.time.get_ticks()
        self.shoe = Shoe()
        self.shoe.change_image(self.deckbackgfxs.get_gfx(self.settings.deckback_gfx_key))
        self.containers.add_container(self.shoe, 1)
        #print 'self.shoe', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.playfield = Playfield()
        self.containers.add_container(self.playfield, 1)
        #print 'self.playfield', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.scoredisplay = ScoreDisplay(self)
        self.containers.add_container(self.scoredisplay, 1)
        #print 'self.scoredisplay', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.dealtcardeventarea = DealtCardEventArea()
        self.containers.add_container(self.dealtcardeventarea, 3)
        #print 'self.dealtcardeventarea', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.dealtcard = DealtCard(self, self.dealtcardeventarea, self.playfield, self.shoe, self.deck)
        self.containers.add_container(self.dealtcard, 2)
        self.playfield.dealtcard_ref = self.dealtcard
        #print 'self.dealtcard', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.leftmenu = LeftMenu(self)
        self.containers.add_container(self.leftmenu, 4)
        #print 'self.leftmenu', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.rightmenuback = RightMenuBack()
        self.containers.add_container(self.rightmenuback, 9)
        #print 'self.rightmenuback', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.helpdisplay = HelpDisplay(self)
        self.containers.add_container(self.helpdisplay, 7)
        #print 'self.helpdisplay', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.rightmenu = RightMenu(self)
        self.containers.add_container(self.rightmenu, 10)
        #print 'self.rightmenu', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.highscoreinput = HighscoreInput(self)
        self.containers.add_container(self.highscoreinput, 5)
        #print 'self.highscoreinput', pygame.time.get_ticks() - prev_time

        if SHOWFPSCOUNTER:
            prev_time = pygame.time.get_ticks()
            self.fpscounter = FpsCounter(self.font, self.clock)
            self.containers.add_container(self.fpscounter, 8)
            #print 'self.fpscounter', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.splash = Splash(self)
        self.containers.add_container(self.splash, 6)
        #print 'self.splash', pygame.time.get_ticks() - prev_time

        prev_time = pygame.time.get_ticks()
        self.containers.disable_all_container_event_handlings()
        self.splash.enable_event_handling()
        self.rightmenu.enable_event_handling()
        #print 'rest of init', pygame.time.get_ticks() - prev_time

    def change_state(self, state):
        self.state = state

        if self.state == self.STATE_NEW_GAME or self.state == self.STATE_CONTINUE:
            self.deck.reset()
            self.playfield.reset()
            self.shoe.reset()
            self.scoredisplay.reset()
            self.highscoreinput.reset()
            
            self.leftmenu.slideout.show()
            self.leftmenu.slidein.show()
            
            if self.state == self.STATE_CONTINUE:
                self.load_savedgame()
                if self.deck.dealt_cards_ct < 25:
                    self.dealtcard.reset(self.savedgame.dealtcard_suit, self.savedgame.dealtcard_value)
                    self.change_state(self.STATE_GAME)
                else:
                    self.dealtcard.reset()
                    self.dealtcard.hide()
                    self.change_state(self.STATE_END_GAME)                
            else:
                self.dealtcard.reset()
                self.change_state(self.STATE_GAME)
        elif self.state == self.STATE_GAME:
            self.containers.disable_all_container_event_handlings()
            self.leftmenu.enable_event_handling() 
            self.rightmenu.enable_event_handling()
            self.dealtcardeventarea.enable_event_handling()
            self.playfield.enable_event_handling()          
        elif self.state == self.STATE_LEFT_MENU:
            self.containers.disable_all_container_event_handlings()
            self.leftmenu.enable_event_handling() 
            self.rightmenu.enable_event_handling()
        elif self.state == self.STATE_END_GAME_AFTER_UPDATE:
            self.containers.disable_all_container_event_handlings()
            self.rightmenu.enable_event_handling()           
        elif self.state == self.STATE_END_GAME:
            self.containers.disable_all_container_event_handlings()
            self.rightmenu.enable_event_handling()
            self.calculate_scores()
            self.leftmenu.slideout.hide()
            self.change_state(self.STATE_DISPLAY_SCORES)
        elif self.state == self.STATE_DISPLAY_SCORES:
            self.containers.disable_all_container_event_handlings()
            self.rightmenu.enable_event_handling()
            self.scoredisplay.enable_event_handling()
        elif self.state == self.STATE_EXIT_SCORE_DISPLAY:
            self.containers.disable_all_container_event_handlings()
            self.rightmenu.enable_event_handling()
            if self.check_if_highscore():
                self.change_state(self.STATE_INPUT_NAME)
            else:
                self.change_state(self.STATE_NEW_GAME)
        elif self.state == self.STATE_INPUT_NAME:
            self.highscoreinput.score.change_text(str(self.total_points)+'$ REWARD', 'center')
            self.highscoreinput.slide_out()
            self.containers.disable_all_container_event_handlings()
            self.rightmenu.enable_event_handling()
            self.highscoreinput.enable_event_handling()
        elif self.state == self.STATE_SAVE_HIGH_SCORE:
            self.containers.disable_all_container_event_handlings()
            self.rightmenu.enable_event_handling()
            self.save_high_score()
            self.leftmenu.update_highscores()
            self.change_state(self.STATE_NEW_GAME)
        elif self.state == self.STATE_EXIT:
            self.save_all()
            self.running = False

    def load_settings(self):
        self.HOMEDIR = os.path.expanduser('~')
        
        self.SAVEDIR = os.path.join(self.HOMEDIR, '.outlawsolitaire')
        if os.path.exists(self.SAVEDIR):
            pass
        else:
            os.mkdir(self.SAVEDIR)

        self.AMERICANHIGHSCORESFILE = os.path.join(self.SAVEDIR, 'american_high_scores')
        if os.path.exists(self.AMERICANHIGHSCORESFILE):
            americanhighscoresfile = open(self.AMERICANHIGHSCORESFILE, 'rb')
            self.americanhighscores = pickle.load(americanhighscoresfile)
            americanhighscoresfile.close()
        else:
            for ct in range(9):
                self.americanhighscores.add_entry('-----', 0, time.strftime('%d.%m.%y', time.localtime()))
            americanhighscoresfile = open(self.AMERICANHIGHSCORESFILE, 'wb')
            pickle.dump(self.americanhighscores, americanhighscoresfile)
            americanhighscoresfile.close()

        self.ENGLISHHIGHSCORESFILE = os.path.join(self.SAVEDIR, 'english_high_scores')
        if os.path.exists(self.ENGLISHHIGHSCORESFILE):
            englishhighscoresfile = open(self.ENGLISHHIGHSCORESFILE, 'rb')
            self.englishhighscores = pickle.load(englishhighscoresfile)
            englishhighscoresfile.close()
        else:
            for ct in range(9):
                self.englishhighscores.add_entry('-----', 0, time.strftime('%d.%m.%y', time.localtime()))
            englishhighscoresfile = open(self.ENGLISHHIGHSCORESFILE, 'wb')
            pickle.dump(self.englishhighscores, englishhighscoresfile)
            englishhighscoresfile.close()

        self.SETTINGSFILE = os.path.join(self.SAVEDIR, 'settings')
        if os.path.exists(self.SETTINGSFILE):
            settingsfile = open(self.SETTINGSFILE, 'rb')
            self.settings = pickle.load(settingsfile)
            settingsfile.close()
        else:        
            settingsfile = open(self.SETTINGSFILE, 'wb')
            pickle.dump(self.settings, settingsfile)
            settingsfile.close()

        self.SAVEDGAMEFILE = os.path.join(self.SAVEDIR, 'saved_game')
        if os.path.exists(self.SAVEDGAMEFILE):
            self.savedgame_exists = True

    def set_window_mode(self):
        self.save_all()
        self.windowed = True
        self.rightmenu.back.show()
        self.rightmenu.backtext.show()
        self.screen = pygame.display.set_mode((SCREENW, SCREENH), 0)

    def set_fullscreen_mode(self):
        self.windowed = False
        self.rightmenu.back.hide()
        self.rightmenu.backtext.hide()
        self.screen = pygame.display.set_mode((SCREENW, SCREENH), pygame.FULLSCREEN)        

    def save_all(self):
        self.save_settings()
        self.save_savedgame()
        
    def save_settings(self):
        self.settings.pointsystem = self.pointsystem
        self.settings.deck_gfx_key = self.deckgfxs.get_gfx_key()
        self.settings.background_gfx_key = self.backgroundgfxs.get_gfx_key()
        self.settings.deckback_gfx_key = self.deckbackgfxs.get_gfx_key()
        settingsfile = open(self.SETTINGSFILE, 'wb')
        pickle.dump(self.settings, settingsfile)
        settingsfile.close()

    def save_savedgame(self):
        if os.path.exists(self.SAVEDGAMEFILE):
            os.remove(self.SAVEDGAMEFILE)

        self.savedgame.dealtcard_suit = self.dealtcard.card.suit
        self.savedgame.dealtcard_value = self.dealtcard.card.value

        game_found = False
        for y_ct in range(5):
            for x_ct in range(5):
                self.savedgame.suit[x_ct][y_ct] = self.playfield.cards[x_ct][y_ct].suit
                self.savedgame.value[x_ct][y_ct] = self.playfield.cards[x_ct][y_ct].value
                if self.savedgame.suit[x_ct][y_ct] > -1:
                    game_found = True
        if game_found:
            savedgamefile = open(self.SAVEDGAMEFILE, 'wb')
            pickle.dump(self.savedgame, savedgamefile)
            savedgamefile.close()

    def load_savedgame(self):
        savedgamefile = open(self.SAVEDGAMEFILE, 'rb')
        self.savedgame = pickle.load(savedgamefile)
        savedgamefile.close()

        for y_ct in range(5):
            for x_ct in range(5):
                if self.savedgame.suit[x_ct][y_ct] > -1:
                    tempcard = self.deck.get_specific_card(self.savedgame.suit[x_ct][y_ct], self.savedgame.value[x_ct][y_ct], True)
                    self.playfield.cards[x_ct][y_ct].copy_from(tempcard)

        for ct in range(int((self.deck.dealt_cards_ct-1) / 8)):
            self.shoe.lower_deck()

    def save_high_score(self):
        if self.pointsystem == self.POINTS_AMERICAN:
            self.americanhighscores.add_entry(self.highscoreinput.playername, self.total_points, time.strftime('%d.%m.%y', time.localtime()))
            americanhighscoresfile = open(self.AMERICANHIGHSCORESFILE, 'wb')
            pickle.dump(self.americanhighscores, americanhighscoresfile)
            americanhighscoresfile.close()
        elif self.pointsystem == self.POINTS_ENGLISH:
            self.englishhighscores.add_entry(self.highscoreinput.playername, self.total_points, time.strftime('%d.%m.%y', time.localtime()))
            englishhighscoresfile = open(self.ENGLISHHIGHSCORESFILE, 'wb')
            pickle.dump(self.englishhighscores, englishhighscoresfile)
            englishhighscoresfile.close()
        else:
            raise RuntimeError('MainApp: save_high_score: Invalid point system!')

    def check_if_highscore(self):
        if self.pointsystem == self.POINTS_AMERICAN:
            self.americanhighscores.sort_entries()
            if self.total_points > self.americanhighscores.entries[8].points:
                return True
            else:
                return False
        elif self.pointsystem == self.POINTS_ENGLISH:
            self.englishhighscores.sort_entries()
            if self.total_points > self.englishhighscores.entries[8].points:
                return True
            else:
                return False            
        else:
            raise RuntimeError('MainApp: check_if_highscore: Invalid point system!')            

    def calculate_scores(self):
        temp_hand_suits = [-1 for ct in range(5)]
        temp_hand_values = [-1 for ct in range(5)]

        #print AMERICANPOINTSDICT
        #print ENGLISHPOINTSDICT
        #print

        self.total_points = 0

        for y_ct in range(5):
            for x_ct in range(5):
                temp_hand_suits[x_ct] = self.playfield.cards[x_ct][y_ct].suit
                temp_hand_values[x_ct] = self.playfield.cards[x_ct][y_ct].value + 1
            #print 'row', y_ct+1, temp_hand_suits, temp_hand_values
            
            if self.pointsystem == self.POINTS_AMERICAN:
                temp_hand_points = AMERICANPOINTSDICT[self.check_hand(temp_hand_suits, temp_hand_values)]
            elif self.pointsystem == self.POINTS_ENGLISH:
                temp_hand_points = ENGLISHPOINTSDICT[self.check_hand(temp_hand_suits, temp_hand_values)]
            else:
                raise RuntimeError('MainApp: calculate_scores: Invalid point system!')
            
            self.total_points += temp_hand_points
            #print temp_hand_points
            #print

            self.scoredisplay.rowpoints[y_ct].change_text(str(temp_hand_points), 'right')

        for x_ct in range(5):
            for y_ct in range(5):
                temp_hand_suits[y_ct] = self.playfield.cards[x_ct][y_ct].suit
                temp_hand_values[y_ct] = self.playfield.cards[x_ct][y_ct].value + 1                
            #print 'col', x_ct+1, temp_hand_suits, temp_hand_values

            if self.pointsystem == self.POINTS_AMERICAN:
                temp_hand_points = AMERICANPOINTSDICT[self.check_hand(temp_hand_suits, temp_hand_values)]
            elif self.pointsystem == self.POINTS_ENGLISH:
                temp_hand_points = ENGLISHPOINTSDICT[self.check_hand(temp_hand_suits, temp_hand_values)]
            else:
                raise RuntimeError('MainApp: calculate_scores: Invalid point system!')
            
            self.total_points += temp_hand_points
            #print temp_hand_points
            #print
           
            self.scoredisplay.colpoints[x_ct].change_text(str(temp_hand_points), 'center')

        self.scoredisplay.total.change_text(str(self.total_points)+'$', 'center')

        if self.check_if_highscore():
            self.scoredisplay.show_new_high_score()
        else:        
            self.scoredisplay.show_tap_to_continue()

    def check_hand(self, s, v):
        s.sort()
        v.sort()

        #print 'chk', ':', s, v        

        temp_hand_name = None
        if (v[0] == 1 and v[1] == 10 and v[2] == 11 and v[3] == 12 and v[4] == 13) and (s[0] == s[1] and s[0] == s[2] and s[0] == s[3] and s[0] == s[4]):
            temp_hand_name = 'royal flush'
        elif (v[1] == v[0] + 1 and v[2] == v[0] + 2 and v[3] == v[0] + 3 and v[4] == v[0] + 4) and (s[0] == s[1] and s[0] == s[2] and s[0] == s[3] and s[0] == s[4]):
            temp_hand_name = 'straight flush'
        elif (v[0] == v[1] and v[1] == v[2] and v[2] == v[3]) or (v[1] == v[2] and v[2] == v[3] and v[3] == v[4]):
            temp_hand_name = 'four of a kind'
        elif (v[1] == v[0] + 1 and v[2] == v[0] + 2 and v[3] == v[0] + 3 and v[4] == v[0] + 4) or (v[0] == 1 and v[1] == 10 and v[2] == 11 and v[3] == 12 and v[4] == 13):
            temp_hand_name = 'straight'
        elif (v[0] == v[1] and v[2] == v[3] and v[3] == v[4]) or (v[3] == v[4] and v[0] == v[1] and v[1] == v[2]):
            temp_hand_name = 'full house'
        elif (v[0] == v[1] and v[1] == v[2]) or (v[1] == v[2] and v[2] == v[3]) or (v[2] == v[3] and v[3] == v[4]):
            temp_hand_name = 'three of a kind'
        elif s[0] == s[1] and s[0] == s[2] and s[0] == s[3] and s[0] == s[4]:
            temp_hand_name = 'flush'
        elif (v[0] == v[1] and v[2] == v[3]) or (v[1] == v[2] and v[3] == v[4]) or (v[0] == v[1] and v[3] == v[4]):
            temp_hand_name = 'two pairs'
        elif v[0] == v[1] or v[1] == v[2] or v[2] == v[3] or v[3] == v[4]:
            temp_hand_name = 'pair'
        else:
            temp_hand_name = 'nothing'

        #print temp_hand_name
        
        return temp_hand_name        

    def update(self):
        self.handle_events()

        if self.windowed:
            self.fps = 1
        else:            
            if self.sleep_timer > 1000:
                self.fps = 1
            elif self.sleep_timer > 500:
                self.fps = 25
            else:
                self.fps = 50

        self.containers.update()
        self.update_rect_list = self.containers.draw(self.screen)       

        #if self.update_rect_list:
        #    print self.update_rect_list, "*"

        self.clock.tick(self.fps)
        
        if self.update_rect_list:
            pygame.display.update(self.update_rect_list)
        #    print '-------------------', pygame.time.get_ticks()
        #print 'xxx'            

        #for temp_widget in self.containers.widget_group:
        #    if temp_widget.parent.event_handling:
        #        pygame.draw.rect(self.screen, (255,0,255), temp_widget.event_area, 1)
        #pygame.display.update()

        if self.state == self.STATE_END_GAME_AFTER_UPDATE:
            self.change_state(self.STATE_END_GAME)

    def handle_events(self):
        self.sleep_timer += 1
        for event in pygame.event.get():
            self.sleep_timer = 0
            
            if event.type == pygame.QUIT:
                self.save_all()
                sys.exit()

            if event.type == pygame.ACTIVEEVENT:
                if event.state == 2 and event.gain == 0 and self.windowed == False:
                    self.set_window_mode()

            if event.type == pygame.KEYDOWN:
                pass
#                if event.key == pygame.K_p:
#                    self.running = False
#
#                if event.key == pygame.K_z:
#                    for y_ct in range(5):
#                        for x_ct in range(5):
#                            self.playfield.cards[x_ct][y_ct].copy_from(self.deck.get_card())
#                            self.playfield.dirty()
#                            
#                if event.key == pygame.K_q:
#                    tempgfx = self.backgroundgfxs.get_prev_gfx()
#                    if tempgfx:
#                        self.background.change_image(tempgfx)
#                elif event.key == pygame.K_a:
#                    tempgfx = self.backgroundgfxs.get_next_gfx()
#                    if tempgfx:
#                        self.background.change_image(tempgfx)
#
#                if event.key == pygame.K_w:
#                    tempgfx = self.deckbackgfxs.get_prev_gfx()
#                    if tempgfx:
#                        self.shoe.change_image(tempgfx)
#                elif event.key == pygame.K_s:
#                    tempgfx = self.deckbackgfxs.get_next_gfx()
#                    if tempgfx:
#                        self.shoe.change_image(tempgfx)
#
#                if event.key == pygame.K_e:
#                    tempgfx = self.deckgfxs.get_prev_gfx()
#                    if tempgfx:
#                        self.deck.set_gfx(tempgfx)
#                        self.playfield.dirty()
#                        self.dealtcard.dirty()
#                elif event.key == pygame.K_d:
#                    tempgfx = self.deckgfxs.get_next_gfx()
#                    if tempgfx:
#                        self.deck.set_gfx(tempgfx)
#                        self.playfield.dirty()
#                        self.dealtcard.dirty()

            if self.windowed:
                self.rightmenu.handle_events(event)
            elif self.helpdisplay.back.visible:
                self.helpdisplay.handle_events(event)
                self.rightmenu.handle_events(event)
            else:
                self.containers.handle_events(event)

if __name__ == '__main__':
    pass                
