/* -*- linux-c -*-
 *
 * Send IPv6 router solicitation message and wait for answer.
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 * Portions taken from MIPL mipv6-2.0.2 project.
 *
 * Author: Jukka Rissanen <jukka.rissanen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <netinet/ip6.h>
#include <netinet/icmp6.h>

/* These two defines are from mipv6-2.0.2, they work only >=2.6.14 kernels
 * so RFC3542 values for socket options must be used
 */
#ifndef IPV6_RECVPKTINFO
#define IPV6_RECVPKTINFO        49
#ifdef IPV6_PKTINFO
#undef IPV6_PKTINFO
#define IPV6_PKTINFO            50
#endif
#endif

#ifndef IPV6_RECVHOPLIMIT
#define IPV6_RECVHOPLIMIT       51
#ifdef IPV6_HOPLIMIT
#undef IPV6_HOPLIMIT
#define IPV6_HOPLIMIT           52
#endif
#endif

#define MAX_PKT_LEN 1540
#define MAX_BUF 255
#define MAX_ERROR_BUF_LEN 255
#define DEFAULT_TIMEOUT (1000*3) /* 3 second timeout (in ms) */
#define DEFAULT_DEVICE "wlan0"

#define DEBUG 1

#define RADV_MANAGED 1
#define RADV_OTHER   2
#define RADV_HA      4

#define IN6ADDR_ALL_NODES_MC_INIT \
        { { { 0xff,0x02,0,0,0,0,0,0,0,0,0,0,0,0,0,0x1 } } } /* ff02::1 */
#define IN6ADDR_ALL_ROUTERS_MC_INIT \
        { { { 0xff,0x02,0,0,0,0,0,0,0,0,0,0,0,0,0,0x2 } } } /* ff02::2 */

static int quit = 0;
static int quiet = 0;
const struct in6_addr in6addr_all_nodes_mc = IN6ADDR_ALL_NODES_MC_INIT;
const struct in6_addr in6addr_all_routers_mc = IN6ADDR_ALL_ROUTERS_MC_INIT;

/* ----------------------------------------------------------------------- */
static int debug_level;

#ifdef DEBUG
#if (__GNUC__ > 2) && ((__GNUC__ > 3) || (__GNUC_MINOR__ > 2))
#define PDEBUG(fmt...) do {						\
		if (debug_level) {					\
			struct timeval tv;				\
			gettimeofday(&tv, 0);				\
			printf("DEBUG[%d]:%ld.%ld:%s:%s():%d: ",	\
			       getpid(),				\
			       tv.tv_sec, tv.tv_usec,			\
			       __FILE__, __FUNCTION__, __LINE__);	\
			printf(fmt);					\
			fflush(stdout);					\
		}							\
	}while(0)
#else
#define PDEBUG(fmt...) do {						\
		if (debug_level) {					\
			struct timeval tv;				\
			gettimeofday(&tv, 0);				\
			printf("DEBUG[%d]:%ld.%ld:%s:%s():%d: ",	\
			       getpid(),				\
			       tv.tv_sec, tv.tv_usec,			\
			       __FILE__, __FUNCTION__, __LINE__);	\
			printf(##fmt);					\
			fflush(stdout);					\
		}							\
	}while(0)
#endif
#else
#define PDEBUG(fmt...)
#endif



/* ----------------------------------------------------------------------- */
static void usage(char *prg, char *str)
{
	if (str)
		printf(str);

	printf("Usage: %s [-D] -d <iface>\n", prg);
	printf("\t-D turn debugging on in this program\n");
	printf("\t-s source address (default is link local)\n");
	printf("\t-d device interface to use\n");
	printf("\t-t timeout (default is %d secs)\n", DEFAULT_TIMEOUT/1000);
	printf("\t-q be quiet\n");
	printf("\n");
	printf("Exit codes (are or'ed together):\n");
	printf("  <0 timeout\n");
	printf("   0 no known flags returned\n");
	printf("   1 MANAGED flag\n");
	printf("   2 OTHER flag\n");
	printf("   4 HA flag\n");
	printf("\n");
	printf("Example:\n\t%s -d wlan0\n\n", prg);
	printf("If return value is 3, then MANAGED and OTHER flags were set.\n");
	exit(-1);
}


/* ----------------------------------------------------------------------- */
void signal_handler(int sig)
{
	switch (sig) {
	case SIGHUP:
	case SIGTERM:
	case SIGINT:
		if (!quiet)
			printf("Quitting\n");
		quit = 1;
		break;
	case SIGALRM:
		if (!quiet)
			printf("Timeout\n");
		quit = 1;
		break;
	default:
		if (!quiet)
			printf("Got signal %d\n", sig);
		break;
	}
}


/* ----------------------------------------------------------------------- */
/* Various functions below are from mipv6-2.0.2 open source project
 * and their license is GPL.
 */

/* Adapted from RFC 1071 "C" Implementation Example */
static uint16_t csum(const void *phdr, const void *data, socklen_t datalen)
{
	register unsigned long sum = 0;
	socklen_t count;
	uint16_t *addr;
	int i;

	/* caller must make sure datalen is even */

	addr = (uint16_t *)phdr;
	for (i = 0; i < 20; i++)
		sum += *addr++;

	count = datalen;
	addr = (uint16_t *)data;

        while (count > 1) {
		sum += *(addr++);
		count -= 2;
	}

	while (sum >> 16)
		sum = (sum & 0xffff) + (sum >> 16);

	return (uint16_t)~sum;
}


/* ----------------------------------------------------------------------- */
static int ndisc_send_unspec(int type, int oif, const struct in6_addr *dest) 
{
	struct _phdr {
		struct in6_addr src;
		struct in6_addr dst;
		uint32_t plen;
		uint8_t reserved[3];
		uint8_t nxt;
	} phdr;

	struct {
		struct ip6_hdr ip;
		union {
			struct icmp6_hdr icmp;
			struct nd_neighbor_solicit ns;
			struct nd_router_solicit rs;
		} i;
	} frame;

	struct msghdr msgh;
	struct cmsghdr *cmsg;
	struct in6_pktinfo *pinfo;
	struct sockaddr_in6 dst;
	char cbuf[CMSG_SPACE(sizeof(*pinfo))];
	struct iovec iov;
	int fd, datalen, ret;

	fd = socket(AF_INET6, SOCK_RAW, IPPROTO_RAW);
	if (fd < 0) {
		perror("socket");
		return -1;
	}

	memset(&frame, 0, sizeof(frame));
	memset(&dst, 0, sizeof(dst));

	datalen = sizeof(frame.i.rs); /* 8, csum() safe */
	dst.sin6_addr = *dest;

	/* Fill in the IPv6 header */
	frame.ip.ip6_vfc = 0x60;
	frame.ip.ip6_plen = htons(datalen);
	frame.ip.ip6_nxt = IPPROTO_ICMPV6;
	frame.ip.ip6_hlim = 255;
	frame.ip.ip6_dst = dst.sin6_addr;
	/* all other fields are already set to zero */

	/* Prepare pseudo header for csum */
	memset(&phdr, 0, sizeof(phdr));
	phdr.dst = dst.sin6_addr;
	phdr.plen = htonl(datalen);
	phdr.nxt = IPPROTO_ICMPV6;

	/* Fill in remaining ICMP header fields */
	frame.i.icmp.icmp6_type = type;
	frame.i.icmp.icmp6_cksum = csum(&phdr, &frame.i, datalen);

	iov.iov_base = &frame;
	iov.iov_len = sizeof(frame.ip) + datalen;

	dst.sin6_family = AF_INET6;
	msgh.msg_name = &dst;
	msgh.msg_namelen = sizeof(dst);
	msgh.msg_iov = &iov;
	msgh.msg_iovlen = 1;
	msgh.msg_flags = 0;

	memset(cbuf, 0, CMSG_SPACE(sizeof(*pinfo)));
	cmsg = (struct cmsghdr *)cbuf;
	pinfo = (struct in6_pktinfo *)CMSG_DATA(cmsg);
	pinfo->ipi6_ifindex = oif;

	cmsg->cmsg_len = CMSG_LEN(sizeof(*pinfo));
	cmsg->cmsg_level = IPPROTO_IPV6;
	cmsg->cmsg_type = IPV6_PKTINFO;
	msgh.msg_control = cmsg;
	msgh.msg_controllen = cmsg->cmsg_len;

	ret = sendmsg(fd, &msgh, 0);
	if (ret < 0) {
		if (!quiet)
			printf("sendmsg: %s\n", strerror(errno));
	}

	close(fd);
	return ret;
}

int ndisc_send_rs(int ifindex, const struct in6_addr *dst)
{
	return ndisc_send_unspec(ND_ROUTER_SOLICIT, ifindex, dst);
}

static inline void ipv6_addr_set(struct in6_addr *addr,
                                 uint32_t w1, uint32_t w2,
                                 uint32_t w3, uint32_t w4)
{
        addr->s6_addr32[0] = w1;
        addr->s6_addr32[1] = w2;
        addr->s6_addr32[2] = w3;
        addr->s6_addr32[3] = w4;
}

static inline void ipv6_addr_llocal(const struct in6_addr *addr,
                                    struct in6_addr *llocal)
{
        ipv6_addr_set(llocal, htonl(0xFE800000), 0,
                      addr->s6_addr32[2], addr->s6_addr32[3]);
}

static inline void ipv6_addr_solict_mult(const struct in6_addr *addr,
                                         struct in6_addr *solicited)
{
        ipv6_addr_set(solicited, htonl(0xFF020000), 0, htonl(0x1),
                      htonl(0xFF000000) | addr->s6_addr32[3]);
}

int if_mc_group(int sock, int ifindex, const struct in6_addr *mc_addr, int cmd)
{
        unsigned int val = 0;
        struct ipv6_mreq mreq;
        int ret = 0;

        memset(&mreq, 0, sizeof(mreq));
        mreq.ipv6mr_interface = ifindex;
        mreq.ipv6mr_multiaddr = *mc_addr;

        ret = setsockopt(sock, IPPROTO_IPV6, IPV6_MULTICAST_LOOP,
                         &val, sizeof(int));

        if (ret < 0) return ret;

        return setsockopt(sock, IPPROTO_IPV6, cmd, &mreq, sizeof(mreq));
}

#define CMSG_BUF_LEN 128

ssize_t icmp6_recv(int sockfd, unsigned char *msg, size_t msglen,
		   struct sockaddr_in6 *addr, struct in6_pktinfo *pkt_info,
		   int *hoplimit)
{
	struct msghdr mhdr;
	struct cmsghdr *cmsg;
	struct iovec iov;
	static unsigned char chdr[CMSG_BUF_LEN];
	ssize_t len;

	iov.iov_len = msglen;
	iov.iov_base = (unsigned char *) msg;

	mhdr.msg_name = (void *)addr;
	mhdr.msg_namelen = sizeof(struct sockaddr_in6);
	mhdr.msg_iov = &iov;
	mhdr.msg_iovlen = 1;
	mhdr.msg_control = (void *)chdr;
	mhdr.msg_controllen = CMSG_BUF_LEN;

	if ((len = recvmsg(sockfd, &mhdr, 0)) < 0)
		return -errno;

        for (cmsg = CMSG_FIRSTHDR(&mhdr); cmsg; 
	     cmsg = CMSG_NXTHDR(&mhdr, cmsg)) {
		if (cmsg->cmsg_level != IPPROTO_IPV6)
			continue;
		switch(cmsg->cmsg_type) {
		case IPV6_HOPLIMIT:
			*hoplimit = *(int *)CMSG_DATA(cmsg);
			break;
		case IPV6_PKTINFO:
			memcpy(pkt_info, CMSG_DATA(cmsg), sizeof(*pkt_info));
			break;
		}
	}
	return len;
}

int ndisc_do_rs(int ifi, struct in6_addr *addr, int do_ll, int timeout, struct nd_router_advert **reply)
{
	struct in6_pktinfo pinfo;
	struct sockaddr_in6 saddr;
	struct nd_router_advert *hdr;
	struct icmp6_filter filter;
	struct in6_addr solicit, ll;
	static unsigned char msg[MAX_PKT_LEN] = {0};
	int hoplimit, sock, ret = 0, val = 1;
	fd_set rset;
	struct timeval tv;

	ICMP6_FILTER_SETBLOCKALL(&filter);
	ICMP6_FILTER_SETPASS(ND_ROUTER_ADVERT, &filter);

	sock = socket(AF_INET6, SOCK_RAW, IPPROTO_ICMPV6);
	if (sock<0) {
		perror("socket");
		return -1;
	}
	setsockopt(sock, IPPROTO_IPV6, IPV6_RECVPKTINFO, &val, sizeof(val));
	setsockopt(sock, IPPROTO_IPV6, IPV6_RECVHOPLIMIT, &val, sizeof(val));
	setsockopt(sock, IPPROTO_ICMPV6, ICMP6_FILTER, &filter,
		   sizeof(struct icmp6_filter));

	ipv6_addr_solict_mult(addr, &solicit);
	if_mc_group(sock, ifi, &in6addr_all_nodes_mc, IPV6_JOIN_GROUP);
	if_mc_group(sock, ifi, &solicit, IPV6_JOIN_GROUP);

	if (do_ll) {
		ipv6_addr_llocal(addr, &ll);
		ndisc_send_unspec(ND_ROUTER_SOLICIT, ifi, &ll);
	} else
		ndisc_send_unspec(ND_ROUTER_SOLICIT, ifi, addr);


	FD_ZERO(&rset);
	FD_SET(sock, &rset);
	tv.tv_sec = timeout;
	tv.tv_usec = 0;
	while (!quit) {
		/* Note on portability: we assume that tv is modified to show
		   the time left which is AFAIK true only in Linux 
		   timeout 
		*/
		if (select(sock+1, &rset, NULL, NULL, &tv) == 0) {
			perror("rs timeout\n");
			ret = -1;
			break;
		}
		if (quit) {
			ret = -1;
			break;
		}

		if (!FD_ISSET(sock, &rset))
			continue;
		/* We got an ICMPv6 packet */
		ret = icmp6_recv(sock, msg, sizeof(msg), &saddr, 
				 &pinfo, &hoplimit);
		if (ret < 0)
			continue;
		hdr = (struct nd_router_advert *)msg;
		if (hdr->nd_ra_code != 0)
			continue;

		/* Figure out stuff we need */
		break;

#if 0
		if (IN6_ARE_ADDR_EQUAL(addr, &hdr->nd_na_target) ||
		    (do_ll && IN6_ARE_ADDR_EQUAL(&ll, &hdr->nd_na_target))) {
			if (!quiet)
				printf("Failure\n");
			ret = -1;
			break;
		}
#endif
	}
	close(sock);

	if (ret>0 && reply) {
		*reply = (struct nd_router_advert *)msg;
	}

	return ret;
}


/* ----------------------------------------------------------------------- */
int main(int argc, char **argv)
{
	int c;
	int ret = -1, st;
	int timeout = DEFAULT_TIMEOUT / 1000;
	struct timeval tv;
	int raw;
	struct in6_addr dst = in6addr_all_routers_mc;
	struct in6_addr src = { 0 };
	int ifindex;
	int sleep_time = DEFAULT_TIMEOUT;
	char *device = DEFAULT_DEVICE;
	int do_ll = 0;  /* undefined source addr i.e., :: */
	struct nd_router_advert *reply = 0;

	debug_level = 0;

	while ((c=getopt(argc, argv, "Dd:lhqt:"))>-1) {

		switch (c) {

		case 'D':
			debug_level = 1;
			break;

		case 'd':
			device = optarg;
			break;

		case 'l':
			do_ll = 1; /* do link layer address, do not use */
			break;

		case 'q':
			quiet = 1;
			break;

		case 't':
			timeout = atoi(optarg);
			break;

		case 'h':
			usage(argv[0], 0);
			break;

		default:
			break;
		}
	}

	if (debug_level>0)
		quiet = 0;

	signal(SIGHUP, signal_handler);
	signal(SIGTERM, signal_handler);
	signal(SIGINT, signal_handler);
	signal(SIGALRM, signal_handler);

	ifindex = if_nametoindex(device);
	if (!ifindex) {
		if (!quiet)
			printf("Cannot get ifindex of %s\n", device);
		goto OUT;
	}

	alarm(timeout);

	/* send solicitation and wait answer */
	ret = ndisc_do_rs(ifindex, &dst, do_ll, sleep_time, &reply);
	if (ret>0) {
		/* we got something back */
		if (!quiet)
			printf("received %d bytes\n", ret);

		if (debug_level) {
			if (!reply) {
				printf("icmp6 reply invalid\n");
				ret = -1;
				goto OUT;
			}

			printf("type = %d\n", reply->nd_ra_type);
			printf("code = %d\n", reply->nd_ra_code);
			printf("cksum = 0x%x\n", ntohs(reply->nd_ra_cksum));
			printf("hoplimit = %d\n", (unsigned char)reply->nd_ra_curhoplimit);
			printf("flags = 0x%x ",  (unsigned char)reply->nd_ra_flags_reserved);
			if (reply->nd_ra_flags_reserved) {
				printf("( ");
				if (reply->nd_ra_flags_reserved & ND_RA_FLAG_MANAGED)
					printf("managed ");
				if (reply->nd_ra_flags_reserved & ND_RA_FLAG_OTHER)
					printf("other ");
				if (reply->nd_ra_flags_reserved & ND_RA_FLAG_HOME_AGENT)
					printf("home agent ");
				printf(")");
			}
			printf("\n");

			printf("lifetime = %d\n", (int)(ntohs(*((unsigned short *)&reply->nd_ra_router_lifetime))));
			printf("reachable time = %d\n", ntohl(reply->nd_ra_reachable));
			printf("retrans time = %d\n", ntohl(reply->nd_ra_retransmit));

			if (ret > sizeof(struct nd_router_advert)) {
				char *ptr = (unsigned char *)reply + sizeof(struct nd_router_advert);
				int left = ret - sizeof(struct nd_router_advert);
				printf("options:\n");
				while (left>0) {
					struct nd_opt_hdr *opt = (struct nd_opt_hdr *)ptr;
					int len = opt->nd_opt_len * 8;
					printf("\ttype = %d, len=%d\n", opt->nd_opt_type, len);

					switch (opt->nd_opt_type) {
					case ND_OPT_SOURCE_LINKADDR:
						printf("\t\tND_OPT_SOURCE_LINKADDR\n");
						break;
					case ND_OPT_PREFIX_INFORMATION: {
						struct nd_opt_prefix_info *prefix = 
							(struct nd_opt_prefix_info *)ptr;
						char straddr[INET6_ADDRSTRLEN];

						printf("\t\tND_OPT_PREFIX_INFORMATION\n");
						printf("\t\tlen = %d\n", (unsigned char)prefix->nd_opt_pi_prefix_len);
						printf("\t\tflags = 0x%x\n", (unsigned char)prefix->nd_opt_pi_flags_reserved);
						printf("\t\tvalid lifetime = 0x%x\n", ntohl(prefix->nd_opt_pi_valid_time));
						printf("\t\tpreferred lifetime = 0x%x\n", ntohl(prefix->nd_opt_pi_preferred_time));
						inet_ntop(AF_INET6, &prefix->nd_opt_pi_prefix, straddr, sizeof(straddr));
						printf("\t\tprefix = %s\n", straddr);
						break;
					}
					case ND_OPT_REDIRECTED_HEADER:
						printf("\t\tND_OPT_REDIRECTED_HEADER\n");
						break;
					case ND_OPT_MTU:
						printf("\t\tND_OPT_MTU\n");
						break;
					case ND_OPT_RTR_ADV_INTERVAL:
						printf("\t\tND_OPT_RTR_ADV_INTERVAL\n");
						break;
					case ND_OPT_HOME_AGENT_INFO:
						printf("\t\tND_OPT_HOME_AGENT_INFO\n");
						break;
					case 24: /* route information */
						printf("\t\troute information (24)\n");
						break;
					default:
						printf("\t\tunknown option\n");
						break;
					}


					left -= len;
					ptr += len;
				}
			}
			
		}


		if (reply) {
			ret = 0;
			if (reply->nd_ra_flags_reserved) {
				if (reply->nd_ra_flags_reserved & ND_RA_FLAG_MANAGED) {
					if (!quiet)
						fprintf(stderr, "MANAGED ");
					ret |= RADV_MANAGED;
				}
				if (reply->nd_ra_flags_reserved & ND_RA_FLAG_OTHER) {
					if (!quiet)
						fprintf(stderr, "OTHER ");
					ret |= RADV_OTHER;
				}
				if (reply->nd_ra_flags_reserved & ND_RA_FLAG_HOME_AGENT) {
					if (!quiet)
						fprintf(stderr, "HA ");
					ret |= RADV_HA;
				}
				if (!quiet)
					fprintf(stderr, "\n");
			}
		}
	}

OUT:
	exit(ret);
}


