/* FMRX Enabler for N900 hardware
 * Copyright (C) 2009 Martin Grimme  <martin.grimme@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "n900-fmrx-enabler.h"
#include "n900-fmrx-enablerGlue.h"


static GMainLoop *loop;
static DBusGConnection *dbusconn;

/* Generate the GObject boilerplate */
G_DEFINE_TYPE(FMRXEnabler, fmrx_enabler, G_TYPE_OBJECT)

static void daemonize();
static gboolean is_bluetooth_allowed();
static gboolean hci_dev_up(int sock, gboolean *had_bluetooth);
static gboolean hci_dev_down(int sock);
static gboolean set_i2c_enabled(int sock, gboolean value);
static gboolean set_rds_enabled(gboolean value);
static gboolean watchdog_handler(gpointer data);
static void on_change_device_mode(DBusGProxy *proxy,
                                  const char *value,
                                  gpointer data);


/* GObject class initializer.
 */
static void
fmrx_enabler_class_init(FMRXEnablerClass *clss) {
    
    dbus_g_object_type_install_info(FMRX_ENABLER_TYPE,
                                    &dbus_glib_fmrx_enabler_object_info);

}

/* GObject object constructor.
 */
static void
fmrx_enabler_init(FMRXEnabler *self) {

    GError *error = NULL;
    DBusGProxy *driver_proxy;
    DBusGProxy *mce_proxy;
    guint32 request_name_ret;
    
    self->hci_socket = 0;
    self->driver_loaded = FALSE;
    self->had_bluetooth = FALSE;
    self->last_request = 0;

    /* set up D-Bus service */
    dbusconn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if (dbusconn == NULL) {
        g_print("could not connect to D-Bus: %s", error->message);
        g_error_free(error);
        return;
    }
    
    dbus_g_connection_register_g_object(dbusconn,
                                        FMRX_OBJECT_PATH,
                                        G_OBJECT(self));
    driver_proxy = dbus_g_proxy_new_for_name(dbusconn,
                                             DBUS_SERVICE_DBUS,
                                             DBUS_PATH_DBUS,
                                             DBUS_INTERFACE_DBUS);

    if (! org_freedesktop_DBus_request_name(driver_proxy, FMRX_SERVICE_NAME,
                                            0, &request_name_ret, &error)) {
        g_print("Failed to request name: %s", error->message);
        g_error_free(error);
        return;
    }
  
    if (request_name_ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
        g_error("Got result code %u from requesting name", request_name_ret);
        return;
    }

    /* watch D-Bus for getting notified about device mode changes */
    /* ... or rather don't as this should be handled by the clients...
    mce_proxy = dbus_g_proxy_new_for_name(dbusconn,
                                          MCE_SERVICE,
                                          MCE_SIGNAL_PATH,
                                          MCE_SIGNAL_IF);
    dbus_g_proxy_add_signal(mce_proxy, MCE_DEVICE_MODE_SIG,
                            G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal(mce_proxy, MCE_DEVICE_MODE_SIG,
                                G_CALLBACK(on_change_device_mode),
                                (gpointer) self,
                                NULL);
    */
    
}



/* Turns this process into a daemon by using the double-(pitch)-fork. :)
 */
static void
daemonize() {

  int devnull;

  if (fork()) exit(0);
  chdir("/");
  setsid();
  umask(0);
  if (fork()) exit(0);

  devnull = open("/dev/null", O_RDWR);
  close(STDIN_FILENO); dup2(STDIN_FILENO, devnull);
  close(STDOUT_FILENO); dup2(STDOUT_FILENO, devnull);
  close(STDERR_FILENO); dup2(STDERR_FILENO, devnull);
  close(devnull);

}


/* Checks whether Bluetooth is allowed at the moment or not.
 */
static gboolean
is_bluetooth_allowed() {

    DBusGProxy *proxy;
    GError *error = NULL;
    char *device_mode;
    
    proxy = dbus_g_proxy_new_for_name(dbusconn,
                                      MCE_SERVICE,
                                      MCE_REQUEST_PATH,
                                      MCE_REQUEST_IF);
    if (! dbus_g_proxy_call(proxy, MCE_DEVICE_MODE_GET, &error,
                            G_TYPE_INVALID,
                            G_TYPE_STRING, &device_mode, G_TYPE_INVALID)) {

        g_print("error retrieving device mode\n");
        g_error_free(error);
        return FALSE;

    }
    
    g_object_unref(proxy);

    /* test for offline or flight mode */
    if (strcmp(device_mode, MCE_FLIGHT_MODE) == 0 ||
        strcmp(device_mode, MCE_OFFLINE_MODE) == 0) {
        
        g_free(device_mode);
        return FALSE;
        
    } else {

        g_free(device_mode);
        return TRUE;
        
    }

}


/* Powers up the Bluetooth device.
 */
static gboolean
hci_dev_up(int sock, gboolean *had_bluetooth) {

    if (ioctl(sock, HCIDEVUP, 0) < 0) {
        if (errno == EALREADY) {
            g_print("[already active] ");
            *had_bluetooth = TRUE;
            ioctl(sock, HCIDEVDOWN, 0);
            ioctl(sock, HCIDEVUP, 0);
            return TRUE;
        } else {
            g_print("[error: %d] ", errno);
            *had_bluetooth = FALSE;
            return FALSE;
        }
    } else {
        *had_bluetooth = FALSE;
        return TRUE;
    }

}


/* Powers down the Bluetooth device.
 */
static gboolean
hci_dev_down(int sock) {

    if (ioctl(sock, HCIDEVDOWN, 0) < 0 && errno != EALREADY) {
        return FALSE;
    } else {
        return TRUE;
    }

}


/* Enables or disables I2C communication on the Bluetooth chip.
 */
static gboolean
set_i2c_enabled(int sock, gboolean value) {

    int rv, opt;
    struct sockaddr_hci addr;

    opt = 0;
    rv = setsockopt(sock, SOL_HCI, HCI_DATA_DIR, &opt, sizeof(opt));
    if (rv < 0) {
        g_printf("[socket opt error: %d] ", rv);
        return FALSE;
    }
    addr.hci_family = AF_BLUETOOTH;
    addr.hci_dev = 0;
    rv = bind(sock, (struct sockaddr *) &addr, sizeof(addr));
    if (rv < 0) {
        g_printf("[socket bind error: %d] ", rv);
        return FALSE;
    }

    write(sock, value ? I2C_ENABLE : I2C_DISABLE, sizeof(I2C_ENABLE));

    return TRUE;

}


/* Enabled or disables RDS support. RDS may only be enabled by root, so it
 * must be done here.
 */
static gboolean
set_rds_enabled(gboolean value) {

    int fd;
    
    fd = open(FMRADIO_SYSFS "/rds", O_WRONLY);
    if (fd != -1) {
        write(fd, value ? "1" : "0", 1);
        close(fd);
        return TRUE;
    } else {
        close(fd);
        return FALSE;
    }

}


/* Signal handler for catching offline mode.
 */
static void
on_change_device_mode(DBusGProxy *proxy,
                      const char *value,
                      gpointer data) {
   
    FMRXEnabler *self = (FMRXEnabler*) data;

    if (strcmp(value, MCE_FLIGHT_MODE) == 0 ||
        strcmp(value, MCE_OFFLINE_MODE) == 0) {

        /* we got into offline mode, switch off FM radio */
        self->driver_loaded = FALSE;
        close(self->hci_socket);
        
        g_print("device entered offline mode\n");
        
    }

}


/* Watchdog for automatically powering down the radio and I2C when not in use
 * for a while.
 */
static gboolean
watchdog_handler(gpointer data) {

    int status;
    GError *error;
    FMRXEnabler *self = (FMRXEnabler*) data;

    /* stop watchdog if driver was unloaded */
    if (! self->driver_loaded) {
        return FALSE;
    }

    g_print("watchdog checking\n");
    if (difftime(time(NULL), self->last_request) > UNLOAD_TIMEOUT) {
    
        self->driver_loaded = FALSE;
        set_i2c_enabled(self->hci_socket, FALSE);
        if (! self->had_bluetooth) {
            hci_dev_down(self->hci_socket);
        }
        close(self->hci_socket);
        
        g_spawn_command_line_sync("/sbin/rmmod radio-bcm2048",
                                  NULL, NULL, &status, &error);
        if (status == 0) {
            g_print("FM Tuner unloaded\n");
            return FALSE;
        } else {
            g_printf("FM Tuner in use, cannot unload\n");
            return TRUE;
        }

    } else {
    
        return TRUE;
    
    }

}


/* D-Bus method: request
 */
gboolean
dbus_fmrx_enabler_request(FMRXEnabler *self,
                          int *result,
                          char **device,
                          GError **error) {

    int status;

    g_print("received client request\n");    

    /* prolong usage time */
    self->last_request = time(NULL);

    if (self->driver_loaded) {
    
        g_print("prolonged usage time\n");
        *result = RESULT_OK;
        
    } else {
        /* check if we're allowed to enable Bluetooth */
        g_print("checking whether Bluetooth may be enabled... ");
        if (is_bluetooth_allowed() == FALSE) {
            g_print("no\n");
            *result = RESULT_BLUETOOTH_DISALLOWED;
            return TRUE;
        }
        g_print("yes\n");

        /* attempt to unload module first */
        g_spawn_command_line_sync("/sbin/rmmod radio-bcm2048",
                            NULL, NULL, &status, &error);

        /* open HCI socket */
        g_print("opening HCI socket... ");
        self->hci_socket = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
        if (self->hci_socket < 0) {
            g_print("failed\n");
            *result = RESULT_BLUETOOTH_FAILURE;
            return TRUE;
        }
        g_print("success\n");
        
        /* power up Bluetooth */
        g_print("powering up Bluetooth device... ");
        if (hci_dev_up(self->hci_socket, &(self->had_bluetooth)) == FALSE) {
            g_print("failed\n");
            *result = RESULT_BLUETOOTH_FAILURE;
            close(self->hci_socket);
            return TRUE;
        }
        g_print("success\n");

        /* power up I2C bus */
        g_print("enabling I2C communication... ");
        if (set_i2c_enabled(self->hci_socket, TRUE) == FALSE) {
            g_print("failed\n");
            *result = RESULT_I2C_FAILURE;
            if (! self->had_bluetooth) {
                hci_dev_down(self->hci_socket);
            }
            close(self->hci_socket);
            return TRUE;
        }
        g_print("success\n");

        /* wait for device */
        usleep(50000);

        /* load driver */
        g_print("loading kernel module... ");
        g_spawn_command_line_sync("/sbin/insmod "
                                  RADIO_BCM2048_MODULE
                                  " radio_nr=1",
                                  NULL, NULL, &status, error);
        if (status != 0) {
            g_print("failed\n");
            *result = RESULT_DRIVER_FAILURE;
            set_i2c_enabled(self->hci_socket, FALSE);
            if (! self->had_bluetooth) {
                hci_dev_down(self->hci_socket);
            }
            g_spawn_command_line_sync("/sbin/rmmod radio-bcm2048",
                                      NULL, NULL, &status, &error);
            return TRUE;
        }
        g_print("success\n");
        
        /* wait for device */
        usleep(250000);
        
        /* check for device */
        g_print("checking for device file... ");
        if (!g_file_test("/dev/radio1", G_FILE_TEST_EXISTS)) {
            g_print("failed\n");
            *result = RESULT_DRIVER_FAILURE;
            set_i2c_enabled(self->hci_socket, FALSE);
            if (! self->had_bluetooth) {
                hci_dev_down(self->hci_socket);
            }
            g_spawn_command_line_sync("/sbin/rmmod radio-bcm2048",
                                      NULL, NULL, &status, &error);
            return TRUE;
        }
        g_print("success\n");
       
        /* enable RDS */
        g_print("enabling RDS... ");
        if (! set_rds_enabled(TRUE)) {
            g_print("failed\n");
        } else {
            g_print("success\n");
        }
       
        /* initialize watchdog */
        g_print("initializing powersave watchdog... ");
        self->driver_loaded = TRUE;
        g_timeout_add(WATCHDOG_TIMEOUT * 1000,
                      watchdog_handler,
                      (gpointer) self);
        g_print("success\n");
        
    }
    
    g_print("FM tuner activated\n");
    *device = g_strdup(FMRADIO_DEVICE);
    
    return TRUE;

}


int
main(int argc, char **argv) {

    GObject *obj;

    if (argc == 1) {
        daemonize();
    }

    g_thread_init(NULL);
    dbus_g_thread_init();

    g_type_init();

    loop = g_main_loop_new(NULL, FALSE);
    obj = g_object_new(FMRX_ENABLER_TYPE, NULL);

    g_main_loop_run(loop);

    return 0;
    
}

