// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "assertedlock.h"

#include <QMutexLocker>
#include <QThread>

AssertedLock::AssertedLock()
    :
#ifndef NDEBUG
    _lock_counter(0),
    _mutex(),
#endif
    _lock(QReadWriteLock::Recursive)
{
}

AssertedLock::~AssertedLock()
{
#ifndef NDEBUG
    QMutexLocker locker(&_mutex);
    Q_ASSERT_X(_lock_counter == 0, __func__, "lock counter must be 0");
#endif
}

void AssertedLock::Lock(LockMode mode)
{
#ifndef NDEBUG
    {
        QMutexLocker locker(&_mutex);
        if (GetLockCounter() > 0) {
            if (GetLockMode() == READ) {
                if (mode != READ) {
                    Q_ASSERT_X(mode == READ, __func__, "unable to change lockmode from READ to WRITE");
                }
            } else {
                if (mode != WRITE) {
                    mode = WRITE;
                }
            }
        } else {
            SetLockMode(mode);
        }
        IncLockCounter();
    }
#endif
    if (mode == READ) {
        _lock.lockForRead();
    } else {
        _lock.lockForWrite();
    }
}

void AssertedLock::Unlock()
{
#ifndef NDEBUG
    {
        QMutexLocker locker(&_mutex);
        Q_ASSERT_X(GetLockCounter() > 0, __func__, "too many unlocks");
        DecLockCounter();
    }
#endif
    _lock.unlock();
}

#ifndef NDEBUG
AssertedLock::LockMode AssertedLock::GetLockMode() const
{
    return _thread_data[QThread::currentThread()]._lock_mode;
}

void AssertedLock::SetLockMode(LockMode mode)
{
    _thread_data[QThread::currentThread()]._lock_mode = mode;
}

int AssertedLock::GetLockCounter() const
{
    return _thread_data[QThread::currentThread()]._lock_counter;
}

void AssertedLock::IncLockCounter()
{
    _thread_data[QThread::currentThread()]._lock_counter += 1;
    ++_lock_counter;
}

void AssertedLock::DecLockCounter()
{
    _thread_data[QThread::currentThread()]._lock_counter -= 1;
    --_lock_counter;
}

void AssertedLock::AssertLockedForMode(LockMode mode) const
{
    QMutexLocker locker(&_mutex);
    if (GetLockCounter() <= 0) {
        Q_ASSERT_X(_lock_counter > 0, __func__, "expected lock");
    }
    if (GetLockMode() == READ) {
        if (mode != READ) {
            Q_ASSERT_X(mode == READ, __func__, "locked for READ but WRITE lock expected");
        }
    }
}

void AssertedLock::AssertNotLocked() const
{
    QMutexLocker locker(&_mutex);
    Q_ASSERT_X(GetLockCounter() == 0, __func__, "unexpected lock.");
}
#endif
