/*
 *  Medard for Maemo.
 *  Copyright (C) 2011 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <QtGui>
#include <QSettings>

#include "aboutdialog.h"
#include "mainwindow.h"

MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{
    m_downloader = new MedardDownloader();

    connect(m_downloader, SIGNAL(downloadFinished(const QString &, const QDateTime &)), this,
            SLOT(downloadedFinished(const QString &, const QDateTime &)));
    connect(m_downloader, SIGNAL(downloadFailed()), this, SLOT(downloadFailed()));

    m_forecast = new ForecastWidget();
    m_forecast->setFixedSize(m_downloader->imageSize());

    m_forecastTypeLabel = new QLabel();
    m_forecastTypeLabel->setAlignment(Qt::AlignCenter);
    m_forecastTypeLabel->setFixedSize(220, 80);

    m_forecastInitialDateLabel = new QLabel();
    m_forecastInitialDateLabel->setAlignment(Qt::AlignCenter);
    m_forecastInitialDateLabel->setDisabled(true);

    m_forecastDateLabel = new QLabel();
    m_forecastDateLabel->setAlignment(Qt::AlignCenter);

    m_downloadRetryButton = new QPushButton(tr("Download again"));

    connect(m_downloadRetryButton, SIGNAL(clicked()), this, SLOT(downloadAgainClicked()));

    m_minusDayButton = new QPushButton(tr("-1 d"));
    m_plusDayButton = new QPushButton(tr("+1 d"));
    m_minusHourButton = new QPushButton(tr("-1 h"));
    m_plusHourButton = new QPushButton(tr("+1 h"));

    connect(m_minusDayButton, SIGNAL(clicked()), this, SLOT(minusDayClicked()));
    connect(m_plusDayButton, SIGNAL(clicked()), this, SLOT(plusDayClicked()));
    connect(m_minusHourButton, SIGNAL(clicked()), this, SLOT(minusHourClicked()));
    connect(m_plusHourButton, SIGNAL(clicked()), this, SLOT(plusHourClicked()));

    setupUi();
    setupMenu();

    loadSettings();
}

MainWindow::~MainWindow()
{
    delete m_downloader;
}

void MainWindow::setupUi()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
#endif
    setWindowTitle(tr("Medard"));

    QWidget *widget = new QWidget;
    setCentralWidget(widget);

    QHBoxLayout *mainLayout = new QHBoxLayout();
    mainLayout->setMargin(8);
    mainLayout->setSpacing(4);
    widget->setLayout(mainLayout);

    mainLayout->addWidget(m_forecast);
    mainLayout->addSpacing(4);

    QVBoxLayout *layout = new QVBoxLayout();
    mainLayout->addLayout(layout);

    layout->addWidget(m_forecastTypeLabel);
    layout->addWidget(m_forecastDateLabel);
    layout->addSpacing(20);
    layout->addWidget(m_forecastInitialDateLabel);
    layout->addSpacing(20);
    layout->addWidget(m_downloadRetryButton);

    QHBoxLayout *dayNavigationBox = new QHBoxLayout();
    dayNavigationBox->addWidget(m_minusDayButton);
    dayNavigationBox->addWidget(m_plusDayButton);
    layout->addLayout(dayNavigationBox);

    QHBoxLayout *hourNavigationBox = new QHBoxLayout();
    hourNavigationBox->addWidget(m_minusHourButton);
    hourNavigationBox->addWidget(m_plusHourButton);
    layout->addLayout(hourNavigationBox);

    hideNavigationButtons(false);
}

void MainWindow::setupMenu()
{
    QMenuBar *menu = new QMenuBar();
    setMenuBar(menu);

    m_domainActionGroup = new QActionGroup(this);
    m_domainActionGroup->setExclusive(true);

    QAction *domainAction;
    domainAction = new QAction(tr("Europe"), m_domainActionGroup);
    domainAction->setCheckable(true);
    domainAction = new QAction(tr("Czech Republic"), m_domainActionGroup);
    domainAction ->setCheckable(true);
    menu->addActions(m_domainActionGroup->actions());
    connect(m_domainActionGroup, SIGNAL(triggered(QAction *)), this, SLOT(forecastDomainChanged(QAction *)));

    QAction *seaLevelPreasureAction = new QAction(tr("Sea Level Pressure"), this);
    menu->addAction(seaLevelPreasureAction);
    connect(seaLevelPreasureAction, SIGNAL(triggered()), this, SLOT(seaLevelPreasureMenuClicked()));

    QAction *precipitationAction = new QAction(tr("Precipitation"), this);
    menu->addAction(precipitationAction);
    connect(precipitationAction, SIGNAL(triggered()), this, SLOT(precipitationMenuClicked()));

    QAction *windVelocityAction = new QAction(tr("Wind Velocity"), this);
    menu->addAction(windVelocityAction);
    connect(windVelocityAction, SIGNAL(triggered()), this, SLOT(windVelocityMenuClicked()));

    QAction *cloudinessAction = new QAction(tr("Cloudiness"), this);
    menu->addAction(cloudinessAction);
    connect(cloudinessAction, SIGNAL(triggered()), this, SLOT(cloudinessMenuClicked()));

    QAction *temperatureAction = new QAction(tr("Temperature"), this);
    menu->addAction(temperatureAction);
    connect(temperatureAction, SIGNAL(triggered()), this, SLOT(temperatureMenuClicked()));

    QAction *aboutAction = new QAction(tr("About"), this);
    menu->addAction(aboutAction);
    connect(aboutAction, SIGNAL(triggered()), this, SLOT(aboutMenuClicked()));
}

void MainWindow::loadSettings()
{
    QSettings settings;

    int forecastDomain = settings.value("ForecastDomain").toInt();
    int forecastType = settings.value("ForecastType").toInt();

    m_downloader->setForecastDomain((MedardDownloader::ForecastDomain) forecastDomain);
    m_domainActionGroup->actions().at(forecastDomain)->setChecked(true);

    switch ((MedardDownloader::ForecastType) forecastType) {
        case MedardDownloader::SeaLevelPressure:
            seaLevelPreasureMenuClicked();
            break;

        case MedardDownloader::Precipitation:
            precipitationMenuClicked();
            break;

        case MedardDownloader::WindVelocity:
            windVelocityMenuClicked();
            break;

        case MedardDownloader::Cloudiness:
            cloudinessMenuClicked();
            break;

        case MedardDownloader::Temperature:
            temperatureMenuClicked();
            break;
    }
}

void MainWindow::showNavigationButtons()
{
    m_downloadRetryButton->hide();

    m_minusDayButton->show();
    m_plusDayButton->show();
    m_minusHourButton->show();
    m_plusHourButton->show();
}

void MainWindow::hideNavigationButtons(bool showRetryButton)
{
    if (showRetryButton)
        m_downloadRetryButton->show();
    else
        m_downloadRetryButton->hide();

    m_minusDayButton->hide();
    m_plusDayButton->hide();
    m_minusHourButton->hide();
    m_plusHourButton->hide();
}

void MainWindow::updateNavigationButtons()
{
    if ((m_downloader->forecastDateOffset() - 24) < m_downloader->minForecastDateOffset()) {
        m_minusDayButton->setDisabled(true);
        m_plusDayButton->setDisabled(false);
    } else if ((m_downloader->forecastDateOffset() + 24) > m_downloader->maxForecastDateOffset()) {
        m_minusDayButton->setDisabled(false);
        m_plusDayButton->setDisabled(true);
    } else {
        m_minusDayButton->setDisabled(false);
        m_plusDayButton->setDisabled(false);
    }

    if ((m_downloader->forecastDateOffset() - 1) < m_downloader->minForecastDateOffset()) {
        m_minusHourButton->setDisabled(true);
        m_plusHourButton->setDisabled(false);
    } else if ((m_downloader->forecastDateOffset() + 1) > m_downloader->maxForecastDateOffset()) {
        m_minusHourButton->setDisabled(false);
        m_plusHourButton->setDisabled(true);
    } else {
        m_minusHourButton->setDisabled(false);
        m_plusHourButton->setDisabled(false);
    }
}

void MainWindow::seaLevelPreasureMenuClicked()
{
    forecastTypeChanged(tr("Sea Level Pressure"), MedardDownloader::SeaLevelPressure);
}

void MainWindow::precipitationMenuClicked()
{
    forecastTypeChanged(tr("Precipitation"), MedardDownloader::Precipitation);
}

void MainWindow::windVelocityMenuClicked()
{
    forecastTypeChanged(tr("Wind Velocity"), MedardDownloader::WindVelocity);
}

void MainWindow::cloudinessMenuClicked()
{
    forecastTypeChanged(tr("Cloudiness"), MedardDownloader::Cloudiness);
}

void MainWindow::temperatureMenuClicked()
{
    forecastTypeChanged(tr("Temperature"), MedardDownloader::Temperature);
}

void MainWindow::aboutMenuClicked()
{
    AboutDialog *dialog = new AboutDialog();
    dialog->exec();
}

void MainWindow::downloadAgainClicked()
{
    m_forecast->clearImage(false);
    m_downloader->downloadImage();

    hideNavigationButtons(false);
}

void MainWindow::plusDayClicked()
{
    forecastDateOffsetChanged(+24);
}

void MainWindow::minusDayClicked()
{
    forecastDateOffsetChanged(-24);
}

void MainWindow::plusHourClicked()
{
    forecastDateOffsetChanged(+1);
}

void MainWindow::minusHourClicked()
{
    forecastDateOffsetChanged(-1);
}

void MainWindow::forecastTypeChanged(const QString label, MedardDownloader::ForecastType type)
{
    m_forecastTypeLabel->setText(label);
    m_forecast->clearImage(false);
    m_downloader->setForecastType(type);
    m_downloader->downloadImage();

    QSettings settings;
    settings.setValue("ForecastType", type);
}

void MainWindow::forecastDateOffsetChanged(int offset)
{
    m_forecast->clearImage(false);
    m_downloader->setForecastDateOffset(m_downloader->forecastDateOffset() + offset);
    m_downloader->downloadImage();
}

void MainWindow::forecastDomainChanged(QAction *action)
{
    int forecastDomain = m_domainActionGroup->actions().indexOf(action);

    m_forecast->clearImage(false);

    if (forecastDomain == 0)
        m_downloader->setForecastDomain(MedardDownloader::Europe);
    else
        m_downloader->setForecastDomain(MedardDownloader::CzechRepublic);

    m_downloader->downloadImage();

    QSettings settings;
    settings.setValue("ForecastDomain", forecastDomain);
}

void MainWindow::downloadedFinished(const QString &filename, const QDateTime &date)
{
    m_forecast->setImage(filename);
    m_forecastInitialDateLabel->setText(tr("Forecast from:\n") +
                                          m_downloader->forecastInitialDate().toString("dd.MM.yyyy hh:mm"));

    /* upcase the first letter of name of day */
    QString temp = date.toString("dddd\ndd.MM.yyyy hh:mm");
    m_forecastDateLabel->setText(temp.replace(0, 1, temp.at(0).toUpper()));

    showNavigationButtons();
    updateNavigationButtons();
}

void MainWindow::downloadFailed()
{
    m_forecast->clearImage(true);
    m_forecastInitialDateLabel->setText("");
    m_forecastDateLabel->setText("");

    hideNavigationButtons(true);
}
