/*
  MathJinni - A simple formular calculator
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>
#include <Lum/Def/MultiView.h>

#include <Lum/Dlg/About.h>

#include <Lum/OS/Main.h>

#include <Lum/Dialog.h>
#include <Lum/Panel.h>
#include <Lum/Tab.h>

#include "Calc.h"
#include "Configuration.h"
#include "ConfigDialog.h"
#include "FormatConversion.h"
#include "Formula.h"
#include "Plot.h"
#include "UnitConversion.h"

#include "config.h"

static Lum::Def::AppInfo info;

class MainWindow : public Lum::Dialog
{
private:
  std::list<ConfigPlugin*> plugins;
  Lum::Model::ActionRef    settingsAction;
  Lum::Model::ActionRef    aboutAction;

public:
  MainWindow()
   : settingsAction(new Lum::Model::Action()),
     aboutAction(new Lum::Model::Action())
  {
    Observe(GetOpenedAction());
    Observe(GetClosedAction());

    Observe(settingsAction);
    Observe(aboutAction);
  }

  void PreInit()
  {
    Calc             *calc;
    FormatConversion *formatConversion;
    FormulaCalc      *formula;
    Plot             *plot;
    UnitConversion   *unitConversion;

    Lum::Def::MultiView multiView(Lum::Def::Desc(L"MathJinni"));

    calc=new Calc();
    calc->SetFlex(true,true);
    plugins.push_back(calc);

    multiView.AddView(Lum::Def::MultiView::View(0,
                                                Lum::Def::Desc(L"Calc"),
                                                calc));

    formatConversion=new FormatConversion();
    formatConversion->SetFlex(true,true);

    multiView.AddView(Lum::Def::MultiView::View(1,
                                                Lum::Def::Desc(L"Format"),
                                                formatConversion));

    unitConversion=new UnitConversion();
    unitConversion->SetFlex(true,true);
    plugins.push_back(unitConversion);

    multiView.AddView(Lum::Def::MultiView::View(2,
                                                Lum::Def::Desc(L"Units"),
                                                unitConversion));

    formula=new FormulaCalc();
    formula->SetFlex(true,true);
    plugins.push_back(formula);

    multiView.AddView(Lum::Def::MultiView::View(3,
                                                Lum::Def::Desc(L"Formula"),
                                                formula));

    plot=new Plot();
    plot->SetFlex(true,true);
    plugins.push_back(plot);

    multiView.AddView(Lum::Def::MultiView::View(2,
                                                Lum::Def::Desc(L"Plot"),
                                                plot));

    Lum::Def::Menu *preMenu=Lum::Def::Menu::Create();

    preMenu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->ActionSettings(settingsAction)
      ->End();

    Lum::Def::Menu *postMenu=Lum::Def::Menu::Create();

    postMenu
      ->GroupHelp()
        //->ActionHelp()
        ->ActionAbout(aboutAction)
      ->End();

    multiView.SetMenu(preMenu,postMenu);

    Lum::OS::display->GetBehaviour()->ApplyMultiViewDlg(this,multiView);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() &&
        GetOpenedAction()->IsFinished()) {
      LoadData(plugins);
      LoadConfig(plugins);
    }
    else if (model==GetClosedAction() &&
             GetClosedAction()->IsFinished()) {
      SaveConfig(plugins);
    }
    else if (model==settingsAction &&
             settingsAction->IsFinished()) {
      ConfigDialog *dialog;

      dialog=new ConfigDialog();
      dialog->SetParent(this);
      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;
    }
    else if (model==aboutAction &&
             aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainWindow>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

   return Lum::OS::MainDialog<MainWindow>::Prepare();
  }

  Main()
  {
    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"Doing some magic with numbers...");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License (GPL)");
  }
};

LUM_MAIN(Main,L"MathJinni")
