/*
 * tangle-style.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-vault.h"
#include <string.h>
#include <gobject/gvaluecollector.h>

/**
 * SECTION:tangle-vault
 * @Short_description: A data structure holding GValues
 * @Title: TangleVault
 *
 * #TangleVault is a data structure that is used to hold
 * references to other objects and data especially in the
 * context of signal handlers. It can be used to transfer
 * multiple values in a single user_data pointer.
 *
 * Here is an example, how to create a new #TangleVault
 * structure:
 *
 * |[
 * ClutterActor* actor;
 * gfloat scaling;
 * TangleVault* vault;
 *
 * ...
 * vault = tangle_vault_new(2,
 *                          CLUTTER_TYPE_ACTOR, actor,
 *                          G_TYPE_FLOAT, scaling);
 * ]|
 *
 * Stored data can be fetched from a tangle vault
 * with the #tangle_vault_get function. It is similar
 * to #g_object_get but values are not referenced
 * by name but by order and type. For example:
 *
 * |[
 * void scale_actor(TangleButton* button, gpointer user_data) {
 * 	ClutterActor* actor;
 * 	gfloat scaling;
 * 	TangleVault* vault;
 *
 * 	vault = TANGLE_VAULT(user_data);
 * 	tangle_vault_get(vault, 2,
 *	                 CLUTTER_TYPE_ACTOR, &actor,
 *	                 G_TYPE_FLOAT, &float);
 * 	...
 * }
 * ]|
 * 
 * #TangleVault stores all references and values as
 * #GValue types, thus also managing reference counting
 * when needed. Remember to free #TangleVault after its
 * usage with #tangle_vault_free.
 *
 * When used together with signal handlers, a
 * the #tangle_signal_connect_vault macro is handy.
 * It associates the vault with the handler so that
 * the #tangle_vault_free is called automatically when
 * the signal handler is disconnected.
 * Simply, construct a new #TangleVault and give it
 * as a user_data pointer to a signal handler. For
 * example:
 *
 * |[
 * tangle_signal_connect_vault(button, "clicked", G_CALLBACK(scale_actor), vault);
 * ]|
 * 
 */

static void free_vault(TangleVault* vault, guint n_values_to_unset);

TangleVault* tangle_vault_new(guint n, GType first_type, ...) {
	TangleVault* vault;
	GType type;
	va_list args;
	guint i;
	gchar* error;
	
	vault = g_slice_new(TangleVault);
	vault->n = n;
	vault->values = (GValue*)g_slice_alloc0(n * sizeof(GValue));

	type = first_type;
	va_start(args, first_type);
	for (i = 0; i < n; i++, type = va_arg(args, GType)) {
		g_value_init(&vault->values[i], type);
		error = NULL;
		G_VALUE_COLLECT(&vault->values[i], args, 0, &error);
		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);
			
			free_vault(vault, i);
			vault = NULL;
			break;
		}
	}

	return vault;
}

void tangle_vault_free(TangleVault* vault) {
	free_vault(vault, vault->n);
}

void tangle_vault_get(TangleVault* vault, guint n, GType first_type, ...) {
	GType type;
	GValue value;
	va_list args;
	guint i;
	gchar* error;

	g_return_if_fail(vault->n >= n);
	
	type = first_type;
	va_start(args, first_type);
	for (i = 0; i < n; i++, type = va_arg(args, GType)) {
		g_return_if_fail(G_VALUE_HOLDS(&vault->values[i], type));
		
		error = NULL;
		G_VALUE_LCOPY(&vault->values[i], args, 0, &error);
		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);
		}
	}
	va_end(args);
}

static void free_vault(TangleVault* vault, guint n_values_to_unset) {
	guint i;
	
	for (i = 0; i < n_values_to_unset; i++) {
		g_value_unset(&vault->values[i]);
	}
	g_slice_free1(vault->n * sizeof(GValue), vault->values);
	g_slice_free(TangleVault, vault);
}
