/*
 * tangle-linear-gradient.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-linear-gradient.h"
#include <math.h>

/**
 * SECTION:tangle-linear-gradient
 * @Short_description: 	An actor that displays linear gradient
 * @Title: TangleLinearGradient
 */

G_DEFINE_TYPE(TangleLinearGradient, tangle_linear_gradient, TANGLE_TYPE_ACTOR);

enum {
	PROP_0,
	PROP_START_COLOR,
	PROP_END_COLOR,
	PROP_ANGLE
};

struct _TangleLinearGradientPrivate {
	ClutterColor start_color;
	ClutterColor end_color;
	gfloat angle;
	CoglTextureVertex* vertices;
	guint n_vertices;
	guchar cached_opacity;
};

static void calculate_vertices(TangleLinearGradient* linear_gradient);

static const ClutterColor default_start_color = { 255, 255, 255, 255 };
static const ClutterColor default_end_color = { 0, 0, 0, 255 };

ClutterActor* tangle_linear_gradient_new(ClutterColor* start_color, ClutterColor* end_color) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_LINEAR_GRADIENT, "start-color", start_color, "end-color", end_color, NULL));
}

void tangle_linear_gradient_get_start_color(TangleLinearGradient* linear_gradient, ClutterColor* color) {
	*color = linear_gradient->priv->start_color;
}

void tangle_linear_gradient_set_start_color(TangleLinearGradient* linear_gradient, const ClutterColor* color) {
	g_return_if_fail(color != NULL);
	
	linear_gradient->priv->start_color = *color;
	g_object_notify(G_OBJECT(linear_gradient), "start-color");
	clutter_actor_queue_redraw(CLUTTER_ACTOR(linear_gradient));
}

void tangle_linear_gradient_get_end_color(TangleLinearGradient* linear_gradient, ClutterColor* color) {
	*color = linear_gradient->priv->end_color;
}

void tangle_linear_gradient_set_end_color(TangleLinearGradient* linear_gradient, const ClutterColor* color) {
	g_return_if_fail(color != NULL);
	
	linear_gradient->priv->end_color = *color;
	calculate_vertices(linear_gradient);
	g_object_notify(G_OBJECT(linear_gradient), "end-color");
	clutter_actor_queue_redraw(CLUTTER_ACTOR(linear_gradient));
}

gfloat tangle_linear_gradient_get_angle(TangleLinearGradient* linear_gradient) {

	return linear_gradient->priv->angle;
}

void tangle_linear_gradient_set_angle(TangleLinearGradient* linear_gradient, gfloat radians) {
	g_return_if_fail(radians >= 0.0 && radians <= 2 * M_PI);

	linear_gradient->priv->angle = radians;
	calculate_vertices(linear_gradient);
	g_object_notify(G_OBJECT(linear_gradient), "angle");
	clutter_actor_queue_redraw(CLUTTER_ACTOR(linear_gradient));
}

static void tangle_linear_gradient_paint(ClutterActor* actor) {
	gdouble scale_x;
	gdouble scale_y;
	ClutterActorBox box;

	cogl_push_matrix();

	g_object_get(actor, "transition-scale-x", &scale_x, "transition-scale-y", &scale_y, NULL);
	cogl_scale(scale_x, scale_y, 0);

	if (TANGLE_LINEAR_GRADIENT(actor)->priv->cached_opacity != clutter_actor_get_paint_opacity(actor)) {
		calculate_vertices(TANGLE_LINEAR_GRADIENT(actor));
	}

	tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &box);
	cogl_clip_push(box.x1, box.y1, box.x2 - box.x1, box.y2 - box.y1);
	cogl_polygon(TANGLE_LINEAR_GRADIENT(actor)->priv->vertices, 4, TRUE);
	cogl_clip_pop();
	
	cogl_pop_matrix();
}

static void tangle_linear_gradient_allocate(ClutterActor* actor, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	CLUTTER_ACTOR_CLASS(tangle_linear_gradient_parent_class)->allocate(actor, box, flags);
	calculate_vertices(TANGLE_LINEAR_GRADIENT(actor));
}

static void tangle_linear_gradient_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleLinearGradient* linear_gradient;
	
	linear_gradient = TANGLE_LINEAR_GRADIENT(object);

	switch (prop_id) {
		case PROP_START_COLOR:
			tangle_linear_gradient_set_start_color(linear_gradient, clutter_value_get_color(value));
			break;
		case PROP_END_COLOR:
			tangle_linear_gradient_set_end_color(linear_gradient, clutter_value_get_color(value));
			break;
		case PROP_ANGLE:
			tangle_linear_gradient_set_angle(linear_gradient, g_value_get_float(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_linear_gradient_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleLinearGradient* linear_gradient;

	linear_gradient = TANGLE_LINEAR_GRADIENT(object);

        switch (prop_id) {
		case PROP_START_COLOR:
			clutter_value_set_color(value, &linear_gradient->priv->start_color);
			break;
		case PROP_END_COLOR:
			clutter_value_set_color(value, &linear_gradient->priv->end_color);
			break;
		case PROP_ANGLE:
			g_value_set_float(value, linear_gradient->priv->angle);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_linear_gradient_finalize(GObject* object) {
	TangleLinearGradient* linear_gradient;
	
	linear_gradient = TANGLE_LINEAR_GRADIENT(object);
	
	if (linear_gradient->priv->vertices) {
		g_slice_free1(linear_gradient->priv->n_vertices * sizeof(CoglTextureVertex), linear_gradient->priv->vertices);
	}

	G_OBJECT_CLASS(tangle_linear_gradient_parent_class)->finalize(object);
}

static void tangle_linear_gradient_class_init(TangleLinearGradientClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);
	ClutterActorClass* actor_class = CLUTTER_ACTOR_CLASS(klass);

	gobject_class->finalize = tangle_linear_gradient_finalize;
	gobject_class->set_property = tangle_linear_gradient_set_property;
	gobject_class->get_property = tangle_linear_gradient_get_property;

	actor_class->paint = tangle_linear_gradient_paint;
	actor_class->allocate = tangle_linear_gradient_allocate;

	/**
	 * TangleLinearGradient:start-color:
	 */
	g_object_class_install_property(gobject_class, PROP_START_COLOR,
	                                clutter_param_spec_color("start-color",
	                                                	 "Start color",
	                                                	 "The starting color of the gradient",
	                                                	 &default_start_color,
	                                                	 G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleLinearGradient:end-color:
	 */
	g_object_class_install_property(gobject_class, PROP_END_COLOR,
	                                clutter_param_spec_color("end-color",
	                                                	 "End color",
	                                                	 "The ending color of the gradient",
	                                                	 &default_end_color,
	                                                	 G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	/**
	 * TangleLinearGradient:angle:
	 */
	g_object_class_install_property(gobject_class, PROP_ANGLE,
	                                g_param_spec_float("angle",
	                                                   "Angle",
	                                                   "The angle of the gradient in radians",
	                                                   0.0, 2 * M_PI, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleLinearGradientPrivate));
}

static void tangle_linear_gradient_init(TangleLinearGradient* linear_gradient) {
	linear_gradient->priv = G_TYPE_INSTANCE_GET_PRIVATE(linear_gradient, TANGLE_TYPE_LINEAR_GRADIENT, TangleLinearGradientPrivate);
}

static void calculate_vertices(TangleLinearGradient* linear_gradient) {
	ClutterActorBox box;
	gfloat angle;
	CoglTextureVertex* vertices;
	gfloat fsin, fcos, dxx, dxy, dyx, dyy;
	ClutterColor start_color;
	ClutterColor end_color;
	ClutterColor* top_left;
	ClutterColor* top_right;
	ClutterColor* bottom_left;
	ClutterColor* bottom_right;
	
	if (linear_gradient->priv->vertices) {
		g_slice_free1(linear_gradient->priv->n_vertices * sizeof(CoglTextureVertex), linear_gradient->priv->vertices);
	}
	linear_gradient->priv->n_vertices = 4;
	linear_gradient->priv->vertices = g_slice_alloc(linear_gradient->priv->n_vertices * sizeof(CoglTextureVertex));

	tangle_actor_get_aligned_allocation(TANGLE_ACTOR(linear_gradient), &box);
	angle = linear_gradient->priv->angle;
	vertices = linear_gradient->priv->vertices;

	linear_gradient->priv->cached_opacity = clutter_actor_get_paint_opacity(CLUTTER_ACTOR(linear_gradient));
	start_color = linear_gradient->priv->start_color;
	start_color.alpha = start_color.alpha * linear_gradient->priv->cached_opacity / 255;
	end_color = linear_gradient->priv->end_color;
	end_color.alpha = end_color.alpha * linear_gradient->priv->cached_opacity / 255;

	if (angle != 0.0) {
		if (angle <= M_PI / 2) {
			top_left = top_right = &start_color;
			bottom_right = bottom_left = &end_color;
		} else if (angle <= M_PI) {
			angle -= M_PI / 2;
			top_left = bottom_left = &end_color;
			top_right = bottom_right = &start_color;
		} else if (angle <= M_PI * 3 / 2) {
			angle -= M_PI;
			top_left = top_right = &end_color;
			bottom_right = bottom_left = &start_color;
		} else {
			angle -= M_PI * 3 / 2;
			top_left = bottom_left = &start_color;
			top_right = bottom_right = &end_color;
		}
		fsin = sin(angle);
		fcos = cos(angle);
		dxx = fsin * fcos * (box.x2 - box.x1);
		dxy = fsin * fsin * (box.x2 - box.x1);
		dyx = fsin * fsin * (box.y2 - box.y1);
		dyy = fsin * fcos * (box.y2 - box.y1);
	} else {
		dxx = dxy = dyx = dyy = 0.0;
		top_left = top_right = &start_color;
		bottom_right = bottom_left = &end_color;
	}
	
	vertices[0].x = box.x1 + dxy;
	vertices[0].y = box.y1 - dyy;
	vertices[0].z = 0.0;
	vertices[0].tx = vertices[0].ty = 0.5;
	cogl_color_set_from_4ub(&vertices[0].color,
	                        top_left->red,
				top_left->green,
				top_left->blue,
				top_left->alpha);

	vertices[1].x = box.x2 + dxx;
	vertices[1].y = box.y1 + dyx;
	vertices[1].z = 0.0;
	vertices[1].tx = vertices[1].ty = 0.5;
	cogl_color_set_from_4ub(&vertices[1].color,
	                        top_right->red,
				top_right->green,
				top_right->blue,
				top_right->alpha);

	vertices[2].x = box.x2 - dxy;
	vertices[2].y = box.y2 + dyy;
	vertices[2].z = 0.0;
	vertices[2].tx = vertices[2].ty = 0.5;
	cogl_color_set_from_4ub(&vertices[2].color,
	                        bottom_right->red,
				bottom_right->green,
				bottom_right->blue,
				bottom_right->alpha);

	vertices[3].x = box.x1 - dxx;
	vertices[3].y = box.y2 - dyx;
	vertices[3].z = 0.0;
	vertices[3].tx = vertices[3].ty = 0.5;
	cogl_color_set_from_4ub(&vertices[3].color,
	                        bottom_left->red,
				bottom_left->green,
				bottom_left->blue,
				bottom_left->alpha);
}

