/*
 * tangle-floating-actor.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-floating-actor.h"

/**
 * SECTION:tangle-floating-actor
 * @Short_description: A wrapper actor that floats on top of other actors
 * @Title: TangleFloatingActor
 */

G_DEFINE_TYPE(TangleFloatingActor, tangle_floating_actor, TANGLE_TYPE_WRAPPER_ACTOR);

enum {
	PROP_0,
	PROP_FLOATING_MODE,
	PROP_FLOATING_X,
	PROP_FLOATING_Y,
	PROP_FLOATING_SCALE_X,
	PROP_FLOATING_SCALE_Y,
	PROP_FLOATING_SCALE_CENTER_X,
	PROP_FLOATING_SCALE_CENTER_Y,
	PROP_PLACEHOLDER_OPACITY,
	PROP_PICK_FLOATING,
	PROP_PICK_PLACEHOLDER
};

struct _TangleFloatingActorPrivate {
	TangleFloatingMode floating_mode;
	gfloat floating_x;
	gfloat floating_y;
	gfloat floating_scale_x;
	gfloat floating_scale_y;
	gfloat floating_scale_center_x;
	gfloat floating_scale_center_y;
	guchar placeholder_opacity;
	CoglMatrix matrix;
	gulong paint_handler_id;
	gulong pick_handler_id;
	guint pick_floating : 1;
	guint pick_placeholder : 1;
};

static void setup_painting(TangleFloatingActor* floating_actor, gboolean force_unsetup);

ClutterActor* tangle_floating_actor_new(ClutterActor* wrapped) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_FLOATING_ACTOR, "wrapped", wrapped, NULL));
}

TangleFloatingMode tangle_floating_actor_get_floating_mode(TangleFloatingActor* floating_actor) {

	return floating_actor->priv->floating_mode;
}

void tangle_floating_actor_set_floating_mode(TangleFloatingActor* floating_actor, TangleFloatingMode floating_mode) {
	if (floating_actor->priv->floating_mode != floating_mode) {
		floating_actor->priv->floating_mode = floating_mode;
		clutter_actor_queue_relayout(CLUTTER_ACTOR(floating_actor));
		setup_painting(floating_actor, FALSE);
		g_object_notify(G_OBJECT(floating_actor), "floating-mode");
	}
}

gfloat tangle_floating_actor_get_floating_x(TangleFloatingActor* floating_actor) {

	return floating_actor->priv->floating_x;
}

void tangle_floating_actor_set_floating_x(TangleFloatingActor* floating_actor, gfloat floating_x) {
	if (floating_actor->priv->floating_x != floating_x) {
		floating_actor->priv->floating_x = floating_x;
		clutter_actor_queue_redraw(clutter_actor_get_stage(CLUTTER_ACTOR(floating_actor)));
		g_object_notify(G_OBJECT(floating_actor), "floating-x");
	}
}

gfloat tangle_floating_actor_get_floating_y(TangleFloatingActor* floating_actor) {

	return floating_actor->priv->floating_y;
}

void tangle_floating_actor_set_floating_y(TangleFloatingActor* floating_actor, gfloat floating_y) {
	if (floating_actor->priv->floating_y != floating_y) {
		floating_actor->priv->floating_y = floating_y;
		clutter_actor_queue_redraw(clutter_actor_get_stage(CLUTTER_ACTOR(floating_actor)));
		g_object_notify(G_OBJECT(floating_actor), "floating-y");
	}
}

void tangle_floating_actor_set_floating_position(TangleFloatingActor* floating_actor, gfloat floating_x, gfloat floating_y) {
	tangle_floating_actor_set_floating_x(floating_actor, floating_x);
	tangle_floating_actor_set_floating_y(floating_actor, floating_y);
}

gfloat tangle_floating_actor_get_floating_scale_x(TangleFloatingActor* floating_actor) {

	return floating_actor->priv->floating_scale_x;
}

void tangle_floating_actor_set_floating_scale_x(TangleFloatingActor* floating_actor, gfloat floating_scale_x) {
	if (floating_actor->priv->floating_scale_x != floating_scale_x) {
		floating_actor->priv->floating_scale_x = floating_scale_x;
		clutter_actor_queue_redraw(clutter_actor_get_stage(CLUTTER_ACTOR(floating_actor)));
		g_object_notify(G_OBJECT(floating_actor), "floating-scale-x");
	}
}

gfloat tangle_floating_actor_get_floating_scale_y(TangleFloatingActor* floating_actor) {

	return floating_actor->priv->floating_scale_y;
}

void tangle_floating_actor_set_floating_scale_y(TangleFloatingActor* floating_actor, gfloat floating_scale_y) {
	if (floating_actor->priv->floating_scale_y != floating_scale_y) {
		floating_actor->priv->floating_scale_y = floating_scale_y;
		clutter_actor_queue_redraw(clutter_actor_get_stage(CLUTTER_ACTOR(floating_actor)));
		g_object_notify(G_OBJECT(floating_actor), "floating-scale-y");
	}
}

void tangle_floating_actor_set_floating_scale(TangleFloatingActor* floating_actor, gfloat floating_scale_x, gfloat floating_scale_y) {
	tangle_floating_actor_set_floating_scale_x(floating_actor, floating_scale_x);
	tangle_floating_actor_set_floating_scale_y(floating_actor, floating_scale_y);
}

gfloat tangle_floating_actor_get_floating_scale_center_x(TangleFloatingActor* floating_actor) {

	return floating_actor->priv->floating_scale_center_x;
}

void tangle_floating_actor_set_floating_scale_center_x(TangleFloatingActor* floating_actor, gfloat floating_scale_center_x) {
	if (floating_actor->priv->floating_scale_center_x != floating_scale_center_x) {
		floating_actor->priv->floating_scale_center_x = floating_scale_center_x;
		clutter_actor_queue_redraw(clutter_actor_get_stage(CLUTTER_ACTOR(floating_actor)));
		g_object_notify(G_OBJECT(floating_actor), "floating-scale-center-x");
	}
}

gfloat tangle_floating_actor_get_floating_scale_center_y(TangleFloatingActor* floating_actor) {

	return floating_actor->priv->floating_scale_center_y;
}

void tangle_floating_actor_set_floating_scale_center_y(TangleFloatingActor* floating_actor, gfloat floating_scale_center_y) {
	if (floating_actor->priv->floating_scale_center_y != floating_scale_center_y) {
		floating_actor->priv->floating_scale_center_y = floating_scale_center_y;
		clutter_actor_queue_redraw(clutter_actor_get_stage(CLUTTER_ACTOR(floating_actor)));
		g_object_notify(G_OBJECT(floating_actor), "floating-scale-center-y");
	}
}

void tangle_floating_actor_set_floating_scale_center(TangleFloatingActor* floating_actor, gfloat floating_scale_center_x, gfloat floating_scale_center_y) {
	tangle_floating_actor_set_floating_scale_center_x(floating_actor, floating_scale_center_x);
	tangle_floating_actor_set_floating_scale_center_y(floating_actor, floating_scale_center_y);
}

guchar tangle_floating_actor_get_placeholder_opacity(TangleFloatingActor* floating_actor) {

	return floating_actor->priv->floating_scale_x;
}

void tangle_floating_actor_set_placeholder_opacity(TangleFloatingActor* floating_actor, guchar placeholder_opacity) {
	if (floating_actor->priv->placeholder_opacity != placeholder_opacity) {
		floating_actor->priv->placeholder_opacity = placeholder_opacity;
		clutter_actor_queue_redraw(clutter_actor_get_stage(CLUTTER_ACTOR(floating_actor)));
		g_object_notify(G_OBJECT(floating_actor), "placeholder_opacity");
	}
}

static void tangle_floating_actor_paint(ClutterActor* actor) {
	TangleFloatingActor* floating_actor;
	guint8 opacity;
	
	floating_actor = TANGLE_FLOATING_ACTOR(actor);
	
	if (floating_actor->priv->floating_mode != TANGLE_FLOATING_MODE_NONE) {
		cogl_get_modelview_matrix(&floating_actor->priv->matrix);
	}
	if (floating_actor->priv->floating_mode != TANGLE_FLOATING_MODE_COLLAPSE) {
		if (floating_actor->priv->floating_mode == TANGLE_FLOATING_MODE_PLACEHOLDER) {
			/* esta notify-signaali */
			opacity = clutter_actor_get_opacity(actor);
			clutter_actor_set_opacity(actor, floating_actor->priv->placeholder_opacity);
		}
		
		CLUTTER_ACTOR_CLASS(tangle_floating_actor_parent_class)->paint(actor);

		if (floating_actor->priv->floating_mode == TANGLE_FLOATING_MODE_PLACEHOLDER) {
			clutter_actor_set_opacity(actor, opacity);
			/* salli signaali */
		}
	}
}

static void tangle_floating_actor_pick(ClutterActor* actor, const ClutterColor* color) {
	TangleFloatingActor* floating_actor;
	
	floating_actor = TANGLE_FLOATING_ACTOR(actor);
	
	if (floating_actor->priv->floating_mode == TANGLE_FLOATING_MODE_NONE ||
	    (floating_actor->priv->floating_mode == TANGLE_FLOATING_MODE_PLACEHOLDER &&
	     floating_actor->priv->pick_placeholder)) {
		CLUTTER_ACTOR_CLASS(tangle_floating_actor_parent_class)->pick(actor, color);
	}
	if (floating_actor->priv->floating_mode != TANGLE_FLOATING_MODE_NONE &&
	    floating_actor->priv->pick_floating) {
		cogl_get_modelview_matrix(&floating_actor->priv->matrix);
	}
}

static void tangle_floating_actor_map(ClutterActor* actor) {
	CLUTTER_ACTOR_CLASS(tangle_floating_actor_parent_class)->map(actor);
	
 	setup_painting(TANGLE_FLOATING_ACTOR(actor), FALSE);
}

static void tangle_floating_actor_unmap(ClutterActor* actor) {
	CLUTTER_ACTOR_CLASS(tangle_floating_actor_parent_class)->unmap(actor);

	setup_painting(TANGLE_FLOATING_ACTOR(actor), FALSE);
}

static void tangle_floating_actor_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	if (TANGLE_FLOATING_ACTOR(actor)->priv->floating_mode != TANGLE_FLOATING_MODE_COLLAPSE) {
		TANGLE_ACTOR_CLASS(tangle_floating_actor_parent_class)->get_preferred_width(actor, for_height, interacting, min_width_p, natural_width_p, max_width_p);
	} else {
		if (min_width_p) {
			*min_width_p = 0;
		}
		if (natural_width_p) {
			*natural_width_p = 0;
		}
		if (max_width_p) {
			*max_width_p = 0;
		}
	}
}

static void tangle_floating_actor_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	if (TANGLE_FLOATING_ACTOR(actor)->priv->floating_mode != TANGLE_FLOATING_MODE_COLLAPSE) {
		TANGLE_ACTOR_CLASS(tangle_floating_actor_parent_class)->get_preferred_height(actor, for_width, interacting, min_height_p, natural_height_p, max_height_p);
	} else {
		if (min_height_p) {
			*min_height_p = 0;
		}
		if (natural_height_p) {
			*natural_height_p = 0;
		}
		if (max_height_p) {
			*max_height_p = 0;
		}
	}
}

static void tangle_floating_actor_allocate_wrapped(TangleWrapperActor* wrapper_actor, ClutterActor* wrapped, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	if (TANGLE_FLOATING_ACTOR(wrapper_actor)->priv->floating_mode != TANGLE_FLOATING_MODE_COLLAPSE) {
		clutter_actor_allocate(wrapped, box, flags);
	}
}

static void tangle_floating_actor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleFloatingActor* floating_actor;
	
	floating_actor = TANGLE_FLOATING_ACTOR(object);

	switch (prop_id) {
		case PROP_FLOATING_X:
			tangle_floating_actor_set_floating_x(floating_actor, g_value_get_float(value));
			break;
		case PROP_FLOATING_Y:
			tangle_floating_actor_set_floating_y(floating_actor, g_value_get_float(value));
			break;
		case PROP_FLOATING_SCALE_X:
			tangle_floating_actor_set_floating_scale_x(floating_actor, g_value_get_float(value));
			break;
		case PROP_FLOATING_SCALE_Y:
			tangle_floating_actor_set_floating_scale_y(floating_actor, g_value_get_float(value));
			break;
		case PROP_FLOATING_SCALE_CENTER_X:
			tangle_floating_actor_set_floating_scale_center_x(floating_actor, g_value_get_float(value));
			break;
		case PROP_FLOATING_SCALE_CENTER_Y:
			tangle_floating_actor_set_floating_scale_center_y(floating_actor, g_value_get_float(value));
			break;
		case PROP_PLACEHOLDER_OPACITY:
			tangle_floating_actor_set_placeholder_opacity(floating_actor, g_value_get_uchar(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_floating_actor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleFloatingActor* floating_actor;

	floating_actor = TANGLE_FLOATING_ACTOR(object);

        switch (prop_id) {
		case PROP_FLOATING_X:
			g_value_set_float(value, floating_actor->priv->floating_x);
			break;
		case PROP_FLOATING_Y:
			g_value_set_float(value, floating_actor->priv->floating_y);
			break;
		case PROP_FLOATING_SCALE_X:
			g_value_set_float(value, floating_actor->priv->floating_scale_x);
			break;
		case PROP_FLOATING_SCALE_Y:
			g_value_set_float(value, floating_actor->priv->floating_scale_y);
			break;
		case PROP_FLOATING_SCALE_CENTER_X:
			g_value_set_float(value, floating_actor->priv->floating_scale_x);
			break;
		case PROP_FLOATING_SCALE_CENTER_Y:
			g_value_set_float(value, floating_actor->priv->floating_scale_y);
			break;
		case PROP_PLACEHOLDER_OPACITY:
			g_value_set_uchar(value, floating_actor->priv->placeholder_opacity);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_floating_actor_dispose(GObject* object) {
	TangleFloatingActor* floating_actor;
	
	floating_actor = TANGLE_FLOATING_ACTOR(object);

	setup_painting(floating_actor, TRUE);
	
	G_OBJECT_CLASS(tangle_floating_actor_parent_class)->dispose(object);
}

static void tangle_floating_actor_class_init(TangleFloatingActorClass* floating_actor_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(floating_actor_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(floating_actor_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(floating_actor_class);
	TangleWrapperActorClass* wrapper_actor_class = TANGLE_WRAPPER_ACTOR_CLASS(floating_actor_class);

	gobject_class->dispose = tangle_floating_actor_dispose;
	gobject_class->set_property = tangle_floating_actor_set_property;
	gobject_class->get_property = tangle_floating_actor_get_property;

	clutter_actor_class->map = tangle_floating_actor_map;
	clutter_actor_class->unmap = tangle_floating_actor_unmap;
	clutter_actor_class->paint = tangle_floating_actor_paint;
	clutter_actor_class->pick = tangle_floating_actor_pick;
	
	actor_class->get_preferred_width = tangle_floating_actor_get_preferred_width;
	actor_class->get_preferred_height = tangle_floating_actor_get_preferred_height;

	wrapper_actor_class->allocate_wrapped = tangle_floating_actor_allocate_wrapped;

	/**
	 * TangleFloatingActor:floating-mode:
	 */
	g_object_class_install_property(gobject_class, PROP_FLOATING_MODE,
	                                g_param_spec_ulong("floating-mode",
	                                                   "Floating mode",
	                                                   "The floating mode used",
	                                                   0, G_MAXULONG, 0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleFloatingActor:floating-x
	 */
	g_object_class_install_property(gobject_class, PROP_FLOATING_X,
	                                g_param_spec_float("floating-x",
	                                                    "Floating X",
	                                                    "The floating transformation along x axis",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleFloatingActor:floating-y
	 */
	g_object_class_install_property(gobject_class, PROP_FLOATING_Y,
	                                g_param_spec_float("floating-y",
	                                                    "Floating Y",
	                                                    "The floating transformation along y axis",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleFloatingActor:floating-sacale-x
	 */
	g_object_class_install_property(gobject_class, PROP_FLOATING_SCALE_X,
	                                g_param_spec_float("floating-scale-x",
	                                                    "Floating scale X",
	                                                    "The floating scaling along x axis",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleFloatingActor:floating-scale-y
	 */
	g_object_class_install_property(gobject_class, PROP_FLOATING_SCALE_Y,
	                                g_param_spec_float("floating-scale-y",
	                                                    "Floating scale Y",
	                                                    "The floating scaling along y axis",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleFloatingActor:floating-sacale-center-x
	 */
	g_object_class_install_property(gobject_class, PROP_FLOATING_SCALE_CENTER_X,
	                                g_param_spec_float("floating-scale-center-x",
	                                                    "Floating scale center X",
	                                                    "The center point of floating scale in X axis",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleFloatingActor:floating-scale-center-y
	 */
	g_object_class_install_property(gobject_class, PROP_FLOATING_SCALE_CENTER_Y,
	                                g_param_spec_float("floating-scale-center-y",
	                                                    "Floating scale center Y",
	                                                    "The center point of floating scale in Y axis",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 1.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleFloatingActor:placeholder-opacity
	 */
	g_object_class_install_property(gobject_class, PROP_PLACEHOLDER_OPACITY,
	                                g_param_spec_uchar("placeholder-opacity",
	                                                    "Placeholder opacity",
	                                                    "The opacity of the floating placeholder",
	                                                   0, 255, 50,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleFloatingActorPrivate));
}

static void tangle_floating_actor_init(TangleFloatingActor* floating_actor) {
	floating_actor->priv = G_TYPE_INSTANCE_GET_PRIVATE(floating_actor, TANGLE_TYPE_FLOATING_ACTOR, TangleFloatingActorPrivate);

	floating_actor->priv->floating_scale_x = 1.0;
	floating_actor->priv->floating_scale_y = 1.0;
	floating_actor->priv->placeholder_opacity = 50;
}

static void on_stage_paint(ClutterActor* actor, gpointer user_data) {
	TangleFloatingActor* floating_actor;
	gboolean has_clip;
	gfloat x, y, width, height;
	
	floating_actor = TANGLE_FLOATING_ACTOR(user_data);
	
	cogl_push_matrix();
	cogl_set_modelview_matrix(&floating_actor->priv->matrix);
	
	has_clip = clutter_actor_has_clip(actor);
	if (has_clip) {
		clutter_actor_get_clip(actor, &x, &y, &width, &height);
		cogl_clip_push(x, y, width, height);
	}
	
	cogl_translate(floating_actor->priv->floating_x, floating_actor->priv->floating_y, 0.0);

	if (floating_actor->priv->floating_scale_x != 1.0 ||
	    floating_actor->priv->floating_scale_y != 1.0) {
		cogl_translate(floating_actor->priv->floating_scale_center_x, floating_actor->priv->floating_scale_center_y, 0.0);
		cogl_scale(floating_actor->priv->floating_scale_x, floating_actor->priv->floating_scale_y, 1.0);
		cogl_translate(-floating_actor->priv->floating_scale_center_x, -floating_actor->priv->floating_scale_center_y, 0.0);
	}

	clutter_actor_paint(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(floating_actor)));
	
	if (has_clip) {
		cogl_clip_pop();
	}
	
	cogl_pop_matrix();
}

static void on_stage_pick(ClutterActor* actor, const ClutterColor* color, gpointer user_data) {
	if (TANGLE_FLOATING_ACTOR(user_data)->priv->pick_floating) {
		on_stage_paint(actor, user_data);
	}	
}

static void setup_painting(TangleFloatingActor* floating_actor, gboolean force_unsetup) {
	ClutterActor* stage;

    	stage = clutter_actor_get_stage(CLUTTER_ACTOR(floating_actor));

	if (!force_unsetup && floating_actor->priv->floating_mode != TANGLE_FLOATING_MODE_NONE &&
	    CLUTTER_ACTOR_IS_MAPPED(CLUTTER_ACTOR(floating_actor))) {
	    	stage = clutter_actor_get_stage(CLUTTER_ACTOR(floating_actor));
		if (!floating_actor->priv->paint_handler_id) {
			floating_actor->priv->paint_handler_id = g_signal_connect_after(stage, "paint", G_CALLBACK(on_stage_paint), floating_actor);
		}
		if (!floating_actor->priv->pick_handler_id) {
			floating_actor->priv->pick_handler_id = g_signal_connect_after(stage, "pick", G_CALLBACK(on_stage_pick), floating_actor);
		}
	} else {
		if (floating_actor->priv->paint_handler_id) {
			g_signal_handler_disconnect(stage, floating_actor->priv->paint_handler_id);
			floating_actor->priv->paint_handler_id = 0;
		}
		if (floating_actor->priv->pick_handler_id) {
			g_signal_handler_disconnect(stage, floating_actor->priv->pick_handler_id);
			floating_actor->priv->pick_handler_id = 0;
		}

	}
}
