/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDITEM_H
#define MICROFEEDITEM_H

#include <microfeed-common/microfeedstore.h>

#include <sys/types.h>

/**
 * @addtogroup common libmicrofeed-common - Common modules used both in the publisher and in the subscriber side
 * @{
 * @addtogroup MicrofeedItem
 *
 * An item in a feed.
 *
 * MicrofeedItem is not thread-safe, so keep one item in one thread only. However, usually the lifetime
 * of an item is short, so there is even no need to access an item from multiple threads. Thus, there is
 * no reference counting available. If you have to access the same item content in multiple threads,
 * use #microfeed_item_duplicate.
 *
 * @{
 */

/**
 * Opaque data type representing Microfeed item.
 */
typedef struct _MicrofeedItem MicrofeedItem;

/**
 * Opaque data type representing Microfeed item iterator that iterates over the properties of the item.
 */
typedef struct _MicrofeedItemIterator MicrofeedItemIterator;

typedef enum {
	MICROFEED_ITEM_STATUS_NONE = 0,
	MICROFEED_ITEM_STATUS_ACTIVE = 1 << 0,
	MICROFEED_ITEM_STATUS_UNREAD = 1 << 1,
	MICROFEED_ITEM_STATUS_MARKED = 1 << 2,
	MICROFEED_ITEM_STATUS_SET = 1 << 7
} MicrofeedItemStatus;

#define MICROFEED_ITEM_STATUS_NEW (MICROFEED_ITEM_STATUS_ACTIVE | MICROFEED_ITEM_STATUS_UNREAD)

typedef enum {
	MICROFEED_ITEM_PERMISSION_NONE = 0,
	MICROFEED_ITEM_PERMISSION_MODIFY = 1 << 0,
	MICROFEED_ITEM_PERMISSION_REMOVE = 1 << 1,
	MICROFEED_ITEM_PERMISSION_REPLY = 1 << 2
} MicrofeedItemPermission;

MicrofeedItem* microfeed_item_new_temporary(void);
MicrofeedItem* microfeed_item_new(const char* uid, time_t timestamp);
MicrofeedItem* microfeed_item_new_with_status(const char* uid, time_t timestamp, MicrofeedItemStatus status);
MicrofeedItem* microfeed_item_new_from_marshalled(const void* data, size_t size);
void microfeed_item_free(MicrofeedItem* item);
MicrofeedItem* microfeed_item_duplicate(MicrofeedItem* item);

const char* microfeed_item_get_uid(MicrofeedItem* item);
void microfeed_item_marshalled_get_uid(const void* data, size_t data_size, const char** uid_return, size_t* uid_size_return);
time_t microfeed_item_get_timestamp(MicrofeedItem* item);
void microfeed_item_marshalled_get_timestamp_uid(const void* data, size_t data_size, const void** timestamp_uid_return, size_t* timestamp_uid_size_return);
void microfeed_item_set_timestamp(MicrofeedItem* item, time_t timestamp);
MicrofeedItemStatus microfeed_item_get_status(MicrofeedItem* item);
MicrofeedItemStatus microfeed_item_marshalled_get_status(const void* data, size_t data_size);
void microfeed_item_set_status(MicrofeedItem* item, MicrofeedItemStatus status);
void microfeed_item_marshalled_set_status(const void* data, size_t data_size, MicrofeedItemStatus status);
const char* microfeed_item_get_property(MicrofeedItem* item, const char* key);
MicrofeedItemIterator* microfeed_item_iterate_properties(MicrofeedItem* item, const char* start_key);
void microfeed_item_marshal(MicrofeedItem* item, const void** data_pointer, size_t* data_size_pointer);
void microfeed_item_marshal_without_properties(MicrofeedItem* item, const void** data_pointer, size_t* data_size_pointer);
void microfeed_item_set_property(MicrofeedItem* item, const char* key, const char* value);
void microfeed_item_set_property_with_length(MicrofeedItem* item, const char* key, const char* value, size_t value_length);
void microfeed_item_set_property_full(MicrofeedItem* item, const char* key, size_t key_length, const char* value, size_t value_length);
void microfeed_item_set_property_from_integer(MicrofeedItem* item, const char* key, long int integer);
char* microfeed_item_get_properties_as_string(MicrofeedItem* item);
int microfeed_item_set_properties_from_string(MicrofeedItem* item, const char* string);
MicrofeedItemPermission microfeed_item_permission_from_string(const char* string);
char* microfeed_item_permission_to_string(MicrofeedItemPermission item_permission);

void microfeed_item_iterator_free(MicrofeedItemIterator* iterator);
int microfeed_item_iterator_get(MicrofeedItemIterator* iterator, const char** key, const char** value);
void microfeed_item_iterator_next(MicrofeedItemIterator* iterator);

/**
 * @}
 * @}
 */

#endif
